/**
 *
 * @file     callbacks.c
 * @author   Aleix Conchillo Flaque <aleix@member.fsf.org>
 * @date     Tue Aug 05, 2003 21:50
 * @brief    XMMS GNOME applet callbacks
 *
 * $Id: callbacks.c,v 1.12 2005/01/05 00:22:54 aleix Exp $
 *
 * @if copyright
 *
 * Copyright (C) 2003, 2004, 2005 Aleix Conchillo Flaque
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * @endif
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "callbacks.h"

#include "interface.h"
#include "playerctrl.h"

#include <gnome.h>

/* Static functions definition (declaration at end of file) */
static gint timer(gpointer data);


static int const TIMEOUT_VALUE = 250;

void
timer_on(XMMSData *xmms)
{
    if (!xmms->timeout_hnd)
    {
        xmms->timeout_hnd = gtk_timeout_add(TIMEOUT_VALUE, timer, xmms);
    }
}

void
timer_off(XMMSData *xmms)
{
    if (xmms->timeout_hnd)
    {
        gtk_timeout_remove(xmms->timeout_hnd);
        xmms->timeout_hnd = 0;
    }
}

void
cb_realize(GtkWidget *w, XMMSData *xmms)
{
    gui_setup(xmms);
    timer_on(xmms);
}

void
cb_destroy(GtkWidget *w, XMMSData *xmms)
{
    gui_destroy(xmms);
}

void
cb_style_set(GtkWidget *w, GtkStyle *style, XMMSData *xmms)
{
    gui_theme_load(xmms, style);
}

void
cb_change_size(GtkWidget *w, gint size, XMMSData *xmms)
{
    xmms->size = size;
    gui_setup(xmms);
}

void
cb_change_orient(GtkWidget *w, PanelAppletOrient orient, XMMSData *xmms)
{
    xmms->orient = orient;
    gui_setup(xmms);
}

void
cb_prev_clicked(GtkWidget *w, XMMSData *xmms)
{
    player_prev(xmms);
}

void
cb_play_pause_clicked(GtkWidget *w, XMMSData *xmms)
{
    player_play_pause(xmms);
}

void
cb_stop_clicked(GtkWidget *w, XMMSData *xmms)
{
    player_stop(xmms);
}

void
cb_next_clicked(GtkWidget *w, XMMSData *xmms)
{
    player_next(xmms);
}

void
cb_eject_clicked(GtkWidget *w, XMMSData *xmms)
{
    player_eject(xmms);
}

gboolean
cb_seek(GtkWidget *w, GdkEventButton *event, gpointer data)
{
    gdouble t;
    XMMSData *xmms;
    GtkWidget *applet = data;
    GtkRequisition requisition;

    if ((event->button == 3) || (event->button == 2))
    {
	gtk_propagate_event(applet, (GdkEvent *) event);

	return TRUE;
    }

    /* Recover XMMSData from applet */
    xmms = g_object_get_data(G_OBJECT(applet), "xmms");

    gtk_widget_size_request(xmms->main_box, &requisition);

    if ((xmms->orient == PANEL_APPLET_ORIENT_DOWN)
        || (xmms->orient == PANEL_APPLET_ORIENT_UP))
    {
        t = event->x / (gdouble) (requisition.width - 1);
    }
    else
    {
        t = event->y / (gdouble) (requisition.height - 1);
    }
    player_seek(xmms, t);

    return FALSE;
}

gboolean
cb_scrollwheel(GtkWidget *w, GdkEventScroll *event, XMMSData *xmms)
{
    switch (event->direction)
    {
    case GDK_SCROLL_UP:
        player_raise_volume(xmms);
        return TRUE;
        break;
    case GDK_SCROLL_DOWN:
        player_lower_volume(xmms);
        return TRUE;
        break;
    default:
        return FALSE;
    }
}

gboolean
cb_button_press(GtkWidget *w, GdkEventButton *event, GtkWidget *applet)
{
    if ((event->button == 3) || (event->button == 2))
    {
	gtk_propagate_event(applet, (GdkEvent *) event);

	return TRUE;
    }

    return FALSE;
}

void
cb_repeat_menu(BonoboUIComponent *uic, XMMSData *xmms, const gchar *verb)
{
    /* empty (Bonobo UI component listener is used instead) */
}

void
cb_shuffle_menu(BonoboUIComponent *uic, XMMSData *xmms, const gchar *verb)
{
    /* empty (Bonobo UI component listener is used instead) */
}

void
cb_toggle_repeat(BonoboUIComponent *uic, const gchar *path,
                 Bonobo_UIComponent_EventType type, const gchar *state,
                 XMMSData *xmms)
{
    player_repeat(xmms, strcmp(state, "0"));
}

void
cb_toggle_shuffle(BonoboUIComponent *uic, const gchar *path,
                  Bonobo_UIComponent_EventType type, const gchar *state,
                  XMMSData *xmms)
{
    player_shuffle(xmms, strcmp(state, "0"));
}


void
cb_main_win_menu(BonoboUIComponent *uic, XMMSData *xmms, const gchar *verb)
{
    player_main_win(xmms, !player_is_main_win(xmms));
}

void
cb_pl_win_menu(BonoboUIComponent *uic, XMMSData *xmms, const gchar *verb)
{
    player_pl_win(xmms, !player_is_pl_win(xmms));
}

void
cb_eq_win_menu(BonoboUIComponent *uic, XMMSData *xmms, const gchar *verb)
{
    player_eq_win(xmms, !player_is_eq_win(xmms));
}

void
cb_show_hide_menu(BonoboUIComponent *uic, XMMSData *xmms, const gchar *verb)
{
    player_show_hide(xmms);
}

void
cb_player_pref_menu(BonoboUIComponent *uic, XMMSData *xmms, const gchar *verb)
{
    player_preferences(xmms);
}

void
cb_quit_menu(BonoboUIComponent *uic, XMMSData *xmms, const gchar *verb)
{
    player_quit(xmms);
}

void
cb_about_menu(BonoboUIComponent *uic, XMMSData *xmms, const gchar *verb)
{
    static GtkWidget *about = NULL;
    static const gchar *authors[] = {
        "Aleix Conchillo Flaque <aleix@member.fsf.org>",
        NULL
    };

    if (about != NULL)
    {
        gdk_window_show(about->window);
        gdk_window_raise(about->window);
        return;
    }
    about = gnome_about_new(
        PACKAGE_NAME, PACKAGE_VERSION, "Copyright (C) 2003, 2004, 2005 Aleix Conchillo Flaque\n",
        _("gxmms lets you control XMMS or Beep Media Player.\n"),
        authors, NULL, NULL,
#ifdef HAVE_XMMS
        gdk_pixbuf_new_from_file(GXMMS_DATADIR"/gxmms.png", 0));
#else
        gdk_pixbuf_new_from_file(GXMMS_BMP_DATADIR"/gxmms_bmp.png", 0));
#endif

    g_signal_connect(G_OBJECT(about), "destroy",
                     G_CALLBACK(gtk_widget_destroyed), &about);
    gtk_widget_show (about);
}

/* Static functions declaration */

static gint
timer(gpointer data)
{
    static gboolean was_playing = FALSE;
    XMMSData *xmms = data;

    // if xmms has been closed or if it was playing
    if (was_playing
        && (!player_is_playing(xmms) || !player_is_running(xmms)))
    {
        gui_update(xmms);
        was_playing = FALSE;
    }
    if (player_is_playing(xmms))
    {
        gui_update(xmms);
        was_playing = TRUE;
    }

    return TRUE;
}
