/* Implementation of the curve view.
 *
 * Copyright (c) 2003 Ole Laursen.
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA.
 */

#include <algorithm>		// for max/min[_element]()
#include <deque>

#include <libgnomecanvasmm/line.h>
#include <libgnomecanvasmm/point.h>
#include <gconfmm/client.h>

#include "curve-view.hpp"
#include "applet.hpp"
#include "monitor.hpp"


//
// class Curve - represents a line curve
//

class Curve
{
public:
  Curve(Monitor *monitor, const Glib::ustring &mon_dir);

  void update(unsigned int max_samples); // gather info from monitor
  void draw(Gnome::Canvas::Canvas &canvas, // redraw curve on canvas
	    Applet *applet, int width, int height, int samples,
	    double time_offset); 

  Monitor *monitor;
  
private:
  std::auto_ptr<Gnome::Canvas::Line> line;

  // we need to store the past values
  typedef std::deque<double> value_sequence;
  typedef value_sequence::iterator value_iterator;
  value_sequence values;

  Glib::ustring monitor_dir;
};

Curve::Curve(Monitor *m, const Glib::ustring &mon_dir)
  : monitor(m), monitor_dir(mon_dir)
{}

void Curve::update(unsigned int max_samples)
{
  double val = monitor->value();
  values.push_front(val);
  
  // get rid of extra samples (there may be more than one if user changes
  // configuration)
  while (values.size() > max_samples)
    values.pop_back();
}

void Curve::draw(Gnome::Canvas::Canvas &canvas,
		 Applet *applet, int width, int height, int samples,
		 double time_offset)
{
  Glib::RefPtr<Gnome::Conf::Client> &client = applet->get_gconf_client();
    
  value_iterator vi = values.begin(), vend = values.end();

  // only one point is pointless
  if (std::distance(vi, vend) < 2) 
    return;

  // make sure line is initialised
  if (line.get() == 0) {
    line.reset(new Gnome::Canvas::Line(*canvas.root()));
    line->property_smooth() = true;
    line->property_join_style() = Gdk::JOIN_ROUND;
  }

  // get drawing attributes
  unsigned int color;
  double line_width;

  Gnome::Conf::Value v;
  
  v = client->get(monitor_dir + "/color");
  if (v.get_type() == Gnome::Conf::VALUE_INT)
    color = v.get_int();
  else {
    color = applet->get_fg_color();
    client->set(monitor_dir + "/color", int(color));
  }
  

  v = client->get(monitor_dir + "/line_width");
  if (v.get_type() == Gnome::Conf::VALUE_FLOAT)
    line_width = v.get_float();
  else {
    line_width = 1.5;		// FIXME: use schema?
    client->set(monitor_dir + "/line_width", line_width);
  }
  

  line->property_fill_color_rgba() = color;
  line->property_width_units() = line_width;
  

  double max = std::max(monitor->max(), 0.000001); // avoid division by zero

  Gnome::Canvas::Points points;
  points.reserve(values.size());

  // one less than no. of samples because two points are end-points
  double pixels_per_sample = double(width) / (samples - 1 - 1);
  
  // start from right
  double x = width + pixels_per_sample - pixels_per_sample * time_offset;
	
  do {
    double y = line_width/2 + (1 - (*vi / max)) * (height - line_width/2);

    points.push_back(Gnome::Art::Point(x, y));
    x -= pixels_per_sample;
  } while (++vi != vend);

  line->property_points() = points;
}


//
// class CurveView
//

CurveView::CurveView()
  : draws_since_update(0)
{
}

CurveView::~CurveView()
{
  for (curve_iterator i = curves.begin(), end = curves.end(); i != end; ++i)
    delete *i;
}

void CurveView::do_update()
{
  CanvasView::do_update();
  
  // first update the configured attributes
  Glib::RefPtr<Gnome::Conf::Client> &client = applet->get_gconf_client();
  Glib::ustring dir = applet->get_gconf_dir();

  if (client->get(dir + "/viewer/samples").get_type() == Gnome::Conf::VALUE_INT)
    samples = client->get_int(dir + "/viewer/samples");
  else {
    samples = 30;
    client->set(dir + "/viewer/samples", int(samples));
  }
  
  // then loop through each curve
  for (curve_iterator i = curves.begin(), end = curves.end(); i != end; ++i)
    (*i)->update(samples);
  
  draws_since_update = 0;
}

void CurveView::do_attach(Monitor *monitor, const Glib::ustring &mon_dir)
{
  curves.push_back(new Curve(monitor, mon_dir));
}

void CurveView::do_detach(Monitor *monitor)
{
  for (curve_iterator i = curves.begin(), end = curves.end(); i != end; ++i)
    if ((*i)->monitor == monitor) {
      delete *i;
      curves.erase(i);
      return;
    }

  g_assert_not_reached();
}

void CurveView::do_draw_loop()
{
  double time_offset =
    double(draws_since_update) * CanvasView::draw_interval
    / applet->get_update_interval();
  
  for (curve_iterator i = curves.begin(), end = curves.end(); i != end; ++i)
    (*i)->draw(*canvas, applet, width(), height(), samples, time_offset);
  
  ++draws_since_update;
}
