-- Pretty printing of an ART expression with a sub-expression highlit.
-- First convert a value (of class HatRep) into an S-expression,
-- then convert the S-expression to a Doc for pretty printing.
module SExp
  ( SExp(..)
  , SFixity(..)	-- used only in module Pattern
  , Label
  , QName(..), showQN
  , fileNode2SExp
  , sExp2Doc
  , arity, child, label, children, rebuild, parent, prune
  , funId, funLabel
  , prettyExpression
  , prettyEquation
  ) where

import LowLevel hiding    (nil)
import qualified LowLevel (nil)
import HighlightStyle     (Highlight(..),Colour(..))
import PrettyLibHighlight as Pretty
                          (Doc,text,(<>),delimiter,fdelimiter,nil,group,parens
                          ,groupNest,pretty,highlight)
import Char               (isAlpha)
import List               (unzip3,isPrefixOf)
import IO                 (hPutStrLn,stderr)
import FFIExtensions      (unsafePerformIO,showHex)

bold = highlight [Bold, Foreground Blue]

errorT  :: String -> a
errorT s = unsafePerformIO (do hPutStrLn stderr s; return (error ""))
debugT  :: (Show a) => String -> a -> a
debugT s x = unsafePerformIO (do hPutStrLn stderr (s++show x); return x)

data QName
  = Plain String
  | Qualified String String

instance Eq QName where
  Plain v       == Plain v'         =   v==v'
  Qualified q v == Plain v'         =   v==v'
  Plain v       == Qualified q' v'  =   v==v'
  Qualified q v == Qualified q' v'  =   v==v' && q==q'

showQN :: Bool -> QName -> String
showQN _ (Plain n)           = showString n ""
showQN False (Qualified q n) = showString n ""
showQN True  (Qualified q n) = (showString q . showChar '.' . showString n) ""

type Label = (String,FileNode)	-- The label on an SExp contains two components,
				-- a unique string, and the file pointer.
				-- The former enables unique highlighting, and
				-- the latter enables highlighting of sharing.

data SExp a
  = SApp a [SExp a]		-- n-ary application of at least 2 expressions
  | SId a QName SFixity		-- an identifier (variable or constructor)
  | SLiteral a String		-- any other kind of basic value
  | SString a String Ellipsis	-- character strings have special sugar
  | SWithin a [SExp a]		-- chains of if/case/guard inside an expression
  | SLambda a			-- a lambda expression
  | SIf a (SExp a)
  | SCase a (SExp a)
  | SGuard a (SExp a)
  | SFieldExpr a (SExp a) [String] [SExp a]	-- constructor value, or update
  | SCut a			-- cut off subexpression (to limit depth)
  | SUnevaluated a		-- underscore
  | SInterrupted a		--  ^C  (expr entered but never completed)
  | SBottom a			-- _|_  (expr entered but never completed)
  | SCycle a String (SExp a)	-- cyclic expression shown as `id where id = ..'
  | SEquation a (SExp a) (SExp a)
			-- an equation only makes sense as the root of an SExp
--deriving Show	-- only for testing/debugging

type Ellipsis = Bool		-- is a character string truncated?

data SFixity = 
  SInfix Int | SInfixL Int | SInfixR Int | SAssoc Int String | SInfixDefault
  -- need own type for some hardcoded operators that are known to be
  -- semantically associative
--deriving (Show)

-- translate fixity from the file representation to the structured fixity type
transFixity :: Int -> SFixity
transFixity f = case f `divMod` 4 of
                  (p,0) -> SInfix p
                  (p,1) -> SInfixR p
                  (p,2) -> SInfixL p
                  (p,3) -> SInfixDefault

-- arity of an S-expression
arity :: SExp a -> Int
arity (SApp _ exps)     = length exps
arity (SWithin _ exps)  = length exps
arity (SIf _ _)         = 1
arity (SCase _ _)       = 1
arity (SGuard _ _)      = 1
arity (SEquation _ _ _) = 2
arity (SFieldExpr _ _ _ upds) = 1 + length upds
arity _                 = 0

-- get child of an S-expression
-- precondition: i < arity exp
child :: Int -> SExp a -> SExp a
child i (SApp _ exps)    = skipCaseIfGuard (exps!!i)
child i (SWithin _ exps) = skipCaseIfGuard (exps!!i)
--child 0 (SIf _ exp)      = exp
--child 0 (SCase _ exp)    = exp
--child 0 (SGuard _ exp)   = exp
child 0 (SEquation _ l r) = l
child 1 (SEquation _ l r) = r
child 0 (SFieldExpr _ e _ upds) = e
child i (SFieldExpr _ e _ upds) = upds!!(i-1)
child i exp               = errorT ("SExp.child: "++show i++" is too large.")

skipCaseIfGuard (SIf _ exp) = exp
skipCaseIfGuard (SCase _ exp) = exp
skipCaseIfGuard (SGuard _ exp) = exp
skipCaseIfGuard exp = exp

label :: SExp a -> a
label (SApp l _)      = l
label (SId l _ _)     = l
label (SLiteral l _)  = l
label (SString l _ _) = l
label (SLambda l)     = l
label (SWithin l _)   = l
label (SIf l _)       = l
label (SCase l _)     = l
label (SGuard l _)    = l
label (SCut l)        = l
label (SUnevaluated l)= l
label (SInterrupted l)= l
label (SBottom l)     = l
label (SCycle l _ _)  = l
label (SEquation l _ _) = l
label (SFieldExpr l _ _ _) = l

children :: SExp a -> [SExp a]
children (SApp _ es)      = es
children (SWithin l es)   = es
children (SIf _ e)        = [e]
children (SCase _ e)      = [e]
children (SGuard _ e)     = [e]
children (SCycle _ _ e)   = [e]
children (SEquation _ e r)= [e,r]
children (SFieldExpr _ e _ upds) = e: upds
children _                = []

rebuild :: SExp a -> [SExp a] -> SExp a
rebuild (SApp l _)       es = SApp l es
rebuild (SWithin l _)    es = SWithin l es
rebuild (SIf l _)       [e] = SIf l e
rebuild (SCase l _)     [e] = SCase l e
rebuild (SGuard l _)    [e] = SGuard l e
rebuild (SCycle l v _)  [e] = SCycle l v e
rebuild (SEquation l _ _) [e,r] = SEquation l e r
rebuild (SFieldExpr l _ labs _) (e:upds) = SFieldExpr l e labs upds
rebuild sexp             _  = sexp

relabel :: a -> SExp a -> SExp a
relabel l (SApp _ es)     = SApp l es
relabel l (SId _ v f)     = SId l v f
relabel l (SLiteral _ s)  = SLiteral l s
relabel l (SString _ s d) = SString l s d
relabel l (SLambda _)     = SLambda l
relabel l (SWithin _ e)   = SWithin l e
relabel l (SIf _ e)       = SIf l e
relabel l (SCase _ e)     = SCase l e
relabel l (SGuard _ e)    = SGuard l e
relabel l (SCut _)        = SCut l
relabel l (SUnevaluated _)= SUnevaluated l
relabel l (SInterrupted _)= SInterrupted l
relabel l (SBottom _)     = SBottom l
relabel l (SCycle _ a b)  = SCycle l a b
relabel l (SEquation _ e r) = SEquation l e r
relabel l (SFieldExpr _ e labs upds) = SFieldExpr l e labs upds

parent :: SExp Label -> FileNode
parent (SWithin _ (x:_)) = parent x
parent (SEquation _ x _) = parent x
parent x                 = getParentNode (snd (label x))

prune :: Int -> SExp a -> SExp a
prune 0 s = SCut (label s)
prune n (SApp a es) = SApp a (head es: map (prune (n-1)) (tail es))
prune n (SWithin a es) = SWithin a (head es: map (prune (n-1)) (tail es))
prune n (SEquation a lhs rhs) = SEquation a (prune n lhs) (prune n rhs)
prune n s = s

funId :: SExp a -> QName
funId (SApp _ es) = funId (head es)
funId (SWithin _ es) = funId (head es)
funId (SEquation _ e _) = funId e
funId (SFieldExpr _ e _ _) = funId e
funId (SString _ _ _) = Qualified "Prelude" ":"
funId (SLambda _) = Plain "(\\..)"
funId (SId _ s _) = s
funId _ = Plain ""

funLabel :: SExp Label -> FileNode
funLabel (SApp _ es) = funLabel (head es)
funLabel (SWithin _ es) = funLabel (head es)
funLabel (SEquation _ e _) = funLabel e
funLabel (SFieldExpr _ e _ _) = funLabel e
funLabel (SString (_,n) _ _) = n
funLabel (SLambda (_,n)) = n
funLabel (SId (_,n) _ _) = n
funLabel _ = LowLevel.nil

fst3 (x,_,_) = x
snd3 (_,x,_) = x
thd3 (_,_,x) = x

-- conversion function
-- If `uneval' boolean is True, then unevaluated arguments appear in full
-- in the result, otherwise they are represented by SUnevaluated.
fileNode2SExp :: Int -> Bool -> Bool -> Bool -> Label -> SExp Label
fileNode2SExp cutoff uneval strings toplevelLHS label =
  case go cutoff uneval strings toplevelLHS [] label of (e,_,_) -> e  
  where
  simple e = (e,[],[])
  go :: Int			-- cutoff depth
       -> Bool			-- show unevaluated args in full?
       -> Bool			-- sugar character strings?
       -> Bool			-- top-level LHS? (implies uneval to one level)
       -> [(FileNode,String)]	-- enclosing nodes w/ variable name for `where'
       -> Label 		-- root node of expression
       -> ( SExp Label		-- expression 
          , [FileNode]		-- nodes that start cycle
          , [String] )		-- variable names occurring (except for cycles)
  go 0     uneval strings top nodesAbove label = simple (SCut label)
  go depth uneval strings top nodesAbove label@(lab,node) =
    if      node == LowLevel.nil then simple $ SUnevaluated label
    else if node == unevaluated then simple $ SUnevaluated label
    else if node == entered then simple $ SBottom label
    else if node == interrupted then simple $ SInterrupted label
    else if node == lambda then simple $ SLambda label
    else
    let r = peekResult node in
    if  r == unevaluated && not (uneval||top) then simple $ SUnevaluated label
    else if r == entered && not top then simple $ SBottom label
    else if r == interrupted && top then simple $ SInterrupted label
    else if r == lambda && not uneval then simple $ SLambda label
    else case simpleNodeType node of
      NodeModule -> "got a Module" `errorAt` node
      NodeSrcPos -> "got an SrcPos" `errorAt` node
      NodeAtom -> -- "got an Atom" `errorAt` node
        let i = getAtom node
            m = getAtomMod node
        in simple (SId label (Qualified m i) (transFixity (getAtomFixity node)))
      NodeApplication -> 
        let partCycles :: [FileNode]
            partCycles = (funCycles ++ concat argsCycles)
            partVars :: [String]
            partVars = funVars ++ concat argsVars
            isCycle = node `elem` partCycles
            var :: String
            var = head . filter (not . (`elem` partVars)) .
                  map (("cyc"++) . show) $ [1..] 
            newNodesAbove :: [(FileNode,String)]
            newNodesAbove = (node,var) : nodesAbove
            subExps :: [FileNode]
            subExps = getSubExprs node
            -- (fun,funCycles,funVars) = ... not accepted by nhc98
            fun = fst3 z
            funCycles = snd3 z
            funVars = thd3 z
            z = let f = head subExps in
                if f==LowLevel.nil then simple (SCut ('f':lab,f))
                else go depth uneval strings False newNodesAbove ('f':lab,f)
            args = fst3 zs
            argsCycles = snd3 zs
            argsVars = thd3 zs
            -- (args,argsCycles,argsVars) = ... not accepted by nhc98
            zs = unzip3 $ map (go (depth-1) uneval strings False newNodesAbove)
                              (zipWith lbl ['1'..] (tail subExps))
            lbl c n = (c:lab, n)
            -- To do strings right, need to peek one level inside a cons.
            z1 = go 1 uneval strings False newNodesAbove
                 ('1':lab, subExps!!1)	-- only used in string cutoff case
            z2 = go 3 uneval strings False newNodesAbove
                 ('2':lab, subExps!!2)	-- only used in string cutoff case

            sexp = case fun of
              -- convert the representation of constructors with fields
              SId n@(_,m) c _ | isConstrFields m ->
                SFieldExpr label (SId n c SInfixDefault)
                                 (getFieldLabels m) args
              -- convert char-list into string
              SId _ (Qualified _ ":") _ | strings && length args == 2 ->
                case args!!0 of
                  SLiteral _ c | not (null c) && head c == '\'' ->
                    case args!!1 of
                      SId _ (Qualified _ "[]") _
                                    -> SString label (init (tail c)) False
                      SString _ s d -> SString label (init (tail c)++s) d
                      _             -> SApp label (fun:args)
                  SCut _ ->	-- peek beyond the cut
                    case fst3 z1 of
                      SLiteral _ c | not (null c) && head c == '\'' ->
                        case fst3 z2 of
                          SId _ (Qualified _ "[]") _
                             -> SString label (init (tail c)) False
                          _  -> SString label (init (tail c)) True
                      _ -> SApp label (fun:args)
                  _ -> SApp label (fun:args)
              -- different bracketing of a char-list
              SApp _ args1@[SId _ (Qualified _ ":") _,SLiteral _ c]
                | strings && length args == 1
                && not (null c) && head c == '\'' ->
                    case args!!0 of
                      SId _ (Qualified _ "[]") _
                                    -> SString label (init (tail c)) False
                      SString _ s d -> SString label (init (tail c)++s) d
                      _             -> SApp label (args1++args)
              SApp _ args1@[SId _ (Qualified _ ":") _,SCut _]
                | strings && length args == 1 ->
                    fst3 (go (depth+1) uneval strings False nodesAbove label)
              -- combine applications
              SApp n args1 -> SApp label (args1++args)
              -- anything else is just a simple application
              _ -> SApp label (fun:args)
        in case lookup node nodesAbove of
             Just var -> (SId label (Plain var) SInfixDefault,[node],[]) 
                         -- `lower' end of cycle
             Nothing -> ( if isCycle then SCycle (lab,node) var sexp else sexp
                        , partCycles, partVars)
      NodeBasicValue ->
          let i = getValue node in simple $ SLiteral label i
      NodeIdentifier ->
          if isLambda node then simple (SLambda label) else
          let i = getValue node
              m = getValueMod node
          in
          ( SId label (Qualified m i) (case i of
                           "."  | m == "Prelude" -> SAssoc 9 i
                           "++" | m == "Prelude" -> SAssoc 5 i
                           "&&" | m == "Prelude" -> SAssoc 3 i
                           "||" | m == "Prelude" -> SAssoc 2 i
                           "*"  | m == "Prelude" -> SAssoc 7 i
                           "+"  | m == "Prelude" -> SAssoc 6 i
                           ">>" | m == "Prelude" -> SAssoc 1 i
                           ">>=" | m == "Prelude" -> SAssoc 1 i
                           _ -> transFixity (getFixity node))
          , []
          , if isConstructor node then [] else [i] )
      NodeCAF ->
          let i = getValue node
              m = getValueMod node
          in simple (SId label (Qualified m i) (transFixity (getFixity node)))
      NodeConditional ->
          let skind = case nodeType node of
                          ExpGuard -> SGuard
                          ExpCase  -> SCase
                          ExpIf    -> SIf
              within :: Label -> (SExp Label->SExp Label) -> SExp Label
                        -> SExp Label -> SExp Label
              within label kind parent exp =
                case parent of SWithin _ ps -> SWithin label (ps++[kind exp])
                               _            -> SWithin label [parent,kind exp]
          in simple $
             within label (skind ('c':lab,node))
                          (fileNode2SExp depth uneval strings True
                                         ('w':lab, getParentNode node))
                          (fileNode2SExp depth uneval strings False
                                         ('v':lab, head (getSubExprs node)))
      NodeSugar -> -- simple $ SLiteral label "{sugar}"
          case nodeType node of
            ExpDoStmt -> simple $ SLiteral label "{do stmt}"
            ExpFieldUpdate ->
                let (exp:updValues) = getSubExprs node
                    updLabels = getFieldLabels node
                in simple $
                   SFieldExpr label
                              (fileNode2SExp (depth-1) uneval strings False
                                             ('u':lab, exp))
                              updLabels
                              (map (fileNode2SExp (depth-1) uneval
                                                  strings False)
                                   (zipWith (\i v-> (i:lab, v))
                                            ['1'..] updValues))
      NodeSpecial ->
          case nodeType node of
            ExpProjection ->
                (\(exp,x,y) -> (relabel ('p':lab,node) exp, x, y) ) $
                go depth uneval strings False nodesAbove
                   ('p':lab, getResult (head (getSubExprs node)) top)
            ExpHidden -> simple $ SLiteral label "{?}"
            ExpForward ->
                go depth uneval strings False nodesAbove
                   (lab, head (getSubExprs node))

  errorAt :: String -> FileNode -> a
  errorAt str node = errorT ("SExp: "++str++" at 0x"++showHex (int node) "")


-- useful document combinators:
-- non-breaking space
(<->) :: Doc -> Doc -> Doc
d1 <-> d2 = d1 <> delimiter " " <> d2 

-- breakable space
(<+>) :: Doc -> Doc -> Doc
d1 <+> d2 = d1 <> fdelimiter " " <> d2

-- breakable non-space
(<|>) :: Doc -> Doc -> Doc
d1 <|> d2 = d1 <> fdelimiter "" <> d2

-- breakable before a comma
(<*>) :: Doc -> Doc -> Doc
d1 <*> d2 = d1 <|> text "," <> d2

-- breakable before or after a cons
(<:>) :: Doc -> Doc -> Doc
d1 <:> d2 = d1 <|> text ":" <|> d2

indentation :: Int
indentation = 2


isOpSym :: String -> Bool
isOpSym ""  = True	-- representation of the unit value ()
isOpSym sym = let c = head sym in not (isAlpha c || c `elem` "[_{") 

funDoc :: Bool -> QName -> Doc
funDoc q qn = (if isOpSym var then parens else id) $ text (showQN q qn)
  where var = case qn of Plain v -> v; Qualified _ v -> v

opDoc :: Bool -> QName -> Doc
opDoc q qn =  text (if isAlpha (head var) then ('`' :showQN q qn++ "`")
                    else showQN q qn)
  where var = case qn of Plain v -> v; Qualified _ v -> v


data ArgPos = ALeft | ARight
isRight ARight = True
isRight ALeft = False
isLeft = not . isRight

-- surround by parentheses if necessary
-- first fixity of surrounding expression, then if left or right argument,
-- then fixity of expression itself
optParens :: SFixity -> ArgPos -> SFixity -> Doc -> Doc
optParens surFixity aPos ownFixity =
  case (priority surFixity) `compare` (priority ownFixity) of
    LT -> if priority surFixity == (-1) then groupNest indentation else id
    GT -> groupNest indentation . parens
    EQ -> if (isInfixR surFixity && isInfixR ownFixity && isRight aPos)
            || (isInfixL surFixity && isInfixL ownFixity && isLeft aPos)
            || sameAssoc surFixity ownFixity 
            then id
            else groupNest indentation . parens

sameAssoc :: SFixity -> SFixity -> Bool
sameAssoc (SAssoc _ var1) (SAssoc _ var2) = True
sameAssoc _ _ = False

foldr0 :: (a -> a -> a) -> a -> [a] -> a
foldr0 f c [] = c
foldr0 f c xs = foldr1 f xs

listDoc :: Eq a => Bool -> (a->Doc->Doc) -> SFixity -> ArgPos -> SExp a -> Doc
listDoc qual high surFixity aPos e =
  group $ text "[" <> commas e
  where
  commas = sExpFold (sExp2Doc True qual high) (text ",") (text "]")

  sExpFold head cons nil (SApp v [SId c (Qualified _ ":") _, hd
                                 , SId n (Qualified _ "[]") _]) =
      high v (head hd <|> high c (high n nil))
  sExpFold head cons nil (SApp v [SId c (Qualified _ ":") _, hd
                                 , SId n (Plain cyc) _])
      | "cyc" `isPrefixOf` cyc =
      high v (head hd <|> high c cons <|> high n (text (cyc++"..."))
      <|> high c (high n nil))
  sExpFold head cons nil (SApp v [SId c (Qualified _ ":") _
                                 , hd@(SCut t), SCut n]) =
      high v (head hd <> high c (high t (text "...")) <|> high n nil)
  sExpFold head cons nil (SApp v [SId c (Qualified _ ":") _, hd, tl]) =
      high v (head hd <|> high c cons <|> sExpFold head cons nil tl)
  sExpFold head cons nil (SId n (Qualified _ "[]") _) = high n nil
  sExpFold head cons nil e@(SUnevaluated _) = sExp2Doc True qual high e <|> nil
  sExpFold head cons nil e@(SLiteral _ "{?}") = sExp2Doc True qual high e<|>nil
  sExpFold head cons nil e              = sExp2Doc True qual high e

priority (SInfix p) = p
priority (SInfixL p) = p
priority (SInfixR p) = p
priority (SAssoc p _) = p
priority SInfixDefault = 9

isInfixL (SInfixL _) = True
isInfixL SInfixDefault = True
isInfixL _ = False

isInfixR (SInfixR _) = True
isInfixR _ = False

isNotInfixDefault SInfixDefault = False
isNotInfixDefault _ = False

considerAsOperator :: QName -> SFixity -> Bool
considerAsOperator qname fixity = isOpSym var || isNotInfixDefault fixity 
  where var = case qname of Plain v -> v; Qualified _ v -> v

-- A central function.  Convert an SExpression to a Document using
-- the pretty-printing combinators.  In sugar mode, lists are shown
-- with [,,,] sugar, rather than in full with cons applications.
sExp2Doc :: Eq a => Bool -> Bool -> (a->Doc->Doc) -> SExp a -> Doc
sExp2Doc sugar qual high = goDoc (SInfix (-1)) ARight 
  where
  -- fixity of surrounding expression and which sort of argument
--goDoc :: SFixity -> ArgPos -> SExp a -> Doc
  goDoc surFixity aPos (SApp va ((SId vt (Qualified _ (',':xs)) _):args)) =
    if length xs + 2 == length args 
    then high va $
         group (text "(" <> foldr1 comma (map (sExp2Doc sugar qual high) args))
                         <> text ")"
           -- print tuple properly
    else high va $
         optParens surFixity aPos ownFixity
         . (high vt (text ("(,"++xs++")")) <+>)
         . foldr1 (<+>) . map (goDoc ownFixity ARight) $ args
           -- partial application of tuple constructor
    where
    ownFixity = SInfix 10
    comma l r = l <|> high vt (text ",") <> r

  goDoc surFixity aPos (e@(SApp va [SId vf (Qualified _ ":") ownFixity,e1,e2]))
    | sugar && not (ambiguous e2)
                = high va $ listDoc qual high surFixity aPos e
    | otherwise = high va $ optParens surFixity aPos ownFixity
                                (goDoc ownFixity ALeft e1
                                <> high vf (text ":")
                                <> goDoc ownFixity ARight e2)
    where -- check whether the final tail of the list is _ or {?} or {^C}.
          ambiguous (SApp _ [SId _ (Qualified _ ":") _, e1, e2]) = ambiguous e2
          ambiguous (SCycle _ _ e) = ambiguous e
          ambiguous (SId _ _ _) = False
          ambiguous (SCut _)    = False
          ambiguous (SLiteral _ "{?}") = True
          ambiguous (SUnevaluated _)   = True
          ambiguous (SInterrupted _)   = True
          ambiguous (SBottom _)        = True
          ambiguous _                  = True	-- shouldn't happen!
  goDoc surFixity aPos (SApp va [SId vf var ownFixity,e1,e2]) 
    | considerAsOperator var ownFixity = 
      high va $
      optParens surFixity aPos ownFixity
        (goDoc ownFixity ALeft e1
        <+> high vf (opDoc qual var)
        <+> goDoc ownFixity ARight e2)
  goDoc surFixity aPos (SApp va [SId vf var ownFixity,e]) 
    | considerAsOperator var ownFixity =
    -- show infix operator with single argument as section
    groupNest indentation . high va . parens $
      goDoc ownFixity ALeft e <-> high vf (opDoc qual var)
  goDoc surFixity aPos (SApp va (fun:args)) =
    high va
    . optParens surFixity aPos ownFixity
    . (goDoc ownFixity ALeft fun <+>)
    .  foldr1 (<+>) . map (goDoc ownFixity ARight) $ args
    where
    ownFixity = SInfix 10
  goDoc _ _ (SLambda v) = high v $ text "(\\..)"
  goDoc _ _ (SId v var fixity) = high v $ funDoc qual var
  goDoc _ _ (SString v s d) = high v $ text "\"" <> text s <>
                              (if d
                               then highlight [Foreground Blue] (text "...")
                               else nil) <>
                              text "\""
  goDoc _ _ (SLiteral v lit) = high v $ text lit
  goDoc _ _ (SWithin v es) = 
    high v $ foldr1 (\a b-> a <+> group (bold (text "|") <+> b))
                    (map (sExp2Doc sugar qual high) es)
  goDoc _ _ (SIf v exp) = 
    (groupNest indentation . high v) (bold (text "if")
                                     <+> sExp2Doc sugar qual high exp)
  goDoc _ _ (SCase v exp) = 
    (groupNest indentation . high v) (bold (text "case")
                                     <+> sExp2Doc sugar qual high exp)
  goDoc _ _ (SGuard v exp) = 
    (groupNest indentation . high v) (sExp2Doc sugar qual high exp)
  goDoc _ _ (SCut v) = high v $ highlight [ReverseVideo] (text " ")
  goDoc _ _ (SUnevaluated v) = high v $ text "_"
  goDoc _ _ (SBottom v) = high v $ text "_|_"
  goDoc _ _ (SInterrupted v) =
      highlight [Foreground Blue] $ high v $ text "{^C}"
  goDoc _ _ (SCycle v var exp) = 
    groupNest indentation . high v . parens $
      text var <+> group (bold (text "where" <+> text var <+> text "=")
                          <+> sExp2Doc sugar qual high exp)
  goDoc _ _ (SEquation v lhs rhs) =
    group (high v (sExp2Doc sugar qual high lhs
                  <+> text "=" <+> sExp2Doc sugar qual high rhs))
  goDoc _ _ (SFieldExpr v e labs upds) =
    high v $ group (sExp2Doc sugar qual high e
                    <> text "{" <> commas (zipWith field labs upds)
                    <> text "}")
    where
      field name value = text name <> text "=" <> sExp2Doc sugar qual high value
      commas [doc] = doc
      commas (doc:docs) = doc <> text "," <+> commas docs


{- Pretty-print an expression with no interior highlighting -}
prettyExpression :: String -> Int -> Int -> Bool -> Bool -> Bool -> Bool
                    -> FileNode -> String
prettyExpression initial width cutoff uneval strSugar listSugar qual node =
  pretty width
         (highlight [Foreground Blue] (text initial)
         <> groupNest (length initial)
                      (sExp2Doc listSugar qual nohigh
                                (fileNode2SExp cutoff uneval strSugar False
                                               ("",node))))
  where nohigh _ doc = doc

{- Pretty-print an equation with no interior highlighting -}
prettyEquation :: String -> String -> Int -> Int -> Bool -> Bool -> Bool -> Bool
                  -> FileNode -> String
prettyEquation initial final width cutoff uneval strSugar listSugar qual node =
  pretty width
         ( highlight [Foreground Blue] (text initial)
         <> groupNest (length initial)
                      ( sExp2Doc listSugar qual nohigh
                                 (fileNode2SExp cutoff uneval strSugar True
                                                ("",node))
                        <-> text "=" <->
                        sExp2Doc listSugar qual nohigh
                                 (fileNode2SExp cutoff uneval strSugar False
                                                ("",getResult node True)))
         <> (if null final then nil
             else delimiter "  " <> highlight [Foreground Blue] (text final)))
  where nohigh _ doc = doc


{-
-- only for testing:
test1 =
  SApp (mkHatNode 0)
    [ SId (mkHatNode 1) "fun" SInfixDefault
    , SLiteral (mkHatNode 2) "24"
    , SLiteral (mkHatNode 3) "True"
    , SApp (mkHatNode 4)
        [ SId (mkHatNode 5) "+" (SInfixL 5)
        , SLiteral (mkHatNode 6) "3"
        , SLiteral (mkHatNode 7) "4"
        ]
    ]

test2 =
  SApp (mkHatNode 10)
    [ SId (mkHatNode 11) "*" (SInfixL 6)
    , SApp (mkHatNode 12)
        [ SId (mkHatNode 13) "+" (SInfixL 5)
        , SApp (mkHatNode 14)
            [ SId (mkHatNode 15) "-" (SInfixL 5)
            , SLiteral (mkHatNode 16) "3"
            , SLiteral (mkHatNode 17) "6"
            ]
        , SApp (mkHatNode 18)
            [ SId (mkHatNode 19) "-" (SInfixL 5)
            , SLiteral (mkHatNode 20) "3"
            , SLiteral (mkHatNode 21) "6"
            ]
        ]
    , test1
    ]
-}
