-- Simple program to display a src reference.
-- Reports the filename and location, and shows a section of the file
-- with the cursor on the actual location.  We try to place the
-- important line as near the centre of the screen as possible.

module Main where

import System
import IO             (hSetBuffering,BufferMode(..),stdin,stdout)
import Char           (isSpace,isDigit)
import Run            (runAndReadStdout)
import HighlightStyle (getTerminalSize,cls,goto,highlight,lineWrap
                      ,Highlight(..))

main = do
    args <- System.getArgs
    case args of
      [filename,sline,scol] ->
              let line = (read sline)::Int
                  column = (read scol)::Int in
              display filename line column
      _ -> do prog <- System.getProgName
              putStrLn ("Usage: "++prog++" srcfile line column")
              exitWith (ExitFailure 1)

display :: FilePath -> Int -> Int -> IO ()
display srcfile line column = do
    len <- runAndReadStdout ("wc -l "++srcfile)
    let n = read (takeWhile isDigit (dropWhile isSpace len))
    case n of
      0 -> do putStrLn ("File "++srcfile++" not found.")
              exitWith (ExitFailure 1)
      _ -> do (width,height) <- getTerminalSize
              f <- readFile srcfile
              hSetBuffering stdin NoBuffering
              hSetBuffering stdout NoBuffering
              let middle= (height `div` 2) - 1
                  trim  = if line < middle then 0 else line - middle
                  fs    = (unlines . take (height-2) . drop trim . lines) f
                  line' = line-trim
              putStr (cls ++ goto 1 1 ++ lineWrap False)
              putStr (highlight [Bold] ("---- "++srcfile++" ---- line: "
                                       ++show line++" ---- column: "
                                       ++show column++" ----"))
              putStr (goto 1 2 ++ fs ++ goto column (line'+1))
              System.system ("stty -icanon min 1 -echo")
              awaitQuit
              System.system ("stty icanon echo")
              putStr (goto 1 height)
              return ()

awaitQuit :: IO ()
awaitQuit = do
  q <- getChar
  case q of
    'q' -> return ()
    'x' -> return ()
    _   -> awaitQuit
