/*
  Hatari - cart.c

  This file is distributed under the GNU Public License, version 2 or at
  your option any later version. Read the file gpl.txt for details.

  Cartridge Pexec program

  To load programs into memory, through TOS, we need to intercept GEMDOS so we can
  relocate/execute programs via GEMDOS call $4B(PExec).
  We have some 68000 assembler, located at 0xFA1000 (cartridge memory), which is used as our
  new GEMDOS handler. This checks if we need to intercept the call.

  The assembler routine can be found in 'cart_asm.s', and has been converted to a byte
  array and stored in 'cart_img[]'.
*/
char Cart_rcsid[] = "Hatari $Id: cart.c,v 1.6 2004/12/19 13:46:33 thothy Exp $";

#include "main.h"
#include "cart.h"
#include "configuration.h"
#include "file.h"
#include "stMemory.h"
#include "vdi.h"


/* Cartridge header with system initialization code */
/* C-Init flag has bit 3 set = before disk boot, but after GEMDOS init */
unsigned char cart_hdr[] =
{
  0xAB,0xCD,0xEF,0x42,        /* C-FLAG */
  0x00,0x00,0x00,0x00,        /* C-NEXT */
  0x08,0xFA,0x00,0x24,        /* C-INIT */
  0x00,0x00,0x00,0x00,        /* C-RUN */
  0x00,0x00,                  /* C-TIME */
  0x00,0x00,                  /* C-DATE */
  0x00,0x00,0x00,0x00,        /* C-BSIZ, offset: $14*/
  'H','A','T','A','R','I','.','I','M','G',0x00,0x00, /* C-NAME */

  /* Code ($FA0024) - Set up GEMDOS handler and initialize some system variables */
  0xA0,0x00,                  /* Line-A init (for VDI resolutions, see OpCode_SysInit()) */
  0x00,0x0A,                  /* SYSINIT_OPCODE */
  0x4E,0x75                   /* RTS */
};


/* This is the assembled code from cart_asm.s with our gemdos handler.
 * NOTE: Remove first 0x1c (PRG_HEADER_SIZE) bytes from the assembled program
 * file or use an assembler like TurboAss that can generate absolute binary images.
 */
unsigned char cart_img[] =
{
 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0c,0x00,0x08,0x69,0x0a,0x66,0x02,
 0x4e,0x73,0x2f,0x3a,0xff,0xf0,0x4e,0x75,0x41,0xef,0x00,0x08,0x08,0x17,0x00,0x05,
 0x66,0x04,0x4e,0x68,0x54,0x48,0x4a,0x50,0x66,0x3c,0x2f,0x0e,0x2c,0x48,0x2f,0x0e,
 0x61,0x48,0x2c,0x5f,0x61,0x00,0x00,0x98,0x61,0x00,0x00,0xb6,0x42,0xae,0x00,0x02,
 0x42,0xae,0x00,0x0a,0x2d,0x40,0x00,0x06,0x3f,0x3c,0x00,0x30,0x4e,0x41,0x54,0x4f,
 0xe0,0x58,0xb0,0x7c,0x00,0x15,0x6c,0x06,0x3c,0xbc,0x00,0x04,0x60,0x04,0x3c,0xbc,
 0x00,0x06,0x2c,0x5f,0x60,0xac,0x0c,0x50,0x00,0x03,0x66,0xa6,0x2f,0x0e,0x2c,0x48,
 0x61,0x08,0x61,0x5a,0x61,0x7a,0x2c,0x5f,0x4e,0x73,0x3f,0x3c,0x00,0x2f,0x4e,0x41,
 0x54,0x4f,0x20,0x40,0x2f,0x18,0x2f,0x18,0x2f,0x18,0x2f,0x18,0x2f,0x18,0x2f,0x18,
 0x2f,0x18,0x2f,0x18,0x2f,0x18,0x2f,0x18,0x2f,0x18,0x2f,0x08,0x3f,0x3c,0x00,0x17,
 0x2f,0x2e,0x00,0x02,0x3f,0x3c,0x00,0x4e,0x4e,0x41,0x50,0x4f,0x20,0x5f,0x21,0x1f,
 0x21,0x1f,0x21,0x1f,0x21,0x1f,0x21,0x1f,0x21,0x1f,0x21,0x1f,0x21,0x1f,0x21,0x1f,
 0x21,0x1f,0x21,0x1f,0x4a,0x80,0x67,0x04,0x58,0x4f,0x60,0xaa,0x4e,0x75,0x2f,0x2e,
 0x00,0x0a,0x2f,0x2e,0x00,0x06,0x42,0xa7,0x3f,0x3c,0x00,0x05,0x3f,0x3c,0x00,0x4b,
 0x4e,0x41,0x4f,0xef,0x00,0x10,0x4a,0x80,0x6b,0x02,0x4e,0x75,0x58,0x4f,0x60,0x86,
 0x48,0xe7,0x03,0x1c,0x2a,0x40,0x42,0x67,0x2f,0x2e,0x00,0x02,0x3f,0x3c,0x00,0x3d,
 0x4e,0x41,0x50,0x4f,0x2c,0x00,0x2f,0x0d,0x06,0x97,0x00,0x00,0x00,0xe4,0x48,0x78,
 0x00,0x1c,0x3f,0x06,0x3f,0x3c,0x00,0x3f,0x4e,0x41,0x4f,0xef,0x00,0x0c,0x2f,0x0d,
 0x06,0x97,0x00,0x00,0x01,0x00,0x48,0x79,0x7f,0xff,0xff,0xff,0x3f,0x06,0x3f,0x3c,
 0x00,0x3f,0x4e,0x41,0x4f,0xef,0x00,0x0c,0x3f,0x06,0x3f,0x3c,0x00,0x3e,0x4e,0x41,
 0x58,0x4f,0x49,0xed,0x00,0x08,0x20,0x0d,0xd0,0xbc,0x00,0x00,0x01,0x00,0x28,0xc0,
 0x20,0x2d,0x00,0xe6,0x28,0xc0,0xd0,0xad,0x00,0x08,0x28,0xc0,0x28,0xed,0x00,0xea,
 0xd0,0xad,0x00,0xea,0x28,0xc0,0x28,0xed,0x00,0xee,0x20,0x0d,0xd0,0xbc,0x00,0x00,
 0x00,0x80,0x2b,0x40,0x00,0x20,0x28,0x6d,0x00,0x18,0xd9,0xed,0x00,0xf2,0x26,0x6d,
 0x00,0x08,0x20,0x0b,0x4a,0x6d,0x00,0xfe,0x66,0x3a,0x1e,0x14,0x42,0x1c,0xe1,0x4f,
 0x1e,0x14,0x42,0x1c,0x48,0x47,0x1e,0x14,0x42,0x1c,0xe1,0x4f,0x1e,0x14,0x42,0x1c,
 0x4a,0x87,0x67,0x20,0xd7,0xc7,0x7e,0x00,0xd1,0x93,0x1e,0x14,0x18,0xfc,0x00,0x00,
 0x4a,0x07,0x67,0x10,0xbe,0x3c,0x00,0x01,0x66,0x06,0x47,0xeb,0x00,0xfe,0x60,0xea,
 0xd6,0xc7,0x60,0xe4,0x20,0x2d,0x00,0x1c,0x67,0x0a,0x20,0x6d,0x00,0x18,0x42,0x18,
 0x53,0x80,0x66,0xfa,0x20,0x0d,0x4c,0xdf,0x38,0xc0,0x4e,0x75
};


/*-----------------------------------------------------------------------*/
/*
  Load ST GEMDOS intercept program image into cartridge memory space.
  This is used as an interface to the host file system and for GemDOS.
*/
void Cart_LoadImage(void)
{
	char *pCartFileName = ConfigureParams.Rom.szCartridgeImageFileName;

	/* "Clear" cartridge ROM space */
	memset(&STRam[0xfa0000], 0xff, 0x20000);

	if (bUseVDIRes || ConfigureParams.HardDisc.bUseHardDiscDirectories)
	{
		/* Copy cartrige header into ST's cartridge memory */
		memcpy(&STRam[0xfa0000], cart_hdr, sizeof(cart_hdr));

		/* Copy 'cart.img' file into ST's cartridge memory */
		memcpy(&STRam[0xfa1000], cart_img, sizeof(cart_img));
	}
	else if (strlen(pCartFileName) > 0)
	{
		/* Check if we can load an external cartridge file: */
		if (bUseVDIRes)
			fprintf(stderr, "Warning: Cartridge can't be used together with extended VDI resolution!\n");
		else if (ConfigureParams.HardDisc.bUseHardDiscDirectories)
			fprintf(stderr, "Warning: Cartridge can't be used together with GEMDOS harddisc emulation!\n");
		else if (!File_Exists(pCartFileName))
			fprintf(stderr, "Cartridge file not found: %s\n", pCartFileName);
		else if (File_Length(pCartFileName) > 0x20000)
			fprintf(stderr, "Cartridge file %s is too big.\n", pCartFileName);
		else
		{
			/* Now we can load it: */
			File_Read(pCartFileName, &STRam[0xfa0000], NULL, NULL);
		}
	}
}
