// measure.c
// ---------
//		Copyright 1988 D.C. Lindsay at Carnegie Mellon University
//
// Part of Hyperswitch simulation. Does measurements & related reporting.
// Assumed that there will only be one instance, so some static data.

#include "node.h"
#include "statistics.h"
#include "histogram.h"
#include "measure.h"
#include "chart.h"
#include <stdlib.h>
#include <stream.h>

extern schedulerclass * scheduler;
extern int message_trace_switch;
extern int message_length;
extern int cube_size;

// local data ( since only one instance will exist )

bool rungenerator = true;	// generate() consults this
int initialmessages = 0;	// reset after this many note's.
int allowedmessages = 10000;	// stop generating after this many more note's

// accumulators - OK is delivered messages, FAIL is undelivered
statistics OKdistance;		// across hypercube the message travelled
statistics OKlatency;		// nanos from comm start to last bit in RAM
statistics OKprobes;		// probes+1 comm chips got involved
statistics OKprobeexcess;	// difference of probes and distance 
				// ( zero when no trouble encountered )
statistics OKwait;		// (if waited) time spent before seized source
statistics OKretries;		// # of times message was retried
statistics FAILdistance;	// across hypercube the message travelled
statistics FAILlatency;		// nanos from comm start to last bit in RAM
statistics FAILprobes;		// probes+1 comm chips got involved
statistics FAILprobeexcess;	// difference of probes and distance 
				// ( zero when no trouble encountered )
statistics FAILwait;		// (if waited) time spent before seized source
statistics FAILretries;		// # of times message was retried

	// histogram is as "statistics" but note() is indexed by distance
histogramclass OKlatencyhisto;
histogramclass OKprobeshisto;
histogramclass OKprobeexcesshisto;

histogramclass FAILlatencyhisto;
histogramclass FAILprobeshisto;
histogramclass FAILprobeexcesshisto;

	// chart just counts how many times the noted value is in ranges
chartclass OKlatencychart( 10000 );
chartclass OKlatencychart2( 1000 );
chartclass FAILlatencychart( 10000 );
chartclass FAILlatencychart2( 1000 );
chartclass OKprobeexcesschart( 1 );

int in_messagecount = 0;	// reset() copies messagecount to here
int in_postmortemcount = 0;	// reset() copies postmortemcount to here
int messagecount;		// # started up
int postmortemcount;		// diff from messagecount is in-transit ones
int completeOKcount;		// message made it thru teardown
int sourcequitcount;		// message killed - all source links busy
int nopathscount;
int failurecount;
int ignoredcount;

timetype reset_time = 0;	// time measurement started
/*****************************************/
measureclass::measureclass()
{
	messagecount = 0;
	postmortemcount = 0;
	completeOKcount = 0;
	sourcequitcount = 0;
	nopathscount = 0;
	failurecount = 0;
	ignoredcount = 0;
}
/*****************************************/
static void reset()
{
	in_messagecount = messagecount;		// keep a copy
	in_postmortemcount = postmortemcount;	// keep a copy
	messagecount = 0;
	postmortemcount = 0;
	completeOKcount = 0;
	sourcequitcount = 0;
	nopathscount = 0;
	failurecount = 0;
	ignoredcount = 0;

	OKdistance.reset();
	OKlatency.reset();
	OKprobes.reset();
	OKprobeexcess.reset();
	OKwait.reset();
	OKretries.reset();

	FAILdistance.reset();
	FAILlatency.reset();
	FAILprobes.reset();
	FAILprobeexcess.reset();
	FAILwait.reset();
	FAILretries.reset();

	OKlatencyhisto.reset();
	OKprobeshisto.reset();
	OKprobeexcesshisto.reset();

	FAILlatencyhisto.reset();
	FAILprobeshisto.reset();
	FAILprobeexcesshisto.reset();

	OKlatencychart.reset();
	OKlatencychart2.reset();
	FAILlatencychart.reset();
	FAILlatencychart2.reset();
	OKprobeexcesschart.reset();
}

/*****************************************/
void measureclass::runsize( int newinitial, int newallowed )
{
	initialmessages = newinitial;
	allowedmessages = newallowed;
}
/*****************************************/
SMessagePtr  measureclass::generate()
{
	if( rungenerator == false ) return NULLO;
	++messagecount;
	if( TRACE ); else if((messagecount&255)==0) cout << ".";
	SMessagePtr smp;
#ifdef REFGC
	smp.New();
#else
	smp = new ( message );
#endif
	return smp;
}
/*****************************************/
//
// Prints summary of measurements.
// Only prints the first time it is called.

static bool printed = false;

static void print()
{
	if( printed == true ) return;
	printed = true;
	cout	<< "\nMessages completed " << postmortemcount NL;

	cout << " Completed ok " << completeOKcount NL;
	if( sourcequitcount )
		cout << " quit(all source links busy) " << sourcequitcount NL;
	if( failurecount )
		cout << " quit(dest busy) " << failurecount NL;
	if( nopathscount )
		cout << " quit(all paths failed) " << nopathscount NL;
	timetype elapsed = scheduler->clock - reset_time;
	cout	<< "Time for " << postmortemcount << " was "
		<< elapsed << " ns == "
		<< (elapsed/postmortemcount) 
		<< " ns/mesg == "
		<< ((elapsed/postmortemcount)*cube_size)
		<< " ns/m/node" NL;

	int meandistance = 
		OKdistance.print( "Delivered-message: Distance: " );

	int meantime = OKlatency.print( "Delivered-message: Latency: " );
	// now compute the time due to just the bandwidth limit of any link
	int bandtime = message_length * LINK_BANDWIDTH;	
	cout << "\tThis is " << 
		( ((100*meantime)+(bandtime/2))/bandtime )
		<< "% of the bandwidth latency " << bandtime NL;
	cout << "\t(Exceeds the bandwidth latency by " 
		<< (meantime-bandtime)<< ")\n";
	// now add in the time due to link traversal (speed of light, etc)
	// and the time due to flowing through crossbars. This has to be
	// multiplied by the mean distance since it happens per-hop.
	bandtime += (meandistance * (LINK_LATENCY + XBAR_LATENCY));
	cout << "\tThis is " << 
		( ((100*meantime)+(bandtime/2))/bandtime )
		<< "% of the hardware latency " << bandtime NL;
	cout << "\t(Exceeds the hardware latency by " 
		<< (meantime-bandtime)<< ")\n";

	OKlatencyhisto.print( " histo:" );

	OKprobes.print( "Delivered-message: Probes: " );
	OKprobeshisto.print( " histo:" );

	OKprobeexcess.print( "Delivered-message: Probes minus Distance: " );
	OKprobeexcesschart.print( " chart:" );
	OKprobeexcesshisto.print( " histo:" );

	OKretries.print( "Delivered-message: Retries: " );

	OKwait.print( "Delivered-message: Wait: " );

	OKlatencychart.print( "Delivered-message: Latency chart: " );
	OKlatencychart2.print( "Finer-grain Latency chart: " );

	FAILdistance.print( "Undelivered message: Distance: " );

	FAILlatency.print( "Undelivered message: Latency: " );
	FAILlatencyhisto.print( " histo:" );

	FAILprobes.print( "Undelivered message: Probes: " );
	FAILprobeshisto.print( " histo:" );

	FAILprobeexcess.print("Undelivered message: Probes minus Distance: ");
	FAILprobeexcesshisto.print( " histo:" );

	FAILretries.print( "Undelivered message: Retries: " );

	FAILwait.print( "Undelivered message: Wait: " );

	FAILlatencychart.print( "Undelivered message: Latency chart: " );
	FAILlatencychart2.print( "Finer grain Latency chart: " );
}
/*****************************************/
// Accumulate statistics on a message and deallocate it.
//
void measureclass::note( SMessagePtr pm, finalstate reason )
{
	postmortemcount++;
	switch( reason ) {
case failure:		failurecount++;		break;
case nopaths:		nopathscount++;		break;
case sourcequit:	sourcequitcount++;	break;
case completeOK: 	completeOKcount++; 	break;

case ignored:	ignoredcount++;
#ifndef REFGC
		delete pm;		// note, another delete below
#endif
		return;
default:	cout << "ERROR: BUG IN POSTMORTEM\n";
		exit(1);
	}
	if( reason == completeOK ) {
		OKdistance.note( pm-> distance );
		int latency = pm-> lastbittime - pm-> starttime;
		OKlatency.note( latency );
		OKlatencyhisto.note( latency, pm-> distance );
		OKlatencychart.note( latency );
		OKlatencychart2.note( latency );
		OKprobes.note( pm-> probecount );
		OKprobeshisto.note( pm-> probecount, pm-> distance );
		OKprobeexcess.note( pm-> probecount - pm-> distance );
		OKprobeexcesshisto.note( pm-> probecount - pm-> distance,
			pm-> distance );
		OKprobeexcesschart.note( pm-> probecount - pm-> distance );
		if( pm-> retrycount > 0 )
			OKretries.note( pm-> retrycount );
		int wait = pm-> activetime - pm-> starttime;
		if( wait > 0 ) OKwait.note( wait );
	}else{
		FAILdistance.note( pm-> distance );
		int latency = scheduler->clock - pm-> starttime;
		FAILlatency.note( latency );
		FAILlatencyhisto.note( latency,	pm-> distance );
		FAILlatencychart.note( latency );
		FAILlatencychart2.note( latency );
		FAILprobes.note( pm-> probecount );
		FAILprobeshisto.note( pm-> probecount, pm-> distance );
		FAILprobeexcess.note( pm-> probecount - pm-> distance );
		FAILprobeexcesshisto.note( pm-> probecount - pm-> distance,
			pm-> distance );
		if( pm-> retrycount > 0 )
			FAILretries.note( pm-> retrycount );
		int wait = pm-> activetime - pm-> starttime;
		if( wait > 0 ) FAILwait.note( wait );
	}


#ifndef REFGC
	delete pm;		// note, case ignored does its own
#endif
	if( postmortemcount == initialmessages ){
		reset_time = scheduler->clock;
		cout << "\nResetting counts at time " << 
			(unsigned long) reset_time << ".\n";
		//			form( "%lu", reset_time ) << ".\n";
		reset();
		initialmessages = 0;	// can only reset once
	}else if( postmortemcount == allowedmessages ) {
		print();
		rungenerator = false;	// no need to add more mesgs to sim
	}
}
/*****************************************/
void measureclass::finalize()
{
	print();		// if already done, wont happen again
	int tot_messagecount = in_messagecount + messagecount;
	cout	<< "Finalize: total run created "
		<< tot_messagecount << " messages.\n";
	int tot_postmortem = in_postmortemcount + postmortemcount;
	int intransit = tot_messagecount - tot_postmortem;
	if( intransit ) cout << "ERROR ************** "
		<< "messages still in transit " << intransit 
		<< "***************\n";
}
/*****************************************/
