 /* bz2.c:  compression module using bz2 for heartbeat.
 *
 * Copyright (C) 2005 Guochun Shi <gshi@ncsa.uiuc.edu>
 *
 * SECURITY NOTE:  It would be very easy for someone to masquerade as the
 * device that you're pinging.  If they don't know the password, all they can
 * do is echo back the packets that you're sending out, or send out old ones.
 * This does mean that if you're using such an approach, that someone could
 * make you think you have quorum when you don't during a cluster partition.
 * The danger in that seems small, but you never know ;-)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */



#define PIL_PLUGINTYPE          HB_COMPRESS_TYPE
#define PIL_PLUGINTYPE_S        HB_COMPRESS_TYPE_S
#define PIL_PLUGIN              bz2
#define PIL_PLUGIN_S            "bz2"
#define PIL_PLUGINLICENSE	LICENSE_LGPL
#define PIL_PLUGINLICENSEURL	URL_LGPL
#include <portability.h>
#include <stdio.h>
#include <pils/plugin.h>
#include <compress.h>
#include <bzlib.h>
#include <clplumbing/cl_log.h>
#include <string.h>


static struct hb_compress_fns bz2Ops;

PIL_PLUGIN_BOILERPLATE2("1.0", Debug)

static const PILPluginImports*  PluginImports;
static PILPlugin*               OurPlugin;
static PILInterface*		OurInterface;
static struct hb_media_imports*	OurImports;
static void*			interfprivate;

#define LOG	PluginImports->log
#define MALLOC	PluginImports->alloc
#define STRDUP  PluginImports->mstrdup
#define FREE	PluginImports->mfree

PIL_rc
PIL_PLUGIN_INIT(PILPlugin*us, const PILPluginImports* imports);

PIL_rc
PIL_PLUGIN_INIT(PILPlugin*us, const PILPluginImports* imports)
{
	/* Force the compiler to do a little type checking */
	(void)(PILPluginInitFun)PIL_PLUGIN_INIT;

	PluginImports = imports;
	OurPlugin = us;

	/* Register ourself as a plugin */
	imports->register_plugin(us, &OurPIExports);  

	/*  Register our interface implementation */
 	return imports->register_interface(us, PIL_PLUGINTYPE_S
	,	PIL_PLUGIN_S
	,	&bz2Ops
	,	NULL		/*close */
	,	&OurInterface
	,	(void*)&OurImports
	,	interfprivate); 
}

static int
bz2_compress(char* dest, size_t* destlen, 
	      const char* _src, size_t srclen)
{
	int ret;
	char* src;
	unsigned int tmpdestlen;	

	memcpy(&src, &_src, sizeof(char*));
	
	tmpdestlen = *destlen; 
	ret = BZ2_bzBuffToBuffCompress(dest, &tmpdestlen, src, srclen, 1, 0, 30);
	if (ret != BZ_OK){
		cl_log(LOG_ERR, "%s: compression failed",
		       __FUNCTION__);
		return HA_FAIL;
	}	

	*destlen = tmpdestlen;

	return HA_OK;
}

static int
bz2_decompress(char* dest, size_t* destlen,
	       const char* _src, size_t srclen)
{
	
	int ret;
	char* src;
        unsigned int tmpdestlen;
	
	memcpy(&src, &_src, sizeof(char*));

        tmpdestlen = *destlen;
	ret = BZ2_bzBuffToBuffDecompress(dest, &tmpdestlen, src, srclen, 1, 0);
	if (ret != BZ_OK){
		cl_log(LOG_ERR, "%s: decompression failed",
		       __FUNCTION__);
		return HA_FAIL;
	}

        *destlen = tmpdestlen;
	
	return HA_OK;
}

static const char*
bz2_getname(void)
{
	return "bz2";	
}

static struct hb_compress_fns bz2Ops ={
	bz2_compress,
	bz2_decompress,
	bz2_getname,
};
