/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: audsymbian.cpp,v 1.23.2.3 2004/07/09 02:01:39 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#include <e32base.h>
#include <e32std.h>

#include "hxassert.h"
#include "hxresult.h"
#include "hxslist.h"
#include "hxcom.h"
#include "hxtick.h"
#include "hxausvc.h"
#include "ihxpckts.h"   

#include "audiosvr/audio_svr.h"
#include "audiosvr/audio_svr_cntxt.h"

#include "audsymbian.h"
static UINT32 Scale(UINT32 v, UINT32 f0, UINT32 f1, UINT32 t0, UINT32 t1);

CHXAudioDevice::CHXAudioDevice()
    : CActive(EPriorityHigh),
      m_lRefCount(0),
      m_deviceOpen(false),
      m_pDeviceResponse(NULL),
      m_pAudioStream(NULL),
      m_pAudioServerContext(NULL),
      m_paused(false),
      m_uMinPlayerVolume(0),
      m_uMaxPlayerVolume(100),
      m_uMinDevVolume(0),
      m_uMaxDevVolume(100)
{
    CActiveScheduler::Add(this);
    TInt err = m_iTimer.CreateLocal();
    HX_ASSERT(KErrNone==err);
}

CHXAudioDevice::~CHXAudioDevice()
{
    Close(TRUE);
    m_iTimer.Close();
}

ULONG32 CHXAudioDevice::AddRef()
{
    return InterlockedIncrement(&m_lRefCount);
}

ULONG32 CHXAudioDevice::Release()
{
    if (InterlockedDecrement(&m_lRefCount) > 0)
    {
        return m_lRefCount;
    }

    delete this;
    return 0;
}

HX_RESULT CHXAudioDevice::QueryInterface(REFIID riid, void** ppvObj)
{
    if (IsEqualIID(riid, IID_IHXAudioDevice))
    {
        AddRef();
        *ppvObj = (IHXAudioDevice*)this;
        return HXR_OK;
    }
    else if (IsEqualIID(riid, IID_IUnknown))
    {
        AddRef();
        *ppvObj = this;
        return HXR_OK;
    }

    *ppvObj = NULL;
    return HXR_NOINTERFACE;
}

HX_RESULT CHXAudioDevice::Open(const HXAudioFormat* pFormat,
                               IHXAudioDeviceResponse* pDeviceResponse)
{
    HX_RESULT res = HXR_OK;

    HX_ASSERT(!m_pDeviceResponse);
    HX_RELEASE(m_pDeviceResponse);
    if( pDeviceResponse )
    {
        m_pDeviceResponse = pDeviceResponse;
        m_pDeviceResponse->AddRef();
    }

    if (HXR_OK != OpenDevice() || HXR_OK != InitDevice(pFormat))
    {
        res = HXR_FAIL;
    }

    return res;
}

HX_RESULT CHXAudioDevice::Close(const BOOL bFlush)
{
    Reset();

    if (m_pAudioStream)
    {
        m_pAudioStream->Close();
        m_pAudioServerContext->Stop();
        m_deviceOpen = false;

        HX_DELETE(m_pAudioStream);
        HX_DELETE(m_pAudioServerContext);
        HX_RELEASE(m_pDeviceResponse);
    }

    return HXR_OK;
}

HX_RESULT CHXAudioDevice::Write(const HXAudioData* pAudioData)
{
    HX_RESULT res = HXR_OK;


    if (pAudioData)
    {
        // add the buffer to the pending write list
        IHXBuffer* pAudioBuf = pAudioData->pData;
        pAudioBuf->AddRef();
        if( m_pAudioStream )
            m_pAudioStream->Write(pAudioBuf);

    }

    return res;
}

HX_RESULT CHXAudioDevice::Reset()
{
    if (m_pAudioStream)
    {
        if (!m_pAudioStream->Stopped())
        {
            m_pAudioStream->Stop();
        }
    }
    
    if (IsActive())
    {
        Cancel();
    }

    return HXR_OK;
}

HX_RESULT CHXAudioDevice::Drain()
{
    HX_ASSERT("Not implemented"==NULL);
    return HXR_FAIL;
}

HX_RESULT CHXAudioDevice::SetVolume( const UINT16 uVolume )
{
    HX_RESULT res = HXR_FAIL;
    
    if( m_pAudioStream )
    {
        m_pAudioStream->SetVolume(TInt(Scale(UINT32(uVolume), 
                                             m_uMinPlayerVolume, m_uMaxPlayerVolume,
                                             m_uMinDevVolume, m_uMaxDevVolume)));
        res = HXR_OK;
    }

    return res;
}

HX_RESULT CHXAudioDevice::GetCurrentAudioTime( ULONG32& ulCurrentTime )
{
    HX_RESULT res = HXR_FAIL;
    
    HX_ASSERT(m_pAudioStream);
    if( m_pAudioStream )
    {
        ulCurrentTime = m_pAudioStream->GetTime();
        res = HXR_OK;
    }

    return res;
}

BOOL CHXAudioDevice::SupportsVolume()
{
    return TRUE;
}

UINT16 CHXAudioDevice::GetVolume()
{
    UINT16 vol = 0;
    if (m_pAudioStream)
    {
        vol = UINT16(Scale(UINT32(m_pAudioStream->GetVolume()),
                           m_uMinDevVolume, m_uMaxDevVolume,
                           m_uMinPlayerVolume, m_uMaxPlayerVolume));
    }

    return vol;
}

short CHXAudioDevice::GetAudioFd( void )
{
    //We don't have file descriptors on symbian for the
    //audio device.
    return 0;
}

HX_RESULT CHXAudioDevice::Seek(ULONG32 ulSeekTime)
{
    HX_ASSERT( "Not implemented"==NULL);
    return HXR_OK;
}

HX_RESULT CHXAudioDevice::Resume()
{
    m_paused = false;
    
    if(m_pAudioStream)
    {
        m_pAudioStream->Play();
    }

    if( !IsActive() )
    {
        m_iTimer.After(iStatus, 1000*100);
        SetActive();
    }
    
    
    return HXR_OK;
}

HX_RESULT CHXAudioDevice::Pause()
{
    HX_RESULT res = HXR_OK;

    m_paused = true;

    if (m_pAudioStream)
    {
        m_pAudioStream->Pause();
    }

    m_iTimer.Cancel();
    
    return res;
}

HX_RESULT CHXAudioDevice::CheckFormat(const HXAudioFormat* pFormat)
{
    HX_RESULT res = HXR_FAIL;

    //Symbian only supports 16-bit PCM.
    if (pFormat->uBitsPerSample == 16 &&
        HXR_OK == OpenDevice() &&
        HXR_OK == InitDevice(pFormat))
    {
        res = HXR_OK;
    }

    return res;
}

UINT16 CHXAudioDevice::NumberOfBlocksRemainingToPlay(void)
{
    return (UINT16)(m_pAudioStream->GetBlocksBuffered());
}

BOOL CHXAudioDevice::InitVolume(const UINT16 uMinVolume,
                                const UINT16 uMaxVolume )
{
    HX_ASSERT(m_uMaxPlayerVolume > m_uMinPlayerVolume);
    
    if (m_uMaxPlayerVolume > m_uMinPlayerVolume)
    {
        m_uMinPlayerVolume = uMinVolume;
        m_uMaxPlayerVolume = uMaxVolume;
    }
    return (m_uMaxPlayerVolume > m_uMinPlayerVolume) && SupportsVolume();
}

//
// CActive methods
//

void CHXAudioDevice::RunL()
{
    if (iStatus != KErrCancel)
    {
        // call back the response object to update time
        if( m_pDeviceResponse && !m_paused)
        {
            ULONG32 ulAudioTime = 0;
            GetCurrentAudioTime(ulAudioTime);
            m_pDeviceResponse->OnTimeSync(ulAudioTime);
        }
    }

    m_iTimer.After(iStatus, 100*1000 );
    SetActive();
}

void CHXAudioDevice::DoCancel()
{
    if (m_pAudioStream)
    {
        m_pAudioStream->CancelWrite();
    }
    m_iTimer.Cancel();
}

//////////////////////////////////////////////////////////////////////
// private methods ///////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////

HX_RESULT CHXAudioDevice::OpenDevice()
{
    HX_RESULT res = HXR_FAIL;

    if (m_deviceOpen)
        return HXR_OK;

    if(!m_pAudioStream)
    {
        m_pAudioStream = new HXSymbianAudioClient;
    }

    if (m_pAudioStream)
    {
        // create the thread context for the audio server to run 
        if (!m_pAudioServerContext)
        {
            m_pAudioServerContext = new HXSymbianAudioServerContext;
        }

        // start the audio sevrver thread
        if (m_pAudioServerContext && !m_pAudioServerContext->Running())
        {
            m_pAudioServerContext->Start();
        }

        // connect to the audio server and share the session
        if (m_pAudioServerContext->Running() &&
            KErrNone == m_pAudioStream->Connect() &&
            KErrNone == m_pAudioStream->Share(RSessionBase::EAutoAttach))
        {
            m_deviceOpen = true;
            res = HXR_OK;
        }
    }

    return res;
}

HX_RESULT CHXAudioDevice::InitDevice(const HXAudioFormat* pFormat)
{
    HX_RESULT res = HXR_FAIL;

    // attempt to init device with given audio format
    if (m_pAudioStream &&  
        (KErrNone == m_pAudioStream->Init(pFormat->ulSamplesPerSec,
                                          pFormat->uChannels)))
    {
        // grab device volume info (first opportunity)
        m_uMinDevVolume = m_pAudioStream->GetMinVolume();
        m_uMaxDevVolume = m_pAudioStream->GetMaxVolume();

        res = HXR_OK;
    }
    return res;
}

//////////////////////////////////////////////////////////////////////
// convenience methods ///////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////

UINT32 Scale(UINT32 v, UINT32 f0, UINT32 f1, UINT32 t0, UINT32 t1)
{
    HX_ASSERT(f1 > f0);
    HX_ASSERT(t1 >= t0);
    HX_ASSERT(v >= f0);

    if (f1 > f0 && v > f0)
    {
        UINT64 tr = t1 - t0;
        UINT64 fr = f1 - f0;
        UINT64 n  = v - f0;
        UINT64 q  = n / fr;
        UINT64 r  = n % fr;

        return t0 + INT64_TO_UINT32((q * tr + (r * tr + (fr >> 1)) / fr));
    }
    return 0;
}
