/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: base_page2pkt.cpp,v 1.2.2.4 2004/07/09 01:58:26 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */
#include "base_page2pkt.h"
#include "oggutil.h"

#include "debug.h" // DPRINTF()
#define D_BASE_P2P 0 //0x4000000

CBasePageToPacket::CBasePageToPacket() :
    m_bHasStreamID(FALSE),
    m_uStreamID(0),
    m_bHasStartTime(FALSE),
    m_startTime(0),
    m_endTime(0),
    m_ulSeekTime(0),
    m_bEndOfStream(FALSE),
    m_state(stStart),
    m_pCCF(NULL),
    m_bSentHeaders(FALSE)
{
#ifdef _DEBUG
    debug_level() |= D_BASE_P2P;
#endif

    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::CBasePageToPacket()\n"));

    ogg_stream_init(&m_os, 0);
}

CBasePageToPacket::~CBasePageToPacket()
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::~CBasePageToPacket()\n"));
    HX_RELEASE(m_pCCF);

    ogg_stream_clear(&m_os);

    FlushQueues();
    FlushHeaders();
}

STDMETHODIMP CBasePageToPacket::Init(IUnknown* pContext)
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::Init()\n"));
    HX_RESULT res = HXR_FAILED;

    if (pContext)
    {
        HX_RELEASE(m_pCCF);
        res = pContext->QueryInterface(IID_IHXCommonClassFactory,
                                       (void**)&m_pCCF);

        if (HXR_OK == res)
        {
            ChangeState(stNeedHeaders);
            m_ulHeadersRemaining = HeaderCount();
        }
    }

    return res;
}

STDMETHODIMP CBasePageToPacket::SetStreamID(UINT16 uStream)
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::SetStreamID(%u)\n", uStream));

    m_bHasStreamID = TRUE;
    m_uStreamID = uStream;

    return HXR_OK;
}


STDMETHODIMP CBasePageToPacket::OnPage(ogg_page* pPage)
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::OnPage()\n"));

    HX_RESULT res = HXR_FAILED;

    if (pPage)
    {
        if (stReady == m_state)
        {
            res = HandleTimedPage(CurrentTime(), pPage);
        }
        else
        {
            res = HandleUntimedPage(pPage);
        }
    }

    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::OnPage() : res %08x\n", res));
    return res;
}

STDMETHODIMP CBasePageToPacket::OnEndOfFile()
{
    m_bEndOfStream = TRUE;

    return HXR_OK;
}

STDMETHODIMP CBasePageToPacket::GetPacket(REF(IHXPacket*) pPkt)
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::GetPacket()\n"));

    HX_RESULT res = HXR_NO_DATA;

    if (!m_packetQueue.IsEmpty())
    {
        if (!m_bSentHeaders)
        {
            res = CreateHeaderPackets();
        }
        else
        {
            res = HXR_OK;
        }

        if (HXR_OK == res)
        {
            pPkt = (IHXPacket*)m_packetQueue.RemoveHead();
        }
    }
    else if (m_bEndOfStream)
    {
        res = HXR_STREAM_DONE;
    }

    return res;
}

STDMETHODIMP CBasePageToPacket::GetNextTimestamp(REF(UINT32) ulTimestamp)
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::GetNextTimestamp()\n"));

    HX_RESULT res = HXR_NO_DATA;

    if (!m_packetQueue.IsEmpty())
    {
        IHXPacket* pPkt = (IHXPacket*)m_packetQueue.GetHead();
        
        ulTimestamp = pPkt->GetTime();

        res = HXR_OK;
    }
    else if (m_bEndOfStream)
    {
        res = HXR_STREAM_DONE;
    }

    return res;
}

STDMETHODIMP CBasePageToPacket::GetTACInfo(REF(IHXValues*) pHeader)
{
    return HXR_FAILED;
}


STDMETHODIMP CBasePageToPacket::OnSeek(UINT32 ulSeekPoint)
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::OnSeek(%u)\n", ulSeekPoint));
    HX_RESULT res = HXR_FAILED;

    if (stReady == m_state)
    {
        FlushQueues();

        ogg_stream_reset(&m_os);
        
        m_bEndOfStream = FALSE;

        m_ulSeekTime = ulSeekPoint;

        InvalidateCurrentTime();

        ChangeState(stSeeking);

        res = HXR_OK;
    }

    return res;
}

STDMETHODIMP CBasePageToPacket::SetEndTime(ogg_int64_t endTime)
{
    m_endTime = endTime;

    return HXR_OK;
}

STDMETHODIMP_(BOOL) CBasePageToPacket::HasStartTime() const
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::HasStartTime() : %d\n",
                         m_bHasStartTime));

    return m_bHasStartTime;
}

STDMETHODIMP_(UINT32) CBasePageToPacket::StartTime() const
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::StartTime() : %lld\n", 
                         m_startTime));
    return INT64_TO_UINT32(m_startTime);
}

STDMETHODIMP_(UINT32) CBasePageToPacket::Duration() const
{
    UINT32 ulRet = 0;

    if (m_endTime >= m_startTime)
    {
        GetTimestamp(m_endTime, ulRet);
    }
    
    return ulRet;
}

STDMETHODIMP_(BOOL) CBasePageToPacket::EndOfStream() const
{
    return m_bEndOfStream && m_packetQueue.IsEmpty();
}


void CBasePageToPacket::ChangeState(State newState)
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::ChangeState() : %d -> %d\n", 
                         m_state, newState));
    m_state = newState;
}

HX_RESULT CBasePageToPacket::CreatePagePacket(UINT32 ulTimestamp,
                                              ogg_page* pPage,
                                              BOOL bIsHeader)
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::CreatePagePacket( %u )\n",
                         ulTimestamp));

    HX_RESULT res = HXR_FAILED;
    
    if (m_pCCF && pPage && m_bHasStreamID)
    {
        IHXPacket* pPacket = NULL;
        res = m_pCCF->CreateInstance(CLSID_IHXPacket, (void**)&pPacket);

        if (HXR_OK == res)
        {
            IHXBuffer* pPayload = NULL;
            
            res = m_pCCF->CreateInstance(CLSID_IHXBuffer, (void**)&pPayload);

            if (HXR_OK == res)
            {
                res = pPayload->SetSize(pPage->header_len + pPage->body_len);

                if (HXR_OK == res)
                {
                    memcpy(pPayload->GetBuffer(), 
                           pPage->header, pPage->header_len);
                    memcpy(pPayload->GetBuffer() + pPage->header_len, 
                           pPage->body, pPage->body_len);

                    res = pPacket->Set(pPayload,
                                       ulTimestamp,
                                       m_uStreamID, // streamID
                                       0, // flags
                                       0); // rule
                    
                    if (HXR_OK == res)
                    {
                        pPacket->AddRef();

                        BOOL bAddFailed = FALSE;

                        if (bIsHeader)
                        {
                            bAddFailed = !m_packetQueue.AddHead(pPacket);
                        }
                        else
                        {
                            bAddFailed = !m_packetQueue.AddTail(pPacket);
                        }

                        if (bAddFailed)
                        {
                            HX_RELEASE(pPacket);

                            res = HXR_OUTOFMEMORY;
                        }
                    }
                }
            }

            HX_RELEASE(pPayload);
        }
        HX_RELEASE(pPacket);
    }

    return res;
}

HX_RESULT CBasePageToPacket::CacheHeader(ogg_page* pPage)
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::CacheHeader()\n"));

    HX_RESULT res = HXR_OUTOFMEMORY;

    if (HaveThisHeader(pPage))
    {
        res = HXR_OK;
    }
    else
    {
        ogg_page* pNewPage = OggUtil::CopyPage(pPage);

        // We add the pages to the head so that they are in
        // the right order when we insert them into the head
        // of the timed packet list
        if (pNewPage && 
            m_headerPages.AddHead(pNewPage))
        {
            DPRINTF(D_BASE_P2P, 
                    ("CBasePageToPacket::CacheHeader() : hdr cached\n"));
            res = HXR_OK;
        }
        else
        {
            OggUtil::DestroyPage(pNewPage);
        }
    }

    return res;
}

BOOL CBasePageToPacket::HaveThisHeader(ogg_page* pPage)
{
    BOOL bRet = FALSE;
    
    CHXSimpleList::Iterator i;

    for (i = m_headerPages.Begin(); !bRet && i != m_headerPages.End(); ++i)
    {
        ogg_page* pHdr = (ogg_page*)(*i);

        if (OggUtil::ComparePages(pHdr, pPage))
        {
            bRet = TRUE;
        }
    }

    DPRINTF(D_BASE_P2P, 
            ("CBasePageToPacket::HaveThisHeader() : bRet %d\n", bRet));

    return bRet;
}

HX_RESULT CBasePageToPacket::CreateHeaderPackets()
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::CreateHeaderPackets()\n"));

    HX_RESULT res = HXR_UNEXPECTED;

    if (!m_headerPages.IsEmpty())
    {
        // Get the timestamp of the first data packet
        UINT32 ulTimestamp;
        res = GetNextTimestamp(ulTimestamp);

        if (HXR_OK == res)
        {
            // Create packets for these header pages and
            // enqueue them before the first data packet.
            CHXSimpleList::Iterator i;
            for (i = m_headerPages.Begin(); 
                 (HXR_OK == res) && i != m_headerPages.End(); ++i)
            {
                ogg_page* pHdr = (ogg_page*)(*i);
                
                res = CreatePagePacket(ulTimestamp, pHdr, TRUE);
            }

            if (HXR_OK == res)
            {
                m_bSentHeaders = TRUE;
            }
        }
    }

    return res;
}

HX_RESULT CBasePageToPacket::EnqueuePage(ogg_page* pPage)
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::EnqueuePage()\n"));

    HX_RESULT res = HXR_FAILED;

    if (pPage)
    {
        ogg_page* pNewPage = OggUtil::CopyPage(pPage);

        if (pNewPage && m_pageQueue.AddTail(pNewPage))
        {
            res = HXR_OK;
        }
        else
        {
            OggUtil::DestroyPage(pPage);
            res = HXR_OUTOFMEMORY;
        }
    }

    return res;
}

HX_RESULT CBasePageToPacket::HandleTimedPage(ogg_int64_t startTime, 
                                             ogg_page* pPage)
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::HandleTimedPage( %lld , %lld) : %d\n",
                         startTime,
                         ogg_page_granulepos(pPage),
                         ogg_page_continued(pPage)));

    HX_RESULT res = HXR_FAILED;

    if (ogg_page_eos(pPage))
    {
        DPRINTF(D_BASE_P2P, ("CBasePageToPacket::HandleTimedPage() : End of stream\n"));
        m_bEndOfStream = TRUE;
    }   

    if (ogg_stream_pagein(&m_os, pPage) == 0)
    {
        ogg_packet op;
        BOOL bDone = FALSE;
        BOOL bHasHeader = FALSE;

        res = HXR_OK;

        while(!bDone)
        {
            int poErr = ogg_stream_packetout(&m_os, &op);
            
            if (poErr > 0)
            {
                switch(m_state) {
                case stReady:
                    if (IsHeader(&op))
                    {
                        bHasHeader = TRUE;
                        res = OnHeader(&op);
                    }
                    else
                    {
                        HX_ASSERT(!bHasHeader);
                        res = OnDataPacket(&op);
                    }
                    break;
                default:
                    bDone = TRUE;
                    res = HXR_FAILED;
                    break;
                };
            }
            else if (poErr == 0)
            {
                bDone = TRUE;
            }
            else
            {
                res = HXR_UNEXPECTED;
            }
        }
        
        if (HXR_OK == res)
        {
            if (bHasHeader)
            {
                res = CacheHeader(pPage);
            }
            else
            {
                UINT32 ulTimestamp = 0;

                GetTimestamp(startTime, ulTimestamp);

                res = CreatePagePacket(ulTimestamp, pPage, FALSE);
            }
        }
    }

    return res;

}

HX_RESULT CBasePageToPacket::HandleUntimedPage(ogg_page* pPage)
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::HandleUntimedPage( %lld ) : %d\n",
                         ogg_page_granulepos(pPage),
                         ogg_page_continued(pPage)));

    HX_RESULT res = HXR_FAILED;

    if (ogg_page_bos(pPage))
    {
        ogg_stream_clear(&m_os);
        ogg_stream_init(&m_os, ogg_page_serialno(pPage));
        
        StartStream();
            
        ChangeState(stNeedHeaders);
        m_ulHeadersRemaining = HeaderCount();
        FlushHeaders();
    }

    if (ogg_page_eos(pPage))
    {
        DPRINTF(D_BASE_P2P, 
                ("CBasePageToPacket::HandleUntimedPage() : End of stream\n"));
        m_bEndOfStream = TRUE;
    }   

    if (ogg_stream_pagein(&m_os, pPage) == 0)
    {
        ogg_packet op;
        BOOL bDone = FALSE;
        BOOL bHasHeader = FALSE;

        // Assume we are going to need
        // more data
        res = HXR_NO_DATA;

        while(!bDone)
        {
            int poErr = ogg_stream_packetout(&m_os, &op);
            
            if (poErr > 0)
            {
                switch(m_state) {
                case stNeedHeaders:
                    if ((m_ulHeadersRemaining == 0) ||
                        (!IsHeader(&op)))
                    {
                        bDone = TRUE;
                        res = HXR_INVALID_FILE;
                    }
                    else
                    {
                        res = OnHeader(&op);
                        if (HXR_OK == res)
                        {
                            bHasHeader = TRUE;

                            m_ulHeadersRemaining--;
                            
                            if (m_ulHeadersRemaining == 0)
                            {
                                ChangeState(stNeedStartTime);
                            }
                        }
                    }
                    break;
                    
                case stNeedStartTime:
                case stSeeking:
                    HX_ASSERT(!bHasHeader);
                    res = OnDataPacket(&op);
                    break;
                default:
                    bDone = TRUE;
                    res = HXR_FAILED;
                    break;
                };
            }
            else if (poErr == 0)
            {
                bDone = TRUE;
            }
            else if (stNeedStartTime != m_state)
            {
                // This indicates that loss has occured.
                // Loss is only ok if we haven't received
                // the start time yet
                res = HXR_UNEXPECTED;
            }
        }
        
        if ((HXR_OK == res) || (HXR_NO_DATA == res))
        {
            HX_RESULT tmpRes = OnPageEnd(pPage);
            
            if (HXR_OK == tmpRes)
            {
                if (bHasHeader)
                {
                    CacheHeader(pPage);
                }
                else if (stReady != m_state)
                {
                    // We need to save the page for later
                    // processing
                    tmpRes =  EnqueuePage(pPage);
                }
                else
                {
                    // The state changed to stReady. We need
                    // to treat this as a timed page now.
                    tmpRes = HandleTimedPage(CurrentTime(), pPage);
                }
            }

            if (HXR_OK != tmpRes)
            {
                res = tmpRes;
            }
        }
    }

    return res;
}

void CBasePageToPacket::FlushQueues()
{
    while(!m_pageQueue.IsEmpty())
    {
        ogg_page* pPage = (ogg_page*)m_pageQueue.RemoveHead();

        OggUtil::DestroyPage(pPage);
    }

    while(!m_packetQueue.IsEmpty())
    {
        IHXPacket* pPkt = (IHXPacket*)m_packetQueue.RemoveHead();

        HX_RELEASE(pPkt);
    }
}

void CBasePageToPacket::FlushHeaders()
{
    while(!m_headerPages.IsEmpty())
    {
        ogg_page* pPage = (ogg_page*)m_headerPages.RemoveHead();

        OggUtil::DestroyPage(pPage);
    }

    m_bSentHeaders = FALSE;
}

IHXCommonClassFactory* CBasePageToPacket::GetCCF()
{
    IHXCommonClassFactory* pRet = m_pCCF;

    if (pRet)
    {
        pRet->AddRef();
    }

    return pRet;
}

void CBasePageToPacket::SetStartTime(ogg_int64_t startTime)
{
    DPRINTF(D_BASE_P2P, ("CBasePageToPacket::SetStartTime(%lld)\n",
                         startTime));

    m_startTime = startTime;
    m_bHasStartTime = TRUE;
}

void CBasePageToPacket::OnCurrentTimeValid()
{
    ogg_stream_reset(&m_os);

    ChangeState(stReady);

    // Process all queued pages
    while(!m_pageQueue.IsEmpty())
    {
        ogg_page* pPage = (ogg_page*)m_pageQueue.RemoveHead();
        
        HandleTimedPage(CurrentTime(), pPage);

        OggUtil::DestroyPage(pPage);
    }
}
