/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: ogg_page_reader.cpp,v 1.1.2.2 2004/07/09 01:58:26 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#include "hlxclib/stdlib.h"
#include "ogg_page_reader.h"

#include "ihxpckts.h" // IHXBuffer

#include "debug.h" // DPRINTF()
#define D_OGG_PAGE_READER 0 //0x4000000

const UINT32 DefaultReadSize = 8192;

COggPageReader::COggPageReader() :
    m_lRefCount(0)
                                 ,m_state(rsStart)
                                 ,m_pResponse(NULL)
                                 ,m_pFile(NULL)
                                 ,m_pStat(NULL)
                                 ,m_bSeekable(FALSE)
                                 ,m_ulFileSize(0)
                                 ,m_ulPendingSeekOffset(0)
                                 ,m_ulCurrentFileOffset(0)
                                 ,m_pOy(NULL)
{
#ifdef _DEBUG
    debug_level() |= D_OGG_PAGE_READER;
#endif

    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::COggPageReader()\n"));
}

COggPageReader::~COggPageReader()
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::~COggPageReader()\n"));

    Close();
}

HX_RESULT COggPageReader::Init(IHXOggPageReaderResponse* pResponse,
                               IHXFileObject* pFileObject)
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::Init()\n"));

    HX_RESULT res = HXR_FAILED;

    if (rsStart != m_state)
    {
        res = HXR_UNEXPECTED;
    }
    else if (pResponse && pFileObject)
    {
        HX_RELEASE(m_pResponse);

        m_pResponse = pResponse;
        m_pResponse->AddRef();
    
        HX_RELEASE(m_pFile);
        HX_RELEASE(m_pStat);
        m_ulFileSize = 0;

        m_pFile = pFileObject;
        m_pFile->AddRef();

        m_pFile->QueryInterface(IID_IHXFileStat, (void**)&m_pStat);


        m_pOy = (ogg_sync_state*) _ogg_malloc(sizeof(ogg_sync_state));

        if (m_pOy)
        {
            ogg_sync_init(m_pOy);
            ChangeState(rsInitPending);
            res = m_pFile->Init(HX_FILE_READ | HX_FILE_BINARY, this);
        }
        else
        {
            res = HXR_OUTOFMEMORY;
        }
    }

    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::Init() : res 0x%08x\n", res));
    return res;
}

HX_RESULT COggPageReader::Seek(ULONG32 ulFileOffset)
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::Seek(%lu)\n", ulFileOffset));
    HX_RESULT res = HXR_OK;

    if (m_bSeekable && (ulFileOffset < m_ulFileSize))
    {
        switch(m_state) {
        case rsReady:
            res = DoSeek(ulFileOffset);
            break;
        case rsSeekPending:
        case rsReadPending:
        case rsDispatchReadPageDone:
        case rsReadDuringDispatch:
            ChangeState(rsSeekDuringPending);
            /* fall through intentional */
        case rsSeekDuringPending:
            m_ulPendingSeekOffset = ulFileOffset;
            break;
        default:
            res = HXR_UNEXPECTED;
            break;
        };
    }
    else
    {
        res = HXR_UNEXPECTED;
    }

    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::Seek() : done %08x\n", res));

    return res;
}

HX_RESULT COggPageReader::ReadNextPage()
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::ReadNextPage()\n"));
    HX_RESULT res = HXR_UNEXPECTED;

    if (rsReady == m_state)
    {
        res = DoReadNextPage(FALSE);
    }
    else if (rsDispatchReadPageDone == m_state)
    {
        ChangeState(rsReadDuringDispatch);
        res = HXR_OK;
    }

    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::ReadNextPage() : done\n"));

    return res;
}

BOOL COggPageReader::IsSeekable() const
{
    return m_bSeekable;
}

ULONG32 COggPageReader::FileSize() const
{
    return m_ulFileSize;
}

HX_RESULT COggPageReader::Close()
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::Close()\n"));
    HX_RELEASE(m_pResponse);

    if (m_pFile)
    {
        m_pFile->Close();
    }
    HX_RELEASE(m_pFile);
    HX_RELEASE(m_pStat);

    if (m_pOy)
    {
        ogg_sync_destroy(m_pOy);
        m_pOy = NULL;
    }

    return HXR_OK;
}
/*
 *      IUnknown methods
 */
STDMETHODIMP COggPageReader::QueryInterface(THIS_ REFIID riid, void** ppvObj)
{
    HX_RESULT res = HXR_OK;

    if (IsEqualIID(riid, IID_IUnknown)) {
        *ppvObj = (IUnknown *)(IHXFileResponse *)this;
    } else if (IsEqualIID(riid, IID_IHXFileResponse)) {
        *ppvObj = (IHXFileResponse *)this;
    } else if (IsEqualIID(riid, IID_IHXFileStatResponse)) {
        *ppvObj = (IHXFileStatResponse *)this;
    }
    else
    {
        *ppvObj = NULL;
        res =  HXR_NOINTERFACE;
    }

    if (HXR_OK == res)
    {
        AddRef();
    }

    return res;
}

STDMETHODIMP_(ULONG32) COggPageReader::AddRef(THIS)
{    
    return InterlockedIncrement(&m_lRefCount);
}

STDMETHODIMP_(ULONG32) COggPageReader::Release(THIS)
{
    if (InterlockedDecrement(&m_lRefCount) > 0)
        return m_lRefCount;

    delete this;
    return 0;
}

/*
 *      IHXFileResponse methods
 */
STDMETHODIMP COggPageReader::InitDone(THIS_ HX_RESULT status)
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::InitDone(0x%08x)\n", status));

    BOOL bCallInitDone = TRUE;
    if (HXR_OK == status)
    {
        /* Only do a Stat() if we have a IHXFileStat object,
         * and the file will allow random access
         */
        HX_RESULT tmpRes = m_pFile->Advise(HX_FILEADVISE_RANDOMACCESS);
        if (m_pStat && (HXR_ADVISE_PREFER_LINEAR != tmpRes))
        {
            ChangeState(rsStatPending);
            status = m_pStat->Stat(this);
        }
        else
        {
            ChangeState(rsReady);

            if (m_pResponse)
            {
                m_pResponse->PageReaderInitDone(HXR_OK);
            }
        }
    }

    if (HXR_OK != status)
    {
        OnError(status);
    }

    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::InitDone() : done\n"));

    return HXR_OK;
}

STDMETHODIMP COggPageReader::CloseDone(THIS_ HX_RESULT status)
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::CloseDone(0x%08x)\n", 
                                status));
    return HXR_OK;
}

STDMETHODIMP COggPageReader::ReadDone(THIS_ HX_RESULT status, 
                                      IHXBuffer* pBuffer)
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::ReadDone(0x%08x, %u)\n", 
                                status, (pBuffer) ? pBuffer->GetSize() : 0));

    // Check for HXR_CANCELLED status. This can happen if a Seek() is
    // issued while a Read() is pending. This case only seems to happen
    // with HTTP
    if ((HXR_CANCELLED == status) &&
        (rsReadPending == m_state))
    {
        ChangeState(rsSeekPending);
        return HXR_OK;
    }

    if (HXR_OK == status)
    {
        switch(m_state) {
        case rsReadPending: 
        {
            /* Handle data */
            UINT32 ulBytes = pBuffer->GetSize();
            BOOL bEndOfFile = (pBuffer->GetSize() < DefaultReadSize);
            char* buffer = ogg_sync_buffer(m_pOy, ulBytes);
            memcpy(buffer, pBuffer->GetBuffer(), ulBytes);
            ogg_sync_wrote(m_pOy, ulBytes);
            
            ChangeState(rsReady);
            status = DoReadNextPage(bEndOfFile);

            if ((HXR_OK == status) && (rsSeekDuringPending == m_state))
            {
                status = DoSeek(m_ulPendingSeekOffset);
            }
        } break;
        case rsSeekDuringPending:
            status = DoSeek(m_ulPendingSeekOffset);
            break;
        default:
            status = HXR_UNEXPECTED;
        };
    }
    
    if (HXR_OK != status)
    {
        OnError(status);
    }

    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::ReadDone() : done\n"));

    return HXR_OK;
}

STDMETHODIMP COggPageReader::WriteDone(THIS_ HX_RESULT status)
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::WriteDone(0x%08x)\n", 
                                status));
    return HXR_UNEXPECTED;
}

STDMETHODIMP COggPageReader::SeekDone(THIS_ HX_RESULT status)
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::SeekDone(0x%08x)\n", status));

    if (HXR_OK == status)
    {
        switch(m_state) {
        case rsSeekPending:
            status = DoRead();
            break;
        case rsSeekDuringPending:
            status = DoSeek(m_ulPendingSeekOffset);
            break;
        default :
            status = HXR_UNEXPECTED;
            break;
        };
    }
    
    if (HXR_OK != status)
    {
        OnError(status);
    }

    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::SeekDone() : done\n"));
    return HXR_OK;
}

/*
 *      IHXFileStat methods
 */

STDMETHODIMP COggPageReader::StatDone(THIS_
                                      HX_RESULT status,
                                      UINT32 ulSize,
                                      UINT32 ulCreationTime,
                                      UINT32 ulAccessTime,
                                      UINT32 ulModificationTime,
                                      UINT32 ulMode)
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::StatDone(0x%08x, %lu)\n", 
                                status, ulSize));

    if (HXR_OK == status)
    {
        m_ulFileSize = ulSize;

        HX_RESULT tmpRes = m_pFile->Advise(HX_FILEADVISE_RANDOMACCESS);
        if (m_ulFileSize && (HXR_ADVISE_PREFER_LINEAR != tmpRes))
        {
            m_bSeekable = TRUE;
        }

        ChangeState(rsReady);
    }
    else
    {
        ChangeState(rsStart);
    }

    if (m_pResponse)
    {
        m_pResponse->PageReaderInitDone(status);
    }

    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::StatDone() : done\n"));

    return HXR_OK;
}

#ifdef _DEBUG
static const char* z_pStateNames[] = {
    "rsStart",
    "rsInitPending",
    "rsStatPending",
    "rsReady",
    "rsSeekPending",
    "rsSeekDuringPending",
    "rsReadPending",
    "rsDispatchReadPageDone",
    "rsReadDuringDispatch"
};
#endif

void COggPageReader::ChangeState(ReaderState newState)
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::ChangeState() : %s -> %s\n", 
                                z_pStateNames[m_state], 
                                z_pStateNames[newState]));
    m_state = newState;
}

HX_RESULT COggPageReader::DoReadNextPage(BOOL bEndOfFile)
{
    DPRINTF(D_OGG_PAGE_READER, 
            ("COggPageReader::DoReadNextPage(%u)\n", bEndOfFile));

    HX_RESULT res = HXR_FAILED;
    BOOL bDone = FALSE;
    
    while(!bDone)
    {
        ogg_page og;
        int err = ogg_sync_pageseek(m_pOy, &og);
        
        if (err < 0)
        {
            /* resyncing */
            DPRINTF(D_OGG_PAGE_READER, 
                    ("COggPageReader::DoReadNextPage() : resyncing %d skipped\n",
                     -err));

            m_ulCurrentFileOffset += -err;
        }
        else if (err == 0)
        {
            DPRINTF(D_OGG_PAGE_READER, 
                    ("COggPageReader::DoReadNextPage() : Need more data\n"));
            bDone = TRUE;

            if (!bEndOfFile)
            {
                // Read more data from the file
                res = DoRead();
            }
        }
        else
        {
            DPRINTF(D_OGG_PAGE_READER, 
                    ("COggPageReader::DoReadNextPage() : synced on %d byte page\n",
                     err));

            res = DispatchPage(m_ulCurrentFileOffset, err, &og);
            m_ulCurrentFileOffset += err;

            if (rsReadDuringDispatch != m_state)
            {
                bDone = TRUE;
            }
        }
    }

    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::DoReadNextPage() : done\n"));

    return res;
}

HX_RESULT COggPageReader::DoSeek(ULONG32 ulFileOffset)
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::DoSeek(%lu)\n", 
                                ulFileOffset));

    ChangeState(rsSeekPending);
    m_ulCurrentFileOffset = ulFileOffset;
    ogg_sync_reset(m_pOy);

    return m_pFile->Seek(ulFileOffset, FALSE);
}

HX_RESULT COggPageReader::DoRead()
{
    ChangeState(rsReadPending);
    return m_pFile->Read(DefaultReadSize);
}

void COggPageReader::OnError(HX_RESULT status)
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::OnError(0x%08x)\n", 
                                status));
    switch(m_state) {
    case rsStart:
    case rsInitPending:
    case rsStatPending:
        ChangeState(rsStart);
        if (m_pResponse)
        {
            m_pResponse->PageReaderInitDone(status);
        }
        break;
    case rsReady:
    case rsSeekPending :
    case rsSeekDuringPending:
    case rsReadPending :
    case rsDispatchReadPageDone:
    case rsReadDuringDispatch:
        ChangeState(rsReady);
        if (m_pResponse)
        {
            m_pResponse->ReadNextPageDone(status, 0, 0, NULL);
        }
        break;
    };
}

HX_RESULT COggPageReader::DispatchPage(ULONG32 ulFileOffset, 
                                       UINT32 ulPageSize, 
                                       ogg_page* pOg)
{
    DPRINTF(D_OGG_PAGE_READER, ("COggPageReader::DispatchPage(%lu, %u)\n", 
                                ulFileOffset, ulPageSize));

    ChangeState(rsDispatchReadPageDone);
    if (m_pResponse)
    {
        m_pResponse->ReadNextPageDone(HXR_OK, ulFileOffset, ulPageSize, pOg);
    }

    if (rsDispatchReadPageDone == m_state)
    {
        ChangeState(rsReady);
    }

    return HXR_OK;
}
