/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: theoravidfmt.cpp,v 1.10 2004/07/09 18:31:54 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */
 
#include "theoravidfmt.h"

#include "debug.h"
#define D_THEORA_VIDFMT 0 //0x4000000

CTheoraVideoFormat::CTheoraVideoFormat(IHXCommonClassFactory* pCCF, 
				       CVideoRenderer* pVidRend) :
    CVideoFormat(pCCF, pVidRend),
    m_state(stStart),
    m_ulFPSNum(0),
    m_ulFPSDenom(0),
    m_currentSerialNum(0),
    m_bNeedKeyframe(TRUE)
{
#ifdef _DEBUG
    debug_level() |= D_THEORA_VIDFMT;
#endif

    DPRINTF(D_THEORA_VIDFMT, ("CTheoraVideoFormat::CTheoraVideoFormat()\n"));

    ogg_sync_init(&m_oy);
    ogg_stream_init(&m_os, m_currentSerialNum);

    theora_info_init(&m_ti);
    theora_comment_init(&m_tc);
    memset(&m_ts, 0, sizeof(m_ts));

    m_lastFrameSize.cx = 0;
    m_lastFrameSize.cy = 0;
}

CTheoraVideoFormat::~CTheoraVideoFormat()
{
    DPRINTF(D_THEORA_VIDFMT, ("CTheoraVideoFormat::~CTheoraVideoFormat()\n"));

    ogg_sync_clear(&m_oy);
    ogg_stream_clear(&m_os);

    theora_clear(&m_ts);
    theora_info_clear(&m_ti);
    theora_comment_clear(&m_tc);

}

CMediaPacket* CTheoraVideoFormat::CreateAssembledPacket(IHXPacket* pCodecData)
{
    DPRINTF(D_THEORA_VIDFMT, ("CTheoraVideoFormat::CreateAssembledPacket(%u)\n",
			      pCodecData));

    CMediaPacket* pRet = NULL;

    if (pCodecData && !pCodecData->IsLost())
    {
	IHXBuffer* pInputBuffer = pCodecData->GetBuffer();

	// Copy data into the ogg_sync_state object
	char* buffer = ogg_sync_buffer(&m_oy, pInputBuffer->GetSize());
	memcpy(buffer, pInputBuffer->GetBuffer(), pInputBuffer->GetSize());
	ogg_sync_wrote(&m_oy, pInputBuffer->GetSize());
	
	HX_RELEASE(pInputBuffer);

	ogg_page og;

	UINT32 ulFrameCount = 0;

	while (1 == ogg_sync_pageout(&m_oy, &og))
	{
	    DPRINTF(D_THEORA_VIDFMT, ("CTheoraVideoFormat::CAP() : Got page\n"));
	    if (ogg_page_serialno(&og) != m_currentSerialNum)
	    {
		m_currentSerialNum = ogg_page_serialno(&og);
		ogg_stream_init(&m_os, m_currentSerialNum);
	    }
	    
	    if (ogg_stream_pagein(&m_os, &og) == 0)
	    {
		ogg_packet op;

		while(1 == ogg_stream_packetout(&m_os, &op))
		{
		    DPRINTF(D_THEORA_VIDFMT, ("CTheoraVideoFormat::CAP() : Got packet\n"));
		    IHXBuffer* pPktBuf = OggPktToIHXBuffer(&op);

		    if (op.b_o_s)
		    {
			HandleIdentHdr(&op);
		    }

		    if (pPktBuf)
		    {
			BOOL bCreatePkt = TRUE;

			UINT32 ulTimestamp = pCodecData->GetTime();
			
			if (!IsHeader(&op))
			{
			    ulTimestamp += CalcTimestamp(ulFrameCount);

			    if (m_bNeedKeyframe)
			    {
				if (IsKeyframe(&op))
				{
				    m_bNeedKeyframe = FALSE;
				}
				else
				{
				    bCreatePkt = FALSE;
				}
			    }
			}

			DPRINTF(D_THEORA_VIDFMT, 
				("CTheoraVideoFormat::CAP() : PKT_TS %u FRAME_TS %u\n", 
				 pCodecData->GetTime(),
				 ulTimestamp));
			
			if (bCreatePkt)
			{
			    CMediaPacket* pMediaPkt = 
				new CMediaPacket(pPktBuf,
						 pPktBuf->GetBuffer(),
						 pPktBuf->GetSize(),
						 pPktBuf->GetSize(),
						 ulTimestamp,
						 MDPCKT_USES_IHXBUFFER_FLAG,
						 (void*)1);

			    if (pMediaPkt)
			    {
				DPRINTF(D_THEORA_VIDFMT, ("CTheoraVideoFormat::CAP() : adding packet\n"));
				ReturnAssembledPacket(pMediaPkt);
				pMediaPkt = NULL;
			    }
			}
			
			HX_RELEASE(pPktBuf);
		    }

		    ulFrameCount++;
		}
	    }
	}
    }

    return pRet;
}

static
void CopyPixelData(REF(UINT8*) pDest, char* pSrc, int width, int height, 
		   int stride)
{
    for (int i = 0; i < height; i++)
    {
	memcpy(pDest, pSrc, width);
	pSrc += stride;
	pDest += width;
    }
}

static 
void DestroySampleDesc(void* pSampleDesc, void* pUserData)
{
    HXxSize* pDims = (HXxSize*)pSampleDesc;
    delete pDims;
}

CMediaPacket* 
CTheoraVideoFormat::CreateDecodedPacket(CMediaPacket* pCodedPacket)
{
    DPRINTF(D_THEORA_VIDFMT, ("CTheoraVideoFormat::CreateDecodedPacket(%p)\n",
			      pCodedPacket));

    CMediaPacket* pRet = NULL;
    
    if (pCodedPacket && pCodedPacket->m_pData)
    {
	yuv_buffer yuv;
	ogg_packet* pOp = (ogg_packet*)pCodedPacket->m_pData;

	DPRINTF(D_THEORA_VIDFMT, 
		("CTheoraVideoFormat::CreateDecodedPacket() : bytes %u\n",
		 pOp->bytes));
    
	int err = 0;

	if (pOp->b_o_s)
	{
            // Clear out old state
            theora_clear(&m_ts);
            theora_info_clear(&m_ti);
            theora_comment_clear(&m_tc);

            // Prep for new state
            theora_info_init(&m_ti);
            theora_comment_init(&m_tc);
            memset(&m_ts, 0, sizeof(m_ts));

	    ChangeState(stNeedIdent);
	}
    
	switch(m_state) {
	case stNeedIdent:
	    if ( 0 == theora_decode_header(&m_ti, &m_tc, pOp))
	    {
		ChangeState(stNeedComments);
	    }
	    break;
	case stNeedComments:
	    if ( 0 == theora_decode_header(&m_ti, &m_tc, pOp))
	    {
		ChangeState(stNeedCodebooks);
	    }
	    break;
	case stNeedCodebooks:
	    if ( 0 == theora_decode_header(&m_ti, &m_tc, pOp))
	    {
		theora_decode_init(&m_ts, &m_ti);
		ChangeState(stReady);
	    }
	    break;
	case stReady: {
	    theora_decode_packetin(&m_ts, pOp);
	    
	    err = theora_decode_YUVout(&m_ts, &yuv);
	    
	    if (0 == err)
	    {
		pRet = CreateYUVPacket(yuv, pCodedPacket->m_ulTime);
	    }

	    DPRINTF(D_THEORA_VIDFMT, ("YUVout err %d\n", err));
	    DPRINTF(D_THEORA_VIDFMT, ("YUVout y_width %d\n", yuv.y_width));
	    DPRINTF(D_THEORA_VIDFMT, ("YUVout y_height %d\n", yuv.y_height));
	    DPRINTF(D_THEORA_VIDFMT, ("YUVout y_stride %d\n", yuv.y_stride));
	    DPRINTF(D_THEORA_VIDFMT, ("YUVout uv_width %d\n", yuv.uv_width));
	    DPRINTF(D_THEORA_VIDFMT, ("YUVout uv_height %d\n", yuv.uv_height));
	    DPRINTF(D_THEORA_VIDFMT, ("YUVout uv_stride %d\n", yuv.uv_stride));
	} break;
	};
    }

    if (pCodedPacket != NULL)
    {
	pCodedPacket->Clear();
	delete pCodedPacket;
    }

    DPRINTF(D_THEORA_VIDFMT, 
	    ("CTheoraVideoFormat::CreateDecodedPacket() : pRet %p\n", pRet));

    return pRet;
}

HX_RESULT 
CTheoraVideoFormat::InitBitmapInfoHeader(HXBitmapInfoHeader &BitmapInfoHeader,
					 CMediaPacket* pVideoPacket)
{
    HX_RESULT res = HXR_FAILED;

    if (pVideoPacket && pVideoPacket->m_pSampleDesc)
    {
	HXxSize* pDims = (HXxSize*)pVideoPacket->m_pSampleDesc;
	BitmapInfoHeader.biWidth = pDims->cx;
	BitmapInfoHeader.biHeight = pDims->cy;
	BitmapInfoHeader.biSizeImage = (BitmapInfoHeader.biWidth * 
					BitmapInfoHeader.biHeight * 
					BitmapInfoHeader.biBitCount / 8);
	res = HXR_OK;
    }
    return res;
}

void CTheoraVideoFormat::Reset()
{
    DPRINTF(D_THEORA_VIDFMT, ("CTheoraVideoFormat::Reset()\n"));
    m_bNeedKeyframe = TRUE;

    m_lastFrameSize.cx = 0;
    m_lastFrameSize.cy = 0;

    ogg_stream_reset(&m_os);
    ogg_sync_reset(&m_oy);

    CVideoFormat::Reset();
}

IHXBuffer* CTheoraVideoFormat::OggPktToIHXBuffer(ogg_packet* pOp)
{
    UINT32 ulDataSize = sizeof(ogg_packet) + pOp->bytes;
    IHXBuffer* pRet = CreateBuffer(ulDataSize);

    if (pRet)
    {
	ogg_packet* pOggPkt = (ogg_packet*)pRet->GetBuffer();
	UINT8* pPktData = pRet->GetBuffer() + sizeof(ogg_packet);

	// Copy ogg_packet struct
	*pOggPkt = *pOp;
	pOggPkt->packet = pPktData;
	
	// Copy packet data
	memcpy(pOggPkt->packet, pOp->packet, pOp->bytes);
    }

    return pRet;
}

CMediaPacket* CTheoraVideoFormat::CreateYUVPacket(yuv_buffer& yuv, 
						  UINT32 ulTimestamp)
{
    UINT32 ulLumaSize = yuv.y_width * yuv.y_height;
    UINT32 ulChromaSize = yuv.uv_width * yuv.uv_height;
    UINT32 ulFrameSize = ulLumaSize + 2 * ulChromaSize;

    CMediaPacket* pRet = NULL;

    if (ulFrameSize > 0)
    {
	HXxSize* pDims = NULL;

	if ((m_lastFrameSize.cx != yuv.y_width) ||
	    (m_lastFrameSize.cy != yuv.y_height))
	{
	    pDims = new HXxSize;

	    if (pDims)
	    {
		DPRINTF(D_THEORA_VIDFMT, ("CreateYUVPacket() : Setting frame size\n"));

		m_lastFrameSize.cx = yuv.y_width;
		m_lastFrameSize.cy = yuv.y_height;
		
		*pDims = m_lastFrameSize;
	    }
	}

	pRet = (CMediaPacket*)m_pFramePool->Get(ulFrameSize);
	UINT8* pData = NULL;

	if (pRet)
	{
	    pData = pRet->m_pData;

	    pRet->Init(pData, ulFrameSize, ulTimestamp, 0, pDims);
	}
	else
	{
	    pData = new UCHAR[ulFrameSize];

	    if (pData)
	    {
		pRet = new CMediaPacket(pData, 
					pData,
					ulFrameSize,
					ulFrameSize,
					ulTimestamp,
					0,
					pDims /* desc */);
		if (!pRet)
		{
		    HX_VECTOR_DELETE(pData);
		}
	    }
	}
	
	if (pRet)
	{
	    if (pDims)
	    {
		pRet->SetSampleDescKiller(&DestroySampleDesc);
	    }

	    // Copy Luma plane
	    CopyPixelData(pData, yuv.y, yuv.y_width, yuv.y_height, 
			  yuv.y_stride);
	    
	    // Copy first chroma plane
	    CopyPixelData(pData, yuv.u, yuv.uv_width, yuv.uv_height, 
			  yuv.uv_stride);
	    
	    // Copy second chroma plane
	    CopyPixelData(pData, yuv.v, yuv.uv_width, yuv.uv_height, 
			  yuv.uv_stride);

	}
	else
	{
	    HX_DELETE(pDims);
	}
    }

    return pRet;
}

IHXBuffer* CTheoraVideoFormat::CreateBuffer(UINT32 ulSize)
{
    IHXBuffer* pRet = NULL;
    
    if (m_pCommonClassFactory &&
	(HXR_OK == m_pCommonClassFactory->CreateInstance(IID_IHXBuffer,
							 (void**)&pRet)) &&
	(HXR_OK != pRet->SetSize(ulSize)))
    {
	HX_RELEASE(pRet);
    }

    return pRet;
}

BOOL CTheoraVideoFormat::IsHeader(ogg_packet* pOp) const
{
    BOOL bRet = FALSE;

    if (pOp && pOp->packet && (pOp->bytes >= 7) &&
	((pOp->packet[0] & 0x80) == 0x80) &&
	(!memcmp(pOp->packet + 1, "theora", 6)))
    {
	bRet = TRUE;
    }

    return bRet;
}

BOOL CTheoraVideoFormat::IsKeyframe(ogg_packet* pOp) const
{
    BOOL bRet = FALSE;

    if (pOp && pOp->packet && ((pOp->packet[0] & 0xc0) == 0))
    {
	bRet = TRUE;
    }

    return bRet;
}

static 
UINT32 gcd(UINT32 a, UINT32 b)
{
    // Compute the greatest common denominator
    while(b != 0)
    {
        UINT32 tmp = b;
        b = a % b;
        a = tmp;
    }

    return a;
}

void CTheoraVideoFormat::HandleIdentHdr(ogg_packet* pOp)
{
    if (IsHeader(pOp) && (pOp->bytes >= 42) && (pOp->packet[0] == 0x80))
    {
	const unsigned char* pIdent = (const unsigned char*)pOp->packet;
	
	// Ident Header
	m_ulFPSNum = ((pIdent[0x16] << 24) | (pIdent[0x17] << 16) |
		      (pIdent[0x18] << 8) | (pIdent[0x19]));
	m_ulFPSDenom = ((pIdent[0x1a] << 24) | (pIdent[0x1b] << 16) |
			(pIdent[0x1c] << 8) | (pIdent[0x1d]));

        // Attempt to reduce the numerator and denominator
        // by finding the greatest common denominator.
        UINT32 tmp = gcd(m_ulFPSNum, m_ulFPSDenom);
        if (tmp > 1)
        {
            m_ulFPSNum /= tmp;
            m_ulFPSDenom /= tmp;
        }
    }
}

UINT32 CTheoraVideoFormat::CalcTimestamp(UINT32 ulFrame) const
{
    UINT32 uRet = 0;

    if (m_ulFPSNum)
    {
        ogg_int64_t q = ulFrame / m_ulFPSNum;
        ogg_int64_t r = ulFrame - (q * m_ulFPSNum);
        ogg_int64_t mult = ((ogg_int64_t)m_ulFPSDenom) * 1000;
        ogg_int64_t ts = ((q * mult) + 
                          ((r * mult) / m_ulFPSNum));
        uRet = INT64_TO_UINT32(ts);
    }

    return uRet;
}

void CTheoraVideoFormat::ChangeState(State newState)
{
    DPRINTF(D_THEORA_VIDFMT, ("CTheoraVideoFormat::ChangeState() : %d -> %d\n",
			      m_state, newState));
    m_state = newState;
}
