#!/bin/sh

# script arguments
INSTDIR=$1
LOGFILE=$2


# no instdir passed in? guess it
if [ -z $INSTDIR ] ; then
    if [ -f ../hxplay.bin -o -f ../realplay.bin ] ; then
	INSTDIR=`( cd ..; pwd )`
    elif [ -f hxplay.bin -o -f realplay.bin ] ; then
	INSTDIR=`pwd`
    else
	echo "cannot find the installation directory--aborting setup!"
	exit 1
    fi
fi


# for which target are we installing?
if [ -f $INSTDIR/realplay ] ; then
    TARGET=realplay
elif [ -f $INSTDIR/hxplay ] ; then
    TARGET=hxplay
else
    echo "cannot find either hxplay or realplay--aborting setup!"
    exit 2
fi


# make sure we know exactly where our logfile is
if [ -n "$LOGFILE" ] ; then
    if [ -f $INSTDIR/$LOGFILE ]; then
	LOGFILE=$INSTDIR/$LOGFILE
    fi
fi


# for logging our commands
eval_cmd()
{
    eval $*
    if [ -n "$LOGFILE" ] ; then
	echo $* >> $LOGFILE
    fi
}


# LOGNAME doesn't always reflect whether you're root or not, but not all
# systems have "whoami" (note: we're not exporting LOGNAME, just changing it
# for the scope of this script.)
if [ -n `which whoami` ] ; then
    LOGNAME=`whoami`
fi

# do root processing
if [ $LOGNAME = "root" ] ; then
    # find system plugins directory
    MOZFOUND=0
    for MOZDIR in "/usr/lib/mozilla/plugins" "/usr/lib/browser-plugins" "/opt/mozilla/lib/plugins" "/usr/lib/firefox/plugins" "/usr/lib/mozilla-firefox/plugins" "/usr/lib/netscape/plugins" "/opt/mozilla/plugins" "/opt/netscape/plugins"; do
	if [ -d "$MOZDIR" ] ; then
	    eval_cmd ln -sf "$INSTDIR/mozilla/nphelix.so" "$MOZDIR"
	    eval_cmd ln -sf "$INSTDIR/mozilla/nphelix.xpt" "$MOZDIR"
	    MOZFOUND=1
	fi
    done
    unset MOZDIR

    if [ $MOZFOUND -eq 0 ] ; then
	echo "No Mozilla plugin directory found."
	echo "Please enter Mozilla's plugins directory to install the Mozilla plug-in,"
	echo "or leave blank to skip plug-in installation: "
	read MOZDIR
    fi

    # setup our symlinks -- use /usr as default, as mozilla and nautlius
    # both rely on realplay being in the path, and if X is started via gdm,
    # /usr/local/bin might not be.
    DEFPREFIX=/usr

    # where icons go
    # http://freedesktop.org/Standards/icon-theme-spec/0.7/ar01s07.html
    ICONPATH=/usr/share/icons/hicolor

    # where pixmaps go
    PIXMAPPATH=/usr/share/pixmaps

    # where our desktop file goes
    for DESKTOPPATH in "/usr/share/applications" "/opt/gnome/share/applications" ; do
	if [ -d "$DESKTOPPATH" ] ; then
	    break
	fi
    done

    # where our applications file goes
    for APPSPATH in "/usr/share/application-registry" "/opt/gnome/share/application-registry" ; do
	if [ -d "$APPSPATH" ] ; then
	    break
	fi
    done

    # where our mo files go
    LOCALEPATH=/usr/share/locale

    echo -n "configure system-wide symbolic links? [Y/n]: "
    read DOSYMLINKS

    if [ -z "$DOSYMLINKS" -o "$DOSYMLINKS" = "y" -o "$DOSYMLINKS" = "Y" ] ; then
	echo -n "enter the prefix for symbolic links [$DEFPREFIX]: "
	read PREFIX

	test -z "$PREFIX" && PREFIX=$DEFPREFIX

	echo "Setting up $TARGET symlinks in $PREFIX..."
	test -L $PREFIX/share/$TARGET && eval_cmd rm -f $PREFIX/share/$TARGET
	eval_cmd ln -s $INSTDIR/share $PREFIX/share/$TARGET
	test -L $PREFIX/bin/$TARGET && eval_cmd rm -f $PREFIX/bin/$TARGET
	eval_cmd ln -s $INSTDIR/$TARGET $PREFIX/bin/$TARGET
	test -L $PREFIX/lib/$TARGET && eval_cmd rm -f $PREFIX/lib/$TARGET
	eval_cmd ln -s $INSTDIR/lib $PREFIX/lib/$TARGET
    fi

    # try to find GNOME mime folder
    for GNOMEMIME in "/usr/share/mime-info" "/opt/gnome/share/mime-info" ; do
	if [ -d "$GNOMEMIME" ] ; then
	    break
	fi
    done

else
    # find user's plugins directory
    if [ -n "$HOME" -a -d "$HOME" ] ; then
	MOZDIR=$HOME/.mozilla/plugins
    elif [ -n "$LOGNAME" ] ; then
	if [ -d "/home/$LOGNAME" ] ; then
	    MOZDIR=/home/$LOGNAME/.mozilla/plugins
	elif [ -d "/export/home/$LOGNAME" ] ; then
	    MOZDIR=/export/home/$LOGNAME/.mozilla/plugins
	fi
    fi

    if [ -n "$MOZDIR" ] ; then
	# make sure it exists
	if [ ! -d $MOZDIR ] ; then
	    eval_cmd mkdir -p $MOZDIR
	fi
    else
	echo "Cannot find \$HOME.  Set your \$HOME environment variable and re-run"
	echo "postinst/postinst.sh to install the mozilla plugin.  Or consult the README"
	echo "for instructions on how to manually install it.  If you think you've"
	echo "encountered this in error, please file a bug report at:"
	echo "http://helixcommunity.org:8080/enter_bug.cgi?product=player"
    fi

    # try to find the user GNOME mime folder
    for MIME in "$HOME/.gnome/mime-info" ; do
	if [ -d $MIME ] ; then
	    GNOMEMIME=$MIME
	    break
	fi
    done
fi

# configure mozilla
if [ -n "$MOZDIR" -a -d "$MOZDIR" ] ; then
    echo "configuring mozilla..."
    eval_cmd rm -f $MOZDIR/nphelix.so
    eval_cmd rm -f $MOZDIR/nphelix.xpt
    eval_cmd ln -s $INSTDIR/mozilla/nphelix.so $MOZDIR/nphelix.so
    eval_cmd ln -s $INSTDIR/mozilla/nphelix.xpt $MOZDIR/nphelix.xpt
fi

# configure icons
if [ -n "$ICONPATH" ] ; then
    echo "configuring icons..."
    for SIZE in "16x16" "32x32" "48x48" "192x192" ; do
	ICON="$INSTDIR/share/icons/${TARGET}_$SIZE.png"
	if [ -f "$ICON" ] ; then
	    if [ -d "$ICONPATH" ] ; then
		eval_cmd mkdir -p "$ICONPATH/$SIZE/apps"
		eval_cmd cp "$ICON" "$ICONPATH/$SIZE/apps/$TARGET.png"
		eval_cmd chmod 644 "$ICONPATH/$SIZE/apps/$TARGET.png"
	    fi
	fi
    done
fi

# configure document icons
if [ -n "$ICONPATH" ] ; then
    echo "configuring document icons..."
    for SIZE in "16x16" "32x32" "48x48" "192x192" ; do
        for MIME in \
            "application-generic" \
            "audio-generic"       \
            "text-realtext"       \
            "video-generic"       \
            "video-ogg"           \
            "application-ogg"     \
            "application-ram"     \
            "application-rpm"     \
            "application-smil"    \
            "application-rm"      \
            "audio-aiff"          \
            "audio-au"            \
            "audio-mp3"           \
            "audio-mp4"           \
            "audio-ogg"           \
            "audio-ra"            \
            "audio-wav"           \
            "video-avi"           \
            "video-mov"           \
            "video-rv"            \
            "video-swf" ; do
        if [ -f $INSTDIR/share/icons/mime-${MIME}_$SIZE.png ] ; then
        	ICON="$INSTDIR/share/icons/mime-${MIME}_$SIZE.png"
  		    eval_cmd mkdir -p "$ICONPATH/$SIZE/mimetypes"
	        eval_cmd rm -f "$ICONPATH/$SIZE/mimetypes/${TARGET}-${MIME}.png"
	        eval_cmd ln -s "$ICON" "$ICONPATH/$SIZE/mimetypes/${TARGET}-${MIME}.png"
	    fi
        done
    done
fi

# configure pixmaps
if [ -n "$PIXMAPPATH" -a -d "$PIXMAPPATH" ] ; then
    echo "configuring pixmaps..."
    eval_cmd rm -f $PIXMAPPATH/$TARGET.png
    eval_cmd ln -s "$INSTDIR/share/$TARGET.png" \
                   "$PIXMAPPATH/$TARGET.png"
fi

# add .mo files
if [ -n "$LOCALEPATH" -a -d "$LOCALEPATH" ] ; then
    echo "configuring locale..."
    for LANGUAGE in \
            "de" \
            "es" \
            "fr" \
            "it" \
            "ja" \
            "ko" \
            "pt_BR" \
            "zh_CN" \
            "zh_TW" ; do
        if [ -f $INSTDIR/share/locale/$LANGUAGE/player.mo ] ; then
            eval_cmd rm -f "$LOCALEPATH/$LANGUAGE/LC_MESSAGES/$TARGET.mo"
            eval_cmd install -m 644 "$INSTDIR/share/locale/$LANGUAGE/player.mo" "$LOCALEPATH/$LANGUAGE/LC_MESSAGES/$TARGET.mo" 
        fi
        if [ -f $INSTDIR/share/locale/$LANGUAGE/widget.mo ] ; then
            eval_cmd rm -f "$LOCALEPATH/$LANGUAGE/LC_MESSAGES/libgtkhx.mo"
            eval_cmd install -m 644 "$INSTDIR/share/locale/$LANGUAGE/widget.mo" "$LOCALEPATH/$LANGUAGE/LC_MESSAGES/libgtkhx.mo" 
        fi
    done
fi

# configure desktop
if [ -n "$DESKTOPPATH" -a -d "$DESKTOPPATH" ] ; then
    echo "configuring desktop..."
    eval_cmd rm -f "$DESKTOPPATH/$TARGET".desktop
    eval_cmd cp "$INSTDIR/share/$TARGET".desktop $DESKTOPPATH
    eval_cmd chmod 644 "$DESKTOPPATH/$TARGET".desktop
fi

# install applications file
if [ -n "$APPSPATH" -a -d "$APPSPATH" ] ; then
    echo "configuring applications..."
    eval_cmd rm -f "$APPSPATH/$TARGET".applications
    eval_cmd cp "$INSTDIR/share/$TARGET".applications $APPSPATH
    eval_cmd chmod 644 "$APPSPATH/$TARGET".applications
fi

# configure GNOME
if [ -n "$GNOMEMIME" -a -d "$GNOMEMIME" ] ; then
    echo "configuring GNOME mime types..."
    eval_cmd cp "$INSTDIR/share/$TARGET".keys $GNOMEMIME
    eval_cmd chmod 644 "$GNOMEMIME/$TARGET".keys
    eval_cmd cp "$INSTDIR/share/$TARGET".mime $GNOMEMIME
    eval_cmd chmod 644 "$GNOMEMIME/$TARGET".mime
fi

# configure the player startup script
if [ -x $INSTDIR/postinst/confscript.sh ] ; then
    $INSTDIR/postinst/confscript.sh $INSTDIR $LOGFILE
fi


# platform-specific scripts
if [ `uname` = "SunOS" -a -x $INSTDIR/postinst/solarisinst.sh ] ; then
    $INSTDIR/postinst/solarisinst.sh $INSTDIR $LOGFILE
fi



