/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: atocolor.cpp,v 1.1.2.1 2004/07/09 01:50:20 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

/////////////////////////////////////////////////////////////////////////////
//
//  ATOCOLOR.CPP
//
//
//  This file contains the functions that are used in RealText rendering to
//  convert a string of characters into a color (which in Windows is a
//  COLORTYPE)
//
//  (1)	BOOL convertColorNameToCOLORTYPE(
//		_CHAR* pColorNm,
//		ULONG32 colorNmLen, 
//		COLORTYPE& colorVal);
//  //  Converts a string, like "Darkblue", into a COLORTYPE value, like
//  //  0x00990000 (which is RGB(0,0,153) in Windows), and returns FALSE,
//  //  with colorVal undefined if pColorNm contains an unrecognized
//  //	color.  colorNmLen is equivalent of strlen(pColorNm), i.e.,
//  //	it does not take into account the terminating '\0' char.
//
//  (2) BOOL convertColorValStringToCOLORTYPE(
//		_CHAR* pColorVal,
//		ULONG32 colorValBufLen, 
//		COLORTYPE& colorVal);
//  //Converts a string like "#F71C00" to a COLORTYPE value, like
//  //  0x00001CF7 (which is RGB(F7,1C,00) in Windows), and returns
//  //  FALSE, with colorVal undefined if colorValBuf contains an
//  //  invalid value.  The leading '#' character is not necessary
//  //  because all values are assumed to be in Hexidecimal.
//	
//

#include "hxtypes.h"
#include "hxassert.h"
#include "rt_types.h" //for _CHAR, RED_GREEN_OR_BLUE, COLORTYPE

#include "rt_string.h" //for stringCompare().

#include "atocolor.h"

#include <stdlib.h>
#include <ctype.h>

#include "hxheap.h"
#ifdef _DEBUG
#undef HX_THIS_FILE		
static char HX_THIS_FILE[] = __FILE__;
#endif



COLORTYPE convertRGBtoCOLORTYPE(
	RED_GREEN_OR_BLUE red,
	RED_GREEN_OR_BLUE green,
	RED_GREEN_OR_BLUE blue)
{
    return (COLORTYPE)(
	    blue |
	    (COLORTYPE(green)<<RED_GREEN_OR_BLUE_SIZE_IN_BITS) | 
	    (COLORTYPE(red)<<(RED_GREEN_OR_BLUE_SIZE_IN_BITS<<1)));
}


/////////////////////////////////////////////////////////////////////////////
//
//  Converts a string, like "Darkblue", into a COLORTYPE value, like
//  0x00990000 (which is RGB(0,0,153) in Windows), and returns FALSE,
//  with colorVal undefined if pColorNm contains an unrecognized
//	color.  colorNmLen is equivalent of strlen(pColorNm), i.e.,
//	it does not take into account the terminating '\0' char.
//
BOOL convertColorNameToCOLORTYPE(_CHAR* pColorNm, ULONG32 colorNmLen, 
	COLORTYPE& colorVal)
{
    _CHAR* pColorName = pColorNm;
    ULONG32 colorNameLen = colorNmLen;
    _CHAR savedEndChar = '\0';

    HX_ASSERT_VALID_PTR(pColorName);
    HX_ASSERT(MIN_COLORNAMELENGTH >= 2);
    if(!pColorName  ||  colorNameLen < MIN_COLORNAMELENGTH)
    {
	return FALSE;
    }

    savedEndChar = pColorName[colorNmLen-1];

    if('\"' == pColorName[0])
    {
	if('\"' == pColorName[colorNameLen-1])
	{
	    pColorName[colorNameLen-1] = '\0'; //get rid of end '\"'.
	    colorNameLen--;
	}
	pColorName++;
	colorNameLen--;
    }

    for(ULONG32 i=0; i< colorNameLen; i++)
    {
	_CHAR* pCh = &pColorName[i];
	*pCh = toupper(*pCh);
	if(*pCh < 'A'  ||  *pCh > 'Z')
	{
	    //Make sure pColorNm is in origninal state:
	    pColorNm[colorNmLen-1] = savedEndChar;
	    return FALSE;
	}
    }
    switch(pColorName[0])
    {
	case 'A':
	    if(!stringCompare(pColorName, colorNameLen,
			    "AQUA", 4))
	    {
		//NOTE: same as "cyan":
		colorVal = convertRGBtoCOLORTYPE(0,255,255);
	    }
	    break;
	case 'B':
	    if(!stringCompare(pColorName, colorNameLen,
			    "BLACK", 5))
	    {
		colorVal = convertRGBtoCOLORTYPE(0,0,0);
	    }
	    else if(!stringCompare(pColorName, colorNameLen,
			    "BROWN", 5))
	    {
		colorVal = convertRGBtoCOLORTYPE(153,0,0);
	    }
	    else if(!stringCompare(pColorName, colorNameLen,
			    "BLUE", 4))
	    {
		colorVal = convertRGBtoCOLORTYPE(0,0,255);
	    }
	    break;

	case 'C':
	    if(!stringCompare(pColorName, colorNameLen,
			    "CYAN", 4))
	    {
		//NOTE: same as "aqua":
		colorVal = convertRGBtoCOLORTYPE(0,255,255);
	    }
	    break;

	case 'D':
	    if(!stringCompare(pColorName, colorNameLen,
			    "DARKBLUE", 8))
	    {
		colorVal = convertRGBtoCOLORTYPE(0,0,153);
	    }
	    break;

	case 'F':
	    if(!stringCompare(pColorName, colorNameLen,
			    "FUCHSIA", 7))
	    {
		//NOTE: same as "magenta":
		colorVal = convertRGBtoCOLORTYPE(255,0,255);
	    }
	    break;

	case 'G':
	    if(!stringCompare(pColorName, colorNameLen,
			    "GRAY", 4))
	    {
		colorVal = convertRGBtoCOLORTYPE(128,128,128);
	    }
	    else if(!stringCompare(pColorName, colorNameLen,
			    "GREEN", 5))
	    {
		colorVal = convertRGBtoCOLORTYPE(0,128,0);
	    }
	    break;

	case 'L':
	    if(!stringCompare(pColorName, colorNameLen,
			    "LIGHTBLUE", 9))
	    {
		colorVal = convertRGBtoCOLORTYPE(204,255,255);
	    }
	    else if(!stringCompare(pColorName, colorNameLen,
			    "LIGHTGREEN", 10)  ||
		    !stringCompare(pColorName, colorNameLen,
			    "LIME", 4)
			    )
	    {
		colorVal = convertRGBtoCOLORTYPE(0,255,0);
	    }
	    break;

	case 'M':
	    if(!stringCompare(pColorName, colorNameLen,
			    "MAGENTA", 7))
	    {
		//NOTE: same as "fuchsia":
		colorVal = convertRGBtoCOLORTYPE(255,0,255);
	    }
	    else if(!stringCompare(pColorName, colorNameLen,
			    "MAROON", 6))
	    {
		colorVal = convertRGBtoCOLORTYPE(128,0,0);
	    }
	    break;

	case 'N':
	    if(!stringCompare(pColorName, colorNameLen,
			    "NAVY", 4))
	    {
		colorVal = convertRGBtoCOLORTYPE(0,0,128);
	    }
	    break;

	case 'O':
	    if(!stringCompare(pColorName, colorNameLen,
			    "OLIVE", 5))
	    {
		colorVal = convertRGBtoCOLORTYPE(128,128,0);
	    }
	    break;

	case 'P':
	    if(!stringCompare(pColorName, colorNameLen,
			    "PURPLE", 6))
	    {
		colorVal = convertRGBtoCOLORTYPE(128,0,128);
	    }
	    break;

	case 'R':
	    if(!stringCompare(pColorName, colorNameLen,
			    "RED", 3))
	    {
		colorVal = convertRGBtoCOLORTYPE(255,0,0);
	    }
	    break;

	case 'S':
	    if(!stringCompare(pColorName, colorNameLen,
			    "SILVER", 6))
	    {
		colorVal = convertRGBtoCOLORTYPE(192,192,192);
	    }
	    break;

	case 'T':
	    if(!stringCompare(pColorName, colorNameLen,
			    "TEAL", 4))
	    {
		colorVal = convertRGBtoCOLORTYPE(0,128,128);
	    }
	    else if(!stringCompare(pColorName, colorNameLen,
			    "TRANSPARENT", 11))
	    {
		colorVal = TRANSPARENT_COLOR;
	    }
	    break;

	case 'W':
	    if(!stringCompare(pColorName, colorNameLen,
			    "WHITE", 5))
	    {
		colorVal = convertRGBtoCOLORTYPE(255,255,255);
	    }
	    break;

	case 'Y':
	    if(!stringCompare(pColorName, colorNameLen,
			    "YELLOW", 6))
	    {
		colorVal = convertRGBtoCOLORTYPE(255,255,0);
	    }
	    break;

	default:
	    //Restore the orignial pColorNm:
	    pColorNm[colorNmLen-1] = savedEndChar;
	    return FALSE; //color name was not recognized.
    }
    //Restore the orignial pColorNm, in case the end quote char was removed:
    pColorNm[colorNmLen-1] = savedEndChar;
    return TRUE;
} //end convertColorNameToCOLORTYPE() function body.



/////////////////////////////////////////////////////////////////////////////
//
//  Converts a string like "#F71C00" to a COLORTYPE value, like
//  0x00001CF7 (which is RGB(F7,1C,00) in Windows), and returns
//  FALSE, with colorVal undefined if colorValBuf contains an
//  invalid value.  The leading '#' character is not necessary
//  because all values are assumed to be in Hexidecimal.
//
BOOL convertColorValStringToCOLORTYPE(
	_CHAR* pColorValBuf, 
	ULONG32 colorValBufLen, 
	COLORTYPE& colorVal)
{
	//Fixed a bug where I was checking "if(!colorVal ...)" instead of
	// "if(!pColorValBuf ...)" and, every once in a while when colorVal
	// just happened to be 0, this function would fail when it shouldn't:
	HX_ASSERT(pColorValBuf != NULL  &&  colorValBufLen>0L);
	if(!pColorValBuf  ||  colorValBufLen < 1L)
	{
	    return FALSE;
	}

	ULONG32 startIndx = 0L;
	/*  Only convert up to this character because there are a
	*  max of 6 chars in a hex number describing a color: */
	ULONG32 maxIndx = 5L;	

	ULONG32 bgrVal = 0L;

	if('\"' == pColorValBuf[0])
	{
	    startIndx++;
	    maxIndx++;
	    if(colorValBufLen<startIndx+1)
		    return FALSE;
	}

	if('#' == pColorValBuf[startIndx])
	{
	    startIndx++;
	    maxIndx++;
	    if(colorValBufLen<startIndx+1)
		    return FALSE;
	}

	_CHAR ch;
	while(startIndx<colorValBufLen  &&  startIndx<=maxIndx)
	{
	    ch = pColorValBuf[startIndx];
	    if(ch>='0'  &&  ch<='9') //convert from 'n' to n:
	    {
		ch -= '0';
	    }
	    else if(ch>='A'  &&  ch<='F')
	    {
		ch = ch - 'A' + 10;
	    }
	    else if(ch>='a'  &&  ch<='f')
	    {
		ch = ch - 'a' + 10;
	    }
	    else if(ch=='\"')
	    {
		break;
	    }
	    else
	    {
		return FALSE; //invalid char in what should be a hex number.
	    }

	    bgrVal <<= 4;//4 bits/character: ASCII representation of hex val.
	    bgrVal |= (RED_GREEN_OR_BLUE)ch;
	    startIndx++;
	}

	colorVal = 
		convertRGBtoCOLORTYPE(
		(RED_GREEN_OR_BLUE)(bgrVal>>16), //red is bytes 4,5
		(RED_GREEN_OR_BLUE)(bgrVal>>8), //green is 2,3
		(RED_GREEN_OR_BLUE)bgrVal ); //and blue is bytes 0,1

	return TRUE;
}



