/***************************************************************************
                          main.cpp  -  description
                             -------------------
    begin                : Die Apr 23 22:16:35 CEST 2002
    copyright            : (C) 2002-2004 by Andr Simon
    email                : andre.simon1@gmx.de


   Highlight is a universal source code to HTML converter. Syntax highlighting
   is formatted by Cascading Style Sheets. It's possible to easily enhance
   highlight's parsing database.

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "main.h"

using namespace std;

void HighlightApp::printVersionInfo()
{
  cout << "\n highlight version "
       << HIGHLIGHT_VERSION
       #ifdef USE_LOCAL_GETOPT
         << "\n\n (Built with USE_LOCAL_GETOPT flag.)\n"
       #endif
       << "\n Copyright (C) 2002-2004 Andre Simon <andre.simon1@gmx.de>"
       << "\n\n Artistic Style Classes (1.15.3)"
       << "\n Copyright (C) 1998-2002 Tal Davidson <davidsont@bigfoot.com>"
       << "\n\n Dirstream Classes (0.4)"
       << "\n Copyright (C) 2002-2004 Benjamin Kaufmann <hume@c-plusplus.de>"
       << "\n\n This software is released under the terms of the GNU General "
       << "Public License."
       << "\n For more information about these matters, see the file named "
       << "COPYING.\n\n";
}

void HighlightApp::printBadInstallationInfo()
{
  cerr << "highlight: Data directory not found. Bad installation or wrong "
       << OPT_DATADIR << " parameter."
       << "\n\nCopy the highlight files into one of the directories listed "
       << "in INSTALL.\nYou may also set the data directory with "
       << OPT_DATADIR << "and " << OPT_ADDDATADIR << ".\n";
}

bool HighlightApp::listInstalledFiles(bool showThemes)
{
  vector <string> filePaths;
  string wildcard=(showThemes)? "*.style":"*.lang";
  unsigned int suffixLength=wildcard.length()-1;

  string searchDir = ((showThemes) ? dataDir.getThemeDir():
                                     dataDir.getLangDefDir()) + wildcard;

  bool directoryOK = Platform::getDirectoryEntries(filePaths, searchDir, true);
  if (!directoryOK) {
    cerr << "highlight: Could not access directory "
         <<  searchDir
         << ", aborted.\n";
     return false;
  }

  cout << "\n  Installed "
       << ((showThemes)? "themes":"language definitions ")
       << "(located in "
       << ((showThemes)?dataDir.getThemeDir():dataDir.getLangDefDir())
       << ") :\n"
       << endl;

  sort(filePaths.begin(), filePaths.end());
  string  temp;

  for (unsigned int i=0;i< filePaths.size(); i++){
    if (showThemes)
      temp = (filePaths[i]).substr(dataDir.getThemeDir().length());
    else
      temp = (filePaths[i]).substr(dataDir.getLangDefDir().length());
    cout << "  "<<temp.substr(0, temp.length()- suffixLength) << endl;
  }
  cout <<"\n  Use name of the desired "
       << ((showThemes)?"theme":"language")
       << " with the --"
       << ((showThemes)? OPT_STYLE : OPT_SYNTAX)
       << " option.\n" << endl;
  return true;
}

void HighlightApp::printDebugInfo(highlight::LanguageDefinition &lang,
                                   const string & langDefPath)
{
   cerr << "\nLoading language definition: " << langDefPath;
   cerr << "\n\nSYMBOLS: "
        << lang.getSymbolString();
   cerr << "\n\nKEYWORDS: ";
   highlight::KeywordMap::iterator it;
   highlight::KeywordMap keys=lang.getKeywords();
   cerr << "\n\nID    Keyword \n";
   for (it=keys.begin(); it!=keys.end();it++){
      cerr << it->second
           << "  <- \""
           << it->first <<"\"\n";
   }
   cerr <<"\n";
}

string HighlightApp::getFileSuffix(const string &fileName) {
  size_t ptPos=fileName.rfind(".");
  return (ptPos == string::npos) ?
         "" : fileName.substr(ptPos+1, fileName.length());
}


bool HighlightApp::loadFileExtensions(){
  string extPath=dataDir.getDir() + "extensions.conf";
  ConfigurationReader extensionsConfig(extPath);
  if (extensionsConfig.found())
  {
    stringstream values;
    string paramName, paramVal;
    for (unsigned int i=0;i<extensionsConfig.getParameterNames().size();i++){
      paramName = extensionsConfig.getParameterNames()[i];
      values.str(extensionsConfig.getParameter(paramName)) ;
      while (values >> paramVal) {
         extensions[ paramVal ] = paramName;
      }
      values.clear();
    }
    return true;
  } else {
    cerr << "highlight: Extension configuration file "
         << extPath
         << " not found.\n";
    return false;
  }
}

string HighlightApp::getFileType(const string& suffix)
{
  return (extensions.count(suffix)) ? extensions[suffix] : suffix ;
}

int HighlightApp::getNumDigits(int i){
  int res=0;
  while (i){
    i/=10;
    ++res;
  }
  return res;
}

void HighlightApp::printProgressBar(int total, int count){
  if (!total) return;
  int p=100*count / total;
  int numProgressItems=p/10;
  cout << "\r[";
  for (int i=0;i<10;i++){
    cout <<((i<numProgressItems)?"#":" ");
  }
  cout<< "] " <<setw(3)<<p<<"%, "<<count << " / " << total << "  " <<flush;
  if (p==100) {
    cout << endl;
  }
}

void HighlightApp::printCurrentAction(const string&outfilePath,
                                      int total, int count, int countWidth){
  cout << "Writing file "
       << setw(countWidth)<< count
       << " of "
       << total
       << ": "
       << outfilePath
       << "\n";
}

void HighlightApp::printIOErrorReport(unsigned int numberErrorFiles,
                                      vector<string> & fileList,
                                      const string &action){
  cerr << "highlight: Could not "
       << action
       << " file"
       << ((numberErrorFiles>1)?"s":"")<<":\n";
  copy (fileList.begin(), fileList.end(), ostream_iterator<string>(cerr, "\n"));
  if (fileList.size() < numberErrorFiles) {
    cerr << "... ["
         << (numberErrorFiles - fileList.size() )
         << " of "
         << numberErrorFiles
         << " failures not shown, use --"
         << OPT_VERBOSE
         << " switch to print all paths]\n";
  }
}


int HighlightApp::run(int argc, char**argv){

  //get command line options
  CmdLineOptions options(argc, argv);

  // set data directory path, where /langDefs and /themes reside
  string highlightRootDir = Platform::getAppPath();
  /*#ifdef _WIN32
    highlightRootDir = argv[0];
    string::size_type pos = highlightRootDir.find_last_of(Platform::pathSeparator);
    highlightRootDir = highlightRootDir.substr(0, pos+1);
  #endif*/

  // determine highlight data directory
  if (! dataDir.searchDataDir((options.dataDirGiven())?
                                options.getDataDir(): highlightRootDir)){
    printBadInstallationInfo();
    return EXIT_FAILURE;
  }

  if (options.additionalDataDirGiven()){
     dataDir.setAdditionalDataDir(options.getAdditionalDataDir());
  }

  loadFileExtensions();

  if (options.printVersion()) {
    printVersionInfo();
    return EXIT_SUCCESS;
  }

  if  (options.printHelp())  {
    Help::printHelp(dataDir.getHelpMsgDir() + options.getHelpLang());
    return EXIT_SUCCESS;
  }

  if (options.showThemes() || options.showLangdefs()) {
    return listInstalledFiles(options.showThemes())?EXIT_SUCCESS:EXIT_FAILURE;
  }

  // list of input files
  const  vector <string> inFileList=options.getInputFileNames();

  string stylePath=dataDir.searchForTheme(options.getStyleName());

  highlight::CodeGenerator *generator =
    highlight::CodeGenerator::getInstance(options.getOutputType(),
                                          stylePath,
                                          options.getCssInFilename(),
                                          options.getCssOutFilename(),
                                          options.includeCssDefinition(),
                                          options.attachLineAnchors(),
                                          options.replaceQuotes(),
                                          options.fopCompatible(),
                                          options.getNumberSpaces(),
                                          options.getWrappingStyle(),
                                          options.printLineNumbers(),
                                          options.fillLineNrZeroes(),
                                          options.fragmentOutput());

  assert (generator!=NULL);

  if (!generator->styleFound()) {
    cerr << "highlight: Could not find style "
         << stylePath
         << ".\n";
    highlight::CodeGenerator::deleteInstance();
    return EXIT_FAILURE;
  }

  if (!options.getIndentScheme().empty()){
    string indentSchemePath =
              dataDir.searchForIndentScheme(options.getIndentScheme()+".indent");
    if (!generator->initIndentationScheme(indentSchemePath)){
        cerr << "highlight: Could not find indentation scheme "
             << indentSchemePath
             << ".\n";
       highlight::CodeGenerator::deleteInstance();
       return EXIT_FAILURE;
    }
  }

  string outDirectory = options.getOutDirectory();
  if (!outDirectory.empty() && !options.quietMode() && !dirstr::directory_exists(outDirectory) ){
     cerr << "highlight: Output directory \""
          << outDirectory
	  << "\" does not exist.\n";
     return EXIT_FAILURE;
  }

  unsigned int fileCount=inFileList.size(),
               fileCountWidth=getNumDigits(fileCount),
               i=0,
               numBadFormatting=0,
               numBadInput=0,
               numBadOutput=0;
  bool initError=false, IOError=false;
  vector<string> badFormattedFiles, badInputFiles, badOutputFiles;
  string outFilePath;
  string suffix, lastSuffix;

  if (options.syntaxGiven()) {
    suffix = getFileType(options.getLanguage());
  }

  while (i < fileCount && !initError) {
    if (!options.syntaxGiven()) {
      suffix = getFileType(getFileSuffix(inFileList[i]));
    }

    if (suffix.empty()) {
            if (!options.enableBatchMode())
                cerr << "highlight: Undefined language definition. Use --"
                    << OPT_SYNTAX << " option.\n";
            initError = true;
            break;
    }

    if (suffix != lastSuffix) {
        string langDefPath=dataDir.searchForLangDef(suffix+".lang");
        highlight::LoadResult loadRes= generator->initLanguage(langDefPath);
        if (loadRes==highlight::LOAD_FAILED){
            cerr << "highlight: Unknown source file extension \""
                << suffix
                << "\".\n";
            initError = true;
            break;
        }
        if (options.printDebugInfo() && loadRes==highlight::LOAD_NEW){
            printDebugInfo(generator->getLanguage(), langDefPath);
        }
        lastSuffix = suffix;
    }

    if (options.enableBatchMode()){
      string::size_type pos=(inFileList[i]).find_last_of(Platform::pathSeparator);
      outFilePath = outDirectory;
      outFilePath += inFileList[i].substr(pos+1);
      outFilePath += options.getOutFileSuffix();

      if (!options.quietMode()) {
        if (options.printProgress()){
           printProgressBar(fileCount, i+1);
        } else {
           printCurrentAction(outFilePath, fileCount, i+1, fileCountWidth);
        }
      }
     } else {
        outFilePath = options.getSingleOutFilename();
     }

     highlight::ParseError error = generator->printOutput(inFileList[i], outFilePath);
     if (error==highlight::BAD_INPUT){
       if (numBadInput++ < IO_ERROR_REPORT_LENGTH || options.printDebugInfo()) {
         badInputFiles.push_back(inFileList[i]);
        }
     } else if (error==highlight::BAD_OUTPUT){
       if (numBadOutput++ < IO_ERROR_REPORT_LENGTH || options.printDebugInfo()) {
         badOutputFiles.push_back(outFilePath);
       }
     }
     if (options.formattingEnabled() && !generator->formattingIsPossible()){
      if (numBadFormatting++ < IO_ERROR_REPORT_LENGTH || options.printDebugInfo()) {
        badFormattedFiles.push_back(outFilePath);
      }
     }
     ++i;
    }

    if (!options.includeCssDefinition() && !options.fragmentOutput()) {
      string cssOutFile=outDirectory + options.getCssOutFilename();
      bool success=generator->printExternalStyle (cssOutFile);
      if (!success){
        cerr << "highlight: Could not write " << cssOutFile <<".\n";
        IOError = true;
      }
    }

    if (options.printIndexFile()){
        bool success=generator -> printIndexFile(inFileList, outDirectory);
        if (!success){
          cerr << "highlight: Could not write index file.\n";
          IOError = true;
        }
    }

    if (numBadInput){
      printIOErrorReport(numBadInput, badInputFiles, "read input");
      IOError = true;
    }
    if (numBadOutput){
      printIOErrorReport(numBadOutput, badOutputFiles, "write output");
      IOError = true;
    }
    if (numBadFormatting){
      printIOErrorReport(numBadFormatting, badFormattedFiles, "reformat");
    }

    highlight::CodeGenerator::deleteInstance();
    return (initError || IOError) ? EXIT_FAILURE : EXIT_SUCCESS;
}

int main(int argc, char **argv) {
  HighlightApp app;
  return app.run(argc, argv);
}
