/***************************************************************************
                          htmlcode.cpp  -  description
                             -------------------
    begin                : Wed Nov 28 2001
    copyright            : (C) 2001 by Andr Simon
    email                : andre.simon1@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "htmlgenerator.h"

using namespace std;

namespace highlight {


HtmlGenerator::HtmlGenerator(void)
{}

string  HtmlGenerator::formatStyleAttributes(const string & elemName,
                                             const ElementStyle & elem)
{
  ostringstream s;
  s  << "\t."<<elemName<<"\t{ color:#"
     << (elem.getColour().getHexRedValue())
     << (elem.getColour().getHexGreenValue())
     << (elem.getColour().getHexBlueValue()  )
     << ( elem.isBold() ?"; font-weight:bold" :"" )
     << ( elem.isItalic() ?"; font-style:italic" :"" )
     << ( elem.isUnderline() ?"; text-decoration:underline" :"" )
     << "; }\n" ;
  return  s.str();
}

string  HtmlGenerator::getOpenTag(const string& styleName ){
  return "<span class=\""+styleName+"\">";
}

HtmlGenerator::HtmlGenerator (
  const string &cssStyle,
  const string &cssInFilename,
  const string &cssOutFilename,
  bool includeCssDef,
  bool withAnchors)
    : CodeGenerator( cssStyle),
    brTag("<br>"),
    hrTag("<hr>"),
    idAttr("name"),
    fileSuffix(".html"),
    HTML_HEADER1(
      "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">"
      "\n<html>\n"
      "<head>\n"
      "\t<meta http-equiv=\"content-type\" content=\"text/html; charset=ISO-8859-1\">\n"
      "\t<title>"),
    HTML_HEADER2("</title>\n"),
    HTML_FOOTER(
      "\n</body>\n</html>\n<!--HTML generated by highlight "
      HIGHLIGHT_VERSION
      ", "
      HIGHLIGHT_URL
      "-->\n"),
    cssInFile(cssInFilename),
    cssOutFile(cssOutFilename),
    includeCss(includeCssDef),
    attachAnchors(withAnchors)
{
  styleTagOpen.push_back("");
  styleTagOpen.push_back(getOpenTag("str"));
  styleTagOpen.push_back(getOpenTag("num"));
  styleTagOpen.push_back(getOpenTag("slc"));
  styleTagOpen.push_back(getOpenTag("com"));
  styleTagOpen.push_back(getOpenTag("esc"));
  styleTagOpen.push_back(getOpenTag("dir"));
  styleTagOpen.push_back(getOpenTag("dstr"));
  styleTagOpen.push_back(getOpenTag("line"));
  styleTagOpen.push_back(getOpenTag("sym"));

  styleTagClose.push_back("");
  for (int i=1;i<NUMBER_BUILTIN_STYLES; i++) {
    styleTagClose.push_back("</span>");
  }

  /*assert (styleTagOpen.size()==styleTagClose.size());
  assert (styleTagOpen.size()==NUMBER_BUILTIN_STYLES);
*/
  newLineTag = "\n";
  spacer = " ";
}

string HtmlGenerator::getStyleDefinition()
{
  if (styleDefinitionCache.empty()){
    ostringstream os;
    os << "\tbody\t{ background-color:#"
       << (docStyle.getBgColour().getHexRedValue())
       << (docStyle.getBgColour().getHexGreenValue())
       << (docStyle.getBgColour().getHexBlueValue())
       << "; }\n";
    os << "\tpre\t{ color:#"
       << (docStyle.getDefaultStyle().getColour().getHexRedValue())
       << (docStyle.getDefaultStyle().getColour().getHexGreenValue())
       << (docStyle.getDefaultStyle().getColour().getHexBlueValue()  )
       << "; background-color:#"
       << (docStyle.getBgColour().getHexRedValue())
       << (docStyle.getBgColour().getHexGreenValue())
       << (docStyle.getBgColour().getHexBlueValue())
       << "; font-size:"
       << docStyle.getFontSize()
       << "pt; font-family:Courier;}\n";
    os << formatStyleAttributes("num", docStyle.getNumberStyle())
       << formatStyleAttributes("esc", docStyle.getEscapeCharStyle())
       << formatStyleAttributes("str", docStyle.getStringStyle())
       << formatStyleAttributes("dstr", docStyle.getDirectiveStringStyle())
       << formatStyleAttributes("slc", docStyle.getSingleLineCommentStyle())
       << formatStyleAttributes("com", docStyle.getCommentStyle())
       << formatStyleAttributes("dir", docStyle.getDirectiveStyle())
       << formatStyleAttributes("sym", docStyle.getSymbolStyle())
       << formatStyleAttributes("line", docStyle.getLineStyle());

    KeywordStyles styles = docStyle.getKeywordStyles();
    for (KSIterator it=styles.begin(); it!=styles.end(); it++){
      os << formatStyleAttributes(it->first, *(it->second));
    }
    styleDefinitionCache=os.str();
  }
  return styleDefinitionCache;
}

string HtmlGenerator::getHeader(const string &title)
{
  ostringstream os;
  os << HTML_HEADER1
     << ((title.empty())?"Source file":title )
     << HTML_HEADER2;
  if (langInfo.getSyntaxHighlight())
  {
    if (includeCss)    //CSS-Definition in HTML-<head> einfuegen
      {
        os << "\t<style type=\"text/css\">\n";
        os << "\t<!--\n";
        os << getStyleDefinition();
        os << readUserStyleDef();
        os << "\t//-->\n";
        os << "\t</style>" << endl;
      }
    else  //Referenz auf CSS-Datei einfuegen
      {
        os << "\t<link rel=\"stylesheet\" type=\"text/css\" href=\""
           << cssOutFile
           << "\""
           << ">\n";
      }
  }
  os << "</head>\n<body>\n<pre>";
  return os.str();
}

string HtmlGenerator::getFooter()
{
  return "</pre>" + HTML_FOOTER;
}


void HtmlGenerator::printBody()
{
  processRootState();
}

bool HtmlGenerator::printExternalStyle(const string &outFile)
{
  if (!includeCss && langInfo.getSyntaxHighlight()) {
      ofstream cssOutFile(outFile.c_str());
      if (cssOutFile) {
         cssOutFile << "/* CSS definition file generated by highlight "
                 << HIGHLIGHT_VERSION
                 << ", "
                 << HIGHLIGHT_URL
                 << " */\n";
         cssOutFile << "\n/* Highlighting theme definition: */\n\n"
                 << getStyleDefinition()
                 << "\n";
         cssOutFile << readUserStyleDef();
         cssOutFile.close();
      } else {
        return false;
      }
  }
  return true;
}

string HtmlGenerator::readUserStyleDef(){
  ostringstream ostr;
  if (!cssInFile.empty()){
    ifstream userStyleDef(cssInFile.c_str());
    if (userStyleDef) {
      ostr << "\n\t/* Content of "<<cssInFile<<": */\n";
      string line;
      while (getline(userStyleDef, line)){
        ostr << "\t"<<line << "\n";
      }
      userStyleDef.close();
    } else {
      ostr << "/* ERROR: Could not include "
           << cssInFile
           << ".*/\n";
     }
  }
  return ostr.str();
}

string HtmlGenerator::maskCharacter(unsigned char c)
{
  switch (c) {
    case '<' :
      return "&lt;";
      break;
    case '>' :
      return "&gt;";
      break;
    case '&' :
      return "&amp;";
      break;
    case '\"' :
      return "&quot;";
      break;
    case '@' :
      return "&#64;";
      break;
    case AUML_LC :
      return "&auml;";
      break;
    case OUML_LC :
      return "&ouml;";
      break;
    case UUML_LC :
      return "&uuml;";
      break;
    case AUML_UC :
      return "&Auml;";
      break;
    case OUML_UC :
      return "&Ouml;";
      break;
    case UUML_UC :
      return "&Uuml;";
      break;
    case SZLIG :
      return "&szlig;";
      break;

    case AACUTE_LC :
      return "&aacute;";
      break;
    case EACUTE_LC:
      return "&eacute;";
      break;
    case OACUTE_LC:
      return "&oacute;";
      break;
    case UACUTE_LC:
      return "&uacute;";
      break;

    case AACUTE_UC:
      return "&Aacute;";
      break;
    case EACUTE_UC:
      return "&Eacute;";
      break;
    case OACUTE_UC:
      return "&Oacute;";
      break;
    case UACUTE_UC :
      return "&Uacute;";
      break;

    case AGRAVE_LC:
      return "&agrave;";
      break;
    case EGRAVE_LC:
      return "&egrave;";
      break;
    case OGRAVE_LC:
      return "&ograve;";
      break;
    case UGRAVE_LC:
      return "&ugrave;";
      break;

    case AGRAVE_UC:
      return "&Agrave;";
      break;
    case EGRAVE_UC:
      return "&Egrave;";
      break;
    case OGRAVE_UC:
      return "&Ograve;";
      break;
    case UGRAVE_UC:
      return "&Ugrave;";
      break;

    // skip  first byte of multibyte characters
    #ifndef _WIN32
    case 195:
      return string();
      break;
    #endif

    default :
      string m;
      return m += c;
    }
}

void HtmlGenerator::insertLineNumber (bool insertNewLine)
{
  if (insertNewLine){
    *out << getNewLine();
  }
  if (showLineNumbers)
    {
      State myState=currentState;
      unsigned int styleID=0;
      // Falls Zustand != Standardzustand, entsprechende Tags schliessen...
      if (myState != STANDARD )
        {
           styleID=getStyleID(myState);
           if (styleID!=_UNKNOWN){
             *out << styleTagClose[styleID];
           }
        }
      if (attachAnchors)
        {
          *out << "<a "
               << idAttr
               << "=\"l_"
                   << lineNumber
                   << "\">";
        }
      ostringstream os;
      if (lineNumberFillZeroes) os.fill('0');
      os <<setw(LINE_NUMBER_WIDTH)<<right<< lineNumber;
      *out << styleTagOpen[LINENUMBER]
           << os.str()
           << spacer
               << styleTagClose[LINENUMBER];
      if (attachAnchors)
        {
          *out << "</a>";
        }

      // ... und wieder oeffnen
      if (myState != STANDARD)
        {
           styleID=getStyleID(myState);
           if (styleID!=_UNKNOWN){
             *out << styleTagOpen[styleID];
           }
        }
    }
}

bool HtmlGenerator::printIndexFile(const vector<string> &fileList,
                                   const string &outPath ){
  string suffix = fileSuffix;
  string outFilePath = outPath + "index" + suffix;
  ofstream indexfile(outFilePath.c_str());

  if (!indexfile.fail()){
     string inFileName;
     string inFilePath, newInFilePath;
     indexfile << HTML_HEADER1 << "Source Index" << HTML_HEADER2;
     indexfile << "</head>\n<body>\n<h1> Source Index</h1>\n"
               << hrTag
               <<  "\n<ul>\n";
     string::size_type pos;
     for (unsigned int i=0; i < fileList.size();  i++){
         pos=(fileList[i]).find_last_of(Platform::pathSeparator);
         if (pos!=string::npos){
             newInFilePath = (fileList[i]).substr(0, pos+1);
         } else {
            newInFilePath=Platform::pathSeparator;
         }
         if (newInFilePath!=inFilePath){
           indexfile << "</ul>\n<h2>";
           indexfile << newInFilePath;
           indexfile << "</h2>\n<ul>\n";
           inFilePath=newInFilePath;
         }
         inFileName = (fileList[i]).substr(pos+1);
         indexfile << "<li><a href=\"" << inFileName << suffix << "\">";
         indexfile << inFileName << suffix <<"</a></li>\n";
     }

     indexfile << "</ul>\n"
               << hrTag << brTag
               << "<small>Generated by highlight "
               << HIGHLIGHT_VERSION
               << ", <a href=\"" << HIGHLIGHT_URL << "\" target=\"new\">"
               << HIGHLIGHT_URL << "</a></small>";
     indexfile << HTML_FOOTER;
  } else {
    return false;
  }
  return true;
}

string HtmlGenerator::getMatchingOpenTag(unsigned int styleID){
  return getOpenTag(langInfo.getKeywordClasses()[styleID]);
 }

string HtmlGenerator::getMatchingCloseTag(unsigned int styleID){
  return "</span>";
}

}
