/***************************************************************************
                          cmdlineoptions.cpp  -  description
                             -------------------
    begin                : Sun Nov 25 2001
    copyright            : (C) 2001 by Andre Simon
    email                : andre.simon1@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cmdlineoptions.h"

using namespace std;

/* Siehe man getopt (3)
   Konstruktor legt Optionen und Argumente fest
*/
CmdLineOptions::CmdLineOptions(int argc, char *argv[]):
    numberSpaces(0),
    lineNrWidth(5),
    lineNrStart(0),
    wrappingStyle(highlight::WRAP_DISABLED),
    outputType (highlight::HTML),
    opt_language (false),
    opt_include_style (false),
    opt_help (false),
    opt_version (false),
    opt_verbose (false),
    opt_linenumbers (false),
    opt_style (false),
    opt_batch_mode (false),
    opt_fragment (false) ,
    opt_attach_line_anchors (false),
    opt_show_themes (false),
    opt_show_langdefs (false),
    opt_printindex(false),
    opt_quiet(false),
    opt_xslfo_fop(false),
    opt_replacequotes(false),
    opt_babel(false),
    opt_print_progress(false),
    opt_fill_zeroes(false),
    opt_stylepath_explicit(false),
    opt_force_output(false),
    opt_ordered_list(false),
    configFileRead(false),
    helpLang("en"),
    encodingName("ISO-8859-1")
{

  loadConfigurationFile();

  int c, option_index = 0;
  static struct option long_options[] =
      {
        {OPT_OUT, 1, 0, S_OPT_OUT},
        {OPT_IN, 1, 0, S_OPT_IN},
        {OPT_SYNTAX, 1, 0, S_OPT_SYNTAX},
        {OPT_VERBOSE, 0, 0, S_OPT_VERBOSE},
        {OPT_INC_STYLE, 0, 0, S_OPT_INC_STYLE},
        {OPT_HELP, 0, 0, S_OPT_HELP},
        {OPT_HELPINT, 1, 0, S_OPT_HELPINT},
        {OPT_LINENO,0,0,S_OPT_LINENO},
        {OPT_STYLE, 1,0,S_OPT_STYLE},
        {OPT_STYLE_OUT, 1, 0,S_OPT_STYLE_OUT},
        {OPT_STYLE_IN, 1, 0,S_OPT_STYLE_IN},
        {OPT_DELTABS,1,0,S_OPT_DELTABS},
        {OPT_XHTML, 0,0,S_OPT_XHTML},
        {OPT_RTF, 0,0,S_OPT_RTF},
        {OPT_TEX,0, 0,S_OPT_TEX},
        {OPT_LATEX,0, 0,S_OPT_LATEX},
        {OPT_XSLFO,0, 0,S_OPT_XSLFO},
        {OPT_ANSI,0, 0,S_OPT_ANSI},
        {OPT_XML,0, 0,S_OPT_XML},
        {OPT_BATCHREC,1,0,S_OPT_BATCHREC},
        {OPT_FRAGMENT,0,0,S_OPT_FRAGMENT},
        {OPT_ANCHORS, 0,0,S_OPT_ANCHORS },
        {OPT_LISTTHEMES, 0,0,S_OPT_LISTTHEMES },
        {OPT_LISTLANGS, 0,0,S_OPT_LISTLANGS },
        {OPT_OUTDIR,1,0,S_OPT_OUTDIR},
        {OPT_VERSION,0,0,0},
        {OPT_FORMATSTYLE,1,0,S_OPT_FORMATSTYLE},
        {OPT_DATADIR,1,0,S_OPT_DATADIR},
        {OPT_ADDDATADIR,1,0,S_OPT_ADDDATADIR},
        {OPT_INDEXFILE,0,0,S_OPT_INDEXFILE},
        {OPT_WRAP,0,0,S_OPT_WRAP},
        {OPT_WRAPSIMPLE,0,0,S_OPT_WRAPSIMPLE},
        {OPT_QUIET,0,0,S_OPT_QUIET},
        {OPT_REPLACE_QUOTES,0,0,S_OPT_REPLACE_QUOTES},
        {OPT_PROGRESSBAR,0,0,S_OPT_PROGRESSBAR},
        {OPT_FILLZEROES,0,0,S_OPT_FILLZEROES},
        {OPT_ENCODING,1,0,S_OPT_ENCODING},
        // -- Kafka begins
        {OPT_BASE_FONT,1,0,S_OPT_BASE_FONT},
        {OPT_BASE_FONT_SIZE,1,0,S_OPT_BASE_FONT_SIZE},
        // -- Kafka ends
        {OPT_ORDERED_LIST,0,0,S_OPT_ORDERED_LIST},
        {OPT_LNR_WIDTH,1,0,S_OPT_LNR_WIDTH},
        {OPT_LNR_START,1,0,S_OPT_LNR_START},
        {OPT_BABEL,0,0,S_OPT_BABEL},

        //remove as soon as APAche fixes the bug in FOP (0.20.5)
        {OPT_FOP,0,0,S_OPT_FOP},

        //deprecated
        {OPT_CSSOUT,1,0,0},
        {OPT_CSSIN,1,0,0},
        {OPT_INC_CSS,0,0,0},
        {OPT_FORCE_OUTPUT,0,0,0},

        {0, 0, 0, 0}
      };

  while (1)
    {
      c = getopt_long (argc, argv,S_OPTIONS_STRING,long_options, &option_index);
      if (c == -1)
        break;

      switch (c)
        {
        case 0:   // long options
          if (long_options[option_index].name==OPT_VERSION) {
              opt_version = true;
            }
            if (long_options[option_index].name==OPT_CSSOUT) {
                styleOutFilename=string(optarg);
                printDeprecatedWarning(OPT_CSSOUT, OPT_STYLE_OUT);
            }
            else if (long_options[option_index].name==OPT_CSSIN) {
                styleInFilename=string(optarg);
                printDeprecatedWarning(OPT_CSSIN, OPT_STYLE_IN);
            }
            else if (long_options[option_index].name==OPT_INC_CSS) {
                opt_include_style = true;
                printDeprecatedWarning(OPT_INC_CSS, OPT_INC_STYLE);
            }
            else if (long_options[option_index].name==OPT_FORCE_OUTPUT) {
                opt_force_output = true;
            }
          break;
        case S_OPT_OUT:
          outFilename=string(optarg);
          break;
        case S_OPT_IN:
          inputFileNames.push_back(string(optarg));
          break;
        case S_OPT_STYLE_OUT:
          styleOutFilename=string(optarg);
          opt_stylepath_explicit=true;
          break;
        case S_OPT_STYLE_IN:
          styleInFilename=string(optarg);
          break;
        case S_OPT_VERBOSE:
          opt_verbose = true;
          break;
        case S_OPT_QUIET:
          opt_quiet = true;
          break;
         case S_OPT_INC_STYLE:
          opt_include_style = true;
          break;
        case S_OPT_HELPINT:
          helpLang=string(optarg);
        case S_OPT_HELP:
          opt_help = true;
          break;
        case S_OPT_LINENO:
          opt_linenumbers = true;
          break;
        case '?':
          //opt_help = true;
          break;
        case S_OPT_STYLE:
          styleName=string(optarg);
          opt_style = true;
          break;
        case S_OPT_SYNTAX:
          language=string(optarg);
          opt_language = true;
          break;
        case S_OPT_DELTABS:
          numberSpaces = StringTools::str2int (string(optarg), true);
          break;
        case S_OPT_XHTML:
          outputType=highlight::XHTML;
          break;
        case S_OPT_RTF:
          outputType=highlight::RTF;
          break;
        case S_OPT_TEX:
          outputType=highlight::TEX;
          break;
        case S_OPT_LATEX:
          outputType=highlight::LATEX;
          break;
        case S_OPT_XSLFO:
          outputType=highlight::XSLFO;
          break;
        case S_OPT_ANSI:
          outputType=highlight::ANSI;
          break;
        case S_OPT_XML:
          outputType=highlight::XML;
          break;
        case S_OPT_BATCHREC:
          opt_batch_mode = true;
          readDirectory(string(optarg));
          break;
        case S_OPT_FRAGMENT:
          opt_fragment = true;
          break;
        case S_OPT_ANCHORS:
          opt_attach_line_anchors = true;
          break;
        case S_OPT_LISTTHEMES:
          opt_show_themes = true;
          break;
        case S_OPT_LISTLANGS:
          opt_show_langdefs = true;
          break;
        case S_OPT_OUTDIR:
          outDirectory = validateDirPath(string(optarg));
          break;
        case S_OPT_FORMATSTYLE:
          indentScheme =string(optarg);
          break;
        case S_OPT_ENCODING:
          encodingName =string(optarg);
          break;
        case S_OPT_DATADIR:
          dataDir=validateDirPath(string(optarg));
          break;
        case S_OPT_ADDDATADIR:
          additionalDataDir=validateDirPath(string(optarg));
          break;
         case S_OPT_INDEXFILE:
          opt_printindex=true;
          break;
         case S_OPT_WRAPSIMPLE:
          wrappingStyle = highlight::WRAP_SIMPLE;
          break;
         case S_OPT_WRAP:
          wrappingStyle = highlight::WRAP_DEFAULT;
          break;
         case S_OPT_FOP:
          opt_xslfo_fop=true;
          break;
         case S_OPT_REPLACE_QUOTES:
          opt_replacequotes=true;
          break;
         case S_OPT_BABEL:
          opt_babel=true;
          break;
         case S_OPT_PROGRESSBAR:
          opt_print_progress=true;
          break;
         case S_OPT_FILLZEROES:
          opt_fill_zeroes=true;
          break;
          // -- Kafka begins
         case S_OPT_BASE_FONT:
          baseFont=string(optarg);
          break;
         case S_OPT_BASE_FONT_SIZE:
          baseFontSize=string(optarg);
          break;
         // -- Kafka ends
         case S_OPT_LNR_WIDTH:
          lineNrWidth=StringTools::str2int(string(optarg), true);
          break;
         case S_OPT_LNR_START:
          lineNrStart=StringTools::str2int(string(optarg), true);
          break;
         case S_OPT_ORDERED_LIST:
            opt_ordered_list = opt_linenumbers = true;
          break;

        default:
          cerr <<"higlight: Unknown option " <<c<< endl;
        }
    }

  if (optind < argc)   //still args left
    {
      if  (inputFileNames.empty()) {
        while (optind < argc){
          inputFileNames.push_back(string(argv[optind++]));
         }
      }
    } else if (inputFileNames.empty()) {
       inputFileNames.push_back("");
    }
  if (printDebugInfo() && configFileRead) {
    cout << "Configuration file \""<<configFilePath<<"\" was read.\n";
  }
}

CmdLineOptions::~CmdLineOptions(){
}

const string &CmdLineOptions::getSingleOutFilename()
  {
   if (!inputFileNames.empty() && !outDirectory.empty()) {
      if (outFilename.empty()) {
        outFilename = outDirectory;
        int delim = getSingleInFilename().find_last_of(Platform::pathSeparator)+1;
        outFilename += getSingleInFilename().substr((delim>-1)?delim:0)
                       + getOutFileSuffix();
      }
   }
   return outFilename;
  }

const string &CmdLineOptions::getSingleInFilename()  const
  {
  return inputFileNames[0];
  }

const string &CmdLineOptions::getOutDirectory()
  {
    if (!outFilename.empty() && !enableBatchMode()){
      outDirectory=getDirName(outFilename);
    }
    return outDirectory;
  }

const string CmdLineOptions::getStyleOutFilename() const
  {
      if (!styleOutFilename.empty()) return styleOutFilename;
      return (outputType==highlight::HTML ||
              outputType==highlight::XHTML)? "highlight.css":"highlight.sty";
  }
const string &CmdLineOptions::getStyleInFilename() const
  {
    return styleInFilename;
  }
int CmdLineOptions::getNumberSpaces() const
  {
    return numberSpaces;
  }
bool CmdLineOptions::printVersion()const
  {
    return opt_version;
  }
bool CmdLineOptions::printHelp()const
  {
    return opt_help;
  }
bool CmdLineOptions::printDebugInfo()const
  {
    return opt_verbose;
  }
bool CmdLineOptions::quietMode()const
  {
    return opt_quiet;
  }
bool CmdLineOptions::includeStyleDef()const
  {
      return opt_include_style;
  }

bool CmdLineOptions::formatSupportsExtStyle(){
      return outputType==highlight::HTML ||
             outputType==highlight::XHTML ||
             outputType==highlight::LATEX ||
             outputType==highlight::TEX;
}

bool CmdLineOptions::printLineNumbers()const
  {
    return opt_linenumbers;
  }

string CmdLineOptions::getStyleName()const
  {
    return ( ( opt_style) ? styleName+".style" : "kwrite.style" );
  }
bool CmdLineOptions::enableBatchMode()const{
    return inputFileNames.size()>1 || opt_batch_mode;
}
bool CmdLineOptions::fragmentOutput()const{
    return opt_fragment;
}
string CmdLineOptions::getOutFileSuffix()const{
    switch (outputType){
      case highlight::XHTML: return ".xhtml";
      case highlight::RTF:   return ".rtf";
      case highlight::TEX:
      case highlight::LATEX: return ".tex";
      case highlight::XSLFO: return ".fo";
      case highlight::XML:   return ".xml";
      default:    return ".html";
    }
}
string CmdLineOptions::getDirName(const string & path) {
  size_t dirNameLength=path.rfind(Platform::pathSeparator);
  return (dirNameLength==string::npos)?string():path.substr(0, dirNameLength+1);
}
bool CmdLineOptions::attachLineAnchors()const{
    return opt_attach_line_anchors;
}
bool CmdLineOptions::showThemes()const{
    return opt_show_themes;
}
bool CmdLineOptions::showLangdefs()const{
    return opt_show_langdefs;
}
bool CmdLineOptions::outDirGiven()const{
    return !outFilename.empty();
}
bool CmdLineOptions::fopCompatible() const {
   return opt_xslfo_fop;
}
bool CmdLineOptions::replaceQuotes() const {
   return opt_replacequotes;
}
bool CmdLineOptions::disableBabelShorthands() const {
  return opt_babel;
}
bool CmdLineOptions::getFlag( const string& paramVal){
   return StringTools::lowerCase(paramVal)=="true";
}
bool CmdLineOptions::formattingEnabled(){
    return !indentScheme.empty();
}
bool CmdLineOptions::dataDirGiven()const {
  return !dataDir.empty();
}
bool CmdLineOptions::additionalDataDirGiven()const {
  return !additionalDataDir.empty();
}

bool CmdLineOptions::orderedList() const {
  return opt_ordered_list;
}

const string &CmdLineOptions::getDataDir() const {
 return dataDir;
}
const string &CmdLineOptions::getIndentScheme() const {
 return indentScheme;
}
const string &CmdLineOptions::getAdditionalDataDir()const{
 return additionalDataDir;
}
const string &CmdLineOptions::getLanguage() const {
 return language;
}
const string&CmdLineOptions::getEncoding() const{
    return encodingName;
}
bool CmdLineOptions::printIndexFile() const{
 return opt_printindex && (outputType==highlight::HTML ||
                           outputType==highlight::XHTML);
}
bool CmdLineOptions::printProgress() const{
 return opt_print_progress;
}
bool CmdLineOptions::fillLineNrZeroes() const {
  return opt_fill_zeroes;
}
bool CmdLineOptions::syntaxGiven() const {
  return opt_language;
}
bool CmdLineOptions::omitEncoding() const{
    return StringTools::lowerCase(encodingName)=="none";
}
bool CmdLineOptions::forceOutput() const {
    return opt_force_output;
}
const string  CmdLineOptions::getHelpLang() const {
  return helpLang+".help";
}

highlight::WrapMode CmdLineOptions::getWrappingStyle() const {
  return wrappingStyle;
}
const vector <string> & CmdLineOptions::getInputFileNames() const{
  return inputFileNames;
}
void CmdLineOptions::readDirectory(const string & wildcard){
  // get matching files, use  recursive search
  bool directoryOK=Platform::getDirectoryEntries(inputFileNames, wildcard, true);
  if (!directoryOK)
    {
       cerr << "highlight: No files matched the pattern \""
            << wildcard << "\"."<< endl;
    }
}
void CmdLineOptions::loadConfigurationFile()
{
  #ifndef _WIN32
    #ifdef CONFIG_FILE_PATH
       configFilePath=CONFIG_FILE_PATH;
    #else
      char* homeEnv=getenv("HOME");
      if (homeEnv==NULL) return;
      configFilePath=string(homeEnv)+"/.highlightrc";
    #endif
  #else
    configFilePath = Platform::getAppPath() + "highlight.conf";
  #endif
  ConfigurationReader presets(configFilePath);

  if (presets.found())
    {
      string paramVal;
      configFileRead=true;

      styleOutFilename = presets.getParameter(OPT_STYLE_OUT);
      styleInFilename = presets.getParameter(OPT_STYLE_IN);
      styleName = presets.getParameter(OPT_STYLE);
      opt_style = !styleName.empty();
      language = presets.getParameter(OPT_SYNTAX);
      opt_language = !language.empty();
      numberSpaces = StringTools::str2int(presets.getParameter(OPT_DELTABS), true);
      indentScheme = presets.getParameter(OPT_FORMATSTYLE);
      baseFontSize = presets.getParameter(OPT_BASE_FONT_SIZE);
      baseFont = presets.getParameter(OPT_BASE_FONT);

      paramVal = presets.getParameter(OPT_DATADIR);
      if (!paramVal.empty()) {
         dataDir=validateDirPath( paramVal);
      }
      paramVal = presets.getParameter(OPT_ADDDATADIR);
      if (!paramVal.empty()) {
        additionalDataDir=validateDirPath(paramVal);
      }
      paramVal = presets.getParameter(OPT_OUTDIR);
      if (!paramVal.empty()) {
        outDirectory=validateDirPath(paramVal);
      }
      paramVal = presets.getParameter(OPT_ENCODING);
      if (!paramVal.empty()) {
          encodingName=paramVal;
      }
      paramVal = presets.getParameter(OPT_LNR_WIDTH);
      if (!paramVal.empty()) {
          lineNrWidth=StringTools::str2int(string(paramVal), true);
      }
      paramVal = presets.getParameter(OPT_LNR_START);
      if (!paramVal.empty()) {
          lineNrStart=StringTools::str2int(string(paramVal), true);
      }

      opt_include_style=getFlag(presets.getParameter(OPT_INC_STYLE));
      opt_verbose=getFlag(presets.getParameter(OPT_VERBOSE));
      opt_ordered_list=getFlag(presets.getParameter(OPT_ORDERED_LIST));
      opt_linenumbers=opt_ordered_list || getFlag(presets.getParameter(OPT_LINENO));
      opt_fragment=getFlag(presets.getParameter(OPT_FRAGMENT));
      opt_attach_line_anchors=getFlag(presets.getParameter(OPT_ANCHORS));
      opt_printindex=getFlag(presets.getParameter(OPT_INDEXFILE));
      opt_quiet=getFlag(presets.getParameter(OPT_QUIET));
      opt_xslfo_fop=getFlag(presets.getParameter(OPT_FOP));
      opt_replacequotes=getFlag(presets.getParameter(OPT_REPLACE_QUOTES));
      opt_print_progress=getFlag(presets.getParameter(OPT_PROGRESSBAR));
      opt_fill_zeroes=getFlag(presets.getParameter(OPT_FILLZEROES));

      if (getFlag(presets.getParameter(OPT_WRAP))) {
          wrappingStyle=highlight::WRAP_DEFAULT;
      }
      if (getFlag(presets.getParameter(OPT_WRAPSIMPLE))) {
          wrappingStyle=highlight::WRAP_SIMPLE;
      }
      if (getFlag(presets.getParameter(OPT_XHTML))) {
          outputType=highlight::XHTML;
      } else if (getFlag(presets.getParameter(OPT_RTF))) {
          outputType=highlight::RTF;
      } else if (getFlag(presets.getParameter(OPT_TEX))) {
          outputType=highlight::TEX;
      } else if (getFlag(presets.getParameter(OPT_LATEX))) {
          outputType=highlight::LATEX;
      } else if (getFlag(presets.getParameter(OPT_XSLFO))) {
          outputType=highlight::XSLFO;
      } else if (getFlag(presets.getParameter(OPT_ANSI))) {
          outputType=highlight::ANSI;
      } else if (getFlag(presets.getParameter(OPT_XML))) {
            outputType=highlight::XML;
      }
    }
}

string CmdLineOptions::validateDirPath(const string & path){
   return (path[path.length()-1] !=Platform::pathSeparator)?
              path+Platform::pathSeparator : path;
}

highlight::OutputType CmdLineOptions::getOutputType() const {
    return outputType;
}

void CmdLineOptions::printDeprecatedWarning(const char *oldOption, const char *newOption){
    cerr << "Warning: Long option \""<<oldOption << "\" is DEPRECATED.";
    cerr << " Use \""<<newOption << "\" instead.\n";
}

// -- Kafka begins

bool CmdLineOptions::hasBaseFont() const {
   return (! baseFont.empty()) ;
}

const string& CmdLineOptions::getBaseFont() const {
   return baseFont ;
}

bool CmdLineOptions::hasBaseFontSize() const {
   return (! baseFontSize.empty()) ;
}

const string& CmdLineOptions::getBaseFontSize() const {
   return baseFontSize ;
}

// -- Kafka ends


int CmdLineOptions::getNumberWidth(){
   return lineNrWidth;
}

int CmdLineOptions::getNumberStart(){
   return lineNrStart;
}
