/***************************************************************************
                          main.cpp  -  description
                             -------------------
    begin                : Die Apr 23 22:16:35 CEST 2002
    copyright            : (C) 2002-2004 by Andre Simon
    email                : andre.simon1@gmx.de


   Highlight is a universal source code to HTML converter. Syntax highlighting
   is formatted by Cascading Style Sheets. It's possible to easily enhance
   highlight's parsing database.

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "main.h"
#include "re/Pattern.h"

#define MAX_LINE__WIDTH       80

using namespace std;

void HighlightApp::printVersionInfo()
{
  cout << "\n highlight version "
       << HIGHLIGHT_VERSION
       << "\n Copyright (C) 2002-2005 Andre Simon <andre.simon1 at gmx.de>"
       << "\n\n Artistic Style Classes (1.15.3)"
       << "\n Copyright (C) 1998-2002 Tal Davidson <davidsont at bigfoot.com>"
       << "\n\n Dirstream Classes (0.4)"
       << "\n Copyright (C) 2002-2004 Benjamin Kaufmann <hume at c-plusplus.de>"
       << "\n\n Regex library (0.02a)"
       << "\n Copyright (C) 2003-2005 Jeffery Stuart <stuart at cs.unr.edu>"
       << "\n\n This software is released under the terms of the GNU General "
       << "Public License."
       << "\n For more information about these matters, see the file named "
       << "COPYING.\n\n";
       #ifdef USE_LOCAL_GETOPT
         cout   << " (Built with USE_LOCAL_GETOPT flag set.)\n";
       #endif
       #ifdef HL_DATA_DIR
        cout    << " (HL_DATA_DIR: \"" <<HL_DATA_DIR<< "\" )\n";
       #endif
}

void HighlightApp::printBadInstallationInfo()
{
  cerr << "highlight: Data directory not found. Bad installation or wrong "
       << OPT_DATADIR << " parameter."
       << "\n\nCopy the highlight files into one of the directories listed "
       << "in INSTALL.\nYou may also set the data directory with "
       << OPT_DATADIR << " and " << OPT_ADDDATADIR << ".\n";
}

bool HighlightApp::listInstalledFiles(bool showThemes)
{
  vector <string> filePaths;
  string wildcard=(showThemes)? "*.style":"*.lang";
  unsigned int suffixLength=wildcard.length()-1;

  string searchDir = ((showThemes) ? dataDir.getThemeDir():
                                     dataDir.getLangDefDir()) + wildcard;

  bool directoryOK = Platform::getDirectoryEntries(filePaths, searchDir, true);
  if (!directoryOK) {
    cerr << "highlight: Could not access directory "
         <<  searchDir
         << ", aborted.\n";
     return false;
  }

  cout << "\n  Installed "
       << ((showThemes)? "themes":"language definitions ")
       << "(located in "
       << ((showThemes)?dataDir.getThemeDir():dataDir.getLangDefDir())
       << ") :\n"
       << endl;

  sort(filePaths.begin(), filePaths.end());
  string  temp;

  for (unsigned int i=0;i< filePaths.size(); i++){
    if (showThemes)
      temp = (filePaths[i]).substr(dataDir.getThemeDir().length());
    else
      temp = (filePaths[i]).substr(dataDir.getLangDefDir().length());
    cout << "  "<<temp.substr(0, temp.length()- suffixLength) << endl;
  }
  cout <<"\n  Use name of the desired "
       << ((showThemes)?"theme":"language")
       << " with the --"
       << ((showThemes)? OPT_STYLE : OPT_SYNTAX)
       << " option.\n" << endl;
  return true;
}

void HighlightApp::printDebugInfo(highlight::LanguageDefinition &lang,
                                   const string & langDefPath)
{
   cerr << "\nLoading language definition: " << langDefPath;
   cerr << "\n\nSYMBOLS:\n" << lang.getSymbolString();
   cerr << "\n\nREGEX:\n";
   highlight::RegexElement *re=NULL;
   for (unsigned int i=0; i<lang.getRegexElements().size(); i++){
      re = lang.getRegexElements()[i];
      cerr << "State "<<re->open<<":\t"<<re->rePattern->getPattern()<<"\n";
   }
   cerr << "\n\nKEYWORDS: ";
   highlight::KeywordMap::iterator it;
   highlight::KeywordMap keys=lang.getKeywords();
   cerr << "\n\nID    Keyword \n";
   for (it=keys.begin(); it!=keys.end();it++){
      cerr << it->second
           << "  <- \""
           << it->first <<"\"\n";
   }
   cerr <<"\n";
}

string HighlightApp::getFileSuffix(const string &fileName) {
  size_t ptPos=fileName.rfind(".");
  return (ptPos == string::npos) ?
         "" : fileName.substr(ptPos+1, fileName.length());
}

bool HighlightApp::loadMapConfig(const string& name, StringMap* map){
    string extPath=dataDir.getConfDir() + name + ".conf";
    ConfigurationReader config(extPath);
    if (config.found() )
    {
        stringstream values;
        string paramName, paramVal;
        for (unsigned int i=0;i<config.getParameterNames().size();i++){
            paramName = config.getParameterNames()[i];
            values.str(config.getParameter(paramName)) ;
            while (values >> paramVal) {
                map->insert(make_pair( paramVal,  paramName));
            }
            values.clear();
        }
        return true;
    } else {
        cerr << "highlight: Configuration file "<< extPath << " not found.\n";
        return false;
    }
}


int HighlightApp::getNumDigits(int i){
  int res=0;
  while (i){
    i/=10;
    ++res;
  }
  return res;
}

void HighlightApp::printProgressBar(int total, int count){
  if (!total) return;
  int p=100*count / total;
  int numProgressItems=p/10;
  cout << "\r[";
  for (int i=0;i<10;i++){
    cout <<((i<numProgressItems)?"#":" ");
  }
  cout<< "] " <<setw(3)<<p<<"%, "<<count << " / " << total << "  " <<flush;
  if (p==100) {
    cout << endl;
  }
}

void HighlightApp::printCurrentAction(const string&outfilePath,
                                      int total, int count, int countWidth){
  cout << "Writing file "
       << setw(countWidth)<< count
       << " of "
       << total
       << ": "
       << outfilePath
       << "\n";
}

void HighlightApp::printIOErrorReport(unsigned int numberErrorFiles,
                                      vector<string> & fileList,
                                      const string &action){
  cerr << "highlight: Could not "
       << action
       << " file"
       << ((numberErrorFiles>1)?"s":"")<<":\n";
  copy (fileList.begin(), fileList.end(), ostream_iterator<string>(cerr, "\n"));
  if (fileList.size() < numberErrorFiles) {
    cerr << "... ["
         << (numberErrorFiles - fileList.size() )
         << " of "
         << numberErrorFiles
         << " failures not shown, use --"
         << OPT_VERBOSE
         << " switch to print all paths]\n";
  }
}

string HighlightApp::analyzeFile(const string& file){
    if (scriptShebangs.empty()) loadMapConfig("scriptre", &scriptShebangs);
    ifstream inFile(file.c_str());
    string firstLine;
    getline (inFile, firstLine);
    StringMap::iterator it;
     for (it=scriptShebangs.begin(); it!=scriptShebangs.end();it++){
       if (Pattern::matches(it->first, firstLine)) return it->second;
     }
    return "";
}

string HighlightApp::guessFileType(const string& suffix, const string &inputFile)
{
    if (extensions.empty()) loadMapConfig("extensions", &extensions);
    string fileType = (extensions.count(suffix)) ? extensions[suffix] : suffix ;
    if (!fileType.empty()) return fileType;
    return analyzeFile(inputFile);
}


int HighlightApp::run(int argc, char**argv){

    //cerr << "***THIS IS A BETA RELEASE***\nPlease use the contact form at www.andre-simon.de to report bugs,\n";
    //cerr <<"especially concerning the new regex features.\n";

  //get command line options
  CmdLineOptions options(argc, argv);

  // set data directory path, where /langDefs and /themes reside
  string highlightRootDir = Platform::getAppPath();

  // determine highlight data directory
  if (! dataDir.searchDataDir((options.dataDirGiven())?
                                options.getDataDir(): highlightRootDir)){
    printBadInstallationInfo();
    return EXIT_FAILURE;
  }

  if (options.additionalDataDirGiven()){
     dataDir.setAdditionalDataDir(options.getAdditionalDataDir());
  }

  if (options.printVersion()) {
    printVersionInfo();
    return EXIT_SUCCESS;
  }

  if  (options.printHelp())  {
    Help::printHelp(dataDir.getHelpMsgDir() + options.getHelpLang());
    return EXIT_SUCCESS;
  }

  if (options.showThemes() || options.showLangdefs()) {
    return listInstalledFiles(options.showThemes())?EXIT_SUCCESS:EXIT_FAILURE;
  }

  // list of input files
  const  vector <string> inFileList=options.getInputFileNames();

  string stylePath=dataDir.searchForTheme(options.getStyleName());

  highlight::CodeGenerator *generator =
    highlight::CodeGenerator::getInstance(options.getOutputType());

  generator->initStyle(stylePath);
  generator->setStyleInputPath(options.getStyleInFilename());
  generator->setStyleOutputPath(options.getStyleOutFilename());
  generator->setIncludeStyle(options.includeStyleDef());
  generator->setPrintLineNumbers(options.printLineNumbers(), options.getNumberStart());
  generator->setPrintZeroes(options.fillLineNrZeroes());
  generator->setFragmentCode(options.fragmentOutput());
  generator->setPreformatting(options.getWrappingStyle(),
                             (generator->getPrintLineNumbers()) ?
                              MAX_LINE__WIDTH - options.getNumberWidth() : MAX_LINE__WIDTH,
                              options.getNumberSpaces() );

  generator->setEncoding(options.getEncoding());
  generator->setBaseFont(options.getBaseFont()) ;
  generator->setBaseFontSize(options.getBaseFontSize()) ;
  generator->setLineNumberWidth(options.getNumberWidth());

  generator->setSpecialOptions(options.attachLineAnchors(),options.orderedList(),
                               options.replaceQuotes(), options.disableBabelShorthands(),
                               options.fopCompatible());

  assert (generator!=NULL);

  bool styleFileWanted = !options.fragmentOutput() || options.styleOutPathDefined();

  if (!generator->styleFound() ) {
    cerr << "highlight: Could not find style "
         << stylePath
         << ".\n";
    highlight::CodeGenerator::deleteInstance();
    return EXIT_FAILURE;
  }

  if (!options.getIndentScheme().empty()){
    string indentSchemePath =
              dataDir.searchForIndentScheme(options.getIndentScheme()+".indent");
    if (!generator->initIndentationScheme(indentSchemePath)){
        cerr << "highlight: Could not find indentation scheme "
             << indentSchemePath
             << ".\n";
       highlight::CodeGenerator::deleteInstance();
       return EXIT_FAILURE;
    }
  }

  string outDirectory = options.getOutDirectory();
  if (!outDirectory.empty() && !options.quietMode() && !dirstr::directory_exists(outDirectory) ){
     cerr << "highlight: Output directory \""
          << outDirectory
	  << "\" does not exist.\n";
     return EXIT_FAILURE;
  }

  bool initError=false, IOError=false;

  if (    !options.includeStyleDef()
       && styleFileWanted
       && options.formatSupportsExtStyle()) {
      string cssOutFile=outDirectory  + options.getStyleOutFilename();
      bool success=generator->printExternalStyle (cssOutFile);
      if (!success){
          cerr << "highlight: Could not write " << cssOutFile <<".\n";
          IOError = true;
      }
  }

  if (options.printIndexFile()){
    bool success=generator -> printIndexFile(inFileList, outDirectory);
    if (!success){
      cerr << "highlight: Could not write index file.\n";
      IOError = true;
    }
  }

  unsigned int fileCount=inFileList.size(),
               fileCountWidth=getNumDigits(fileCount),
               i=0,
               numBadFormatting=0,
               numBadInput=0,
               numBadOutput=0;

  vector<string> badFormattedFiles, badInputFiles, badOutputFiles;
  string outFilePath;
  string suffix, lastSuffix;

  if (options.syntaxGiven()) {  // user defined language definition, valid for all files
      suffix = guessFileType(options.getLanguage());
  }

  while (i < fileCount && !initError) {
    if (!options.syntaxGiven()) {  // determine file type for each file
       suffix = guessFileType(getFileSuffix(inFileList[i]), inFileList[i]);
    }
    if (suffix.empty()) {
      if (!options.enableBatchMode())
        cerr << "highlight: Undefined language definition. Use --"
             << OPT_SYNTAX << " option.\n";
        if (!options.forceOutput()){
          initError = true;
          break;
        }
    }

    if (suffix != lastSuffix) {
        string langDefPath=dataDir.searchForLangDef(suffix+".lang");
        highlight::LoadResult loadRes= generator->initLanguage(langDefPath);
        if (loadRes==highlight::LOAD_FAILED){
            cerr << "highlight: Unknown source file extension \""
                << suffix
                << "\".\n";
            if (!options.forceOutput()){
              initError = true;
              break;
            }
        }
        if (options.printDebugInfo() && loadRes==highlight::LOAD_NEW){
            printDebugInfo(generator->getLanguage(), langDefPath);
        }
        lastSuffix = suffix;
    }

    if (options.enableBatchMode()){
      string::size_type pos=(inFileList[i]).find_last_of(Platform::pathSeparator);
      outFilePath = outDirectory;
      outFilePath += inFileList[i].substr(pos+1);
      outFilePath += options.getOutFileSuffix();

      if (!options.quietMode()) {
        if (options.printProgress()){
           printProgressBar(fileCount, i+1);
        } else {
           printCurrentAction(outFilePath, fileCount, i+1, fileCountWidth);
        }
      }
     } else {
        outFilePath = options.getSingleOutFilename();
     }

     highlight::ParseError error = generator->generateFile(inFileList[i], outFilePath);

     if (error==highlight::BAD_INPUT){
       if (numBadInput++ < IO_ERROR_REPORT_LENGTH || options.printDebugInfo()) {
         badInputFiles.push_back(inFileList[i]);
        }
     } else if (error==highlight::BAD_OUTPUT){
       if (numBadOutput++ < IO_ERROR_REPORT_LENGTH || options.printDebugInfo()) {
         badOutputFiles.push_back(outFilePath);
       }
     }
     if (options.formattingEnabled() && !generator->formattingIsPossible()){
      if (numBadFormatting++ < IO_ERROR_REPORT_LENGTH || options.printDebugInfo()) {
        badFormattedFiles.push_back(outFilePath);
      }
     }
     ++i;
    }

    if (numBadInput){
      printIOErrorReport(numBadInput, badInputFiles, "read input");
      IOError = true;
    }
    if (numBadOutput){
      printIOErrorReport(numBadOutput, badOutputFiles, "write output");
      IOError = true;
    }
    if (numBadFormatting){
      printIOErrorReport(numBadFormatting, badFormattedFiles, "reformat");
    }

    highlight::CodeGenerator::deleteInstance();
    return (initError || IOError) ? EXIT_FAILURE : EXIT_SUCCESS;
}


int main(int argc, char **argv) {
  HighlightApp app;
  return app.run(argc, argv);
}
