/* Copyright (C) 2005-2015 Holger Ruckdeschel <holger@hoicher.de>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 */
#ifndef QUEUE_H
#define QUEUE_H

#include "common.h"

#include <list>

#if defined(HAVE_PTHREAD)
# include <pthread.h>
#elif defined(WIN32)
# include <windows.h>
#else
# error no thread support is available
#endif


template <typename T>
class Queue {
      private:
#if defined(HAVE_PTHREAD)
	pthread_mutex_t mtx;
	pthread_cond_t cv;
#elif defined(WIN32)
	CRITICAL_SECTION cs;
	HANDLE sem;
#endif

	std::list<T> queue;

      public:
	Queue();
	~Queue();

      public:
	void put(const T& elem);
	T get();
	void clear();
};


template <typename T>
Queue<T>::Queue()
{
#if defined(HAVE_PTHREAD)
	pthread_mutex_init(&mtx, NULL);
	pthread_cond_init(&cv, NULL);
#elif defined(WIN32)
	InitializeCriticalSection(&cs);
	sem = CreateSemaphore(NULL, 0, INT_MAX, NULL);
#endif
}

template <typename T>
Queue<T>::~Queue()
{
#if defined(HAVE_PTHREAD)
	pthread_mutex_destroy(&mtx);
	pthread_cond_destroy(&cv);
#elif defined(WIN32)
	DeleteCriticalSection(&cs);
	CloseHandle(sem);
#endif
}

template <typename T>
void Queue<T>::put(const T& elem)
{
#if defined(HAVE_PTHREAD)
	pthread_mutex_lock(&mtx);
	queue.push_back(elem);
	pthread_cond_signal(&cv);
	pthread_mutex_unlock(&mtx);
#elif defined(WIN32)
	EnterCriticalSection(&cs);
	queue.push_back(elem);
	ReleaseSemaphore(sem, 1, NULL);
	LeaveCriticalSection(&cs);
#endif
}

template <typename T>
T Queue<T>::get()
{
#if defined(HAVE_PTHREAD)
	pthread_mutex_lock(&mtx);
	while (queue.empty()) {
		pthread_cond_wait(&cv, &mtx);
	}
	T elem = queue.front();
	queue.pop_front();
	pthread_mutex_unlock(&mtx);
	return elem;
#elif defined(WIN32)
	WaitForSingleObject(sem, INFINITE);
	EnterCriticalSection(&cs);
	ASSERT(!queue.empty());
	T elem = queue.front();
	queue.pop_front();
	LeaveCriticalSection(&cs);
	return elem;
#endif
}

template <typename T>
void Queue<T>::clear()
{
#if defined(HAVE_PTHREAD)
	pthread_mutex_lock(&mtx);
	queue.erase(queue.begin(), queue.end());
	pthread_mutex_unlock(&mtx);
#elif defined(WIN32)
	EnterCriticalSection(&cs);
	/* There seems to be no way to simply set the semaphore to zero,
	 * thus we cannot just clear the queue. Instead we remove each
	 * element individually, decrementing the semaphore each time
	 * (like the normal get() procedure does). */
	while (queue.size() != 0) {
		WaitForSingleObject(sem, INFINITE);
		queue.pop_front();
	}
	LeaveCriticalSection(&cs);
#endif
}

#endif // QUEUE_H
