/* 
 * hotsmtpd.c -- ESMTP-HTTPMail gateway. Uses hotwayd code to send email
 *
 * Created: Trever Adams <tadams-lists@myrealbox.com>, 26-Jan-2004
 *
 * This code uses libxml2 and the hotwayd modified libghttp-1.0.9.
 * Code based on hotwayd/hotimapd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <string.h>
#include <sasl/sasl.h>
#include <sasl/saslutil.h>
#include "../httpmail.h"
#include "../hotwayd.h" 
#include "hotsmtpd.h"
#include "../inet.h"
#include <sys/time.h>
#include "../libghttp-1.0.9-mod/ghttp.h"
#include "../libghttp-1.0.9-mod/http_resp.h"
#include "../libghttp-1.0.9-mod/http_global.h"
#include "../libghttp-1.0.9-mod/http_req.h"
#include "../libghttp-1.0.9-mod/http_lazy.h"
#include "commands_smtp.h"
#include "../commands.h"

#define AUTHORIZATION_TIMEOUT 15
#define AUTHENTICATED_TIMEOUT 30

static int state; /* current smtp state */
static int timeout, timed_out = 0;

/* Global variables */
sasl_conn_t *sasl_conn;
char input[N_BUFLEN];
char username[N_BUFLEN];
int log_level = 1; /* log level of 0-2 depending how much detail you want */
char *prog; /* program name */
LISTS HEADERS;
LISTS MAILDATA;

/* extern FILE *stdin; */
extern int validate_username(char *);
extern cmdtable *cached_command(cmdtable *command, int argc, char **argv);
char *folder=NULL; /* which folder of our mailbox to work on */
char buffer[N_BUFFER];
extern char version[]; /* version number of hotway, set in VERSION */

void main_loop(void);
void usage(void);
int args_sanity_check(void);

/****** MISC FUNCTIONS ******/
/* get the next command from the client 
 * and put it in the global input buffer
 */
void get_input(void)
{
  alarm(timeout);
  memset(input, 0, N_BUFLEN);
  if (fgets(input, N_BUFLEN, stdin) == NULL) {
    if (errno != EINTR) {
      timed_out = errno;
      alarm(0);
      strlcpy(input, "timeout\n", N_BUFLEN);
    } else { 
      if (timed_out) {
	strlcpy(input, "timeout\n", N_BUFLEN);
      }
    }
  }

  alarm(0);
}

static void sig_alarm(int sig)
{
  timed_out = -1;
}	

/***** COMMAND HANDLERS ******/

void not_spoken_cmd( cmdtable *c, int argc, char *argv[])
{
  PFSOUT("502 Command %s not implemented", argv[0]);
  PCRLF;
  /*  register_command(NULL, NULL, 0, NULL, NULL); */
}

int stat_cmd_no= 0;

/** do_login is the callback to authenticate via sasl.  Do not change the
 * prototype. */
int do_login(sasl_conn_t *conn, void *context, const char *uname, const char *pwd, unsigned passlen,
	     struct propctx *propctx)
{
  username[0] = '\0'; /* reset username */

  switch (validate_username((char *)uname)) {
  case E_LOGIN_NO_DOMAIN:
    PFSOUT("501 Username invalid, specify your full address, e.g. %s@hotmail.com\n", uname);
    return SASL_NOUSER;
    break;
  case E_LOGIN_UNRECOGNISED_DOMAIN:
    PFSOUT("501 Domain must be hotmail/msn/lycos/spray) in %s\n", uname);
    return SASL_NOUSER;
    break;
  case E_LOGIN_OK:
    {
      char *domain_begin;
      strlcpy(username,uname, N_BUFLEN); 
      domain_begin = index( username, '@');
      /* for hotmail.com addresses remove domain, otherwise leave it */
      if (domain_begin && (strcasecmp("@hotmail.com",domain_begin) == 0)) {
	*domain_begin = '\0'; 
      }
      /* flush the cache */
      /*       register_command(NULL, NULL, 0, NULL, NULL); */
    }
    break;
  default:
    PFSOUT("451 httpmail login BAD %s[%d] Internal error", __FILE__, __LINE__);
    PCRLF;
    return SASL_NOUSER;
  }

  if (username[0] == '\0') {
    PFSOUT("501 Invalid username in %s", uname);
    PCRLF;
    return SASL_NOUSER;
  }

#ifdef DEBUG
  fprintf(stderr,"attempting to login with %s\n",username);
#endif

  switch (httpmail_authenticate_user(username, (char *)pwd)) {
  case INVALIDUSER:
    if (log_level > 0)
      LOG("AUTH failed, server said username %s invalid or server busy, host %s\n", username, inet_client_name);
    PSOUT("421 Mail server too busy (or possibly userid invalid)");
    PCRLF;
    return SASL_NOUSER;
  case PASSWORDFAILED:
    if (log_level > 0)
      LOG("AUTH failed, server said password for user %s invalid, host=%s\n", username, inet_client_name);
    PSOUT("501 Mail server said password was invalid");
    PCRLF;
    return SASL_NOUSER;
  case ERROR:
    if (log_level > 0)
      LOG("AUTH failed, unexpected response from server");
    PSOUT("451 Remote server returned unexpected status");
    PCRLF;
    return SASL_NOUSER;
  case 0:
    if (log_level > 0)
      LOG("AUTH failed, couldn't find folder %s", folder);
    snprintf(buffer, N_BUFFER, "550 Unable to find folder %s on remote server", folder);
    PSOUT(buffer);
    PCRLF;
    return SASL_NOUSER;
  case 1:
    break; /* we logged in ok */
  default:
    if (log_level > 0)
      LOG("AUTH failed, internal error");
    PSOUT("451 Internal error");
    PCRLF;
    return SASL_NOUSER;
  }

  /* log it */
  if (log_level > 1)
    LOG("  IN user=%s host=%s\n", username, inet_client_name);
  return SASL_OK;
}

static void hello_cmd( cmdtable *c, int argc, char *argv[])
{
  int result;
  const char *result_string;
  char supported[100]={'\0'};

  if(strlen(argv[1]) > MAX_DOMAIN_LEN)
    {
      PSOUT("501 domain name too long");
      PCRLF;
      return;
    }

  if(argc < 1) goto no_domain;
  if(strlen(argv[1]) < 3) 
    {
    no_domain:
      if(strcasecmp(argv[0], "EHLO") ==0 ) PSOUT("501 Syntax: EHLO <domain>");
      else PSOUT("501 Syntax: HELO <domain>");
      PCRLF;
      return;
    }

  /* Make a new context for this connection */
  result=sasl_server_new("hotsmtpd", NULL, NULL, NULL, NULL, NULL, 0, &sasl_conn);

  result=sasl_listmech(sasl_conn, NULL, NULL, " ", NULL, &result_string, NULL, NULL);
	
  if(strstr(result_string, "LOGIN")) strcpy(supported, "LOGIN ");
  if(strstr(result_string, "PLAIN")) strcat(supported, "PLAIN");

  PFSOUT("250-%s Pleased to meet you", argv[1]);
  PCRLF;
  if(strcasecmp(argv[0], "EHLO")==0) {
    /* EHLO is ESMTP and needs to show supported authentication and extensions */ 
    PFSOUT("250-AUTH %s", supported);
    PCRLF;
    PFSOUT("250-AUTH=%s", supported);
    PCRLF;
    PSOUT("250 8BITMIME");
    PCRLF;
  }
  state = HELLO;
}

/* Many want better security... sorry.  We need plain text passwords and user
 * names so we can authenticate against hotmail.  Therefore LOGIN and PLAIN
 * are our only options. */ 
static void auth_cmd( cmdtable *c, int argc, char *argv[])
{
  int result;
  char var64[MAX_COMMAND_LEN]={'\0'};
  unsigned var64len;
  const char *out;
  unsigned outlen;

  sasl_decode64(argv[2], strlen(argv[2]), var64, MAX_COMMAND_LEN, &var64len);
  result=sasl_server_start(sasl_conn, argv[1], var64, var64len, &out, &outlen);

  while(result == SASL_CONTINUE) /* Fall through for anything but a continue */
    {
      /* All our responses to and from the client in SASL are base64 encoded. */
      if(sasl_encode64(out, outlen, var64, MAX_COMMAND_LEN, &var64len) == SASL_BUFOVER) {
	PSOUT("501 Buffer overflow attempted in SASL, REJECTING CONNECTION");
	PCRLF;
      }
      PFSOUT("334 %*s", var64len, var64); /* Hey, give me input */
      PCRLF;
      get_input();
      sasl_decode64(input, strlen(input), var64, MAX_COMMAND_LEN, &var64len);
      result=sasl_server_step(sasl_conn, var64, var64len, &out, &outlen);
    }
  sasl_dispose(&sasl_conn);
  if (result == SASL_OK)
    {
      PSOUT("235 Authentication successful!");
      PCRLF;
      state = AUTHENTICATED;
      return;
    }
  if (result !=SASL_OK)
    {
      PSOUT("501 Authentication failed!");
      PCRLF;
      return;
    }
}

static void noop_cmd( cmdtable *c, int argc, char *argv[])
{
  PSOUT("250 NOOP OK");
  PCRLF;
}

static void logout_cmd( cmdtable *c, int argc, char *argv[])
{
  PSOUT("221 Service closing transmission channel");
  PCRLF;
  if (state == AUTHENTICATED) {
    delete_list(&HEADERS);
    delete_list(&MAILDATA);
    state = LOGOUT;
  } else {
    state = TERMINATE;
  }
}

static void rset_cmd( cmdtable *c, int argc, char *argv[])
{
  delete_list(&HEADERS);
  PSOUT("250 OK");
  PCRLF;
  state = HELLO;
}

static void data_cmd( cmdtable *c, int argc, char *argv[])
{
  PSOUT("354 End data with <CR><LF>.<CR><LF>");
  PCRLF;
  state = DATA;
  process_mail(&MAILDATA, &state);
  delete_list(&HEADERS);
  delete_list(&MAILDATA);
}

void mail_cmd( cmdtable *c, int argc, char *argv[])
{
  if(strncasecmp(argv[1], "FROM:", 5)!=0) {
    PSOUT("501 Syntax: MAIL FROM: <address>");
    PCRLF;
    return;
  }
  if(strlen(input)>MAX_COMMAND_LEN) {
    PSOUT("501 MAIL line too long");
    PCRLF;
    return;
  }
  unbork_input(argc, argv);
  process_line(&HEADERS, &state);
  PSOUT("250 Ok");
  PCRLF;
  state = MAILFROM;
}

void rcpt_cmd( cmdtable *c, int argc, char *argv[])
{
  if(strncasecmp(argv[1], "TO:", 3)!=0) {
    PSOUT("501 Syntax: RCPT TO: <address>");
    PCRLF;
    return;
  }
  if(strlen(input) > MAX_COMMAND_LEN) {
    PSOUT("501 RCPT line too long");
    PCRLF;
    return;
  }
  unbork_input(argc, argv);
  process_line(&HEADERS, &state);
  PSOUT("250 Ok");
  PCRLF;
  state = RCPTTO;
}

void exit_cmd( cmdtable *c, int argc, char *argv[])
{
  PSOUT("221 Timed out. Service closing transmission channel");
  PCRLF;
  state = TERMINATE;
}

/* 
 * Global jumptable for commands
 */
const cmdtable commands[] = {
  { "AUTH", (HELLO), &auth_cmd },
  { "HELO", (NEW), &hello_cmd },
  { "EHLO", (NEW), &hello_cmd },
  { "MAIL", (AUTHENTICATED), &mail_cmd },
  { "RCPT", (MAILFROM | RCPTTO), &rcpt_cmd },
  { "DATA", (RCPTTO), &data_cmd },
  { "RSET", (AUTHENTICATED | RCPTTO | MAILFROM), &rset_cmd },
  { "NOOP", (AUTHENTICATED | RCPTTO | MAILFROM | NEW), &noop_cmd },
  { "QUIT", (HELLO | AUTHENTICATED | RCPTTO | MAILFROM | NEW), &logout_cmd },
  { "", 0, NULL }
};

/* Function: main_loop()
 * Wait for SMTP commands. Takes care of transition between states (defined in
 * SMTP standard) and uses the command jump table (commands[]) to determine
 * which function is called upon a given command.
 */
void main_loop(void)
{
  int   argc, quote_cnt = 0;
  char *argv[MAXARGC], *x;
  int   cmd_implemented;
  cmdtable *c;

  state = NEW; 
  username[0] = '\0';

  PFSOUT("220 %s SMTP %s v%s. ESMTP-HTTPMail Gateway based on hotwayd.", inet_server_name, prog, version);
  PCRLF;
		
  while(state < LOGOUT) {	
    /* make sure we've flushed all output before waiting on client */
    PFLUSH;

    /* set timeout according to our state */
    switch (state) {
    case NEW:
      timeout = AUTHORIZATION_TIMEOUT;
      break;

    case AUTHENTICATED:
      timeout = AUTHENTICATED_TIMEOUT;
      break;
    }

    /* get the input from the client */	
    get_input();

#ifdef DEBUG
    if (log_level > 1)
      LOG("Command: %s",input);
#endif
	
    /* init before continuing */
    cmd_implemented = 0;

    /* init argc, argv */
    for (argc = MAXARGC; argc; argv[--argc] = NULL);

    /* strip command from args --
     *  	assumes the cmd points to a properly null terminated string 
     *  	we'll strip off the crlf as well!! 
     */
    for(x = input; *x != '\0'; x++) {
      if (*x == ' ' || *x == '\n' || *x == '\r') {
	*x = '\0';
	if (argv[argc] == NULL) { continue; }
	else {
	  argc++;
	  if (argc >= MAXARGC) {
	    break;
	  }
	}
      } else {
	if (argv[argc] == NULL)
	  argv[argc] = x;
      }
    }

    /* check for empty command */
    if (!argc) {
      PSOUT("500 Error: Empty command");
      PCRLF;
      continue;
    }	

    
    /* cycle through jumptable */
    for (c = (cmdtable *)commands; c->handler != NULL; c++) {
      if (!strcasecmp(c->name, argv[0])) {
	cmd_implemented++;
	if (c->states & state) {
	  (c->handler)(c, argc, argv); /* run the command handler */
	} else {
	  if(!(state & AUTHENTICATED)) PFSOUT("503 Authenticate first! - %s command is not available right now.", argv[0]);
	  else PFSOUT("503 Bad sequence of commands - %s command is not available right now.", argv[0]);
	  PCRLF;
	}
      }
    }

    if (!cmd_implemented) {
      PFSOUT("502 Error: %s Command not implemented", argv[0]);
      PCRLF;
    }
  }
  PFLUSH;

  snprintf(input, N_BUFLEN, "user=%s host=%s", username, inet_client_name);

  if (timed_out) {
    if ((timed_out == 1) || (timed_out == 9)) {
      if (log_level > 0)
	LOG("TIMEOUT %s (%d) client hangup?\n", input, timed_out);
    } else {
      if (log_level > 0)
	LOG("TIMEOUT %s (%d)\n", input, timed_out);
    }
    return;
  }

  if (state == TERMINATE) {
    if (log_level > 1)
      LOG("EXIT host=%s\n", inet_client_name);
    return;
  }

  if (state == LOGOUT) {
    if (log_level > 1)
      LOG(" OUT\n");
    state = TERMINATE;
  }
}


/* Function: main
 * Entry point for hotsmtpd. Takes some command line arguments, processes them
 * and then calls main_loop() to wait for commands. Once main_loop() returns
 * we cleanup our mess and exit.
 */
int main(int argc, char **argv)
{
  cmdtable *c= (cmdtable *) commands;
  int		opt;
  char *prog_name;

  prog_name = strrchr(argv[0], '/'); /* remove slashes from progname */

  if (prog_name && prog_name+1 != '\0')
    prog=strdup(prog_name+1);
  else
    prog=strdup(argv[0]);
 
  if(init_sasl() != 0)
    {
      PSOUT("451 Unable to initialize sasl library!");
      PCRLF;
      exit(-1);
    }
 
  while ((opt = getopt(argc, argv, "a:hl:p:u:q:v")) != -1) {
    char *access_list, *proxy, *proxy_username, *proxy_password;
    
    switch (opt) {

    case 'a':
      access_list = (char *)malloc(strlen(optarg)+1);
      strlcpy(access_list, optarg, strlen(optarg)+1);
      set_access_list(access_list);
      break;
	  
    case 'h': /* display the usage page */
      usage();
      return 0;

    case 'l': /* setup proxy usage */
      log_level = str2val(optarg, "log level", 0, 3);
      break;

    case 'p': /* setup proxy usage */
      proxy = (char *) malloc (strlen(optarg)+1);
      strlcpy(proxy, optarg, strlen(optarg)+1);
      set_proxy(proxy);
      break;

    case 'u':
      proxy_username = (char *) malloc (strlen(optarg)+1);
      strlcpy(proxy_username, optarg, strlen(optarg)+1);
      set_proxy_username(proxy_username);
      break;

    case 'q':
      proxy_password = (char *) malloc (strlen(optarg)+1);
      strlcpy(proxy_password, optarg, strlen(optarg)+1);
      set_proxy_password(proxy_password);
      break;

    case 'v':
      PFSOUT("220 %s v%s - http://hotwayd.sourceforge.net/\n", prog, version);
      return 0;	  
	  
    default:
      usage();
      return -1;
    }
  }

  /* check validity of command line arguments */
  if (!proxy_sanity_check()) {
    usage();
    return -1;
  }

  OPENLOG;
  inet_init();
  signal(SIGALRM, sig_alarm);

  /*  lazy_set(&lazy_handler);*/ /* We don't want lazy handlers */

  while( c && c->handler ){
    if( strcmp( c->name, "stat")== 0 ){
      stat_cmd_no= (int) (c- commands);
    }
    c++;
  }

  main_loop();

  /* ok let's clean up our mess as much as we can :-) */
  httpmail_destroy();
  sasl_done();
  CLOSELOG;
  DPRINTF("exiting.\n");
  return(0);
}


void usage(void) {
  printf("Usage: %s [options and arguments]\n"
	 "  -a <accesslist> specify a file listing permitted HTTPMail accounts\n"
	 "  -h print usage page\n"
	 "  -l <loglevel> specify the logging level (0-2)\n"
	 "  -p [proxy:port] specify proxy\n"
	 "  -r set messages as read after downloading\n"
	 "  -u <username> specify proxy username\n"
	 "  -q <password> specify proxy password\n"
	 "  -v display version information\n"
	 "Example: %s -p http://proxy:8080 -u dave -q proxypass\n"
	 "Note: proxy can be specified without a username or password\n", prog, prog);
}

int init_sasl(void)
{
  int result;

  static sasl_callback_t callbacks[] = {
    {
      SASL_CB_SERVER_USERDB_CHECKPASS, do_login, NULL /* We want to check the user/pass ourselves! */
    }, {
      SASL_CB_LIST_END, NULL, NULL
    }
  };

  /* Initialize SASL */
  result = sasl_server_init(callbacks, "hotsmtpd");

  /* Were we successful? */
  switch(result)
    {
    case SASL_OK:
      return 0;
      break;
    default: /* Couldn't initialize sasl... we have a BIG problem */
      return -1;
      break;
    }
}
