package lib::plugin::buffer;

use lib::plugin::message;
use Fcntl;

use strict;

# Constructors

sub new {
  my($class, %init) = @_;

  my $self;

  $self = bless {
    'fd' => undef,
    'buffer_in' => undef,
    'buffer_out' => undef,
    'len' => 0,
    'timeout' => 0.2,
    'send_queue' => [],
    'recv_queue' => [],
    'lock' => 0,
    'send_now' => undef
  }, $class;

  foreach (keys %init) {
    if(exists $self->{$_}) {
      $self->{$_}=$init{$_};
    }
  }

  if(exists $self->{'fd'}) {
    fcntl($self->{'fd'}, F_SETFL, O_NONBLOCK) or die "can't fcntl F_SETFL: $!";
  }

  lib::debug("Created new buffer to fd $self->{'fd'} - ".$init{'fd'});

  return $self;
}

sub read {
  my $self = shift;
  my $bytes_read=0;
  my $total_read=0;

  my $found;

  my $r_fds="";

  vec($r_fds, fileno($self->{'fd'}), 1)=1;

  $found = select($r_fds, undef, undef, $self->{'timeout'});

  if($found) {
    flock $self->{'fd'}, 2;
    while(($bytes_read=sysread($self->{'fd'}, $self->{'buffer_in'}, 1024,  $self->{'len'}))) {
      $self->{'len'}+=$bytes_read;
      $total_read+=$bytes_read;
#      lib::debug("microRead $bytes_read - ".$self->{'buffer_in'}." - ".$self->{'len'});
    }
    flock $self->{'fd'}, 8;
#    lib::debug("Returned $bytes_read");
  }

#  lib::debug("Read $self->{'fd'} $total_read - $found - ".$self->{'buffer_in'});

  if(index($self->{'buffer_in'}, "\n\n") >=0 ) {
    $self->process_message();
  }

  return $total_read;
}

sub write {
  my($self, $data) = @_;
  my $bytes_written=0;
  my $total_written=0;

  my ($found, $message, $text, $len, @failed);

  my $w_fds="";

  flock $self->{'fd'}, 2;

  if($self->{'send_now'}) {
    while(($self->{'send_now'}) && ($bytes_written=syswrite($self->{'fd'}, $self->{'send_now'}, length($self->{'send_now'})))) {
      $self->{'send_now'} = substr($self->{'send_now'}, $bytes_written);
      lib::debug("high priority semiwritten $bytes_written (left $self->{'len'})");
    }

    if($self->{'send_now'}) {
      return 0;
    }
  }

  while($message=shift(@{$self->{'send_queue'}})) {
    lib::debug("transferring message [method $message->{'method'}] from send queue to write buffer");
    $text=$message->as_string()."\n";
    $len=length($text);

    $self->{'buffer_out'}.=$text;
    $self->{'len'}+=$len;

#    lib::debug("which is now <$self->{'buffer_out'}>");

    flock $self->{'fd'}, 2;

    while(($self->{'len'} > 0) && ($bytes_written=syswrite($self->{'fd'}, $self->{'buffer_out'}, $self->{'len'}))) {
      $self->{'len'}-=$bytes_written;
      $self->{'buffer_out'} = substr($self->{'buffer_out'}, $bytes_written, $self->{'len'});
      $total_written+=$bytes_written;
      lib::debug("semiwritten $bytes_written (left $self->{'len'})");
    }

    lib::debug("Written $total_written");

    if(($self->{'len'})) {
      # Pipe was full and we couldn't send more data down it.
      # Store the data as high-priority and let the caller know, just in case
      $self->{'send_now'}.=$self->{'buffer_out'};
      $self->{'buffer_out'}="";
      $self->{'len'}=0;
      lib::debug("storing unsent ".length($self->{'send_now'})." bytes as high priority");
      return($total_written);
    }


    flock $self->{'fd'}, 8;
  }

  return($total_written);
}

sub flush {
  my($self, $data) = @_;
  my $bytes_flushed=0;
  my $total_flushed=0;

  my $found;

  my $w_fds="";
 
  vec($w_fds, fileno($self->{'fd'}), 1)=1;

  $found = select(undef, $w_fds, undef, $self->{'timeout'});

  if($found) {
    flock $self->{'fd'}, 2;
    while(($self->{'len'} > 0) && ($bytes_flushed=syswrite($self->{'fd'}, $self->{'buffer_out'}, $self->{'len'}))) {
      $self->{'len'}-=$bytes_flushed;
      $self->{'buffer_out'} = substr($self->{'buffer_out'}, 0 - $self->{'len'} + 1);
      $total_flushed+=$bytes_flushed;
    }
    flock $self->{'fd'}, 8;
  }


  return($total_flushed);
}

sub push {
  my ($self, $message) = @_;

#  lib::debug("Pushing ".$message->as_string);

  flock $self->{'fd'}, 2; # ?

  push(@{$self->{'send_queue'}}, $message);
  my $count=@{$self->{'send_queue'}};
  lib::debug("pushing message to send_queue, now there are: ".$count);

  flock $self->{'fd'}, 8; # ?

  $self->write;

  return 1;
}

sub pop {
  my $self = shift;

  $self->read;

  flock $self->{'fd'}, 2; # ?

  my $count=@{$self->{'recv_queue'}};
  my $message=undef;

  if($count > 0) {
    $message = shift(@{$self->{'recv_queue'}});
    lib::debug("popping message from recv_queue, left: ".($count-1)." method: $message->{'method'}");
  } else {
#    lib::debug("No one waiting in the queue");
  }
  flock $self->{'fd'}, 8; # ?

  return $message;
}

sub process_message {
  my $self = shift;

#  lib::debug("processing data <$self->{'buffer_in'}>");

  my ($pos, $current, $line, $key, $val);


#  $self->read;

#  lib::debug("in process message ".$self->{'buffer_in'});

  flock $self->{'fd'}, 2; # ?

  $pos=index($self->{'buffer_in'},"\n\n");

  while($pos >= 0) {
     my $message = new lib::plugin::message;
#    lib::debug("pos -> $pos");

    $current=substr($self->{'buffer_in'}, 0, $pos+1);

    if(($pos + 2) < length($self->{'buffer_in'}) ) {
      $self->{'buffer_in'}=substr($self->{'buffer_in'},$pos+2);
    } else {
      $self->{'buffer_in'}="";
    }

    $self->{'len'}=length($self->{'buffer_in'});

#    lib::debug($self->{'buffer_in'}." - $current");

    foreach $line (split("\n", $current)) {
#      lib::debug("-> $line");
      if(! $message->{'method'} || ! $message->{'version'}) {
       if($line=~/([^\s]+)\s+([\d\.]+)/) {
         $message->{'method'}=$1;
         $message->{'version'}=$2;
         }
      } else {
        if(($key,$val)=split(":",$line,2)) {
          if($val=lib::plugin::decode($val)) {
             $message->add_header($key, $val);
           }
        }
#        lib::debug(">$key - $val<");
      }
    }

    push(@{$self->{'recv_queue'}}, $message);
    $pos=index($self->{'buffer_in'},"\n\n");
#    lib::debug("Adding message (".$message->as_string.")");
  }

  flock $self->{'fd'}, 8; # ?
#  lib::debug("after process <$self->{'buffer_in'}>");

  return 1;
}

return 1;
