/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <connio.h>

#include <openssl/ssl.h>
#include <openssl/err.h>

#include <mdb.h>
#include <hulautil.h>

#include "webadminp.h"

#if !defined(A_INTERNET_EMAIL_ADDRESS)
#define	A_INTERNET_EMAIL_ADDRESS	"Internet EMail Address"
#endif

unsigned long		DefaultConnectionTimeout	=	360;
unsigned long		DefaultLogoID					=	0;
unsigned long		DefaultLanguage				=	4;
unsigned long		DefaultTemplate				=	0;
unsigned char		DefaultDateFormatShort[128]= "%m/%d/%y";
unsigned char		DefaultDateFormatLong[128]	= "%A, %B %d, %Y";
unsigned char		DefaultTimeFormat[128]		= "%I:%M %p";
long					DefaultTimezoneOffset		=	0;
unsigned long		DefaultTimezoneID				=	0;
unsigned char		*DefaultMonthShort[12]		= { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
unsigned char		*DefaultMonthLong[12]		= { "January", "February", "March", "April", "May", "June", "July", "August", "September", "Octobert", "November", "December"};
unsigned char		*DefaultWDayShort[7]			= { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
unsigned char		*DefaultWDayLong[7]			= { "Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"};
unsigned char		*DefaultAMPM[2]				= { "AM", "PM"};
unsigned long		SessionMonitorInterval		=	5*60;

SessionStruct		*SessionDB[MAX_SESSION_ENTRIES];
XplSemaphore		SessionMutex[MAX_SESSION_ENTRIES];
BOOL					SessionHasKey[MAX_SESSION_ENTRIES];
unsigned long		SessionStack[MAX_SESSION_ENTRIES];
long					SessionStackPtr = MAX_SESSION_ENTRIES - 1;

TSessionStruct		*TSessionDB[MAX_TSESSION_ENTRIES];
XplSemaphore		TSessionMutex[MAX_TSESSION_ENTRIES];
unsigned long		TSessionStack[MAX_TSESSION_ENTRIES];
long					TSessionStackPtr = MAX_TSESSION_ENTRIES - 1;

BOOL
WAConnectUserToNMAPServer(SessionStruct *Session)
{
	return(FALSE);
}

void
SessionMonitor(void)
{
	unsigned long	Counter;
	int 				NotUsed;
	SessionStruct	*Session;
	TSessionStruct	*TSession;

	XplSignalLocalSemaphore(ExitSemaphore);

	XplRenameThread(XplGetThreadID(), "SessionExpire Monitor");

	while (!Exiting) {
		for (Counter = 0; (Counter < SessionMonitorInterval) && !Exiting; Counter++) {
			XplDelay(1000);
		}

		if (!Exiting) {
			for (Counter = 0; Counter < MAX_SESSION_ENTRIES; Counter++) {
				if (SessionDB[Counter] == NULL) {
					continue;
				}

				XplExamineLocalSemaphore(SessionMutex[Counter], NotUsed);
				/* Do not try to use a connection that's in use */

				if (NotUsed > 0) {
					XplWaitOnLocalSemaphore(SessionMutex[Counter]);

					if ((SessionDB[Counter]->Timestamp + SessionDB[Counter]->ConnectionTimeout) < time(NULL)) {
						Session = SessionDB[Counter];
						SessionDB[Counter] = NULL;

						/* Note: We still hold the mutex lock; WADestroySession will release it for us */
						WADestroySession(Session);
					} else {
						XplSignalLocalSemaphore(SessionMutex[Counter]);
					}
				}
			}
		}

		if (!Exiting) {
			for (Counter = 0; Counter < MAX_TSESSION_ENTRIES; Counter++) {
				if (TSessionDB[Counter] == NULL) {
					continue;
				}

				XplExamineLocalSemaphore(TSessionMutex[Counter], NotUsed);
				/* Do not try to use a connection that's in use */

				if (NotUsed > 0) {
					XplWaitOnLocalSemaphore(TSessionMutex[Counter]);
				
					if ((TSessionDB[Counter]->Timestamp + 300) < time(NULL)) {
						TSession = TSessionDB[Counter];
						TSessionDB[Counter] = NULL;

						/* Note: We still hold the mutex lock; DestroyTSession will release it for us */
						DestroyTSession(TSession);
					} else {
						XplSignalLocalSemaphore(TSessionMutex[Counter]);
					}
				}
			}
		}
	}

	XplWaitOnLocalSemaphore(ExitSemaphore);

	return;
}

BOOL
WADestroySession(SessionStruct *Session)
{
	unsigned long	i;

	if (!Session) {
		return(FALSE);
	}

	if (Session->User) {
		free(Session->User);
	}

	if (Session->UserDN) {
		free(Session->UserDN);
	}

	if (Session->Title) {
		free(Session->Title);
	}

	if (Session->NMAPs!=-1) {
		WASendNMAPServer(Session, "QUIT\r\n", 6);
		IPclose(Session->NMAPs);
		Session->NMAPs=-1;
	}

	if (Session->ObjectListCache) {
		FreeObjectList(Session->ObjectListCache);
	}

	if (Session->BrowseListCache) {
		FreeObjectList(Session->BrowseListCache);
	}

	FreeClassList(Session);

	if (Session->V) {
		MDBDestroyValueStruct(Session->V);
		Session->V = NULL;
	}

	if (Session->AllowedTemplates) {
		MDBDestroyValueStruct(Session->AllowedTemplates);
		Session->AllowedTemplates = NULL;
	}

	if (Session->CachedValues) {
		MDBDestroyValueStruct(Session->CachedValues);
	}

	if (Session->LastFormNames) {
		MDBDestroyValueStruct(Session->LastFormNames);
	}

	if (Session->LastFormValues) {
		MDBDestroyValueStruct(Session->LastFormValues);
	}

	if (Session->ErrorNames) {
		MDBDestroyValueStruct(Session->ErrorNames);
	}

	if (Session->ErrorStrings) {
		MDBDestroyValueStruct(Session->ErrorStrings);
	}

	if (Session->ModuleData) {
		for (i = 0; i < TModuleCount; i++) {
			if (TModules[i].DestroySession) {
				TModules[i].DestroySession(Session, Session->ModuleData[i]);
			}
		}

		for (i = 0; i < OModuleCount; i++) {
			if (OModules[i].DestroySession) {
				OModules[i].DestroySession(Session, Session->ModuleData[TModuleCount + i]);
			}
		}

		for (i = 0; i < SModuleCount; i++) {
			if (SModules[i].DestroySession) {
				SModules[i].DestroySession(Session, Session->ModuleData[TModuleCount + OModuleCount + i]);
			}
		}

		for (i = 0; i < DModuleCount; i++) {
			if (DModules[i].DestroySession) {
				DModules[i].DestroySession(Session, Session->ModuleData[TModuleCount + OModuleCount + SModuleCount + i]);
			}
		}

		for (i = 0; i < RModuleCount; i++) {
			if (RModules[i].DestroySession) {
				RModules[i].DestroySession(Session, Session->ModuleData[TModuleCount + OModuleCount + SModuleCount + DModuleCount + i]);
			}
		}

		free(Session->ModuleData);
	}

	SessionHasKey[Session->SessionID] = FALSE;

	XplWaitOnLocalSemaphore(SessionDBSemaphore);

	SessionDB[Session->SessionID] = NULL;
	SessionStack[++SessionStackPtr] = Session->SessionID;

	XplSignalLocalSemaphore(SessionMutex[Session->SessionID]);
	XplSignalLocalSemaphore(SessionDBSemaphore);

	if (Session->AuthHandle) {
		MDBRelease(Session->AuthHandle);
	}

	free(Session);

	return(TRUE);
}

BOOL
DestroyAllSessions(void)
{
	unsigned long	i;

	for (i = 0; i < MAX_SESSION_ENTRIES; i++) {
		if (SessionDB[i] != NULL) {
			XplWaitOnLocalSemaphore(SessionMutex[i]);
			WADestroySession(SessionDB[i]);
		}
	}

	for (i = 0; i < MAX_TSESSION_ENTRIES; i++) {
		if (TSessionDB[i] != NULL) {
			XplWaitOnLocalSemaphore(TSessionMutex[i]);
			DestroyTSession(TSessionDB[i]);
		}
	}

	return(TRUE);
}

BOOL
CreateSession(ConnectionStruct *Client, unsigned char *Username, unsigned char *Password, BOOL *Disabled)
{
	SessionStruct			*Session;
	unsigned char			UserDN[MDB_MAX_OBJECT_CHARS + 1];
	MDBValueStruct			*Config;
	int						i;

	Session = malloc(sizeof(SessionStruct));
	if (!Session) {
		return(FALSE);
	}

	memset(Session, 0, sizeof(SessionStruct));
	Client->Session = Session;
	Session->NMAPs = -1;

	snprintf(UserDN, sizeof(UserDN), "WA:%s\n", Username);
	XplRenameThread(XplGetThreadID(), UserDN);

	WAX500toMDB(Username, UserDN, NULL);

	Session->AuthHandle = MDBAuthenticate("WebAdmin", UserDN, Password);
	if (!Session->AuthHandle) {
		unsigned char		adminName[MDB_MAX_OBJECT_CHARS + 1];

		/* Make sure Username doesn't have any context provided */
		if (!strchr(Username, '.')) {
			/* Since we failed to login, and we have no context, try the server context */

			snprintf(adminName, sizeof(adminName), "%s\\%s", DefaultContext, Username);
			HulaStrNCpy(UserDN, adminName, sizeof(UserDN));
			Session->AuthHandle = MDBAuthenticate("WebAdmin", UserDN, Password);
		}

		if (!Session->AuthHandle) {
			i=sizeof(Client->cs);
			IPgetpeername(Client->s, (struct sockaddr *)&(Client->cs), &i);
			// syslog(LOG_INFO, "Unrecognized user %s login attempt from %d.%d.%d.%d", Username, Client->cs.sin_addr.s_net, Client->cs.sin_addr.s_host, Client->cs.sin_addr.s_lh, Client->cs.sin_addr.s_impno);
			free(Session);
			Client->Session=NULL;
			return(FALSE);
		}
	}

	if (!QuickCmp(UserDN, "\\Tree\\Context\\Admin")) {
	    free(Session);
	    Client->Session = NULL;
	    return FALSE;
	}

	Config = MDBCreateValueStruct(Session->AuthHandle, NULL);
	Session->V = MDBCreateValueStruct(Session->AuthHandle, NULL);

	Session->CachedValues = MDBShareContext(Session->V);
	Session->LastFormNames = MDBShareContext(Session->V);
	Session->LastFormValues = MDBShareContext(Session->V);

	Session->ErrorNames = MDBShareContext(Session->V);
	Session->ErrorStrings = MDBShareContext(Session->V);

	MDBGetServerInfo(NULL, Session->CurrentTree, Session->V);

	Session->User = strdup(Username);
	Session->UserDN = strdup(UserDN);

	/* We now have the user's DN, real username and NMAP store address; let's load the other defaults */
	Session->SessionUID = time(NULL);

	/* Connection Timeout */
	if (MDBRead(UserDN, "Novonyx:Timeout", Config) > 0) {
		Session->ConnectionTimeout = atol(Config->Value[0]) * 60;
	}

	if (Session->ConnectionTimeout < 1) {
		Session->ConnectionTimeout = DefaultConnectionTimeout;
	} else if (Session->ConnectionTimeout > 40 * 60) {
		Session->ConnectionTimeout = 40 * 60;
		MDBFreeValues(Config);
		MDBAddValue("40", Config);
		MDBWrite(UserDN, "Novonyx:Timeout", Config);
	}

	/* Language */
	MDBFreeValues(Config);
	if (MDBRead(UserDN, "Novonyx:Language", Config) > 0) {
		Session->Language = atol(Config->Value[0]);
	} else {
		Session->Language = DefaultLanguage;

		/* Determine language; check Client->Language */
		WAFindLanguage(Client->Language, Session->Language);
	}
	MDBFreeValues(Config);

	/*	Create the list of allowed templates	*/
	Session->AllowedTemplates = MDBCreateValueStruct(Session->AuthHandle, NULL);

	if (!IgnoreConfig && !IgnoreRoles) {
		MDBEnumStruct		*Roles;
		unsigned char		*CurrentRole;

		Roles = MDBCreateEnumStruct(Session->V);
		do {
			CurrentRole = (unsigned char *)MDBEnumerateObjectsEx((const unsigned char *)ConfigDN, NULL, NULL, TRUE, Roles, Config);
			if (CurrentRole) {
				unsigned char		*ptr;

				ptr = strrchr(CurrentRole, '\\');
				if (ptr) {
					ptr++;

					MDBAddValue(ptr, Session->AllowedTemplates);
				}
			}
		} while (CurrentRole);
		MDBDestroyEnumStruct(Roles, Session->V);
	} else {
		unsigned long		i;

		for (i = 0; i < TemplateCount; i++) {
			MDBAddValue(Templates[i]->Name, Session->AllowedTemplates);
		}
	}

	if (Session->AllowedTemplates->Used == 0) {
		MDBAddValue("Default", Session->AllowedTemplates);
	}

	WASetSessionTemplate(DefaultTemplate, Session->Language, Session);

	/* Since the template can be changed on the fly we keep the original around in case we want to get back */
	Session->InitialTemplate = Session->TemplateID;

	if (Session->LogoID == 0) {
		Session->LogoID = DefaultLogoID;
	}

	if (Session->ItemsPerPage == 0) {
		Session->ItemsPerPage = 400;
	}

	if (Session->Colors[COLOR_PAGE_FG][0] == '\0') {
		/* Use the colors from the template, since the user' colors are not set */
		snprintf(Session->Colors[COLOR_PAGE_FG], sizeof(Session->Colors[COLOR_PAGE_FG]), "%06x", Templates[Session->TemplateID]->Colors[0][0]);
		snprintf(Session->Colors[COLOR_PAGE_BG], sizeof(Session->Colors[COLOR_PAGE_BG]), "%06x", Templates[Session->TemplateID]->Colors[0][1]);
		snprintf(Session->Colors[COLOR_BORDER_FG], sizeof(Session->Colors[COLOR_BORDER_FG]), "%06x", Templates[Session->TemplateID]->Colors[1][0]);
		snprintf(Session->Colors[COLOR_BORDER_BG], sizeof(Session->Colors[COLOR_BORDER_BG]), "%06x", Templates[Session->TemplateID]->Colors[1][1]);
		snprintf(Session->Colors[COLOR_SECTION_FG], sizeof(Session->Colors[COLOR_SECTION_FG]), "%06x", Templates[Session->TemplateID]->Colors[2][0]);
		snprintf(Session->Colors[COLOR_SECTION_BG], sizeof(Session->Colors[COLOR_SECTION_BG]), "%06x", Templates[Session->TemplateID]->Colors[2][1]);
		snprintf(Session->Colors[COLOR_FIELDNAME_FG], sizeof(Session->Colors[COLOR_FIELDNAME_FG]), "%06x", Templates[Session->TemplateID]->Colors[3][0]);
		snprintf(Session->Colors[COLOR_FIELDNAME_BG], sizeof(Session->Colors[COLOR_FIELDNAME_BG]), "%06x", Templates[Session->TemplateID]->Colors[3][1]);
		snprintf(Session->Colors[COLOR_FIELDBODY_FG], sizeof(Session->Colors[COLOR_FIELDBODY_FG]), "%06x", Templates[Session->TemplateID]->Colors[4][0]);
		snprintf(Session->Colors[COLOR_FIELDBODY_BG], sizeof(Session->Colors[COLOR_FIELDBODY_BG]), "%06x", Templates[Session->TemplateID]->Colors[4][1]);
	}

#if 0
	/* Prepare MDB related settings */
	Session->CurrentObject[0] = '\\';
	MDBGetServerInfo(NULL, Session->CurrentObject + 1, Session->V);
	MDBGetObjectDetails(Session->CurrentObject, Session->CurrentClass, NULL, NULL, Session->V);
#endif

	/* Clean up ValueStruct */	
	MDBDestroyValueStruct(Config);

	Session->Timestamp = time(NULL);

	Session->ModuleData = malloc(sizeof(void *) * (TModuleCount + OModuleCount + SModuleCount + DModuleCount + RModuleCount));
	if (Session->ModuleData == NULL && (TModuleCount + OModuleCount + SModuleCount + DModuleCount + RModuleCount) > 0) {
		WADestroySession(Session);
		Client->Session=NULL;
		return(FALSE);
	} else {
		memset(Session->ModuleData, 0, sizeof(void *) * (TModuleCount + OModuleCount + SModuleCount + DModuleCount + RModuleCount));
	}
	
	/* We need to insert the session into our array */
	XplWaitOnLocalSemaphore(SessionDBSemaphore);
	if (SessionStackPtr >= 0) {
		Session->SessionID = SessionStack[SessionStackPtr--];
		SessionDB[Session->SessionID] = Session;
		SessionHasKey[Session->SessionID] = FALSE;

		XplWaitOnLocalSemaphore(SessionMutex[Session->SessionID]);
		XplSignalLocalSemaphore(SessionDBSemaphore);
	} else {
		/* Session Database is full, should clean up */
		XplSignalLocalSemaphore(SessionDBSemaphore);

		// FIXME - need error handling
		WADestroySession(Session);
		Client->Session = NULL;
		return(FALSE);
	}

	/* Init Template modules */
	for (i = 0; i < TModuleCount; i++) {
		if (TModules[i].InitSession) {
			TModules[i].InitSession(Session, &(Session->ModuleData[i]));
		}
	}

	/* Init object modules */
	for (i = 0; i < OModuleCount; i++) {
		if (OModules[i].InitSession) {
			OModules[i].InitSession(Session, &(Session->ModuleData[TModuleCount + i]));
		}
	}

	/* Init object modules */
	for (i = 0; i < SModuleCount; i++) {
		if (SModules[i].InitSession) {
			SModules[i].InitSession(Session, &(Session->ModuleData[TModuleCount + OModuleCount + i]));
		}
	}

	/* Init object modules */
	for (i = 0; i < DModuleCount; i++) {
		if (DModules[i].InitSession) {
			DModules[i].InitSession(Session, &(Session->ModuleData[TModuleCount + OModuleCount + SModuleCount + i]));
		}
	}

	/* Init object modules */
	for (i = 0; i < RModuleCount; i++) {
		if (RModules[i].InitSession) {
			RModules[i].InitSession(Session, &(Session->ModuleData[TModuleCount + OModuleCount + SModuleCount + DModuleCount + i]));
		}
	}

	/* We leave with the session mutex accquired */

	return(TRUE);
}

SessionStruct
*RetrieveSession(unsigned long SessionID, unsigned long SessionUID)
{
	SessionStruct	*Session;

	if (SessionID >= MAX_SESSION_ENTRIES) {
		return(NULL);
	}

	XplWaitOnLocalSemaphore(SessionMutex[SessionID]);
	Session = SessionDB[SessionID];
	
	if (!Session || Session->SessionUID != SessionUID) {
		XplSignalLocalSemaphore(SessionMutex[SessionID]);
		return(NULL);
	}

	XplRenameThread(XplGetThreadID(), Session->User);

	Session->Timestamp = time(NULL);

	/* We leave with the session mutex accquired */

	return(Session);
}

/*
	If a key was specified when the session was created, we allow the same
	session to be retrieved on a new login, rather than creating a new one.
*/
SessionStruct
*RetrieveSessionByKey(unsigned char *Key, unsigned char *UserDN, unsigned char *Pass)
{
	MDBHandle				AuthHandle;
	unsigned long			i;
	SessionStruct			*Session = NULL;
	unsigned char			adminName[MDB_MAX_OBJECT_CHARS + 1];

	AuthHandle = MDBAuthenticate("WebAdmin", UserDN, Pass);
	if (!AuthHandle) {
		unsigned char		*ptr;

		ptr = strrchr(UserDN, '\\');
		if (ptr) {
			ptr++;
		} else {
			ptr = UserDN;
		}

		/* We failed to login, so now we have to assume that we have no context */
		snprintf(adminName, sizeof(adminName), "%s\\%s", DefaultContext, ptr);
		/* Make sure the check below works */
		UserDN = adminName;

		AuthHandle = MDBAuthenticate("WebAdmin", UserDN, Pass);
		/* We don't log here, because we will be trying again in just a second */
	}

	if (AuthHandle) {
		/*
			Now that we know the user has rights we are going to try
			and find the session with a matching user name and key.
		*/

		for (i = 0; i < MAX_SESSION_ENTRIES && Session == NULL; i++) {
			if (SessionHasKey[i] == TRUE) {
				int	UsedCount;

				/* Don't try if the session is in use */
				XplExamineLocalSemaphore(SessionMutex[i], UsedCount);
				if (UsedCount > 0) {
					XplWaitOnLocalSemaphore(SessionMutex[i]);

					if (SessionDB[i]->UserDN &&
						WAQuickCmp(SessionDB[i]->UserDN, UserDN) &&
						WAQuickCmp(SessionDB[i]->Key, Key)) {

						/* We found our session - We leave it locked */
						Session = SessionDB[i];
					} else {
						XplSignalLocalSemaphore(SessionMutex[i]);
					}
				}
			}
		}
		MDBRelease(AuthHandle);
	}

	return(Session);
}

BOOL
SetSessionKey(SessionStruct *Session, unsigned char *Key)
{
	SessionHasKey[Session->SessionID] = TRUE;
	HulaStrNCpy(Session->Key, Key, sizeof(Session->Key));

	return(TRUE);
}

BOOL
DestroyTSession(TSessionStruct *TSession)
{
	unsigned char	Path[XPL_MAX_PATH+1];

	if (!TSession) {
		return(FALSE);
	}

	snprintf(Path, sizeof(Path), "%s/T%lu", WorkDir, TSession->SessionID);
	unlink(Path);

	XplWaitOnLocalSemaphore(TSessionDBSemaphore);

	TSessionDB[TSession->SessionID] = NULL;
	TSessionStack[++TSessionStackPtr] = TSession->SessionID;
	XplSignalLocalSemaphore(TSessionMutex[TSession->SessionID]);
	XplSignalLocalSemaphore(TSessionDBSemaphore);

	free(TSession);

	return(TRUE);
}

TSessionStruct
*RetrieveTSession(unsigned long SessionID, unsigned long SessionUID)
{
	TSessionStruct	*TSession;

	if (SessionID>=MAX_TSESSION_ENTRIES) {
		return(NULL);
	}

	XplWaitOnLocalSemaphore(TSessionMutex[SessionID]);
	TSession = TSessionDB[SessionID];
	
	if (!TSession || TSession->SessionUID!=SessionUID) {
		XplSignalLocalSemaphore(TSessionMutex[SessionID]);
		return(NULL);
	}

	return(TSession);
}


TSessionStruct
*CreateTSession(void)
{
	TSessionStruct			*TSession;

	TSession=malloc(sizeof(TSessionStruct));

	if (!TSession) {
		return(NULL);
	}

	memset(TSession, 0, sizeof(TSessionStruct));
	TSession->SessionUID=time(NULL);
	TSession->Timestamp=time(NULL);

	/* We need to insert the session into our array */
	XplWaitOnLocalSemaphore(TSessionDBSemaphore);
	if (TSessionStackPtr>=0) {
		TSession->SessionID=TSessionStack[TSessionStackPtr--];
		TSessionDB[TSession->SessionID]=TSession;
		XplWaitOnLocalSemaphore(TSessionMutex[TSession->SessionID]);
		XplSignalLocalSemaphore(TSessionDBSemaphore);
	} else {
		/* Session Database is full, should clean up */
		XplSignalLocalSemaphore(TSessionDBSemaphore);

		free(TSession);
		return(NULL);
	}

	return(TSession);
}


BOOL
SessionDBInit(void)
{
	unsigned long	i;

	memset(&SessionDB, 0, sizeof(SessionStruct *) * MAX_SESSION_ENTRIES);
	memset(&SessionHasKey, 0, sizeof(BOOL) * MAX_SESSION_ENTRIES);

	memset(&TSessionDB, 0, sizeof(TSessionStruct *) * MAX_TSESSION_ENTRIES);

	for (i = 0; i < MAX_SESSION_ENTRIES; i++) {
		SessionStack[i] = i;
		XplOpenLocalSemaphore(SessionMutex[i], 1);
	}

	for (i = 0; i < MAX_TSESSION_ENTRIES; i++) {
		TSessionStack[i] = i;
		XplOpenLocalSemaphore(TSessionMutex[i], 1);
	}
	return(TRUE);
}

BOOL
SessionDBShutdown(void)
{
	unsigned long	i;

	for (i = 0; i < MAX_SESSION_ENTRIES; i++) {
		XplCloseLocalSemaphore(SessionMutex[i]);
	}

	for (i = 0; i < MAX_TSESSION_ENTRIES; i++) {
		XplCloseLocalSemaphore(TSessionMutex[i]);
	}

	return(TRUE);
}
