/** 
* @file engine.c
* @brief smoothing second part
* @author serge guelton
* @date 2008-02-13
*/
/*
 * This file is part of hyantes.
 *
 * hyantes is free software; you can redistribute it and/or modify
 * it under the terms of the CeCILL-C License
 *
 * You should have received a copy of the CeCILL-C License
 * along with this program.  If not, see <http://www.cecill.info/licences>.
 */

#include "hs_config.h"
#include "engine.h"
#include "interpolation.h"

#include <math.h>
#include <float.h>
#include <assert.h>
#include <error.h>
#include <stdlib.h>
#include <string.h>

/* program constant */
typedef 
#ifdef USE_FLOAT
    float
#else
    double
#endif
        internal_type;

/** 
* @brief Earth radius given in kilometers
*/
static const internal_type EARTH_RADIUS = 6368.f;

/** 
* @brief threshold to decide if we should stop the computing of this branch of the tree
*/
static const internal_type THRESHOLD = 0.01f;


/** 
* @brief compute the geodesique distance between two points A and B given their precomputed cos / sin
* 
* @param coslatA cosinus of the latitude of A
* @param sinlatA sinus of the latitude of A
* @param coslatB cosinus of the latitude of A
* @param sinlatB sinus of the latitude of A
* @param dlon   absolute value of the difference of longitude between A and B i.e. abs(lonA-lonB) 
* 
* @return geodesique distance between points A and B
*/
#define compute_distance(coslatA,  sinlatA,  coslatB,  sinlatB,  dlon ) \
    EARTH_RADIUS * \
    acos(\
      (coslatA) * (coslatB) * cos ( (dlon) )\
      +\
      (sinlatA) * (sinlatB)\
    )

/**
 * computes the influence of all valuable stocks on a potential
 */

#define incr_iter() ++extra_iter; ++iter
#define jump_iter() extra_iter+=iter->size; iter+=iter->size

/** 
* @brief computes the potential of one point of the out map given the whole tree as parameter
* 
* @param out pointer to an hs_potential_t variable with filled lon / lat fields but 0 pot field
* @param param parameters used for the computation
*/
void make_pot(hs_potential_t * out, const struct make_pot_param *param)
{
    internal_type temp;
    const planartree *iter = param->ptrees;
    const extra_t *extra_iter = param->extra;
    const internal_type COSLAT_A = cos(out->lat);
    const internal_type SINLAT_A = sin(out->lat);
    internal_type COSLAT_B, SINLAT_B;
    internal_type tmp_lon;

    /* checks */
    assert(param->ptrees != NULL);
    assert(g_inter != NULL);

    /* iterate through the array, jumping when a node contains too few values */
    while(iter->size > 0)
    {
        /* node  */
        if(iter->size > 1)
        {
            /* compute dist(potential, stock) */
            if(iter->coo.mLat - out->lat > FLT_EPSILON)
            {
                COSLAT_B = extra_iter->cos[0];
                SINLAT_B = extra_iter->sin[0];
            }
            else if(out->lat - iter->coo.MLat > FLT_EPSILON)
            {
                COSLAT_B = extra_iter->cos[2];
                SINLAT_B = extra_iter->sin[2];
            }
            else
            {
                COSLAT_B = COSLAT_A;
                SINLAT_B = SINLAT_A;
            }


            if(iter->coo.mLon - out->lon > FLT_EPSILON)
            {
                tmp_lon = iter->coo.mLon;
            }
            else if(out->lon - iter->coo.MLon > FLT_EPSILON)
            {
                tmp_lon = iter->coo.MLon;
            }
            else
            {
                tmp_lon = out->lon;
            }


            temp =
               compute_distance(COSLAT_A, SINLAT_A, COSLAT_B, SINLAT_B,
               fabs(out->lon - tmp_lon));
            temp = (*g_inter) (temp);
            temp *= (iter->value);

            /* evaluate the influence of the node */
            if(temp - THRESHOLD > FLT_EPSILON)
            {
                incr_iter();
            }
            /* jump to next node */
            else
            {
                jump_iter();
            }
        }

        /* leaf, always computed */
        else
        {

            temp =
               compute_distance(COSLAT_A, SINLAT_A, extra_iter->cos[0],
               extra_iter->sin[0], fabs(out->lon - iter->coo.mLon));
            temp = (*g_inter) (temp);
            temp *= (iter->value);
            out->pot += temp;
            incr_iter();
        }
    }
}

/* legacy code */
#if 0
double compute_distance(double coslatA, double sinlatA, double coslatB,
   double sinlatB, double dlon)
{
    return
       EARTH_RADIUS *
       acos(cos(latA) * cos(latB) * cos(dlon) + sin(latA) * sin(latB));
}
#endif
