/** 
* @file interpolation.c
* @brief smoothing fucntions part
* @author serge guelton
* @date 2008-02-13
*/
/*
 * This file is part of hyantes.
 *
 * hyantes is free software; you can redistribute it and/or modify
 * it under the terms of the CeCILL-C License
 *
 * You should have received a copy of the CeCILL-C License
 * along with this program.  If not, see <http://www.cecill.info/licences>.
 */

#include "hs_config.h"
#include "interpolation.h"
#include "hyantes.h"

#include <math.h>
#include <float.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>



/******************************************************************************\
*
*       declare globals
*
\******************************************************************************/
/** 
* @brief global pointer to the chosen smoothing fuction
*/
/*@null@*/ func_interpolation g_inter;

static double g_param = 0.f;


/******************************************************************************\*
*
*       declare routines
*
\******************************************************************************/
static double disk(double d);
static double amortized_disk(double d);
static double gaussian(double d);
static double exponential(double d);
static double pareto(double d);

/** 
* @brief static array containg function names.
* each function must also be listed in func_ptrs in the same order
*/
static const char *func_names[] = {
    "disk",
    "amortized_disk",
    "gaussian",
    "exponential",
    "pareto"
};

/** 
* @brief static array containg function pointers.
* each function must also be listed in func_names in the same order
*/
static func_interpolation func_ptrs[] = {
    &disk,
    &amortized_disk,
    &gaussian,
    &exponential,
    &pareto
};


/******************************************************************************\
*
*      extern function
*
\******************************************************************************/


/** 
* @brief set the fucntion in use in for the computation, given its name and radius
* if the name is not found, default is to use disk function
* @param function_name name of the function
* @param function_param radisu used
*/
void set_func_inter(const char *function_name, double function_param)
{
    size_t i;
    g_param = function_param;
    assert(sizeof(func_ptrs) / sizeof(*func_ptrs) ==
       sizeof(func_names) / sizeof(*func_names));

    for(i = 0; i < sizeof(func_ptrs) / sizeof(*func_ptrs); ++i)
    {
        if(strncmp(func_names[i], function_name,
              sizeof(func_names[i])) == 0)
        {
            g_inter = func_ptrs[i];
            break;
        }
    }

    if(i == (sizeof(func_ptrs) / sizeof(*func_ptrs)))
    {
        fprintf(stderr,
           "# required function does not exist, switching to disk \navailable functions are:\n");
        for(i = 0; i < sizeof(func_ptrs) / sizeof(*func_ptrs); ++i)
        {
            fprintf(stderr, "# - %s\n", func_names[i]);
        }
        g_inter = &disk;
    }
}

/** 
* @brief get the array of available function names
* 
* @param sz size of the returned array
* 
* @return array of smoothing functions name
*/
const char **list_smoothing(size_t * sz)
{
    static const size_t names_count =
       sizeof(func_names) / sizeof(*func_names);

    *sz = names_count;
    return func_names;
}

/******************************************************************************\
*
*      routines
*
\******************************************************************************/

/** 
* @brief disk smoothing
* 
* @param d distance to cut
* 
* @return cut distance 
*/
double disk(double d)
{
    const double s = /*1.f */ M_PI * g_param * g_param;

    return (d - g_param > FLT_EPSILON) ? 0.f : 1.f / s;
}

/** 
* @brief amortized disk smoothing
*   
* 
* @param d distance to cut
* 
* @return cut distance 
*/
double amortized_disk(double d)
{
    const double s = /*1.f */ M_PI * g_param * g_param / 3.f;
    if(d - g_param > FLT_EPSILON)
    {
        return 0.f;
    }
    else
    {
        return (g_param - d) / (g_param * s);
    }
}

/** 
* @brief gaussian smoothing
* 
* @param d distance to cut
* 
* @return cut distance 
*/
double gaussian(double d)
{
    const double s = /*1.f */ M_PI * g_param * g_param;
    return exp(-d * d / (g_param * g_param)) / s;
}


/** 
* @brief exponential smoothing
* 
* @param d distance to cut
* 
* @return cut distance 
*/
double exponential(double d)
{
    const double s = /*1.f */ M_PI * 2.f * g_param * g_param;
    return exp(-d / g_param) / s;
}

/** 
* @brief pareto smoothing
* 
* @param d distance to cut
* 
* @return cut distance 
*/
double pareto(double d)
{
    const double s = /*1.f */ M_PI * g_param * g_param / 2.f;
    const double p = d / g_param;
    const double tmp = p * p;
    return 1 / (s * (1 + tmp * tmp));
}
