/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: Mixer.cpp,v 1.83 2004/02/20 14:09:13 comix Exp $
 *
 */


#include "Mixer.h"
#include "lib/Hydrogen.h"
#include "lib/Song.h"

#include "HydrogenApp.h"

#define MIXER_STRIP_WIDTH	44

/**
 * Constructor
 */
Mixer::Mixer( QWidget* parent ) : QWidget( parent, "Mixer" ), Object( "Mixer" )
{
//	cout << "Mixer INIT" << endl;

	mixerHeight = 270;
	nMinimumVisibleFadersSize = 5 + MIXER_STRIP_WIDTH * 16 + 5;
	uint fadersSize = 5 + MIXER_STRIP_WIDTH * MAX_INSTRUMENTS;

	mixerWidth = 5 + MIXER_STRIP_WIDTH * 4 + 70 + 50;	// 4 instruments visible
	setMinimumSize( mixerWidth, mixerHeight );

	setMaximumSize( fadersSize +  70 + 50, mixerHeight );
	mixerWidth = 5 + MIXER_STRIP_WIDTH * 16 + 70 + 50;	// 16 instruments visible
	resize( mixerWidth, mixerHeight );
	setCaption( trUtf8( "Mixer" ) );
	setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );

	// Background image
	string background_path = string( IMG_PATH ) + string( "/img/mixerPanel/mixer_background.png" );
	bool ok = background.load( background_path.c_str() );
	if( ok == false ){
		errorLog( "Error loading pixmap " + background_path );
	}

	// Left frame
	m_pLeftFrame = new QFrame( this );
	m_pLeftFrame->resize( 50, mixerHeight );
	m_pLeftFrame->setBackgroundPixmap( background );

	string showMixer_on_path = string(IMG_PATH).append("/img/mixerPanel/showMixer_on.png");
	string showMixer_off_path = string(IMG_PATH).append("/img/mixerPanel/showMixer_off.png");
	string showMixer_over_path = string(IMG_PATH).append("/img/mixerPanel/showMixer_over.png");
	m_pShowFaderPanelBtn = new ToggleButton( m_pLeftFrame, QSize(40, 12), showMixer_on_path, showMixer_off_path, showMixer_over_path );
	m_pShowFaderPanelBtn->move( 5, 20 );
	m_pShowFaderPanelBtn->setPressed(true);
	QToolTip::add( m_pShowFaderPanelBtn, trUtf8( "Show faders panel" ) );
	connect( m_pShowFaderPanelBtn, SIGNAL(clicked(Button*)), this, SLOT(showFadersPanelClicked(Button*)));

	string showFX_on_path = string(IMG_PATH).append("/img/mixerPanel/showFX_on.png");
	string showFX_off_path = string(IMG_PATH).append("/img/mixerPanel/showFX_off.png");
	string showFX_over_path = string(IMG_PATH).append("/img/mixerPanel/showFX_over.png");
	m_pShowFXPanelBtn = new ToggleButton( m_pLeftFrame, QSize(40, 12), showFX_on_path, showFX_off_path, showFX_over_path );
	m_pShowFXPanelBtn->move( 5, 40 );
	m_pShowFXPanelBtn->setPressed(false);
	QToolTip::add( m_pShowFXPanelBtn, trUtf8( "Show FX panel" ) );
	connect( m_pShowFXPanelBtn, SIGNAL(clicked(Button*)), this, SLOT( showFXPanelClicked(Button*)));

	string showPeaks_on_path = string(IMG_PATH).append("/img/mixerPanel/showPeaks_on.png");
	string showPeaks_off_path = string(IMG_PATH).append("/img/mixerPanel/showPeaks_off.png");
	string showPeaks_over_path = string(IMG_PATH).append("/img/mixerPanel/showPeaks_over.png");
	m_pShowPeaksBtn = new ToggleButton( m_pLeftFrame, QSize(40, 12), showPeaks_on_path, showPeaks_off_path, showPeaks_over_path );
	m_pShowPeaksBtn->move( 5, 60 );
	m_pShowPeaksBtn->setPressed( (PreferencesMng::getInstance())->showInstrumentPeaks() );
	QToolTip::add( m_pShowPeaksBtn, trUtf8( "Show instrument peaks" ) );
	connect( m_pShowPeaksBtn, SIGNAL(clicked(Button*)), this, SLOT( showPeaksBtnClicked(Button*)));



	// fader scroll view
	m_pFaderScrollView = new QScrollView( this );
	m_pFaderScrollView->move( 50, 0 );
	m_pFaderScrollView->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pFaderScrollView->setHScrollBarMode( QScrollView::AlwaysOn );
	m_pFaderScrollView->resize( nMinimumVisibleFadersSize, mixerHeight );
//	m_pFaderScrollView->viewport()->setBackgroundPixmap( background );

	// fader frame
	m_pFaderFrame = new QFrame( m_pFaderScrollView->viewport() );
	m_pFaderFrame->resize( fadersSize, mixerHeight );
	m_pFaderScrollView->addChild( m_pFaderFrame );
	m_pFaderFrame->setBackgroundPixmap( background );


	// fx scroll view
	m_pFXScrollView = new QScrollView( this );
	m_pFXScrollView->move( 50, 0 );
	m_pFXScrollView->setVScrollBarMode( QScrollView::AlwaysOff );
	m_pFXScrollView->setHScrollBarMode( QScrollView::AlwaysOn );
	m_pFXScrollView->resize( nMinimumVisibleFadersSize, mixerHeight );
//	m_pFXScrollView->viewport()->setBackgroundPixmap( background );

	// fX frame
	m_pFXFrame = new QFrame( m_pFXScrollView->viewport() );
	m_pFXFrame->resize( fadersSize, mixerHeight );
	m_pFXScrollView->addChild( m_pFXFrame );
	m_pFXFrame->setBackgroundPixmap( background );





	this->setBackgroundPixmap( background );

	m_pShowFaderPanelBtn->setPressed(true);

	m_pFaderScrollView->show();
	m_pFXScrollView->hide();

	setupMixer();

	timer = new QTimer( this );
	connect(timer, SIGNAL( timeout() ), this, SLOT( updateMixer() ) );

	( Hydrogen::getInstance() )->addEngineListener( this );
}




/**
 * Destructor
 */
Mixer::~Mixer() {
//	cout << "Mixer DESTROY" << endl;
}




/**
 * Setup the mixer strips
 */
void Mixer::setupMixer()
{
	uint master_X = mixerWidth - MIXER_STRIP_WIDTH - 5;

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	uint xPos = 0;
	uint nInstruments = instrList->getSize();
	for ( uint i = 0; i < MAX_INSTRUMENTS; i++ ) {	// MIXER LINE
		xPos = 5 + MIXER_STRIP_WIDTH * i;

		float volume = 0.2;
		bool mute = false;
		bool solo = false;

		Instrument *instr = NULL;
		if (i < nInstruments) {
			instr = instrList->get( i );
			volume = instr->getVolume();
			mute = instr->isMuted();
		}

		mixerLine[i] = new MixerLine( m_pFaderFrame );
		mixerLine[i]->move( xPos, 4 );
		mixerLine[i]->setVolume( volume );
		mixerLine[i]->setMuteClicked( mute );
		mixerLine[i]->setSoloClicked( solo );
		mixerLine[i]->updateMixerLine();

		connect( mixerLine[i], SIGNAL( noteOnClicked(MixerLine*) ), this, SLOT( noteOnClicked(MixerLine*) ) );
		connect( mixerLine[i], SIGNAL( noteOffClicked(MixerLine*) ), this, SLOT( noteOffClicked(MixerLine*) ) );
		connect( mixerLine[i], SIGNAL( muteBtnClicked(MixerLine*) ), this, SLOT( muteClicked(MixerLine*) ) );
		connect( mixerLine[i], SIGNAL( soloBtnClicked(MixerLine*) ), this, SLOT( soloClicked(MixerLine*) ) );
		connect( mixerLine[i], SIGNAL( loadBtnClicked(MixerLine*) ), this, SLOT( loadClicked(MixerLine*) ) );
		connect( mixerLine[i], SIGNAL( volumeChanged(MixerLine*) ), this, SLOT( volumeChanged(MixerLine*) ) );
		connect( mixerLine[i], SIGNAL( instrumentNameClicked(MixerLine*) ), this, SLOT( nameClicked(MixerLine*) ) );
		connect( mixerLine[i], SIGNAL( instrumentNameSelected(MixerLine*) ), this, SLOT( nameSelected(MixerLine*) ) );
		connect( mixerLine[i], SIGNAL( panChanged(MixerLine*) ), this, SLOT( panChanged( MixerLine*) ) );
		connect( mixerLine[i], SIGNAL( knobChanged(MixerLine*, int) ), this, SLOT( knobChanged( MixerLine*, int) ) );
	}

	swingFactorLine = new FxMixerLine( m_pFXFrame );
	swingFactorLine->move( 5, 4 );
	swingFactorLine->setName( trUtf8( "Swing factor" ) );
	connect( swingFactorLine, SIGNAL( volumeChanged(FxMixerLine*) ), this, SLOT( swingChanged(FxMixerLine*) ) );
	connect( swingFactorLine, SIGNAL( activeBtnClicked(FxMixerLine*) ), this, SLOT( activeBtnChanged( FxMixerLine*) ) );

	humanizeTimeLine = new FxMixerLine( m_pFXFrame );
	humanizeTimeLine->move( 5 + MIXER_STRIP_WIDTH, 4 );
	humanizeTimeLine->setName( trUtf8( "Human time" ) );
	connect( humanizeTimeLine, SIGNAL( volumeChanged(FxMixerLine*) ), this, SLOT( humanizeChanged(FxMixerLine*) ) );
	connect( humanizeTimeLine, SIGNAL( activeBtnClicked(FxMixerLine*) ), this, SLOT( activeBtnChanged( FxMixerLine*) ) );

	humanizeVelocityLine = new FxMixerLine( m_pFXFrame );
	humanizeVelocityLine->move( 5 + MIXER_STRIP_WIDTH * 2, 4 );
	humanizeVelocityLine->setName( trUtf8( "Human velocity" ) );
	connect( humanizeVelocityLine, SIGNAL( volumeChanged(FxMixerLine*) ), this, SLOT( humanizeChanged(FxMixerLine*) ) );
	connect( humanizeVelocityLine, SIGNAL( activeBtnClicked(FxMixerLine*) ), this, SLOT( activeBtnChanged( FxMixerLine*) ) );

	// LADSPA
	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		m_pLadspaFXLine[nFX] = new LadspaFXMixerLine( m_pFXFrame );
		m_pLadspaFXLine[nFX]->move( 5 + MIXER_STRIP_WIDTH * (4 + nFX), 4 );
		connect( m_pLadspaFXLine[nFX], SIGNAL( activeBtnClicked(LadspaFXMixerLine*) ), this, SLOT( ladspaActiveBtnClicked( LadspaFXMixerLine*) ) );
		connect( m_pLadspaFXLine[nFX], SIGNAL( editBtnClicked(LadspaFXMixerLine*) ), this, SLOT( ladspaEditBtnClicked( LadspaFXMixerLine*) ) );
		connect( m_pLadspaFXLine[nFX], SIGNAL( volumeChanged(LadspaFXMixerLine*) ), this, SLOT( ladspaVolumeChanged( LadspaFXMixerLine*) ) );
		connect( m_pLadspaFXLine[nFX], SIGNAL( ladspaNameClicked(LadspaFXMixerLine*) ), this, SLOT( ladspaEditBtnClicked( LadspaFXMixerLine*) ) );
	}
	// ~LADSPA


//	uint masterVolume = 50;
	masterLine = new MasterMixerLine( this );
	masterLine->move( master_X, 4 );
//	masterLine->setVolume(masterVolume);
	connect( masterLine, SIGNAL( volumeChanged(MasterMixerLine*) ), this, SLOT( masterVolumeChanged(MasterMixerLine*) ) );

}




void Mixer::muteClicked(MixerLine* ref) {
	int nLine = findMixerLineByRef(ref);
	bool isMuteClicked = ref->isMuteClicked();

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	Instrument *instr = instrList->get(nLine);
	instr->setMuted(isMuteClicked);

	//update the pattern editor
	(HydrogenApp::getInstance())->getPatternEditor()->getPatternEditor()->updateEditor(true);

	PatternEditorPanel *pPatternEditorPanel = ( HydrogenApp::getInstance() )->getPatternEditor();
	if ( pPatternEditorPanel->getSelectedInstrument() != nLine ) {
		pPatternEditorPanel->setSelectedInstrument( nLine );
	}
}




void Mixer::soloClicked(MixerLine* ref) {
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();


	int nLine = findMixerLineByRef(ref);
	bool isSoloClicked = ref->isSoloClicked();

	if (isSoloClicked) {
		for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
			mixerLine[i]->setSoloClicked( false );
			mixerLine[i]->setMuteClicked( true );

			Instrument *instr = instrList->get( i );
			instr->setMuted( true );
		}
		mixerLine[nLine]->setSoloClicked( true );
		mixerLine[nLine]->setMuteClicked( false );
		Instrument *instr = instrList->get( nLine );
		instr->setMuted( false );
	}
	else {
		for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
			mixerLine[i]->setMuteClicked( false );
			mixerLine[i]->setSoloClicked( false );

			Instrument *instr = instrList->get( i );
			instr->setMuted( false );
		}
	}
	//update the pattern editor
	(HydrogenApp::getInstance())->getPatternEditor()->getPatternEditor()->updateEditor(true);

	PatternEditorPanel *pPatternEditorPanel = ( HydrogenApp::getInstance() )->getPatternEditor();
	if ( pPatternEditorPanel->getSelectedInstrument() != nLine ) {
		pPatternEditorPanel->setSelectedInstrument( nLine );
	}
}


// used in PatternEditorInstrumentList
void Mixer::soloClicked(uint nLine)
{
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();


	bool isSoloClicked = (mixerLine[nLine])->isSoloClicked();

	if (!isSoloClicked) {
		for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
			mixerLine[i]->setSoloClicked( false );
			mixerLine[i]->setMuteClicked( true );

			Instrument *instr = instrList->get( i );
			instr->setMuted( true );
		}
		mixerLine[nLine]->setSoloClicked( true );
		mixerLine[nLine]->setMuteClicked( false );
		Instrument *instr = instrList->get( nLine );
		instr->setMuted( false );
	}
	else {
		for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
			mixerLine[i]->setMuteClicked( false );
			mixerLine[i]->setSoloClicked( false );

			Instrument *instr = instrList->get( i );
			instr->setMuted( false );
		}
	}

}


void Mixer::loadClicked(MixerLine* ref) {
	int nLine = findMixerLineByRef(ref);

	(HydrogenApp::getInstance())->loadNewInstrument(nLine);

	PatternEditorPanel *pPatternEditorPanel = ( HydrogenApp::getInstance() )->getPatternEditor();
	if ( pPatternEditorPanel->getSelectedInstrument() != nLine ) {
		pPatternEditorPanel->setSelectedInstrument( nLine );
	}
}



void Mixer::noteOnClicked( MixerLine* ref )
{
	int nLine = findMixerLineByRef( ref );

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	Note *note = new Note( 0, 1.0, 1.0, 1.0 );
	note->setInstrument( instrList->get(nLine) );

	engine->noteOn( note );

	PatternEditorPanel *pPatternEditorPanel = ( HydrogenApp::getInstance() )->getPatternEditor();
	if ( pPatternEditorPanel->getSelectedInstrument() != nLine ) {
		pPatternEditorPanel->setSelectedInstrument( nLine );
	}
}




// Play sample button, right-clicked (note off)
 void Mixer::noteOffClicked( MixerLine* ref )
{
	int nLine = findMixerLineByRef( ref );

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	Note *note = new Note( 0, 1.0, 1.0, 1.0 );
	note->setInstrument( instrList->get( nLine ) );

	engine->noteOff( note );

	PatternEditorPanel *pPatternEditorPanel = ( HydrogenApp::getInstance() )->getPatternEditor();
	if ( pPatternEditorPanel->getSelectedInstrument() != nLine ) {
		pPatternEditorPanel->setSelectedInstrument( nLine );
	}
}




/**
 *
 */
uint Mixer::findMixerLineByRef(MixerLine* ref) {
	for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
		if (mixerLine[i] == ref) {
			return i;
		}
	}
	return 0;
}




void Mixer::volumeChanged(MixerLine* ref) {

	int nLine = findMixerLineByRef(ref);

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	Instrument *instr = instrList->get(nLine);

	instr->setVolume(ref->getVolume());

	PatternEditorPanel *pPatternEditorPanel = ( HydrogenApp::getInstance() )->getPatternEditor();
	if ( pPatternEditorPanel->getSelectedInstrument() != nLine ) {
		pPatternEditorPanel->setSelectedInstrument( nLine );
	}
}




/**
 *
 */
void Mixer::masterVolumeChanged(MasterMixerLine* ref) {
	float volume = ref->getVolume();
	Song *song = (HydrogenApp::getInstance())->getSong();
	song->setVolume(volume);
}




/**
 *
 */
void Mixer::updateMixer() {
	if(!isVisible()) {
		return;
	}

	PreferencesMng *pPref = PreferencesMng::getInstance();
	bool bShowPeaks = pPref->showInstrumentPeaks();

	Hydrogen *engine = Hydrogen::getInstance();
	Song *pSong = engine->getSong();
	InstrumentList *instrList = pSong->getInstrumentList();

	//float fallOff = 1.08;	// slow
	//float fallOff = 1.1;	// normal
	//float fallOff = 1.3;	// fast

	uint nSelectedInstr = ( HydrogenApp::getInstance() )->getPatternEditor()->getSelectedInstrument();

	float fallOff = (PreferencesMng::getInstance())->getMixerFalloffSpeed();

	uint nMuteClicked = 0;
	uint nInstruments = instrList->getSize();
	for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
		Instrument *instr = instrList->get(i);

		// fader
		float oldPeak_L = mixerLine[i]->getPeak_L();
		float newPeak_L = instr->getPeak_L();
		instr->setPeak_L( 0.0f );	// reset instrument peak

		float oldPeak_R = mixerLine[i]->getPeak_R();
		float newPeak_R = instr->getPeak_R();
		instr->setPeak_R( 0.0f );	// reset instrument peak

		if (!bShowPeaks) {
			newPeak_L = 0.0f;
			newPeak_R = 0.0f;
		}

		if ( newPeak_L >= oldPeak_L) {	// LEFT peak
			mixerLine[i]->setPeak_L( newPeak_L );
		}
		else {
			mixerLine[i]->setPeak_L( oldPeak_L / fallOff );
		}
		if ( newPeak_R >= oldPeak_R) {	// Right peak
			mixerLine[i]->setPeak_R( newPeak_R );
		}
		else {
			mixerLine[i]->setPeak_R( oldPeak_R / fallOff );
		}

		// fader position
		float newVolume = instr->getVolume();
		mixerLine[i]->setVolume(newVolume);

		// mute
		bool muted = instr->isMuted();
		if (muted) {
			nMuteClicked++;
		}
		mixerLine[i]->setMuteClicked( muted );

		// instr name
		mixerLine[i]->setName( instr->getName().c_str() );

		// pan
		float panValue = 0.0;
		float pan_L = instr->getPan_L();
		float pan_R = instr->getPan_R();
		if (pan_R == 1.0) {
			panValue = 1.0 - (pan_L / 2.0);
		}
		else {
			panValue = pan_R / 2.0;
		}
		panValue = panValue * 100.0;

		mixerLine[i]->setPan((int)panValue);	// FIXME: perche' setPan prende un'intero???

		// activity
		if ( mixerLine[i]->getActivity() > 0 ) {
			mixerLine[i]->setActivity( mixerLine[i]->getActivity() - 30 );
			mixerLine[i]->setPlayClicked( true );
		}
		else {
			mixerLine[i]->setPlayClicked( false );
		}

		for (uint nFX = 0; nFX < MAX_FX; nFX++) {
			mixerLine[i]->setFXLevel( nFX, instr->getFXLevel( nFX ) );
		}

		if ( i == nSelectedInstr) {
			mixerLine[i]->setSelected( true );
		}
		else {
			mixerLine[i]->setSelected( false );
		}

		mixerLine[i]->updateMixerLine();
	}

	if (nMuteClicked == MAX_INSTRUMENTS - 1) {
		// find the not muted button
		for (uint i = 0; i < nInstruments; i++) {
			Instrument *instr = instrList->get(i);
			if (instr->isMuted() == false) {
				mixerLine[i]->setSoloClicked(true);
				break;
			}
		}
	}
	else {
		for (uint i = 0; i < nInstruments; i++) {
			mixerLine[i]->setSoloClicked(false);
		}
	}


	// update MasterPeak
	float oldPeak_L = masterLine->getPeak_L();
	float newPeak_L = engine->getMasterPeak_L();
	engine->setMasterPeak_L(0.0);
	float oldPeak_R = masterLine->getPeak_R();
	float newPeak_R = engine->getMasterPeak_R();
	engine->setMasterPeak_R(0.0);

	if (!bShowPeaks) {
		newPeak_L = 0.0;
		newPeak_R = 0.0;
	}

	if (newPeak_L >= oldPeak_L) {
		masterLine->setPeak_L( newPeak_L );
	}
	else {
		masterLine->setPeak_L( oldPeak_L / fallOff );
	}
	if (newPeak_R >= oldPeak_R) {
		masterLine->setPeak_R(newPeak_R);
	}
	else {
		masterLine->setPeak_R( oldPeak_R / fallOff );
	}




	// set master fader position
	float newVolume = pSong->getVolume();
	float oldVolume = masterLine->getVolume();
	if (oldVolume != newVolume) {
		masterLine->setVolume(newVolume);
	}
	masterLine->updateMixerLine();


	// set humanize fx
	humanizeTimeLine->setFxActive( pSong->isHumanizeTimeEnabled() );
	if (humanizeTimeLine->getVolume() != pSong->getHumanizeTimeValue() ) {
		humanizeTimeLine->setVolume( pSong->getHumanizeTimeValue() );
	}
	humanizeTimeLine->setPeak_L( pSong->getHumanizeTimeValue() );
	humanizeTimeLine->setPeak_R( pSong->getHumanizeTimeValue() );
	humanizeTimeLine->updateMixerLine();


	humanizeVelocityLine->setFxActive( pSong->isHumanizeVelocityEnabled() );
	if ( humanizeVelocityLine->getVolume() != pSong->getHumanizeVelocityValue() ) {
		humanizeVelocityLine->setVolume( pSong->getHumanizeVelocityValue() );
	}
	humanizeVelocityLine->setPeak_L( pSong->getHumanizeVelocityValue() );
	humanizeVelocityLine->setPeak_R( pSong->getHumanizeVelocityValue() );
	humanizeVelocityLine->updateMixerLine();

	// swing factor
	swingFactorLine->setFxActive( pSong->isSwingEnabled() );
	if ( swingFactorLine->getVolume() != pSong->getSwingFactor() ) {
		swingFactorLine->setVolume( pSong->getSwingFactor() );
	}
	swingFactorLine->setPeak_L ( pSong->getSwingFactor() );
	swingFactorLine->setPeak_R ( pSong->getSwingFactor() );
	swingFactorLine->updateMixerLine();


	// LADSPA
	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		LadspaFX *pFX = pSong->getLadspaFX( nFX );
		if ( pFX ) {
			m_pLadspaFXLine[nFX]->setName( pFX->getPluginName().c_str() );
			float fNewPeak_L = 0.0;
			float fNewPeak_R = 0.0;
			engine->getLadspaFXPeak( nFX, &fNewPeak_L, &fNewPeak_R );
			engine->setLadspaFXPeak( nFX, 0.0, 0.0 );	// reset

			float fOldPeak_L = 0.0;
			float fOldPeak_R = 0.0;
			m_pLadspaFXLine[nFX]->getPeaks( &fOldPeak_L, &fOldPeak_R );

			if (fNewPeak_L < fOldPeak_L)	fNewPeak_L = fOldPeak_L / fallOff;
			if (fNewPeak_R < fOldPeak_R)	fNewPeak_R = fOldPeak_R / fallOff;
			m_pLadspaFXLine[nFX]->setPeaks( fNewPeak_L, fNewPeak_R );
			m_pLadspaFXLine[nFX]->setFxActive( pFX->isEnabled() );
			m_pLadspaFXLine[nFX]->setVolume( pFX->getVolume() );
		}
		else {
			m_pLadspaFXLine[nFX]->setName( trUtf8("No plugin") );
			m_pLadspaFXLine[nFX]->setFxActive( false );
			m_pLadspaFXLine[nFX]->setVolume( 0.0 );
		}
	}
	// ~LADSPA
}




/**
 *
 */
void Mixer::updateStart(bool start) {
	if (start) {
//		cout << "mixer update start" << endl;
		timer->start(50);	// update mixer at 20 fps
	}
	else {
//		cout << "mixer update stop" << endl;
		timer->stop();
	}
}




/**
 * show event
 */
void Mixer::showEvent ( QShowEvent *ev ) {
	updateMixer();
	updateStart(true);
}




/**
 * hide event
 */
void Mixer::hideEvent ( QHideEvent *ev ) {
	updateStart(false);
}




void Mixer::nameClicked(MixerLine* ref) {
	int nLine = findMixerLineByRef(ref);
	PatternEditorPanel *pPatternEditorPanel = ( HydrogenApp::getInstance() )->getPatternEditor();
	if ( pPatternEditorPanel->getSelectedInstrument() != nLine ) {
		pPatternEditorPanel->setSelectedInstrument( nLine );
	}


	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	Instrument *instr = instrList->get(nLine);

	InstrumentPropertiesDialog *dialog = new InstrumentPropertiesDialog(this, instr);
	if (dialog->exec() == QDialog::Accepted) {
		song->setModified(true);
	}
	delete dialog;
	dialog = NULL;

}


void Mixer::nameSelected(MixerLine* ref)
{
	int nLine = findMixerLineByRef(ref);
	PatternEditorPanel *pPatternEditorPanel = ( HydrogenApp::getInstance() )->getPatternEditor();
	if ( pPatternEditorPanel->getSelectedInstrument() != nLine ) {
		pPatternEditorPanel->setSelectedInstrument( nLine );
	}
}





void Mixer::panChanged(MixerLine* ref) {
	float panValue = ref->getPan();

	float pan_L = (100.0 - panValue) / 100.0;
	float pan_R = panValue / 100.0;

	panValue = panValue / 100.0;

	if (panValue >= 0.5) {
		pan_L = (1.0 - panValue) * 2;
		pan_R = 1.0;
	}
	else {
		pan_L = 1.0;
		pan_R = ( 1.0 - ( 1.0 - panValue) ) * 2;
	}

	int nLine = findMixerLineByRef(ref);
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	Instrument *instr = instrList->get(nLine);
	instr->setPan_L( pan_L );
	instr->setPan_R( pan_R );

	PatternEditorPanel *pPatternEditorPanel = ( HydrogenApp::getInstance() )->getPatternEditor();
	if ( pPatternEditorPanel->getSelectedInstrument() != nLine ) {
		pPatternEditorPanel->setSelectedInstrument( nLine );
	}

}



void Mixer::knobChanged(MixerLine* ref, int nKnob) {
	int nLine = findMixerLineByRef(ref);
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();
	Instrument *pInstr = instrList->get(nLine);
	pInstr->setFXLevel( nKnob, ref->getFXLevel(nKnob) );
	QString sInfo = trUtf8( "Set FX %1 level ").arg( nKnob );
	( HydrogenApp::getInstance() )->setStatusBarMessage( sInfo+ QString( "[%1]" ).arg( ref->getFXLevel(nKnob), 0, 'f', 2 ), 2000 );

	PatternEditorPanel *pPatternEditorPanel = ( HydrogenApp::getInstance() )->getPatternEditor();
	if ( pPatternEditorPanel->getSelectedInstrument() != nLine ) {
		pPatternEditorPanel->setSelectedInstrument( nLine );
	}

}





/**
 *
 */
void Mixer::humanizeChanged(FxMixerLine* ref) {
	Hydrogen *engine = Hydrogen::getInstance();
	ref->setPeak_L( ref->getVolume() );
	ref->setPeak_R( ref->getVolume() );

	char faderPos[100];
	float value = ref->getVolume();
	sprintf( faderPos, "%#.2f",  value);

	if (ref == humanizeTimeLine) {
		float humanizeValue = ref->getVolume();
		engine->lockEngine("Mixer::humanizeChanged");
		engine->getSong()->setHumanizeTimeValue( humanizeValue );
		engine->unlockEngine();
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Set humanize time parameter [%1]").arg( faderPos ), 2000 );
	}
	else if (ref == humanizeVelocityLine) {
		float humanizeValue = ref->getVolume();
		engine->lockEngine("Mixer::humanizeChanged");
		engine->getSong()->setHumanizeVelocityValue( humanizeValue );
		engine->unlockEngine();
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Set humanize velocity parameter [%1]").arg( faderPos ), 2000 );
	}


}





/**
 *
 */
void Mixer::swingChanged( FxMixerLine* ref) {
	ref->setPeak_L( ref->getVolume() );
	ref->setPeak_R( ref->getVolume() );
	Hydrogen *engine = Hydrogen::getInstance();
	float swingFactor = ref->getVolume();
	engine->lockEngine("Mixer::swingChanged");
	engine->getSong()->setSwingFactor( swingFactor );
	engine->unlockEngine();

	char faderPos[100];
	float value = ref->getVolume();
	sprintf( faderPos, "%#.2f",  value);
	( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Set swing factor [%1]").arg( faderPos ), 2000 );
}






/**
 * This method is called from another thread (audio engine)
 */
void Mixer::noteOn( Note *note ) {
	H2TextEvent *ev = new H2TextEvent( "noteOn", note );
	QApplication::postEvent(this, ev);
}



/**
 *
 */
void Mixer::customEvent( QCustomEvent *ev ) {
	if ( ev->type() != H2_TEXT_EVENT ) {	// Must be a H2TextEvent
		return;
	}
	QString message = ( (H2TextEvent*) ev )->getText();

	if (message == QString( "noteOn" )) {
		Note *note = ( (H2TextEvent*) ev )->getNote();

		Hydrogen *engine = Hydrogen::getInstance();
		Song *song = engine->getSong();
		InstrumentList *instrList = song->getInstrumentList();

		// search the instrument line
		int instrLine = -1;
		for (uint i = 0; i < instrList->getSize(); i++) {
			Instrument* instr = instrList->get(i);
			if (note->getInstrument() == instr) {
				instrLine = i;
				break;
			}
		}
		if (instrLine != -1) {
			mixerLine[instrLine]->setActivity( 100 );
		}

		delete note;
		note = NULL;
	}

}




/**
 *
 */
void Mixer::activeBtnChanged( FxMixerLine *ref ) {
	bool isActive = ref->isFxActive();
	Song *song = ( Hydrogen::getInstance() )->getSong();
	if ( ref == humanizeTimeLine ) {
		song->setHumanizeTimeEnabled( isActive );
		if (isActive) {
			( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Humanize Time FX enabled" ), 2000 );
		}
		else {
			( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Humanize Time FX disabled" ), 2000 );
		}
	}
	else if ( ref == humanizeVelocityLine ) {
		song->setHumanizeVelocityEnabled( isActive );
		if (isActive) {
			( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Humanize Velocity FX enabled" ), 2000 );
		}
		else {
			( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Humanize Velocity FX disabled" ), 2000 );
		}
	}
	else if ( ref == swingFactorLine ) {
		song->setSwingEnabled( isActive );
		if (isActive) {
			( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Swing factor FX enabled" ), 2000 );
		}
		else {
			( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Swing factor FX disabled" ), 2000 );
		}
	}
	else {
		cout << "Mixer::activeBtnChanged ??" << endl;
	}
}


/**
 *
 */
void Mixer::resizeEvent ( QResizeEvent *ev )
{
	m_pFaderScrollView->resize( width() - 60 - 50 - 5, mixerHeight );
	m_pFXScrollView->resize( width() - 60 - 50 -5, mixerHeight );

	uint master_X = width() - MIXER_STRIP_WIDTH - 5;

	masterLine->move( master_X, 4 );
	resize(width(), mixerHeight);	// qt bug workaround
}


/**
 *
 */
void Mixer::showFadersPanelClicked(Button* ref)
{
	m_pShowFaderPanelBtn->setPressed(true);
	m_pShowFXPanelBtn->setPressed(false);
	m_pFaderScrollView->show();
	m_pFXScrollView->hide();
}



/**
 *
 */
void Mixer::showFXPanelClicked(Button* ref)
{
	m_pShowFXPanelBtn->setPressed(true);
	m_pShowFaderPanelBtn->setPressed(false);
	m_pFXScrollView->show();
	m_pFaderScrollView->hide();
}



/**
 *
 */
void Mixer::showPeaksBtnClicked(Button* ref)
{
	PreferencesMng *pPref = PreferencesMng::getInstance();

	if ( ref->isPressed() ) {
		pPref->setInstrumentPeaks( true );
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Show instrument peaks = On"), 2000 );
	}
	else {
		pPref->setInstrumentPeaks( false );
		( HydrogenApp::getInstance() )->setStatusBarMessage( trUtf8( "Show instrument peaks = Off"), 2000 );
	}
}



void Mixer::ladspaActiveBtnClicked( LadspaFXMixerLine* ref )
{
	bool bActive = ref->isFxActive();

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		if (ref == m_pLadspaFXLine[ nFX ] ) {
			LadspaFX *pFX = song->getLadspaFX(nFX);
			if (pFX) {
				pFX->setEnabled( bActive );
			}
			break;
		}
	}
}



void Mixer::ladspaEditBtnClicked( LadspaFXMixerLine *ref )
{
	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		if (ref == m_pLadspaFXLine[ nFX ] ) {
			( HydrogenApp::getInstance() )->getLadspaFXProperties(nFX)->hide();
			( HydrogenApp::getInstance() )->getLadspaFXProperties(nFX)->show();
		}
	}
	(Hydrogen::getInstance() )->getSong()->setModified( true );
}




void Mixer::ladspaVolumeChanged( LadspaFXMixerLine* ref)
{
	Song *pSong = (Hydrogen::getInstance() )->getSong();
	pSong->setModified( true );

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		if (ref == m_pLadspaFXLine[ nFX ] ) {
			LadspaFX *pFX = pSong->getLadspaFX(nFX);
			if (pFX) {
				pFX->setVolume( ref->getVolume() );
				QString sInfo = trUtf8( "Set LADSPA FX ( %1 ) volume").arg( QString(pFX->getPluginName().c_str()) );
				( HydrogenApp::getInstance() )->setStatusBarMessage( sInfo+ QString( " [%1]" ).arg( ref->getVolume(), 0, 'f', 2 ), 2000 );
			}
		}
	}


}


