/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: PatternEditor.cpp,v 1.78 2004/03/01 10:33:34 comix Exp $
 *
 */


#include "PatternEditor.h"

#include "lib/Song.h"
#include "lib/Hydrogen.h"

///
/// Constructor
///
PatternEditor::PatternEditor(QWidget* parent, PatternEditorPanel *pPanel)  : QWidget(parent), Object( "PatternEditor" )
{
//	cout << "PatternEditor INIT" << endl;

	m_pPatternEditorPanel = pPanel;
	m_nGridWidth = (PreferencesMng::getInstance())->getPatternEditorGridWidth();
	m_nGridHeight = (PreferencesMng::getInstance())->getPatternEditorGridHeight();

	m_nEditorWidth = 5 + m_nGridWidth * MAX_NOTES + 5;
	m_nEditorHeight = m_nGridHeight * MAX_INSTRUMENTS;

	m_nResolution = 8;
	m_bUseTriplets = false;
	currentPattern = NULL;

	changed = true;
	notesChanged = true;

	// Prepare pixmaps
	string background_path = IMG_PATH;
	background_path.append( "/img/patternEditor/editor_background.png" );
	bool ok = background.load( background_path.c_str() );
	if( ok == false ){
		qWarning( "PatternEditor: Error loading pixmap" );
	}
	createBackground( &background );
	setBackgroundPixmap( background );

	resize( m_nEditorWidth, m_nEditorHeight );
	setMinimumSize( m_nEditorWidth, m_nEditorHeight );
	setMaximumSize( m_nEditorWidth, m_nEditorHeight );

	background.resize(m_nEditorWidth, m_nEditorHeight );

	(Hydrogen::getInstance())->addEngineListener( this );
}





/**
 * Destructor
 */
PatternEditor::~PatternEditor()
{
//	cout << "PatternEditor DESTROY" << endl;
}




void PatternEditor::updateEditor(bool forceRepaint)
{
	if(!isVisible()) {
		return;
	}

	Hydrogen* engine = Hydrogen::getInstance();

	// check engine state
	int state = engine->getState();
	if ( (state != READY) && (state != PLAYING) ) {
		cerr << "FIXME: skip pattern editor update (state shoud be READY or PLAYING)" << endl;
		return;
	}

	Pattern* enginePattern = engine->getCurrentPattern();
	if (enginePattern != NULL) {

		if (currentPattern != enginePattern) {
			currentPattern = enginePattern;
			createBackground(&background);
			changed = true;
			notesChanged = true;
		}
	}
	else {
		if (currentPattern != enginePattern) {
			currentPattern = enginePattern;
			createBackground(&background);
			changed = true;
			notesChanged = true;
		}
	}

	if (forceRepaint) {
		createBackground(&background);
		changed = true;
		notesChanged = true;
	}

	// redraw all
	update();
}





void PatternEditor::mousePressEvent(QMouseEvent *ev) {
	if (currentPattern == NULL) {
		return;
	}

	int row = MAX_INSTRUMENTS - 1 - (ev->y()  / (int)m_nGridHeight);
	if (row >= MAX_INSTRUMENTS) {
		return;
	}

	int nBase;
	if (m_bUseTriplets) {
		nBase = 3;
	}
	else {
		nBase = 4;
	}

	int width = (m_nGridWidth * 4 * MAX_NOTES) / (nBase * m_nResolution);

	int x = ev->x();
	int column;
	column = x - 10 + (width / 2);
	column = column / width;
	column = (column * 4 * MAX_NOTES) / (nBase * m_nResolution);

	if (column >= MAX_NOTES) {
		update();
		return;
	}

	( Hydrogen::getInstance() )->lockEngine("PatternEditor::mousePressEvent");	// lock the audio engine
	SequenceList *sequenceList = currentPattern->getSequenceList();
	Sequence *seq = sequenceList->get(row);

	if (seq->noteList[column] != NULL) {
		Note *note = seq->noteList[column];
		delete note;
		seq->noteList[column] = NULL;
		notesChanged = true;
	}
	else {
		// create the new note
		uint position = column;
		float velocity = 0.8;
		float pan_L = 1.0;
		float pan_R = 1.0;
		Note *note = new Note(position, velocity, pan_L, pan_R);
		Song *song = (Hydrogen::getInstance())->getSong();
		Instrument *instrRef = (song->getInstrumentList())->get(row);
		note->setInstrument(instrRef);
		seq->noteList[column] = note;
		notesChanged = true;

		// hear note
		PreferencesMng *pref = PreferencesMng::getInstance();
		if ( pref->getHearNewNotes() ) {
			Note *note2 = new Note(0, velocity, pan_L, pan_R);
			note2->setInstrument(instrRef);
			( Hydrogen::getInstance() )->noteOn( note2 );
		}
	}
	Song *song = (Hydrogen::getInstance())->getSong();
	song->setModified(true);
	( Hydrogen::getInstance() )->unlockEngine(); // unlock the audio engine

	// update the selected line
	int nSelectedInstrument = m_pPatternEditorPanel->getSelectedInstrument();
	if (nSelectedInstrument != row) {
		m_pPatternEditorPanel->setSelectedInstrument(row);
	}
	else {
		changed = true;
		notesChanged = true;
		createBackground(&background);
		update();
		m_pVelocityRuler->updateEditor();
	}
}



/// Draw a note
void PatternEditor::drawNote(Note *note, uint nSequence, QPixmap *pixmap) {
	QPainter p(pixmap);
	QColor black(0, 0, 0);

	p.setPen(black);

	uint pos = note->getPosition() % MAX_NOTES;


	uint x_pos = 10 + (pos * m_nGridWidth) - m_nGridWidth;
	uint y_pos = (MAX_INSTRUMENTS  * m_nGridHeight) - ((nSequence + 1) * m_nGridHeight) + (m_nGridHeight / 2) - 3;

	// draw the "dot"
	p.drawLine(x_pos + 3, y_pos, x_pos + 6, y_pos + 3);
	p.drawLine(x_pos + 3, y_pos, x_pos, y_pos + 3);
	p.drawLine(x_pos + 3, y_pos + 6, x_pos + 6, y_pos + 3);
	p.drawLine(x_pos, y_pos + 3, x_pos + 3, y_pos + 6);

	p.drawLine(x_pos + 3, y_pos + 1, x_pos + 5, y_pos + 3);
	p.drawLine(x_pos + 3, y_pos + 1, x_pos + 1, y_pos + 3);
	p.drawLine(x_pos + 3, y_pos + 5, x_pos + 5, y_pos + 3);
	p.drawLine(x_pos + 1, y_pos + 3, x_pos + 3, y_pos + 5);

	p.drawLine(x_pos + 3, y_pos + 2, x_pos + 4, y_pos + 3);
	p.drawLine(x_pos + 3, y_pos + 2, x_pos + 2, y_pos + 3);
	p.drawLine(x_pos + 3, y_pos + 4, x_pos + 4, y_pos + 3);
	p.drawLine(x_pos + 2, y_pos + 3, x_pos + 3, y_pos + 4);

/*

	// test - draw a single note
	uint x = 10 + (pos * m_nGridWidth);
	uint y = (MAX_INSTRUMENTS  * m_nGridHeight) - ((nSequence + 1) * m_nGridHeight) + (m_nGridHeight / 100.0 * 20.0);
	int w = m_nGridWidth;
	int h = m_nGridHeight - ((m_nGridHeight / 100.0 * 20.0) * 2.0) + 1;

	p.fillRect( x, y, w, h, QColor(50, 50, 100) );
	p.drawRect( x, y, w, h );
*/

}





/**
 *
 */
void PatternEditor::createBackground(QPixmap *pixmap)
{
	QColor highlighted( 209, 233, 245 );
	QColor highlightedMute( 169, 183, 195 );
	QColor muted( 200, 200, 200 );
	QColor alternate( 240, 240, 255 );

	pixmap->fill();		// fill the pixmap with white color
	QPainter p(pixmap);

	// start y of velocity ruler
	uint velRuler_y_start = m_nGridHeight * MAX_INSTRUMENTS;

	int nSelectedInstrument = m_pPatternEditorPanel->getSelectedInstrument();
	// horizontal lines (only fill color)
	for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
		uint y = m_nGridHeight * i;
		Song *song = (Hydrogen::getInstance())->getSong();
		Instrument *pInstr = (song->getInstrumentList())->get( (MAX_INSTRUMENTS -1) - i );

		if ( ((MAX_INSTRUMENTS -1) - i) == (uint)nSelectedInstrument) {	// highlighted instrument line
			if ( pInstr->isMuted() ) {
				p.setPen( highlightedMute );	// color for highlighted and muted instrument line
			}
			else {
				p.setPen( highlighted );	// color for highlighted instrument line
			}
			// fill line
			for (uint j = 1; j < m_nGridHeight; j++) {
				p.drawLine( 0, y + j, (10 + MAX_NOTES * m_nGridWidth), y + j);
			}
		}
		else if ( pInstr->isMuted() ) {
			// Gray line
			p.setPen( muted );
			for (uint j = 1; j < m_nGridHeight; j++) {
				p.drawLine( 0, y + j, (10 + MAX_NOTES * m_nGridWidth), y + j);
			}
		}
		else {
			// fill line
			p.setPen( alternate );
			for (uint j = 1; j < m_nGridHeight; j++) {
				if (i % 2 != 0) {
					p.drawLine( 0, y + j, (10 + MAX_NOTES * m_nGridWidth), y + j);
				}
			}
		}
	}

	// vertical lines
	QColor res_1(0, 0, 0);
	QColor res_2(100, 100, 100);
	QColor res_3(150, 150, 150);
	QColor res_4(200, 200, 200);
	QColor res_5(230, 230, 230);

//	QColor test(200, 0, 0);

	p.setPen(res_1);
	p.drawLine(0, 0, 0, velRuler_y_start );

	int nBase;
	if (m_bUseTriplets) {
		nBase = 3;
	}
	else {
		nBase = 4;
	}

	int n4th = 4 * MAX_NOTES / (nBase * 4);
	int n8th = 4 * MAX_NOTES / (nBase * 8);
	int n16th = 4 * MAX_NOTES / (nBase * 16);
	int n32th = 4 * MAX_NOTES / (nBase * 32);
	int n64th = 4 * MAX_NOTES / (nBase * 64);

	if (!m_bUseTriplets) {

		for (uint i = 0; i < MAX_NOTES + 1; i++) {
			uint x = 10 + i * m_nGridWidth;

			if ( (i % n4th) == 0 ) {
				if (m_nResolution >= 4) {
					p.setPen(res_1);
					p.drawLine(x, 0, x, velRuler_y_start );
				}
			}
			else if ( (i % n8th) == 0 ) {
				if (m_nResolution >= 8) {
					p.setPen(res_2);
					p.drawLine(x, 0, x, velRuler_y_start);
				}
			}
			else if ( (i % n16th) == 0 ) {
				if (m_nResolution >= 16) {
					p.setPen(res_3);
					p.drawLine(x, 0, x, velRuler_y_start );
				}
			}
			else if ( (i % n32th) == 0 ) {
				if (m_nResolution >= 32) {
					p.setPen(res_4);
					p.drawLine(x, 0, x, velRuler_y_start );
				}
			}
			else if ( (i % n64th) == 0 ) {
				if (m_nResolution >= 64) {
					p.setPen(res_5);
					p.drawLine(x, 0, x, velRuler_y_start );
				}
			}
		}
	}
	else {	// Triplets
		uint nCounter = 0;
		int nSize = 4 * MAX_NOTES / (nBase * m_nResolution);

		for (uint i = 0; i < MAX_NOTES + 1; i++) {
			uint x = 10 + i * m_nGridWidth;

			if ( (i % nSize) == 0) {
				if ((nCounter % 3) == 0) {
					p.setPen(res_1);
				}
				else {
					p.setPen(res_3);
				}
				p.drawLine(x, 0, x, velRuler_y_start );
				nCounter++;
			}
		}

	}


	// horizontal lines
	p.setPen(QColor(0, 0, 0));
	for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
		uint y = m_nGridHeight * i;
		p.drawLine( 0, y, (10 + MAX_NOTES * m_nGridWidth), y);
	}
	// end of instruments

	p.setPen(QColor(0, 0, 0));
	p.drawLine( 0, m_nGridHeight * MAX_INSTRUMENTS, (10 + MAX_NOTES * m_nGridWidth), m_nGridHeight * MAX_INSTRUMENTS);
}





/**
 * Paint method
 */
void PatternEditor::paintEvent( QPaintEvent*)
{
	if (!isVisible()) {
		return;
	}

	if (changed) {
		changed = false;
		if (notesChanged) {	// ridisegno tutto solo se sono cambiate le note
			drawPattern(&background);
			notesChanged = false;
		}
		setErasePixmap(background);
		// copio l'immagine definitiva
//		bitBlt(this, 0, 0, &background, 0, 0, m_nEditorWidth, m_nEditorHeight, CopyROP);
	}
}





/**
 *
 */
void PatternEditor::drawPattern(QPixmap *pixmap)
{
	Hydrogen *engine = Hydrogen::getInstance();
	Pattern *currentPattern = engine->getCurrentPattern();

	if (currentPattern == NULL) {
		return;
	}
//	cout << "*** patternEditor:drawpattern" << endl;

	SequenceList *sequenceList = currentPattern->getSequenceList();

	for (uint i = 0; i < sequenceList->getSize(); i++) 	{
		Sequence *seq = sequenceList->get(i);
		for (uint j = 0; j < MAX_NOTES; j++) {
			Note *note = seq->noteList[j];
			if (note != NULL) {
				drawNote(note, i, pixmap);
			}
		}
	}
}




/**
 * show event
 */
void PatternEditor::showEvent ( QShowEvent *ev ) {
	updateEditor();
//	updateStart(true);
}




/**
 * hide event
 */
void PatternEditor::hideEvent ( QHideEvent *ev ) {
//	updateStart(false);
}




/**
 *
 */
void PatternEditor::setResolution(uint res, bool bUseTriplets) {
	this->m_nResolution = res;
	this->m_bUseTriplets = bUseTriplets;

	// redraw all
	createBackground(&background);
	changed = true;
	notesChanged = true;
	update();
	m_pVelocityRuler->updateEditor();
	//FIXME: [PatternEditor::setResolution] aggiornare la risoluzione del Ruler in alto."
}



/**
 * This method is called from another thread (audio engine)
 */
void PatternEditor::patternChanged() {
	H2TextEvent *ev = new H2TextEvent("patternChanged");
	QApplication::postEvent(this, ev);

}



/**
 *
 */
void PatternEditor::customEvent( QCustomEvent *ev ) {
	if ( ev->type() != H2_TEXT_EVENT ) {	// Must be a H2TextEvent
		return;
	}
	QString message = ((H2TextEvent*)ev)->getText();

	if (message == QString( "patternChanged" ) ) {
		updateEditor();
	}
}







///////////////////////////////////



/**
 *
 */
PatternEditorRuler::PatternEditorRuler(QWidget* parent)  : QWidget(parent), Object( "PatternEditorRuler" )
{
//	cout << "PatternEditorRuler INIT" << endl;

	m_nGridWidth = (PreferencesMng::getInstance())->getPatternEditorGridWidth();

	m_nRulerWidth = 5 + m_nGridWidth * MAX_NOTES + 5;
	m_nRulerHeight = 15;

	setMinimumSize( m_nRulerWidth, m_nRulerHeight );
	setMaximumSize( m_nRulerWidth, m_nRulerHeight );
	resize( m_nRulerWidth, m_nRulerHeight );

	temp.resize( m_nRulerWidth, m_nRulerHeight );

	string tickPosition_path = string( IMG_PATH ) + string( "/img/patternEditor/tickPosition.png" );
	bool ok = tickPosition.load(tickPosition_path.c_str());
	if( ok == false ){
		qWarning( "PatternEditor: Error loading pixmap" );
	}

	background.resize( m_nRulerWidth, m_nRulerHeight );
	QColor backgroundColor(230, 230, 230);
	background.fill( backgroundColor );


	setBackgroundPixmap(background);

	changed = true;

	timer = new QTimer(this);
	connect(timer, SIGNAL(timeout()), this, SLOT(updateEditor()));
}





/**
 *
 */
PatternEditorRuler::~PatternEditorRuler() {
//	cout << "PatternEditorRuler DESTROY" << endl;
}





/**
 *
 */
void PatternEditorRuler::updateStart(bool start) {
	if (start) {
		timer->start(100);	// update ruler at 10 fps
	}
	else {
		timer->stop();
	}
}




/**
 * show event
 */
void PatternEditorRuler::showEvent ( QShowEvent *ev ) {
	updateEditor();
	updateStart(true);
}




/**
 * hide event
 */
void PatternEditorRuler::hideEvent ( QHideEvent *ev ) {
	updateStart(false);
}






/**
 *
 */
void PatternEditorRuler::updateEditor( bool bRedrawAll ) {
	static uint oldNTicks = 0;

	Hydrogen* engine = Hydrogen::getInstance();

	int state = engine->getState();
	if ( state == PLAYING ) {
		nTicks = engine->getTickPosition();
	}
	else {
		nTicks = 0;
	}


	if (oldNTicks != nTicks) {
		// redraw all
		changed = true;
		update();
	}
	oldNTicks = nTicks;

	if (bRedrawAll) {
		changed = true;
		update();
	}
}



/**
 * Paint method
 */
void PatternEditorRuler::paintEvent( QPaintEvent*) {
	if (!isVisible()) {
		return;
	}

	if (changed) {
//		cout << "PatternEditorRuler PAINT" << endl;

		changed = false;
		bitBlt(&temp, 0, 0, &background, 0, 0, m_nRulerWidth, m_nRulerHeight, CopyROP);

		// gray background for unusable section of pattern
		QPainter p(&temp);
		p.setPen( QColor(170,170,170) );
		Hydrogen* engine = Hydrogen::getInstance();
		Pattern *pPattern = engine->getCurrentPattern();
		if (pPattern) {
			int nXStart = 10 + pPattern->getSize() * m_nGridWidth;
			for (uint i = 0; i < m_nRulerHeight; ++i) {
				p.drawLine( nXStart, i, m_nRulerWidth, i);
			}
		}

		// numbers
		PreferencesMng *pref = PreferencesMng::getInstance();
		QString family = pref->getApplicationFontFamily().c_str();
		int size = pref->getApplicationFontPointSize();
		QFont font( family, size );
		p.setFont(font);
		p.setPen( QColor( 0, 0, 0 ) );
		p.drawLine( 0, 0, m_nRulerWidth, 0 );
		p.drawLine( 0, 0, 0, m_nRulerHeight );
		p.drawLine( m_nRulerWidth - 1, 0, m_nRulerWidth - 1, m_nRulerHeight );
		p.drawLine( 0, m_nRulerHeight - 1, m_nRulerWidth - 1, m_nRulerHeight - 1);

		uint nQuarter = 48;
		QColor res_2(100, 100, 100);
		p.setPen( res_2 );

		uint nText_x = 10 + 0 * m_nGridWidth;
		p.drawText( nText_x - m_nGridWidth, 0, m_nGridWidth * 2, m_nRulerHeight, Qt::AlignCenter, "1" );

		nText_x = 10 + nQuarter * m_nGridWidth;
		p.drawText( nText_x - m_nGridWidth, 0, m_nGridWidth * 2, m_nRulerHeight, Qt::AlignCenter, "2" );

		nText_x = 10 + nQuarter * 2 * m_nGridWidth;
		p.drawText( nText_x - m_nGridWidth, 0, m_nGridWidth * 2, m_nRulerHeight, Qt::AlignCenter, "3" );

		nText_x = 10 + nQuarter * 3 * m_nGridWidth;
		p.drawText( nText_x - m_nGridWidth, 0, m_nGridWidth * 2, m_nRulerHeight, Qt::AlignCenter, "4" );



		// draw tickPosition
		uint x = 10 + nTicks * m_nGridWidth - 5;
		bitBlt( &temp, x, 4, &tickPosition, 0, 0, 11, m_nRulerHeight, CopyROP );



		setErasePixmap(temp);

		// copio l'immagine definitiva
//		bitBlt(this, 0, 0, &temp, 0, 0, 18, 100, CopyROP);
	}
}
















//////////////////////////////////



///
/// Constructor
///
PatternEditorInstrumentList::PatternEditorInstrumentList( QWidget *parent, PatternEditor *pPatternEditor, PatternEditorPanel *pPanel ) : QWidget( parent ), Object( "PatternEditorInstrumentList" )
{

//	cout << "PatternEditorInstrumentList INIT" << endl;
 	m_pPatternEditorPanel = pPanel;
	m_nGridHeight = 12;
	m_nGridHeight = (PreferencesMng::getInstance())->getPatternEditorGridHeight();

	m_pPatternEditor = pPatternEditor;
	m_nEditorWidth = 100;
	m_nEditorHeight = m_nGridHeight * MAX_INSTRUMENTS - 1;
	changed = true;

	background.resize(m_nEditorWidth, m_nEditorHeight );
	temp.resize(m_nEditorWidth, m_nEditorHeight );

	createBackground( &background );

	resize( m_nEditorWidth, m_nEditorHeight );
	setMinimumSize( m_nEditorWidth, m_nEditorHeight );
	setMaximumSize( m_nEditorWidth, m_nEditorHeight );


	string genericIcon_path = IMG_PATH;
	genericIcon_path.append( "/img/songEditor/patternIcon.png" );
	genericIcon.load( genericIcon_path.c_str() );

	// Popup menu
	functionPopup  = new QPopupMenu( this, "patternSequence_patternPopupMenu" );
	functionPopup->insertItem( genericIcon, "Properties",  this, SLOT( functionProperties() ) );
	functionPopup->insertItem( genericIcon, "Load sample",  this, SLOT( functionLoad() ) );
	functionPopup->insertItem( genericIcon, "Mute",  this, SLOT( functionMute() ) );
	functionPopup->insertItem( genericIcon, "Solo",  this, SLOT( functionSolo() ) );
	functionPopup->insertItem( genericIcon, "Clear notes",  this, SLOT( functionClearNotes() ) );
	functionPopup->insertItem( genericIcon, "Fill notes",  this, SLOT( functionFillNotes() ) );




	(Hydrogen::getInstance())->addEngineListener( this );
}




///
/// Destructor
///
PatternEditorInstrumentList::~PatternEditorInstrumentList()
{
//	cout << "PatternEditorInstrumentList DESTROY" << endl;
}




/**
 * Paint method
 */
void PatternEditorInstrumentList::paintEvent( QPaintEvent*) {
	if (!isVisible()) {
		return;
	}

	if (changed) {
		changed = false;
//		cout << "----------- PatternEditorInstrumentList PAINT --------------" << endl;
//		bitBlt(&temp, 0, 0, &background, 0, 0, m_nEditorWidth, m_nEditorHeight, CopyROP);
//		setErasePixmap(temp);
		setErasePixmap(background);

		// copio l'immagine definitiva
//		bitBlt(this, 0, 0, &temp, 0, 0, m_nEditorWidth, m_nEditorHeight, CopyROP);
	}
}





void PatternEditorInstrumentList::createBackground(QPixmap *pixmap) {
//	cout << "[PatternEditorInstrumentList::createBackground()]" << endl;
	pixmap->fill();		// fill the pixmap with white color

	PreferencesMng *pref = PreferencesMng::getInstance();
	QString family = pref->getApplicationFontFamily().c_str();
	int size = pref->getApplicationFontPointSize();
	QFont font( family, size );

	QPainter p(pixmap);
	p.setFont(font);

	Song *song = (Hydrogen::getInstance())->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	int nSelectedInstrument = m_pPatternEditorPanel->getSelectedInstrument();

	// horizontal lines (only fill color)
	for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
		uint y = m_nGridHeight * i;

		if ( ((MAX_INSTRUMENTS -1) - i) == (uint)nSelectedInstrument ) {
			// highlighted instrument line
			p.setPen(QColor(209, 233, 245));	// color for highlighted instrument line
			// fill line
			for (uint j = 1; j < m_nGridHeight; j++) {
				p.drawLine( 0, y + j, (100 + m_nEditorWidth), y + j);
			}
		}
		else {
			// fill line
			for (uint j = 1; j < m_nGridHeight; j++) {
				if (i % 2 != 0) {
					p.setPen(QColor(240, 240, 255));
					p.drawLine( 0, y + j, (100 + m_nEditorWidth), y + j);
				}
			}
		}
	}

	p.setPen( QColor(0,0,0) );
	p.drawLine(0, 0, 0, m_nEditorHeight);

	// horizontal lines
	for (uint i = 0; i < MAX_INSTRUMENTS; i++) {
		uint y = m_nGridHeight * i;
		p.drawLine( 0, y, (100 + m_nEditorWidth), y);

		uint text_y = (m_nGridHeight * MAX_INSTRUMENTS) - (i * m_nGridHeight);
		string trackName = "";

		Instrument *instr = instrList->get(i);
		trackName = instr->getName();
		p.drawText( 5, text_y - 1 - m_nGridHeight, 100, m_nGridHeight + 2, Qt::AlignVCenter, trackName.c_str() );
	}

	p.drawLine( 0, m_nGridHeight * MAX_INSTRUMENTS, m_nEditorWidth, m_nGridHeight * MAX_INSTRUMENTS);
	p.drawLine(m_nEditorWidth - 1 , 0, m_nEditorWidth - 1 , m_nEditorHeight);
}




/**
 * This method is called from another thread (audio engine)
 */
void PatternEditorInstrumentList::patternChanged() {
	H2TextEvent *ev = new H2TextEvent("patternChanged");
	QApplication::postEvent(this, ev);

}



void PatternEditorInstrumentList::customEvent( QCustomEvent *ev ) {
	if ( ev->type() != H2_TEXT_EVENT ) {	// Must be a H2TextEvent
		return;
	}
	QString message = ((H2TextEvent*)ev)->getText();

	if (message == QString( "patternChanged" ) ) {
		updateEditor();
	}

}




void PatternEditorInstrumentList::updateEditor() {
	if(!isVisible()) {
		return;
	}

	Hydrogen* engine = Hydrogen::getInstance();

	// check engine state
	int state = engine->getState();
	if ( (state != READY) && (state != PLAYING) ) {
		errorLog( "[PatternEditorInstrumentList::updateEditor] FIXME: skip pattern editor update (state shoud be READY or PLAYING" );
		return;
	}

	createBackground(&background);
	changed = true;
	update();
}






void PatternEditorInstrumentList::mousePressEvent(QMouseEvent *ev) {
	int row = MAX_INSTRUMENTS - 1 - (ev->y()  / (int)m_nGridHeight);
	if (row >= MAX_INSTRUMENTS) {
		return;
	}

	m_pPatternEditorPanel->setSelectedInstrument( row );
	updateEditor();
	m_pPatternEditor->updateEditor();
	m_pPatternEditorPanel->getVelocityRuler()->updateEditor();

	if (ev->button() == LeftButton ) {
		float velocity = 0.8;
		float pan_L = 1.0;
		float pan_R = 1.0;
		Song *song = (Hydrogen::getInstance())->getSong();
		Instrument *instrRef = (song->getInstrumentList())->get(row);
		Note *note = new Note(0, velocity, pan_L, pan_R);
		note->setInstrument(instrRef);
		( Hydrogen::getInstance() )->noteOn( note );
	}
	else if (ev->button() == RightButton ) {
		functionPopup->popup( QPoint( ev->globalX(), ev->globalY() ) );
	}
	return;

}




/**
 *
 */
void PatternEditorInstrumentList::functionClearNotes() {
	Hydrogen *engine = (Hydrogen::getInstance());
	engine->lockEngine("PatternEditorInstrumentList::functionClearNotes");	// lock the audio engine

	Pattern* currentPattern = engine->getCurrentPattern();
	SequenceList *sequenceList = currentPattern->getSequenceList();
	int nSelectedInstrument = m_pPatternEditorPanel->getSelectedInstrument();

	Sequence *seq = sequenceList->get( nSelectedInstrument );

	for (uint i = 0; i < MAX_NOTES; i++) {
		if (seq->noteList[i] != NULL) {
			Note *note = seq->noteList[i];
			delete note;
			seq->noteList[i] = NULL;
		}
	}
	engine->unlockEngine();	// unlock the audio engine

	changed = true;
	update();
	m_pPatternEditor->updateEditor(true);
	(HydrogenApp::getInstance())->getPatternEditor()->getVelocityRuler()->updateEditor();
}




/**
 *
 */
void PatternEditorInstrumentList::functionFillNotes()
{
	Hydrogen *engine = (Hydrogen::getInstance());

	float velocity = 0.8;
	float pan_L = 1.0;
	float pan_R = 1.0;

	int nBase;
	if ( m_pPatternEditor->isUsingTriplets() ) {
		nBase = 3;
	}
	else {
		nBase = 4;
	}
	int nResolution = 4 * MAX_NOTES / ( nBase * m_pPatternEditor->getResolution() );


	engine->lockEngine("PatternEditorInstrumentList::functionFillNotes");	// lock the audio engine

	Pattern* currentPattern = engine->getCurrentPattern();
	SequenceList *sequenceList = currentPattern->getSequenceList();
	int nSelectedInstrument = m_pPatternEditorPanel->getSelectedInstrument();
	Sequence *seq = sequenceList->get( nSelectedInstrument );
	Song *song = ( Hydrogen::getInstance() )->getSong();
	Instrument *instrRef = (song->getInstrumentList())->get( nSelectedInstrument );

	for ( uint i = 0; i < MAX_NOTES; i += nResolution ) {
		if (seq->noteList[i] == NULL ) {
			// create the new note
			Note *note = new Note( i, velocity, pan_L, pan_R );
			note->setInstrument(instrRef);
			seq->noteList[i] = note;
		}
	}

	engine->unlockEngine();	// unlock the audio engine

	changed = true;
	update();
	m_pPatternEditor->updateEditor(true);
	(HydrogenApp::getInstance())->getPatternEditor()->getVelocityRuler()->updateEditor();
}



/**
 *
 */
void PatternEditorInstrumentList::functionProperties()
{
	int nLine = m_pPatternEditorPanel->getSelectedInstrument();

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	Instrument *instr = instrList->get(nLine);

	InstrumentPropertiesDialog *dialog = new InstrumentPropertiesDialog(this, instr);
	if (dialog->exec() == QDialog::Accepted) {
		song->setModified(true);
	}
	delete dialog;
	dialog = NULL;

	changed = true;
	updateEditor();
}


/**
 *
 */
void PatternEditorInstrumentList::functionMute()
{
	int nLine = m_pPatternEditorPanel->getSelectedInstrument();

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	InstrumentList *instrList = song->getInstrumentList();

	Instrument *instr = instrList->get(nLine);
	instr->setMuted( ! instr->isMuted() );

	m_pPatternEditor->updateEditor( true );
}



/**
 *
 */
void PatternEditorInstrumentList::functionSolo()
{
	int nLine = m_pPatternEditorPanel->getSelectedInstrument();
	( (HydrogenApp::getInstance())->getMixer() )->soloClicked( nLine );
	m_pPatternEditor->updateEditor(true);
}


/**
 *
 */
void PatternEditorInstrumentList::functionLoad()
{
	int nLine = m_pPatternEditorPanel->getSelectedInstrument();
	(HydrogenApp::getInstance())->loadNewInstrument(nLine);
}



///////////////////////////






///
/// Constructor
///
PatternEditorVelocityRuler::PatternEditorVelocityRuler( QWidget *parent, PatternEditor *pPatternEditor, PatternEditorPanel *pPanel ) : QWidget( parent ), Object( "PatternEditorVelocityRuler" )
{
//	cout << "**** PAtternEditorVelocityRuler INIT" << endl;
	m_pCurrentPattern = NULL;
	m_pPatternEditorPanel = pPanel;
	m_pPatternEditor = pPatternEditor;

	m_nGridWidth = (PreferencesMng::getInstance())->getPatternEditorGridWidth();

	m_nEditorWidth = 5 + m_nGridWidth * MAX_NOTES + 5;
	m_nEditorHeight = 50;

	resize( m_nEditorWidth, m_nEditorHeight );
	setMinimumSize( m_nEditorWidth, m_nEditorHeight );
	setMaximumSize( m_nEditorWidth, m_nEditorHeight );

	m_background.resize( m_nEditorWidth, m_nEditorHeight );

	m_bChanged = true;

	updateEditor();
	(Hydrogen::getInstance())->addEngineListener( this );
}


///
/// Destructor
///
PatternEditorVelocityRuler::~PatternEditorVelocityRuler()
{
//	cout << "**** PAtternEditorVelocityRuler DESTROY" << endl;
}



///
///
///
void PatternEditorVelocityRuler::mousePressEvent(QMouseEvent *ev)
{
//	infoLog( "mousePressEvent()" );

	int nBase;
	if (m_pPatternEditor->isUsingTriplets()) {
		nBase = 3;
	}
	else {
		nBase = 4;
	}

	int width = (m_nGridWidth * 4 *  MAX_NOTES) / ( nBase * m_pPatternEditor->getResolution());
	int x_pos = ev->x();
	int column;
	column = (x_pos - 10) + (width / 2);
	column = column / width;
	column = (column * 4 * MAX_NOTES) / ( nBase * m_pPatternEditor->getResolution() );

	float val = (int)m_nEditorHeight - ev->y();
	if (val > 50.0) {
		val = 50.0;
	}
	else if (val < 0.0) {
		val = 0.0;
	}
	val = val * 2;
	val = val / 100.0;
	SequenceList *sequenceList = m_pCurrentPattern->getSequenceList();

	int nSelectedInstrument = m_pPatternEditorPanel->getSelectedInstrument();

	Sequence *seq = sequenceList->get( nSelectedInstrument );
	for (uint j = 0; j < MAX_NOTES; j++) {
		Note *note = seq->noteList[j];
		if ( (note != NULL) && ((int)(note->getPosition()) == column) ) {
			note->setVelocity( val );
			m_bChanged = true;
			updateEditor();
			Song *song = (Hydrogen::getInstance())->getSong();
			song->setModified(true);

			char valueChar[100];
			sprintf( valueChar, "%#.2f",  val);
			( HydrogenApp::getInstance() )->setStatusBarMessage( QString("Set note velocity [%1]").arg( valueChar ), 2000 );
			break;
		}
	}
}



 void PatternEditorVelocityRuler::mouseMoveEvent( QMouseEvent *ev ) {
//	infoLog( "mouse move" );
	mousePressEvent( ev );
}






/**
 * Paint method
 */
void PatternEditorVelocityRuler::paintEvent( QPaintEvent*) {
	if (!isVisible()) {
		return;
	}

	if (m_bChanged) {
		m_bChanged = false;
		setErasePixmap(m_background);
	}
}



/**
 *
 */
void PatternEditorVelocityRuler::createBackground(QPixmap *pixmap) {
//	cout << "**** [PatternEditorVelocityRuler::createBackground]" << endl;

	QColor backgroundColor(230, 230, 230);
	pixmap->fill( backgroundColor );		// fill the pixmap with white color
	QPainter p(pixmap);

	// vertical lines
	QColor res_1(0, 0, 0);
	QColor res_2(100, 100, 100);
	QColor res_3(150, 150, 150);
	QColor res_4(200, 200, 200);
	QColor res_5(230, 230, 230);

	int nBase;
	if (m_pPatternEditor->isUsingTriplets()) {
		nBase = 3;
	}
	else {
		nBase = 4;
	}

	int n4th = 4 * MAX_NOTES / (nBase * 4);
	int n8th = 4 * MAX_NOTES / (nBase * 8);
	int n16th = 4 * MAX_NOTES / (nBase * 16);
	int n32th = 4 * MAX_NOTES / (nBase * 32);
	int n64th = 4 * MAX_NOTES / (nBase * 64);

	int nResolution = m_pPatternEditor->getResolution();

	if ( !m_pPatternEditor->isUsingTriplets() ) {

		for (uint i = 0; i < MAX_NOTES + 1; i++) {
			uint x = 10 + i * m_nGridWidth;

			if ( (i % n4th) == 0 ) {
				if (nResolution >= 4) {
					p.setPen(res_1);
					p.drawLine(x, 0, x, m_nEditorHeight);
				}
			}
			else if ( (i % n8th) == 0 ) {
				if (nResolution >= 8) {
					p.setPen(res_2);
					p.drawLine(x, 0, x, m_nEditorHeight);
				}
			}
			else if ( (i % n16th) == 0 ) {
				if (nResolution >= 16) {
					p.setPen(res_3);
					p.drawLine(x, 0, x, m_nEditorHeight);
				}
			}
			else if ( (i % n32th) == 0 ) {
				if (nResolution >= 32) {
					p.setPen(res_4);
					p.drawLine(x, 0, x, m_nEditorHeight);
				}
			}
			else if ( (i % n64th) == 0 ) {
				if (nResolution >= 64) {
					p.setPen(res_5);
					p.drawLine(x, 0, x, m_nEditorHeight);
				}
			}
		}
	}
	else {	// Triplets
		uint nCounter = 0;
		int nSize = 4 * MAX_NOTES / (nBase * nResolution);

		for (uint i = 0; i < MAX_NOTES + 1; i++) {
			uint x = 10 + i * m_nGridWidth;

			if ( (i % nSize) == 0) {
				if ((nCounter % 3) == 0) {
					p.setPen(res_1);
				}
				else {
					p.setPen(res_3);
				}
				p.drawLine(x, 0, x, m_nEditorHeight);
				nCounter++;
			}
		}
	}

	if (m_pCurrentPattern == NULL) {
		m_pCurrentPattern = (Hydrogen::getInstance())->getCurrentPattern();
	}

	if (m_pCurrentPattern != NULL) {
		SequenceList *sequenceList = m_pCurrentPattern->getSequenceList();
		int nSelectedInstrument = m_pPatternEditorPanel->getSelectedInstrument();

		for (uint i = 0; i < sequenceList->getSize(); i++) {
			if ( (int)i == nSelectedInstrument ) {	// draw the velocity

				Sequence *seq = sequenceList->get(i);
				for (uint j = 0; j < MAX_NOTES; j++) {
					Note *note = seq->noteList[j];
					if (note != NULL) {
						uint pos = note->getPosition() % MAX_NOTES;
						uint x_pos = 10 + pos * m_nGridWidth - 3;

						QColor center(80, 97, 117);
						QColor side(123, 149, 180);

						uint line_end = 50;

						uint velocity = (uint)(note->getVelocity() * 50.0);
						uint line_start = line_end - velocity;

						p.setPen(side);
						p.drawLine(x_pos + 2, line_start, x_pos + 2, line_end);	// left side
						p.drawLine(x_pos + 4, line_start, x_pos + 4, line_end);	// right side

						p.setPen(center);
						p.drawLine(x_pos + 3, line_start, x_pos + 3, line_end);	// center
					}
				}
			}
		}
	}
	p.setPen(res_1);
	p.drawLine(0, 0, m_nEditorWidth, 0);
	p.drawLine(0, m_nEditorHeight - 1, m_nEditorWidth, m_nEditorHeight - 1);

	m_bChanged = true;
}




void PatternEditorVelocityRuler::updateEditor()
{
	createBackground(&m_background);

	// redraw all
	update();
}




/**
 * This method is called from another thread (audio engine)
 */
void PatternEditorVelocityRuler::patternChanged() {
	H2TextEvent *ev = new H2TextEvent("patternChanged");
	QApplication::postEvent(this, ev);
}




void PatternEditorVelocityRuler::customEvent( QCustomEvent *ev ) {
	if ( ev->type() != H2_TEXT_EVENT ) {	// Must be a H2TextEvent
		return;
	}
	QString message = ((H2TextEvent*)ev)->getText();

	if (message == QString( "patternChanged" ) ) {
		m_pCurrentPattern = (Hydrogen::getInstance())->getCurrentPattern();

		updateEditor();
	}

}
