#include <iostream>
#include <map>
#include <string>
#include <cassert>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <vector>
#include <set>
#include <fstream>
#include <fnmatch.h>

#define ARCHLIST "/var/lib/ia32-libs-tools/arch_all.list"
#define RENAMELIST "/etc/ia32-libs-tools/rename.list"
#ifndef ARCH
#error ARCH undefined
#endif
#ifndef PKGPREFIX
#error PKGPREFIX undefined
#endif
#ifndef VERSION
#error VERSION undefined
#endif

#ifdef __ia64__
bool ia64_mode = true;
#else
bool ia64_mode = false;
#endif

class Archlist {
public:
    Archlist(std::string filename = ARCHLIST) : file(filename) {
	dirty = false;
	std::ifstream in(filename.c_str());
	std::string s;
	while(in >> s) {
//	    std::cerr << "arch: reading " << s << std::endl;
	    names.insert(s);
	}
    }
    ~Archlist() {
	if (dirty) {
	    std::ofstream out(file.c_str());
	    if (!out) {
		std::cerr << "Warning: arch_all.list is dirty but can't write.\n";
		return;
	    }
	    std::set<std::string>::const_iterator it;
	    for(it = names.begin(); it != names.end(); ++it) {
		out << *it << std::endl;
	    }
	}
    }
    void set(std::string name, std::string arch) {
	if (arch == "all") {
	    if (names.find(name) == names.end()) {
		std::cerr << "arch_all.list: adding " << name << " " << arch << std::endl;
		dirty = true;
		names.insert(name);
	    }
	} else {
	    if (names.find(name) != names.end()) {
		std::cerr << "arch_all.list: deleting " << name << " " << arch << std::endl;
		dirty = true;
		names.erase(name);
	    }
	}
    }
    bool is_arch_all(const std::string& name) const {
	return (names.find(name) != names.end());
    }
private:
    std::string file;
    std::set<std::string> names;
    bool dirty;
};

class Renamelist {
public:
    Renamelist(std::string filename = RENAMELIST) {
	std::ifstream in(filename.c_str());
	if (!in) {
	    std::cerr << "Error opening " << filename << std::endl;
	    exit(1);
	}
	std::string line;
	int linenum = 0;
	while(getline(in, line)) {
	    ++linenum;
	    unsigned int i, j;
	    if (line.length() == 0) continue;
	    if (line[0] == '#') continue;
	    for(i = 0; i < line.length(); ++i) {
		if (line[i] == ' ' || line[i] == '\t') break;
	    }
	    for(j = i + 1; j < line.length(); ++j) {
		if (line[j] != ' ' && line[j] != '\t') break;
	    }
	    if (j >= line.length()) {
		std::cerr << filename << ": Ignoring syntax error in line " << linenum << ": " << line << std::endl;
		continue;
	    }
	    std::string pat = line.substr(0, i);
	    std::string name = line.substr(j, line.length() - j);
	    std::pair<std::string, std::string>pair(pat, name);
	    patterns.push_back(pair);
	}
    }
    std::string rename(const std::string& name, const std::string& arch) const {
	if (arch == "all") return name;
	std::vector<std::pair<std::string, std::string> >::const_iterator it;
	for(it = patterns.begin(); it != patterns.end(); ++it) {
	    if (fnmatch(it->first.c_str(), name.c_str(), 0) == 0) {
		if (it->second == "-") return name;
		if (!ia64_mode && it->second != "+") return it->second;
		return PKGPREFIX + name;
	    }
	}
	return name;
    }
private:
    std::vector<std::pair<std::string, std::string> > patterns;
};


class Package;

class KeyValue {
public:
    size_t parse(char* buf) {
	char *t1 = strstr(buf, ":");
	assert(t1 != NULL);
	*t1 = 0;
	key = buf;
	while(t1[1] == ' ') ++t1;
	char *t2 = strstr(&t1[1], "\n");
	while(t2 != NULL && (t2[1] == ' ' || t2[1] == '\t')) t2 = strstr(&t2[1], "\n");
	assert(t2 != NULL);
	*t2 = 0;
	value = &t1[1];
	return t2 - buf + 1;
    }
    std::string key;
    std::string value;
};

bool is_name(char c) {
    if (c >= 'a' && c <= 'z') return true;
    if (c >= 'A' && c <= 'Z') return true;
    if (c >= '0' && c <= '9') return true;
    if (c == '+') return true;
    if (c == '-') return true;
    if (c == '_') return true;
    if (c == '.') return true;
    return false;
}

class PkgDep {
public:
    std::string name;
    std::string version;
//    PkgDep(std::string n, std::string v) : name(n), version(v) { }
    size_t parse(std::string& s, size_t offset) { 
	size_t i = offset;
	while(i < s.length() && is_name(s[i])) ++i;
	name = s.substr(offset, i - offset);
	while(i < s.length() && s[i] == ' ') ++i;
	if (i < s.length() && s[i] == '(') {
	    ++i;
	    offset = i;
	    while(s[i] != ')') ++i;
	    version = s.substr(offset, i - offset); 
	    ++i;
	} else {
	    version = "";
	}
//	std::cerr << "PkgDep: '" << name << "' '" << version << "'";
//	if (i < s.length()) {
//	    std::cerr << " '" << s.substr(i, s.length() - i) << "'";
//	}
//	std::cerr << std::endl;
	return i;
    }
    void mangle(const Archlist& archlist, const Renamelist& renamelist);
    void print(std::ostream& out) const { 
	out << name;
	if (version != "") {
	    out << " (" << version << ")";
	}
    }
};

class PkgDepOr {
public:
    std::vector<PkgDep> pkgOr;
    size_t parse(std::string& s, size_t offset) {
	while(offset < s.length() && s[offset] != ',') {
	    if (s[offset] == ' ' || s[offset] == '|') {
		++offset;
		continue;
	    }
	    assert(is_name(s[offset]));
	    PkgDep dep;
	    offset = dep.parse(s, offset);
	    pkgOr.push_back(dep);
	}
	return offset;
    }
    void mangle(const Archlist& archlist, const Renamelist& renamelist);
    void print(std::ostream& out) const {
	std::vector<PkgDep>::const_iterator it;
	std::string sep = "";
	for(it = pkgOr.begin(); it != pkgOr.end(); ++it) {
	    out << sep;
	    it->print(out);
	    sep = " | ";
	}
    }
};

class PkgDepAnd {
public:
    std::vector<PkgDepOr> pkgAnd;
    size_t parse(std::string& s, size_t offset) {
	while(offset < s.length()) {
            if (s[offset] == ' ' || s[offset] == ',') {
                ++offset;
                continue;
            }
	    assert(is_name(s[offset]));
            PkgDepOr dep;
            offset = dep.parse(s, offset);
            pkgAnd.push_back(dep);
        }
        return offset;
    }
    void mangle(const Archlist& archlist, const Renamelist& renamelist);
    void print(std::ostream& out) const {
	std::vector<PkgDepOr>::const_iterator it;
	std::string sep = "";
	for(it = pkgAnd.begin(); it != pkgAnd.end(); ++it) {
	    out << sep;
	    it->print(out);
	    sep = ", ";
	}
    }
};

/*
bool need_rename(const std::string& name, const std::string& arch) {
  // FIXME: sync with rename
  // FIXME: use some conffile
    if (arch == "all") return false;
    if (fnmatch("lib*", name.c_str(), 0) == 0) {
	return true;
    }
    if (name == "iptables") {
	return true;
    }
    return false;
}
*/

class Package {
public:
    size_t parse(char* buf) {
	char *p = buf;
	if (*p == 0) return 0;
	while(*p != 0 && *p != '\n') {
	    KeyValue kv;
	    p += kv.parse(p);
	    if (kv.key == "Package") {
		name = kv.value;
	    } else if (kv.key == "Architecture") {
		if (kv.value == "all") {
		    arch = kv.value;
		} else {
		    arch = ARCH;
		}
	    } else if (kv.key == "Version") {
		version = kv.value;
	    } else if (kv.key == "Conflicts") { 
                deps["Conflicts"] = parse_deps(kv.value);
	    } else if (kv.key == "Depends") { 
                deps["Depends"] = parse_deps(kv.value);
	    } else if (kv.key == "Pre-Depends") { 
                deps["Pre-Depends"] = parse_deps(kv.value);
	    } else if (kv.key == "Provides") { 
                deps["Provides"] = parse_deps(kv.value);
	    } else if (kv.key == "Recommends") { 
                deps["Recommends"] = parse_deps(kv.value);
	    } else if (kv.key == "Replaces") { 
                deps["Replaces"] = parse_deps(kv.value);
	    } else if (kv.key == "Suggests") { 
                deps["Suggests"] = parse_deps(kv.value);
            } else {
		data[kv.key] = kv.value;
	    }
	}
	if (*p == '\n') ++p;
	return p - buf;
    }
    PkgDepAnd parse_deps(std::string& s) {
	PkgDepAnd pkgAnd;
	pkgAnd.parse(s, 0);
	return pkgAnd;
    }
    void print(std::ostream& out) const {
	out << "Package: " << name << std::endl;
	out << "Architecture: " << arch << std::endl;
	out << "Version: " << version << std::endl;
	std::map<std::string, PkgDepAnd>::const_iterator depIt;
	for(depIt = deps.begin(); depIt != deps.end(); ++depIt) {
	    out << depIt->first << ": ";
	    depIt->second.print(out);
	    out << std::endl;
	}
	std::map<std::string, std::string>::const_iterator it;
	for(it = data.begin(); it != data.end(); ++it) {
	    out << it->first << ": " << it->second << std::endl;
	}
	out << std::endl;
    }
    void mangle(const Archlist& archlist, const Renamelist& renamelist) {
	std::map<std::string, PkgDepAnd>::iterator depIt; 
        for(depIt = deps.begin(); depIt != deps.end(); ++depIt) { 
	    depIt->second.mangle(archlist, renamelist);
	}
	std::string src;
	if (data.find("Source") == data.end()) {
	    src = name + " (" + version + ")";
	} else {
	    src = data["Source"];
	    if (src[src.length()-1] != ')') {
		src += " (" + version + ")";
	    }
	}
	data["Source"] = src;
	name = renamelist.rename(name, arch);
// Always change version so native packages are newer than foreign
//	if (need_rename(name, arch)) {
	if (arch != "all") {
	    if (version != "") { 
		// FIXME: 1.2-3+b1 -> 1.2-3~5+b1
		version += VERSION;
	    }
	}
    }
    std::string name;
    std::string arch;
    std::string version;
    std::map<std::string, PkgDepAnd> deps;
    std::map<std::string, std::string> data;
};

void PkgDep::mangle(const Archlist& archlist, const Renamelist& renamelist) {
    std::string arch = (archlist.is_arch_all(name)) ? "all" : ARCH;
    name = renamelist.rename(name, arch);
// Always change version so native packages are newer than foreign
//    if (need_rename(name, arch)) {
    if (arch != "all") {
	if (version != "") {
	    // FIXME: 1.2-3+b1 -> 1.2-3~5+b1
	    version += VERSION;
	}
    }
} 

void PkgDepOr::mangle(const Archlist& archlist, const Renamelist& renamelist) {
    std::vector<PkgDep>::iterator it; 
    for(it = pkgOr.begin(); it != pkgOr.end(); ++it) { 
	it->mangle(archlist, renamelist);
    }
}

void PkgDepAnd::mangle(const Archlist& archlist, const Renamelist& renamelist) {
    std::vector<PkgDepOr>::iterator it;
    for(it = pkgAnd.begin(); it != pkgAnd.end(); ++it) {
	it->mangle(archlist, renamelist);
    }
}

void mangle(Archlist& archlist, Renamelist& renamelist) {
    std::map<std::string, Package> packages;
    char *buf = (char*)malloc(1024);
    size_t buf_size = 1024;
    size_t buf_used = 0;
    size_t len;
    int num_packages = 0;

    // Read Packages file into memory
    assert(buf != NULL);
    while ((len = read(STDIN_FILENO, &buf[buf_used], buf_size - buf_used - 3)) > 0) {
	buf_used += len;
	if (buf_size <= buf_used + 3) {
	    buf_size *= 2;
	    buf = (char*)realloc(buf, buf_size);
	    assert(buf != NULL);
	}
    }
    assert(buf_used > 1);
    while(buf[buf_used-2] != '\n') {
	buf[buf_used++] = '\n';
    }
    buf[buf_used] = 0;
    // std::cerr << "Read " << buf_used << " byte" << std::endl;

    // Parse packages
    while(true) {
	Package pkg;
	len = pkg.parse(buf);
	if (len <= 0) break;
	buf += len;
	++num_packages;
	packages[pkg.name] = pkg;
    }
    // std::cerr << "Parsed " << num_packages << " packages" << std::endl;

    // Update Archlist
    std::map<std::string, Package>::iterator it; 
    for(it = packages.begin(); it != packages.end(); ++it) {
	archlist.set(it->second.name, it->second.arch);
    }

    // Mangle depends and output
    for(it = packages.begin(); it != packages.end(); ++it) {
	it->second.mangle(archlist, renamelist);
	it->second.print(std::cout);
    }
}

int main(int argc, char *argv[]) {
    Archlist archlist;
    Renamelist renamelist;
    int base = 1;

    if (argc > 2 && std::string(argv[1]) == "--arch") {
	base = 3;
	if (std::string(argv[2]) == "ia64") {
	    ia64_mode = true;
	} else {
	    ia64_mode = false;
	}
    }

    if (argc == base + 1) {
	if (std::string(argv[base]) == "--index") {
	    mangle(archlist, renamelist);
	    return 0;
	} else if (std::string(argv[base]) == "--shlibs") {
	    std::cerr << "shlibs" << std::endl;
	    return 0;
	} else if (std::string(argv[base]) == "--symbols") {
	    std::cerr << "symbols" << std::endl;
	    return 0;
	}
    } else if (argc == base + 2 && std::string(argv[base]) == "--rename") {
      std::string name(argv[base + 1]);
      std::string arch = (archlist.is_arch_all(name)) ? "all" : ARCH;
      std::cout << renamelist.rename(name, arch) << std::endl;
      return 0;
    }

    std::cerr << "Usage: " << argv[0] << " [--arch <arch>] --index < index-file\n";
    std::cerr << "       " << argv[0] << " [--arch <arch>] --rename name\n";
    std::cerr << "       " << argv[0] << " [--arch <arch>] --shlibs < shlibs-file\n";
    std::cerr << "       " << argv[0] << " [--arch <arch>] --symbols < symbols-file\n";
    exit(1);
}
