/*
 * SpanDSP - a series of DSP components for telephony
 *
 * t38_gateway_to_terminal_tests.c - Joint tests for the T.38 FoIP terminal and gateway modules.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2005 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t38_gateway_to_terminal_tests.c,v 1.10 2006/05/18 14:15:36 steveu Exp $
 */

/*! \file */

/*! \page t38_gateway_to_terminal_tests_page T.38 mixed gateway and termination tests
\section t38_gateway_to_terminal_tests_page_sec_1 What does it do?
These tests exercise the path

    FAX machine -> T.38 gateway -> T.38 termination
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <inttypes.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <tgmath.h>
#include <math.h>
#include <assert.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#define __USE_MISC
#include <arpa/inet.h>
#include <sys/select.h>
#include <sys/time.h>
#include <audiofile.h>
#include <tiffio.h>

#include "spandsp.h"

#define SAMPLES_PER_CHUNK       160

#define INPUT_FILE_NAME         "itutests.tif"
#define OUTPUT_FILE_NAME        "t38.tif"
#define OUTPUT_FILE_NAME_T30    "t38_gateway_to_terminal_t30.wav"
#define OUTPUT_FILE_NAME_T38    "t38_gateway_to_terminal_t38.wav"

fax_state_t fax_state_a;
t38_state_t t38_state_a;
t38_state_t t38_state_b;

queue_t queue_a_to_b;
queue_t queue_b_to_a;

int done[2] = {FALSE, FALSE};
int succeeded[2] = {FALSE, FALSE};

int current_samples = 0;

static void phase_b_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    
    i = (int) (intptr_t) user_data;
    printf("%c: Phase B handler on channel %c - (0x%X) %s\n", i, i, result, t30_frametype(result));
}
/*- End of function --------------------------------------------------------*/

static void phase_d_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    t30_stats_t t;
    char ident[21];

    i = (int) (intptr_t) user_data;
    printf("%c: Phase D handler on channel %c - (0x%X) %s\n", i, i, result, t30_frametype(result));
    t30_get_transfer_statistics(s, &t);
    printf("%c: Phase D: bit rate %d\n", i, t.bit_rate);
    printf("%c: Phase D: pages transferred %d\n", i, t.pages_transferred);
    printf("%c: Phase D: image size %d x %d\n", i, t.width, t.length);
    printf("%c: Phase D: image resolution %d x %d\n", i, t.x_resolution, t.y_resolution);
    printf("%c: Phase D: bad rows %d\n", i, t.bad_rows);
    printf("%c: Phase D: longest bad row run %d\n", i, t.longest_bad_row_run);
    printf("%c: Phase D: coding method %s\n", i, t4_encoding_to_str(t.encoding));
    printf("%c: Phase D: image size %d\n", i, t.image_size);
    t30_get_local_ident(s, ident);
    printf("%c: Phase D: local ident '%s'\n", i, ident);
    t30_get_far_ident(s, ident);
    printf("%c: Phase D: remote ident '%s'\n", i, ident);
}
/*- End of function --------------------------------------------------------*/

static void phase_e_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    t30_stats_t t;
    char ident[21];
    
    i = (int) (intptr_t) user_data;
    done[i - 'A'] = TRUE;
    printf("%c: Phase E handler on channel %c - (%d) %s\n", i, i, result, t30_completion_code_to_str(result));
    t30_get_transfer_statistics(s, &t);
    printf("%c: Phase E: bit rate %d\n", i, t.bit_rate);
    printf("%c: Phase E: pages transferred %d\n", i, t.pages_transferred);
    printf("%c: Phase E: image size %d x %d\n", i, t.width, t.length);
    printf("%c: Phase E: image resolution %d x %d\n", i, t.x_resolution, t.y_resolution);
    printf("%c: Phase E: bad rows %d\n", i, t.bad_rows);
    printf("%c: Phase E: longest bad row run %d\n", i, t.longest_bad_row_run);
    printf("%c: Phase E: coding method %s\n", i, t4_encoding_to_str(t.encoding));
    printf("%c: Phase E: image size %d bytes\n", i, t.image_size);
    t30_get_local_ident(s, ident);
    printf("%c: Phase E: local ident '%s'\n", i, ident);
    t30_get_far_ident(s, ident);
    printf("%c: Phase E: remote ident '%s'\n", i, ident);
    succeeded[i - 'A'] = (result == T30_ERR_OK)  &&  (t.pages_transferred == 12);
}
/*- End of function --------------------------------------------------------*/

static int tx_packet_handler_a(t38_state_t *s, void *user_data, const uint8_t *buf, int len, int count)
{
    int delay[2];

    /* This routine queues messages between two instances of T.38 processing */
    span_log(&s->logging, SPAN_LOG_FLOW, "Send seq %d, len %d\n", s->tx_seq_no, len);

    delay[0] = current_samples + 800 + (rand() & 0x7FF);
    delay[1] = t38_state_b.tx_seq_no++;
    queue_write_msg(&queue_a_to_b, (uint8_t *) delay, sizeof(delay));
    queue_write_msg(&queue_a_to_b, buf, len);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int tx_packet_handler_b(t38_state_t *s, void *user_data, const uint8_t *buf, int len, int count)
{
    int delay[2];

    /* This routine queues messages between two instances of T.38 processing */
    span_log(&s->logging, SPAN_LOG_FLOW, "Send seq %d, len %d\n", s->tx_seq_no, len);

    delay[0] = current_samples + 800 + (rand() & 0x7FF);
    delay[1] = t38_state_a.tx_seq_no++;
    queue_write_msg(&queue_b_to_a, (uint8_t *) delay, sizeof(delay));
    queue_write_msg(&queue_b_to_a, buf, len);
    return 0;
}
/*- End of function --------------------------------------------------------*/

int main(int argc, char *argv[])
{
    int16_t t38_amp_a[SAMPLES_PER_CHUNK];
    int16_t t30_amp_a[SAMPLES_PER_CHUNK];
    int t38_len_a;
    int t30_len_a;
    int msg_len;
    uint8_t msg[1024];
    int delay_a[2];
    int delay_b[2];
    int log_audio;
    int outframes;
    int t38_version;
    AFfilesetup filesetup;
    AFfilehandle t30_handle;
    AFfilehandle t38_handle;

    log_audio = TRUE;
    t38_version = 1;
    t30_handle = AF_NULL_FILEHANDLE;
    t38_handle = AF_NULL_FILEHANDLE;
    if (log_audio)
    {
        filesetup = afNewFileSetup();
        if (filesetup == AF_NULL_FILESETUP)
        {
            fprintf(stderr, "    Failed to create file setup\n");
            exit(2);
        }
        afInitSampleFormat(filesetup, AF_DEFAULT_TRACK, AF_SAMPFMT_TWOSCOMP, 16);
        afInitRate(filesetup, AF_DEFAULT_TRACK, (float) SAMPLE_RATE);
        afInitFileFormat(filesetup, AF_FILE_WAVE);
        afInitChannels(filesetup, AF_DEFAULT_TRACK, 1);

        t30_handle = afOpenFile(OUTPUT_FILE_NAME_T30, "w", filesetup);
        if (t30_handle == AF_NULL_FILEHANDLE)
        {
            fprintf(stderr, "    Cannot create wave file '%s'\n", OUTPUT_FILE_NAME_T30);
            exit(2);
        }
        t38_handle = afOpenFile(OUTPUT_FILE_NAME_T38, "w", filesetup);
        if (t38_handle == AF_NULL_FILEHANDLE)
        {
            fprintf(stderr, "    Cannot create wave file '%s'\n", OUTPUT_FILE_NAME_T38);
            exit(2);
        }
    }
    fax_init(&fax_state_a, FALSE, NULL);
    t30_set_local_ident(&fax_state_a.t30_state, "11111111");
    t30_set_tx_file(&fax_state_a.t30_state, INPUT_FILE_NAME, -1, -1);
    t30_set_phase_b_handler(&fax_state_a.t30_state, phase_b_handler, (void *) (intptr_t) 'B');
    t30_set_phase_d_handler(&fax_state_a.t30_state, phase_d_handler, (void *) (intptr_t) 'B');
    t30_set_phase_e_handler(&fax_state_a.t30_state, phase_e_handler, (void *) (intptr_t) 'B');
    fax_state_a.logging.level = SPAN_LOG_DEBUG | SPAN_LOG_SHOW_TAG;
    fax_state_a.logging.tag = "FAX-B ";
    fax_state_a.t30_state.logging.level = SPAN_LOG_DEBUG | SPAN_LOG_SHOW_TAG;
    fax_state_a.t30_state.logging.tag = "FAX-B ";
    memset(t30_amp_a, 0, sizeof(t30_amp_a));

    if (t38_gateway_init(&t38_state_a, FALSE, tx_packet_handler_a, &t38_state_b) == NULL)
    {
        fprintf(stderr, "Cannot start the T.38 channel\n");
        exit(2);
    }
    t38_set_t38_version(&t38_state_a, t38_version);
    t38_state_a.logging.level = SPAN_LOG_DEBUG | SPAN_LOG_SHOW_TAG;
    t38_state_a.logging.tag = "T.38-B";
    t38_state_a.t30_state.logging.level = SPAN_LOG_DEBUG | SPAN_LOG_SHOW_TAG;
    t38_state_a.t30_state.logging.tag = "T.38-B";
    memset(t38_amp_a, 0, sizeof(t38_amp_a));

    if (t38_terminal_init(&t38_state_b, TRUE, tx_packet_handler_b, &t38_state_a) == NULL)
    {
        fprintf(stderr, "Cannot start the T.38 channel\n");
        exit(2);
    }
    t38_set_t38_version(&t38_state_b, t38_version);
    t38_state_b.logging.level = SPAN_LOG_DEBUG | SPAN_LOG_SHOW_TAG;
    t38_state_b.logging.tag = "T.38-A";
    t38_state_b.t30_state.logging.level = SPAN_LOG_DEBUG | SPAN_LOG_SHOW_TAG;
    t38_state_b.t30_state.logging.tag = "T.38-A";

    t30_set_local_ident(&t38_state_b.t30_state, "22222222");
    t30_set_rx_file(&t38_state_b.t30_state, OUTPUT_FILE_NAME, -1);
    t30_set_phase_b_handler(&t38_state_b.t30_state, phase_b_handler, (void *) (intptr_t) 'A');
    t30_set_phase_d_handler(&t38_state_b.t30_state, phase_d_handler, (void *) (intptr_t) 'A');
    t30_set_phase_e_handler(&t38_state_b.t30_state, phase_e_handler, (void *) (intptr_t) 'A');

    if (queue_create(&queue_a_to_b, 32768, QUEUE_WRITE_ATOMIC | QUEUE_READ_ATOMIC) < 0)
        return -1;
    if (queue_create(&queue_b_to_a, 32768, QUEUE_WRITE_ATOMIC | QUEUE_READ_ATOMIC) < 0)
        return -1;
    delay_a[0] = -1;
    delay_b[0] = -1;

    for (;;)
    {
        t38_send_timeout(&t38_state_b, SAMPLES_PER_CHUNK);

        {
            t30_len_a = fax_tx(&fax_state_a, t30_amp_a, SAMPLES_PER_CHUNK);
            /* The receive side always expects a full block of samples, but the
               transmit side may not be sending any when it doesn't need to. We
               may need to pad with some silence. */
            if (t30_len_a < SAMPLES_PER_CHUNK)
            {
                memset(t30_amp_a + t30_len_a, 0, sizeof(int16_t)*(SAMPLES_PER_CHUNK - t30_len_a));
                t30_len_a = SAMPLES_PER_CHUNK;
            }
            if (log_audio)
            {
                outframes = afWriteFrames(t30_handle, AF_DEFAULT_TRACK, t30_amp_a, t30_len_a);
                if (outframes != t30_len_a)
                    break;
            }
            if (t38_rx(&t38_state_a, t30_amp_a, t30_len_a))
                break;
    
            t38_len_a = t38_tx(&t38_state_a, t38_amp_a, SAMPLES_PER_CHUNK);
            if (t38_len_a < SAMPLES_PER_CHUNK)
            {
                memset(t38_amp_a + t38_len_a, 0, sizeof(int16_t)*(SAMPLES_PER_CHUNK - t38_len_a));
                t38_len_a = SAMPLES_PER_CHUNK;
            }
            if (log_audio)
            {
                outframes = afWriteFrames(t38_handle, AF_DEFAULT_TRACK, t38_amp_a, t38_len_a);
                if (outframes != t38_len_a)
                    break;
            }
            if (fax_rx(&fax_state_a, t38_amp_a, SAMPLES_PER_CHUNK))
                break;
            t38_send_timeout(&t38_state_a, SAMPLES_PER_CHUNK);
            current_samples += SAMPLES_PER_CHUNK;
        }

        if (!queue_empty(&queue_a_to_b))
        {
            if (delay_a[0] < 0)
                msg_len = queue_read_msg(&queue_a_to_b, (uint8_t *) delay_a, sizeof(delay_a));
            if (!queue_empty(&queue_a_to_b)  &&  delay_a[0] <= current_samples)
            {
                delay_a[0] = -1;
                msg_len = queue_read_msg(&queue_a_to_b, msg, 1024);
                t38_rx_ifp_packet(&t38_state_b, delay_a[1], msg, msg_len);
            }
        }
        if (!queue_empty(&queue_b_to_a))
        {
            if (delay_b[0] < 0)
                msg_len = queue_read_msg(&queue_b_to_a, (uint8_t *) delay_b, sizeof(delay_b));
            if (!queue_empty(&queue_b_to_a)  &&  delay_b[0] <= current_samples)
            {
                delay_b[0] = -1;
                msg_len = queue_read_msg(&queue_b_to_a, msg, 1024);
                t38_rx_ifp_packet(&t38_state_a, delay_b[1], msg, msg_len);
            }
        }
        if (done[0]  &&  done[1])
            break;
    }
    if (log_audio)
    {
        if (afCloseFile(t30_handle) != 0)
        {
            fprintf(stderr, "    Cannot close wave file '%s'\n", OUTPUT_FILE_NAME_T30);
            exit(2);
        }
        if (afCloseFile(t38_handle) != 0)
        {
            fprintf(stderr, "    Cannot close wave file '%s'\n", OUTPUT_FILE_NAME_T38);
            exit(2);
        }
        afFreeFileSetup(filesetup);
    }
    if (!succeeded[0]  ||  !succeeded[1])
    {
        printf("Tests failed\n");
        exit(2);
    }
    printf("Tests passed\n");
    return  0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
