/*
 * mount.c
 * - mount authentication file stuff
 *
 * Copyright (c) 1999 Jack Moffitt, Barath Raghavan, and Alexander Havng
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#ifdef _WIN32
#include <win32config.h>
#else
#include <config.h>
#endif
#endif

#include "definitions.h"
#include <stdio.h>
#include "definitions.h"

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

#include <string.h>
#include <sys/types.h>
#include <time.h>
#include <stdlib.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifndef _WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#endif

#include "avl.h"
#include "threads.h"
#include "icetypes.h"
#include "icecast.h"
#include "utility.h"
#include "ice_string.h"
#include "connection.h"
#include "log.h"
#include "sock.h"
#include "avl_functions.h"
#include "restrict.h"
#include "match.h"
#include "memory.h"
#include "commands.h"
#include "basic.h"
#include "mount.h"
#include "group.h"
#include "admin.h"

extern server_info_t info;

extern mutex_t authentication_mutex;
extern mounttree_t *mounttree;
extern usertree_t *usertree;
extern grouptree_t *grouptree;

void parse_mount_authentication_file()
{
	int fd;
	char *mountfile = get_icecast_file(info.mountfile, conf_file_e, R_OK);
	mount_t *mount;
	char line[BUFSIZE];

	if (!mountfile || ((fd = open_for_reading(mountfile)) == -1)) {
		if (mountfile)
			nfree(mountfile);
		xa_debug(1, "WARNING: Could not open mount authentication file");
		return;
	}
	while (fd_read_line(fd, line, BUFSIZE)) {
		if (line[0] == '#' || line[0] == ' ')
			continue;

		mount = create_mount_from_line(line);

		if (mount)
			add_authentication_mount(mount);
	}

	if (mountfile)
		nfree(mountfile);
	fd_close(fd);
}

mount_t *
 create_mount_from_line(char *line)
{
	mount_t *mount;
	group_t *group;
	char name[BUFSIZE], cgroup[BUFSIZE];
	int go_on = 1;

	if (!line) {
		xa_debug(1, "WARNING: create_mount_from_line() called with NULL pointer");
		return NULL;
	}
	if (!splitc(name, line, ':')) {
		xa_debug(1, "ERROR: Syntax error in mount file, with line [%s]", line);
		return NULL;
	}
	mount = create_mount();

	mount->name = nstrdup(clean_string(name));

	do {
		if (splitc(cgroup, line, ',') == NULL) {
			strcpy(cgroup, line);
			go_on = 0;
		}
		group = find_group_from_tree(grouptree, clean_string(cgroup));

		if (!group) {
			write_log(LOG_DEFAULT, "WARNING: Unrecognized group [%s] specified for mount [%s]",
				  cgroup, name);
		} else {
			avl_insert(mount->grouptree, group);
		}
	} while (go_on);

	return mount;
}

mount_t *
 create_mount()
{
	mount_t *mount = (mount_t *) nmalloc(sizeof (mount_t));

	mount->grouptree = create_group_tree();
	mount->name = NULL;
	return mount;
}

mounttree_t *
 create_mount_tree()
{
	mounttree_t *gt = avl_create(compare_mounts, &info);

	return gt;
}

void add_authentication_mount(mount_t * mount)
{
	mount_t *out;

	if (!mount || !mounttree || !mount->name || !mount->grouptree) {
		xa_debug(1, "ERROR: add_authentication_mount() called with NULL pointers");
		return;
	}
	out = avl_replace(mounttree, mount);

	if (out) {
		write_log(LOG_DEFAULT, "WARNING: Duplicate mount record %s, using latter", mount->name);
		nfree(out->name);
		free_group_tree(out->grouptree);
		nfree(out);
	}
	xa_debug(1, "DEBUG: add_authentication_mount(): Inserted mount [%s]", mount->name);
}

void free_mount_tree(mounttree_t * mt)
{
	avl_traverser trav =
	{0};
	mount_t *mount, *out;

	if (!mt)
		return;

	while ((mount = avl_traverse(mt, &trav))) {
		out = avl_delete(mt, mount);

		if (!out) {
			xa_debug(1, "WARNING: Weird ass things going on in mounttree!");
			continue;
		}
		nfree(mount->name);
		avl_destroy(mount->grouptree, NULL);
		nfree(mount);
	}
}

grouptree_t *
 get_grouptree_for_mount(const char *mountname)
{
	mount_t *mount;
	avl_traverser trav =
	{0};

	if (!mountname)
		return NULL;

	while ((mount = avl_traverse(mounttree, &trav))) {
		if (mount->name && (ice_strcmp(mountname, mount->name) == 0))
			return mount->grouptree;
	}

	return NULL;
}

void con_display_mounts(com_request_t * req)
{
	avl_traverser trav =
	{0};
	avl_traverser grouptrav =
	{0};
	mount_t *mount;
	group_t *group;
	int listed = 0;

	admin_write_line(req, ADMIN_SHOW_AUTH_MOUNT_START, "Listing mount points in the authentication module:");

	thread_mutex_lock(&authentication_mutex);

	while ((mount = avl_traverse(mounttree, &trav))) {
		zero_trav(&grouptrav);

		admin_write(req, ADMIN_SHOW_AUTH_MOUNT_ENTRY, "%s: ", mount->name ? mount->name : "(null)");

		while ((group = avl_traverse(mount->grouptree, &grouptrav)))
			admin_write(req, -1, "%s ", group->name);
		admin_write_line(req, -1, "");
		listed++;
	}

	thread_mutex_unlock(&authentication_mutex);

	admin_write_line(req, ADMIN_SHOW_AUTH_MOUNT_END, "End of mount point listing (%d listed)", listed);
}

int runtime_add_mount_with_group(const char *name, char *groups)
{
	char line[BUFSIZE];
	char *mountfile, *s;
	int fd;

	if (!name || !groups || !name[0] || !groups[0])
		return ICE_ERROR_INVALID_SYNTAX;

	while ((s = strchr(groups, ' ')))
		*s = ',';

#ifdef _WIN32
	sprintf(line, "%s:%s\r\n", name, groups);
#else
	sprintf(line, "%s:%s\n", name, groups);
#endif

	thread_mutex_lock(&authentication_mutex);

	if (get_grouptree_for_mount(name)) {
		thread_mutex_unlock(&authentication_mutex);
		return ICE_ERROR_DUPLICATE;
	}
	mountfile = get_icecast_file(info.mountfile, conf_file_e, W_OK);

	if (!mountfile || ((fd = open_for_append(mountfile)) == -1)) {
		if (mountfile)
			nfree(mountfile);
		xa_debug(1, "WARNING: Could not open mount authentication file for writing");
		thread_mutex_unlock(&authentication_mutex);
		return ICE_ERROR_FILE;
	}
	fd_write_line(fd, "%s", line);
	fd_close(fd);
	if (mountfile)
		nfree(mountfile);
	thread_mutex_unlock(&authentication_mutex);
	return 1;
}

int runtime_add_mount(const char *name)
{
	char line[BUFSIZE];
	char *mountfile;
	int fd;

	if (!name || !name[0])
		return ICE_ERROR_INVALID_SYNTAX;
#ifdef _WIN32
	sprintf(line, "%s:\r\n", name);
#else
	sprintf(line, "%s:\n", name);
#endif

	thread_mutex_lock(&authentication_mutex);

	if (get_grouptree_for_mount(name)) {
		thread_mutex_unlock(&authentication_mutex);
		return ICE_ERROR_DUPLICATE;
	}
	mountfile = get_icecast_file(info.mountfile, conf_file_e, W_OK);

	if (!mountfile || ((fd = open_for_append(mountfile)) == -1)) {
		if (mountfile)
			nfree(mountfile);
		xa_debug(1, "WARNING: Could not open mount authentication file for writing");
		thread_mutex_unlock(&authentication_mutex);
		return ICE_ERROR_FILE;
	}
	fd_write_line(fd, "%s", line);
	fd_close(fd);
	if (mountfile)
		nfree(mountfile);
	thread_mutex_unlock(&authentication_mutex);
	return 1;
}
