/*****************************************************************************
 *
 * GETCGI.C -  Icinga CGI Input Routines
 *
 * Copyright (c) 1999-2009 Ethan Galstad (egalstad@nagios.org)
 * Copyright (c) 2009-2015 Icinga Development Team (http://www.icinga.org)
 *
 * License:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 *****************************************************************************/

#include "../include/config.h"
#include "../include/getcgi.h"
#include "../include/cgiutils.h"
#include <stdio.h>
#include <stdlib.h>


#undef PARANOID_CGI_INPUT


/* Remove potentially harmful characters from CGI input that we don't need or want */
void sanitize_cgi_input(char **cgivars) {
	char *strptr;
	int x, y, i;
	int keep;

	/* don't strip for now... */
	return;

	for (strptr = cgivars[i=0]; strptr != NULL; strptr = cgivars[++i]) {

		for (x = 0, y = 0; strptr[x] != '\x0'; x++) {

			keep = 1;

			/* remove potentially nasty characters */
			if (strptr[x] == ';' || strptr[x] == '|' || strptr[x] == '&' || strptr[x] == '<' || strptr[x] == '>')
				keep = 0;
#ifdef PARANOID_CGI_INPUT
			else if (strptr[x] == '/' || strptr[x] == '\\')
				keep = 0;
#endif
			if (keep == 1)
				strptr[y++] = strptr[x];
		}

		strptr[y] = '\x0';
	}

	return;
}


/* convert encoded hex string (2 characters representing an 8-bit number) to its ASCII char equivalent */
unsigned char hex_to_char(char *input) {
	unsigned char outchar = '\x0';
	unsigned int outint;
	char tempbuf[3];

	/* NULL or empty string */
	if (input == NULL)
		return '\x0';
	if (input[0] == '\x0')
		return '\x0';

	tempbuf[0] = input[0];
	tempbuf[1] = input[1];
	tempbuf[2] = '\x0';

	sscanf(tempbuf, "%X", &outint);

	/* only convert "normal" ASCII characters - we don't want the rest.  Normally you would
	   convert all characters (i.e. for allowing users to post binary files), but since we
	   aren't doing this, stay on the cautious side of things and reject outsiders... */
#ifdef PARANOID_CGI_INPUT
	if (outint < 32 || outint > 126)
		outint = 0;
#endif

	outchar = (unsigned char)outint;

	return outchar;
}



/* unescape hex characters and plus in CGI input */
void unescape_cgi_input(char *input) {
	int x, y;
	int len;

	if (input == NULL)
		return;

	len = strlen(input);
	for (x = 0, y = 0; x < len; x++, y++) {

		if (input[x] == '\x0') {
			break;

		// RB 2013-04-07
		// only allow hex conversion if '%' is follow by a valid character
		} else if (input[x] == '%' && (
			// 0 - 9
			(input[x+1] >= 48 && input[x+1] <= 57) ||
			// A - F
			(input[x+1] >= 65 && input[x+1] <= 70) ||
			// a - f
			(input[x+1] >= 97 && input[x+1] <= 102))
			) {

			input[y] = hex_to_char(&input[x+1]);
			x += 2;

		// RB 2011-09-08
		// convert plus as well that it can bu used in service and host names
		} else if (input[x] == '+') {
			input[y] = ' ';
		} else
			input[y] = input[x];
	}
	input[y] = '\x0';

	return;
}

html_request *getcgivars(void) {
	char *request_method;
	char *content_type;
	char *content_length_string;
	int content_length;
	char *cgiinput;
	char *nvpair;
	char *eqpos;
	char *temp_pair;
	html_request *new_html_request_list = NULL;
	html_request *new_request_item = NULL;
	html_request *last_request_item = NULL;

	/* initialize char variable(s) */
	cgiinput = "";

	/* depending on the request method, read all CGI input into cgiinput */

	request_method = getenv("REQUEST_METHOD");
	if (request_method == NULL)
		request_method = "";

	if (!strcmp(request_method, "GET") || !strcmp(request_method, "HEAD")) {

		/* check for NULL query string environment variable - 04/28/00 (Ludo Bosmans) */
		if (getenv("QUERY_STRING") == NULL) {
			cgiinput = (char *)malloc(1);
			if (cgiinput != NULL) {
				cgiinput[0] = '\x0';
			}
		} else {
			cgiinput = strdup(getenv("QUERY_STRING"));
			if (cgiinput == NULL) {
				printf("getcgivars(): Could not allocate memory for CGI input.\n");
				exit(1);
			}
		}
	}

	else if (!strcmp(request_method, "POST") || !strcmp(request_method, "PUT")) {

		/* if CONTENT_TYPE variable is not specified, RFC-2068 says we should assume it is "application/octet-string" */
		/* mobile (WAP) stations generate CONTENT_TYPE with charset, we we should only check first 33 chars */

		content_type = getenv("CONTENT_TYPE");
		if (content_type == NULL)
			content_type = "";

		if (strlen(content_type) && strncasecmp(content_type, "application/x-www-form-urlencoded", 33)) {
			printf("getcgivars(): Unsupported Content-Type.\n");
			exit(1);
		}

		content_length_string = getenv("CONTENT_LENGTH");
		if (content_length_string == NULL)
			content_length_string = "0";

		if (!(content_length = atoi(content_length_string))) {
			printf("getcgivars(): No Content-Length was sent with the POST request.\n") ;
			exit(1);
		}
		/* suspicious content length */
		if ((content_length < 0) || (content_length >= INT_MAX - 1)) {
			printf("getcgivars(): Suspicious Content-Length was sent with the POST request.\n");
			exit(1);
		}

		if (!(cgiinput = (char *)malloc(content_length + 1))) {
			printf("getcgivars(): Could not allocate memory for CGI input.\n");
			exit(1);
		}
		if (!fread(cgiinput, content_length, 1, stdin)) {
			printf("getcgivars(): Could not read input from STDIN.\n");
			exit(1);
		}
		cgiinput[content_length] = '\0';

	} else {

		printf("getcgivars(): Unsupported REQUEST_METHOD -> '%s'\n", request_method);
		printf("\n");
		printf("I'm guessing you're trying to execute the CGI from a command line.\n");
		printf("In order to do that, you need to set the REQUEST_METHOD environment\n");
		printf("variable to either \"GET\", \"HEAD\", or \"POST\".  When using the\n");
		printf("GET and HEAD methods, arguments can be passed to the CGI\n");
		printf("by setting the \"QUERY_STRING\" environment variable.  If you're\n");
		printf("using the POST method, data is read from standard input.  Also of\n");
		printf("note: if you've enabled authentication in the CGIs, you must set the\n");
		printf("\"REMOTE_USER\" environment variable to be the name of the user you're\n");
		printf("\"authenticated\" as.\n");
		printf("\n");

		exit(1);
	}

	nvpair = my_strtok(cgiinput, "&");
	while (nvpair) {

		temp_pair = strdup(nvpair);
		if(temp_pair == NULL) {
			printf("getcgivars(): Could not allocate memory for name-value pair element %s.\n", nvpair);
			exit(1);
		}

		/* allocating new memory */
		new_request_item = (html_request *)malloc(sizeof(html_request));
		if(new_request_item == NULL) {
			printf("getcgivars(): Could not allocate memory for new html_request element.\n");
			my_free(temp_pair);
			exit(1);
		}

		new_request_item->option = NULL;
		new_request_item->value = NULL;
		new_request_item->is_valid = FALSE;
		new_request_item->next = NULL;

		/* get value */
		if ((eqpos = strchr(temp_pair, '=')) != NULL) {
			*eqpos = '\0';
			new_request_item->value = strdup(eqpos + 1);
			if(new_request_item->value == NULL) {
				printf("getcgivars(): Could not allocate memory for cgi param value: %s=%s.\n", temp_pair,eqpos + 1);
				exit(1);
			}
			unescape_cgi_input(new_request_item->value);
			/* do some basic length checking */
			if (strlen(new_request_item->value) >= MAX_INPUT_BUFFER - 1) {
				printf("getcgivars(): length of cgi param value exceeds MAX_INPUT_BUFFER: %d.\n", MAX_INPUT_BUFFER);
				exit(1);
			}
		}

		/* get option name
		   just reuse the temp_pair pointer without allocating new memory
		*/
		new_request_item->option = temp_pair;
		unescape_cgi_input(new_request_item->option);
		if (strlen(new_request_item->option) >= MAX_INPUT_BUFFER - 1) {
			printf("getcgivars(): length of cgi param option exceeds MAX_INPUT_BUFFER: %d.\n", MAX_INPUT_BUFFER);
			exit(1);
		}

		if (new_html_request_list == NULL) {
			new_html_request_list = new_request_item;
			new_html_request_list->next = NULL;
			last_request_item = new_html_request_list;
		} else {
			last_request_item->next = new_request_item;
			last_request_item = new_request_item;
			last_request_item->next = NULL;
		}

		nvpair = my_strtok(NULL, "&");
	}

	/* free allocated memory */
	free(cgiinput);

	/* return the list of name-value strings */
	return new_html_request_list;
}

/* free() memory allocated to storing the CGI request data */
void free_html_request(html_request *html_request_list) {
	html_request *this_html_request = NULL;
	html_request *next_html_request = NULL;

	/* free memory for html request list */
	for (this_html_request = html_request_list; this_html_request != NULL; this_html_request = next_html_request) {
		next_html_request = this_html_request->next;
		my_free(this_html_request->option);
		my_free(this_html_request->value);
		my_free(this_html_request);
	}

	html_request_list = NULL;

	return;
}
