# -*- coding: iso-8859-1 -*-
#
#-------------------------------------------------------------------------------
#
#     This file is part of the Code_Saturne User Interface, element of the
#     Code_Saturne CFD tool.
#
#     Copyright (C) 1998-2008 EDF S.A., France
#
#     contact: saturne-support@edf.fr
#
#     The Code_Saturne User Interface is free software; you can redistribute it
#     and/or modify it under the terms of the GNU General Public License
#     as published by the Free Software Foundation; either version 2 of
#     the License, or (at your option) any later version.
#
#     The Code_Saturne User Interface is distributed in the hope that it will be
#     useful, but WITHOUT ANY WARRANTY; without even the implied warranty
#     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
#
#     You should have received a copy of the GNU General Public License
#     along with the Code_Saturne Kernel; if not, write to the
#     Free Software Foundation, Inc.,
#     51 Franklin St, Fifth Floor,
#     Boston, MA  02110-1301  USA
#
#-------------------------------------------------------------------------------

"""
This module defines the 'Meshes and Enveloppe' page.

This module contains the following classes:
- ProfilesView
"""

#-------------------------------------------------------------------------------
# Standard modules
#-------------------------------------------------------------------------------

import string
import logging

#-------------------------------------------------------------------------------
# Third-party modules
#-------------------------------------------------------------------------------

from PyQt4.QtCore import *
from PyQt4.QtGui  import *

#-------------------------------------------------------------------------------
# Application modules import
#-------------------------------------------------------------------------------

from Base.Toolbox import GuiParam
from ProfilesForm import Ui_ProfilesForm
from Base.QtPage import IntValidator, DoubleValidator,  ComboModel
from Pages.ProfilesModel import ProfilesModel

#-------------------------------------------------------------------------------
# log config
#-------------------------------------------------------------------------------

logging.basicConfig()
log = logging.getLogger("ProfilesView")
log.setLevel(GuiParam.DEBUG)

#-------------------------------------------------------------------------------
# StandarItemModel class
#-------------------------------------------------------------------------------

class StandardItemModelProfile(QStandardItemModel):

    def __init__(self):
        """
        """
        QStandardItemModel.__init__(self)
        self.setColumnCount(2)
        self.dataProfile = []


    def data(self, index, role):
        if not index.isValid():
            return QVariant()
        if role == Qt.DisplayRole:
            return QVariant(self.dataProfile[index.row()][index.column()])
        return QVariant()


    def flags(self, index):
        if not index.isValid():
            return Qt.ItemIsEnabled
        else:
            return Qt.ItemIsEnabled | Qt.ItemIsSelectable


    def headerData(self, section, orientation, role):
        if orientation == Qt.Horizontal and role == Qt.DisplayRole:
            if section == 0:
                return QVariant(self.tr("Profile name"))
            elif section == 1:
                return QVariant(self.tr("Variables"))
        return QVariant()


    def setData(self, index, value, role):
        self.emit(SIGNAL("dataChanged(const QModelIndex &, const QModelIndex &)"), index, index)
        return True


    def addItem(self, label, prof):
        """
        Add a row in the table.
        """
        item = [label, prof]
        self.dataProfile.append(item)
        row = self.rowCount()
        self.setRowCount(row+1)


    def replaceItem(self, row, ):
        """
        Replace a row in the table.
        """
        item = []
        self.dataProfile[row] = item


    def deleteRow(self, row):
        """
        Delete the row in the model
        """
        del self.dataProfile[row]
        row = self.rowCount()
        self.setRowCount(row-1)


    def getItem(self, row):
        """
        Returns the name of the mesh file.
        """
        return self.dataProfile[row]


    def getLabel(self, row):
        """
        Returns the name of the mesh file.
        """
        return self.dataProfile[row][0]

#-------------------------------------------------------------------------------
# 
#-------------------------------------------------------------------------------

class DragListView(QListView):
    def __init__(self, parent=None):
        QListView.__init__(self, parent)
        self.setObjectName("DragList")
        self.setAlternatingRowColors(True)
        self.setDragEnabled(True)
        self.setAcceptDrops(True)
        self.setDragDropOverwriteMode(False)
        self.setDragDropMode(QAbstractItemView.DragDrop)


#    def dragEnterEvent(self,event):
#        if event.mimeData().hasFormat("application/x-ics"):
#            if event.source() in self.children():
#                event.setDropAction(Qt.MoveAction)
#                event.accept()
#            else:
#                event.acceptProposedAction()
#
#        elif event.mimeData().hasText():
#            event.acceptProposedAction()
#        else:
#            event.ignore()

#    def dragMoveEvent(self,event):
#        event.ignore()
#
#    def dragLeaveEvent(self,event):
#        event.accept()
#        print "1 drag leave"
#
#    def dropEvent(self,event):
#        event.acceptProposedAction()
#        print "1 drag drop"


#    def mouseMoveEvent(self, event):
#        self.QtBaseClass.mousePressEvent(self, event)
#        if self.enable_drag:
#            if event.buttons() & Qt.LeftButton != Qt.LeftButton:
#                return
#            if (event.pos() - self._drag_start_position).manhattanLength() < \
#                                            QApplication.startDragDistance():
#                return
#            self.start_drag(self._drag_start_position)
#
#
#    def dragEnterEvent(self, event):
#        if event.mimeData().hasFormat("text/uri-list"):
#            event.acceptProposedAction()
#
#    def dragMoveEvent(self, event):
#        event.acceptProposedAction()
#
#
#    def dropEvent(self, event):
#        files = self._get_r_ok_files(event)
#        if files:
#            try:
#                event.setDropAction(Qt.CopyAction)
#                if self.files_dropped(files, event):
#                    event.accept()
#            except Exception, e:
#                Error("There was an error processing the dropped files.", e)
#                raise e


class DropListView(QListView):
    def __init__(self, parent=None):
        QListView.__init__(self, parent)
        self.setObjectName("DragList")
        self.setAlternatingRowColors(True)
        self.setDragEnabled(True)
        self.setAcceptDrops(True)
        self.setDragDropOverwriteMode(False)
        self.setDragDropMode(QAbstractItemView.DragDrop)


#    def dragEnterEvent(self,event):
#        if event.mimeData().hasFormat("application/x-ics"):
#            print "1"
#            if event.source() in self.children():
#                event.setDropAction(Qt.MoveAction)
#                event.accept()
#            else:
#                event.acceptProposedAction()
#
#        elif event.mimeData().hasText():
#            print "2"
#            event.acceptProposedAction()
#        else:
#            print "3"
#            event.ignore()

#    def dragMoveEvent(self,event):
#        event.ignore()
#
#    def dragLeaveEvent(self,event):
#        event.accept()
#        print "2 drag leave"

#    def dropEvent(self,event):
#        if event.mimeData().hasText():
#            text = event.mimeData().text()
#            self.model().append(text)
#            event.acceptProposedAction()
#        else:
#            event.ignore()

#-------------------------------------------------------------------------------
# Main class
#-------------------------------------------------------------------------------

class ProfilesView(QWidget, Ui_ProfilesForm):
    """
    """
    def __init__(self, parent, case, stbar):
        """
        Constructor
        """
        QWidget.__init__(self, parent)

        Ui_ProfilesForm.__init__(self)
        self.setupUi(self)

        self.case = case

        self.var_prof = []

        self.mdl = ProfilesModel(self.case)

        #  Initialize variables concerning the display of the Hlist

        self.entriesNumber = 0

        # Models
        self.modelProfile = StandardItemModelProfile()
        self.treeViewProfile.setModel(self.modelProfile)

        # QListView layout
        self.gridlayout1 = QGridLayout(self.widgetDrag)
        self.gridlayout1.setMargin(0)
        self.DragList = DragListView(self.widgetDrag)
        self.gridlayout1.addWidget(self.DragList,0,0,1,1)

        self.gridlayout2 = QGridLayout(self.widgetDrop)
        self.gridlayout2.setMargin(0)
        self.DropList = DropListView(self.widgetDrop)
        self.gridlayout2.addWidget(self.DropList,0,0,1,1)

        self.modelDrag = QStringListModel()
        self.modelDrop = QStringListModel()
        self.DragList.setModel(self.modelDrag)
        self.DropList.setModel(self.modelDrop)

        # Combo items
        self.modelFreq = ComboModel(self.comboBoxFreq, 2, 1)
        self.modelFreq.addItem(self.tr("at the end of the calculation"), "end")
        self.modelFreq.addItem(self.tr("at each 'n' time steps"), "frequency")

        # Connections
        self.connect(self.treeViewProfile,  SIGNAL("pressed(const QModelIndex &)"), self.selectProfile)

        self.connect(self.pushButtonAdd,    SIGNAL("clicked()"), self.slotAddProfile)
        self.connect(self.pushButtonEdit,   SIGNAL("clicked()"), self.slotEditProfile)
        self.connect(self.pushButtonDelete, SIGNAL("clicked()"), self.slotDeleteProfile)

        self.connect(self.comboBoxFreq,  SIGNAL("activated(const QString&)"), self.getFrequencyType)

        # Validators
        validatorFreq = IntValidator(self.lineEditFreq, min=0)
        validatorFreq.setExclusiveMin(True)
        self.lineEditFreq.setValidator(validatorFreq)

        validatorFloat = DoubleValidator(self.lineEditX1)
        self.lineEditX1.setValidator(validatorFloat)
        self.lineEditY1.setValidator(validatorFloat)
        self.lineEditZ1.setValidator(validatorFloat)
        self.lineEditX2.setValidator(validatorFloat)
        self.lineEditY2.setValidator(validatorFloat)
        self.lineEditZ2.setValidator(validatorFloat)

        #update list of variables, properties, scalars ...
        #self.mdl.updateDicoLabelName()
        liste_label = QStringList()
        for label in self.mdl.dicoLabel2Name.keys():
            liste_label.append(label)
        self.modelDrag.setStringList(liste_label)

        #update list of profiles for view from xml file
        for lab in self.mdl.getProfilesLabelsList():
            self.entriesNumber = self.entriesNumber + 1
            label, list, freq, x1, y1, z1, x2, y2, z2 = self.mdl.getProfileData(lab)
            self.insertProfile(self.entriesNumber, label, list)

        self.eraseEntries()


    def getLabel(self):
        """
        Return label of average.
        """
        label = str(self.lineEditProfileName.text())
        if label in self.mdl.getProfilesLabelsList():
            #windowTitle = "Checking of label"
            default = {}
            default['label'] = label
            default['list'] = self.mdl.getProfilesLabelsList()
            from VerifyExistenceLabelDialogView import VerifyExistenceLabelDialogView
            dialog = VerifyExistenceLabelDialogView(self, default)
            if dialog.exec_():
                result = dialog.get_result() # User clicked 'OK'
                label = result['label']
                if result['label'] == default['label']:
                    label = ""
        return label


    def getFrequencyType(self):
        """
        Input choice for frequency for profile.
        """
        txt = str(self.comboBoxFreq.currentText())
        choice = self.modelFreq.dicoV2M[txt]
        self.freq = choice

        if choice == "end":
            nfreq = -1
            self.lineEditFreq.setText(QString(str(nfreq)))
            self.lineEditFreq.setDisabled(True)

        if choice == "frequency":
            self.lineEditFreq.setEnabled(True)
            nfreq = self.getFrequency()
            self.lineEditFreq.setText(QString(str(nfreq)))

        return choice, nfreq


    def getFrequency(self):
        """
        Input the frequency of the listing output
        """
        nfreq = 1

        if self.freq == "frequency":
            nfreq = int(self.lineEditFreq.text())
            if nfreq > 0: self.nfreq = nfreq

        return nfreq


    def setFrequency(self, nfreq):
        """
        Put frequency' choice and values on view
        """
        if nfreq == -1 or nfreq == None:
            self.modelFreq.setItem(str_model='end')
            self.lineEditFreq.setDisabled(True)
        else:
            self.modelFreq.setItem(str_model='frequency')
            self.lineEditFreq.setEnabled(True)
        self.lineEditFreq.setText(QString(str(nfreq)))


    def profileInfo(self, row):
        """
        Return info from the argument entry.
        """
        label = self.modelProfile.getLabel(row)
        lab, liste, freq, x1, y1, z1, x2, y2, z2 = self.mdl.getProfileData(label)
        return label, liste, freq, x1, y1, z1, x2, y2, z2


    def insertProfile(self, num, label, list):
        """
        Insert values in table view.
        """
        prof = string.join(list,' ; ')

        index = 'item' + str(num)
        self.modelProfile.addItem(label, prof)


    @pyqtSignature("")
    def slotAddProfile(self):
        """
        Set in view label and variables to see on profile
        """
        label_ok = 1
        self.var_prof = [str(s) for s in self.modelDrop.stringList()]
        liste = string.join(self.var_prof, ';')
        if liste == '':
            title = self.tr("Warning")
            msg   = self.tr("You must select at least one variable or property from list")
            QMessageBox.information(self, title, msg)
            label_ok = 0

        choice, freq = self.getFrequencyType()

        self.X1, ok = self.lineEditX1.text().toDouble()
        self.Y1, ok = self.lineEditY1.text().toDouble()
        self.Z1, ok = self.lineEditZ1.text().toDouble()
        self.X2, ok = self.lineEditX2.text().toDouble()
        self.Y2, ok = self.lineEditY2.text().toDouble()
        self.Z2, ok = self.lineEditZ2.text().toDouble()

        if label_ok == 0:
            return
        else:
            label = self.getLabel()
            if label in self.mdl.getProfilesLabelsList():
                msg = self.tr("This label already exists: another is choosen")
		print msg

            self.entriesNumber = self.entriesNumber + 1
            if label == '':
                label = 'profile' + repr(self.entriesNumber)

            self.insertProfile(self.entriesNumber, label, self.var_prof)
            var_prof = string.split(liste,';')

            self.mdl.setProfile(label, var_prof, freq,
                                self.X1, self.Y1, self.Z1,
                                self.X2, self.Y2, self.Z2)

        self.eraseEntries()


    @pyqtSignature("")
    def slotEditProfile(self):
        """
        Edit profile to modify its characteristics.
        """
        try:
            row = self.treeViewProfile.currentIndex().row()
            label, list, freq, x1, y1, z1, x2, y2, z2 = self.profileInfo(row)
        except:
            title = self.tr("Warning")
            msg   = self.tr("To modify, you must select one time average")
            QMessageBox.information(self, title, msg)
            return

        self.setEditedProfile(label, list, freq, x1, y1, z1, x2, y2, z2)


    @pyqtSignature("")
    def slotDeleteProfile(self):
        """
        Delete the profile from the list (one by one).
        """

        #self.selectProfile()
        ln = len(self.mdl.getProfilesLabelsList())

        try:
            row = self.treeViewProfile.currentIndex().row()
            label, list, freq, x1, y1, z1, x2, y2, z2 = self.profileInfo(row)
        except:
            title = self.tr("Warning")
            msg   = self.tr("You must select at least one variable or property from list")
            QMessageBox.information(self, title, msg)
            return

        self.modelProfile.deleteRow(row)
        self.mdl.deleteProfile(label)

        self.eraseEntries()


    def setEditedProfile(self, old_label, old_list, old_freq, old_x1, old_y1,
                         old_z1, old_x2, old_y2, old_z2):
        """
        Modify the selected moment from the list.
        """
        iok = 1
        new_label = self.getLabel()
        if new_label in self.mdl.getProfilesLabelsList() and new_label != old_label:
            msg = self.tr("This label already exists: the old label is kept")
            print msg
            new_label = ''

        if new_label == '':
            new_label = old_label

        if len(self.var_prof) == 0 or self.var_prof[0] == '':
            title = self.tr("Warning")
            msg   = self.tr("You must select at least one variable or property from list")
            QMessageBox.information(self, title, msg)
            iok = 0

        choice, new_freq = self.getFrequencyType()

        if iok == 0:
            return
        else:
            new_prof = self.var_prof
            self.replaceProfile(new_label, new_prof)

            self.X1, ok = self.lineEditX1.text().toDouble()
            self.Y1, ok = self.lineEditY1.text().toDouble()
            self.Z1, ok = self.lineEditZ1.text().toDouble()
            self.X2, ok = self.lineEditX2.text().toDouble()
            self.Y2, ok = self.lineEditY2.text().toDouble()
            self.Z2, ok = self.lineEditZ2.text().toDouble()

            self.mdl.replaceProfile(old_label, new_label, new_prof, self.nfreq,
                                    self.X1, self.Y1, self.Z1,
                                    self.X2, self.Y2, self.Z2)


    def replaceProfile(self, label, prof):
        """
        Replace values on the selected moment.
        """
        # TODO
        prof = string.join(prof,';')


    def selectProfile(self):
        """
        Return the selected item from the list.
        """
        log.debug("selectProfile")
        #row = self.treeViewProfile.currentIndex().row()
        selectionModel = self.treeViewProfile.selectionModel()
        for index in selectionModel.selectedIndexes():
            row = index.row()

##             self.lineEditProfileName.setText(QString(str("")))
##             self.modelFreq.setItem(str_model='end')
##             self.lineEditX1.setText(QString(str("")))
##             self.lineEditY1.setText(QString(str("")))
##             self.lineEditZ1.setText(QString(str("")))
##             self.lineEditX2.setText(QString(str("")))
##             self.lineEditY2.setText(QString(str("")))
##             self.lineEditZ2.setText(QString(str("")))

            label, liste, freq, x1, y1, z1, x2, y2, z2 = self.profileInfo(row)

            self.lineEditProfileName.setText(QString(str(label)))

            if freq >= 1:
                self.modelFreq.setItem(str_model='frequency')
                self.lineEditFreq.setEnabled(True)
                self.lineEditFreq.setText(QString(str(freq)))
            else:
                self.modelFreq.setItem(str_model='end')
                self.lineEditFreq.setText(QString(str("-1")))
                self.lineEditFreq.setDisabled(True)

            self.lineEditX1.setText(QString(str(x1)))
            self.lineEditY1.setText(QString(str(y1)))
            self.lineEditZ1.setText(QString(str(z1)))
            self.lineEditX2.setText(QString(str(x2)))
            self.lineEditY2.setText(QString(str(y2)))
            self.lineEditZ2.setText(QString(str(z2)))

            self.modelDrop.setStringList(QStringList())
            liste = [QString(s) for s in liste]
            self.modelDrop.setStringList(liste)


    def cancelSelection(self):
        """
        Fonction for the Popup Menu.
        Blank the selection.
        """
        self.eraseEntries()


    def eraseEntries(self):
        """
        Delete all caracters in the entries.
        """
        self.lineEditProfileName.setText(QString(str("")))
        self.lineEditX1.setText(QString(str("")))
        self.lineEditY1.setText(QString(str("")))
        self.lineEditZ1.setText(QString(str("")))
        self.lineEditX2.setText(QString(str("")))
        self.lineEditY2.setText(QString(str("")))
        self.lineEditZ2.setText(QString(str("")))

        self.modelFreq.setItem(str_model='end')
        self.lineEditFreq.setText(QString(str("-1")))
        self.lineEditFreq.setDisabled(True)

        self.var_prof = []
        self.modelDrop.setStringList(QStringList())


    def tr(self, text):
        """
        Translation
        """
        return text


#-------------------------------------------------------------------------------
# Testing part
#-------------------------------------------------------------------------------


if __name__ == "__main__":
    pass


#-------------------------------------------------------------------------------
# End
#-------------------------------------------------------------------------------
