/*
#   mp3dec.c: decodes mp3 file format for xlplayer
#   Copyright (C) 2007 Stephen Fairchild
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "../config.h"
#ifdef HAVE_MAD

#include <stdio.h>
#include <string.h>
#include <jack/jack.h>
#include <mad.h>
#include "xlplayer.h"
#include "mp3dec.h"

#define TRUE 1
#define FALSE 0
#define ACCEPTED 1
#define REJECTED 0

#define BSIZ 16384
#define MAD_SCALE ((float)(1L << MAD_F_SCALEBITS))

static void skip_id3v2_tag(FILE *fp)
   {
   long start = ftell(fp);
   long tagsize;
   
   if (fgetc(fp) == 'I' && fgetc(fp) == 'D' && fgetc(fp) == '3') 	/* check for ID3 signature */
      {
      fgetc(fp); fgetc(fp); fgetc(fp);	/* skip tag version number and flags */
      tagsize = fgetc(fp) & 0x7F;	/* 28 bits of tag size info packed into 4 bytes - big endian */
      tagsize <<= 7;			/* most significant bit discarded - should be zero */
      tagsize |= fgetc(fp) & 0x7F;
      tagsize <<= 7;
      tagsize |= fgetc(fp) & 0x7F;
      tagsize <<= 7;
      tagsize |= fgetc(fp) & 0x7F;
      fseek(fp, tagsize, SEEK_CUR);	/* skip over the tag - note the relative jump */
      }
   else
      fseek(fp, start, SEEK_SET);	/* not ID3 so restore the file pointer */
   }

static inline float scale(mad_fixed_t sample)
   {
   return (float)sample / MAD_SCALE;
   }

static int mp3decode_get_frame(struct xlplayer *xlplayer)
   {
   struct mp3decode_vars *self = xlplayer->dec_data;
   size_t nb;
   
   if (mad_frame_decode(&(self->frame), &(self->stream)) != 0)
      {
      if (self->stream.next_frame)
         {
         nb = self->read_buffer + self->bytes_in_buffer - self->stream.next_frame;
         memmove(self->read_buffer, self->stream.next_frame, nb);
         self->bytes_in_buffer = nb;
         }
      switch (self->stream.error)
         {
         case MAD_ERROR_BUFLEN:
            self->bytes_in_buffer += (nb = fread(self->read_buffer + self->bytes_in_buffer, 1, BSIZ - self->bytes_in_buffer, self->fp));
            if (nb == 0 || ferror(self->fp))
               {
               return 0;
               break;
               }
            mad_stream_buffer(&(self->stream), self->read_buffer, self->bytes_in_buffer);
            break;
         default:
            mad_stream_buffer(&(self->stream), self->read_buffer, self->bytes_in_buffer);
         }
      return -1;
      }
   else
      return 1;
   }

static void mp3decode_eject(struct xlplayer *xlplayer)
   {
   struct mp3decode_vars *self = xlplayer->dec_data;
 
   if (self->resample)
      {
      if (xlplayer->src_data.data_in)
         free(xlplayer->src_data.data_in);
      if (xlplayer->src_data.data_out)
         free(xlplayer->src_data.data_out);
      xlplayer->src_state = src_delete(xlplayer->src_state);
      }
   mad_synth_finish(&(self->synth));
   mad_stream_finish(&(self->stream));
   mad_frame_finish(&(self->frame));
   fclose(self->fp);
   free(self->read_buffer);
   free(self);
   }

static void mp3decode_init(struct xlplayer *xlplayer)
   {
   struct mp3decode_vars *self = xlplayer->dec_data;
   off_t start, end, offset;
   int retcode, dump;
   int src_error;

   if (xlplayer->seek_s)	/* seek over the file by means of estimation - sometimes accurate other times not */
      {
      start = (float)ftell(self->fp);
      fseek(self->fp, 0, SEEK_END);
      end = (float)ftell(self->fp);
      offset = (end - start) * (float)xlplayer->seek_s / (float)xlplayer->size + start;
      fseek(self->fp, (long)offset, SEEK_SET);
      
      self->bytes_in_buffer = fread(self->read_buffer, 1, BSIZ, self->fp);
      if (self->bytes_in_buffer == 0 || ferror(self->fp))
         {
         fprintf(stderr, "mp3decode_init: seeked to end of input file\n");
         mp3decode_eject(xlplayer);
         xlplayer->playmode = PM_STOPPED;
         xlplayer->command = CMD_COMPLETE;
         return;
         }
      mad_stream_buffer(&(self->stream), self->read_buffer, self->bytes_in_buffer);
      for (dump = 0; dump < 2; dump++)
         {
         while ((retcode = mp3decode_get_frame(xlplayer)) < 0);
         if (retcode == 0)
            {
            mp3decode_eject(xlplayer);
            xlplayer->playmode = PM_STOPPED;
            xlplayer->command = CMD_COMPLETE;
            return;
            }
         }
      }
   mad_synth_frame(&(self->synth), &(self->frame));
   self->nchannels = self->synth.pcm.channels;
   self->samplerate = self->synth.pcm.samplerate;
   if ((self->resample = self->samplerate != xlplayer->samplerate))
      {
      fprintf(stderr, "Configuring resampler\n");
      xlplayer->src_data.output_frames = 0;
      xlplayer->src_data.data_in = NULL;
      xlplayer->src_data.data_out = NULL;
      xlplayer->src_data.src_ratio = (double)xlplayer->samplerate / (double)self->samplerate;
      xlplayer->src_data.end_of_input = 0;
      xlplayer->src_state = src_new(xlplayer->rsqual, self->nchannels, &src_error);
      if (src_error)
         {
         fprintf(stderr, "mp3decode_init: %s src_new reports - %s\n", xlplayer->playername, src_strerror(src_error));
         self->resample = 0;
         mp3decode_eject(xlplayer);
         xlplayer->playmode = PM_STOPPED;
         xlplayer->command = CMD_COMPLETE;
         }
      }
   }
   
static void mp3decode_play(struct xlplayer *xlplayer)
   {
   struct mp3decode_vars *self = xlplayer->dec_data;
   struct mad_pcm *pcm;
   jack_default_audio_sample_t *lp, *rp, *dp, gain;
   mad_fixed_t const *left_ch, *right_ch;
   int nchannels, nsamples, frame_code;
   SRC_DATA *src_data = &(xlplayer->src_data);
   
   pcm = &(self->synth.pcm);
   left_ch = pcm->samples[0];
   if ((nchannels = pcm->channels) == 2)
      right_ch = pcm->samples[1];
   else
      right_ch = NULL;
   
   if (self->resample)
      {
      src_data->end_of_input = (pcm->length == 0);
      src_data->input_frames = pcm->length;
      src_data->data_in = dp = realloc(src_data->data_in, pcm->length * nchannels * sizeof (float));
      src_data->output_frames = (int)(src_data->input_frames * src_data->src_ratio) + 2 + (512 * src_data->end_of_input);
      src_data->data_out = realloc(src_data->data_out, src_data->output_frames * nchannels * sizeof (float));
      for (nsamples = pcm->length; nsamples; nsamples--)
         {
         *dp++ = scale(*left_ch++);
         if (nchannels == 2)
            *dp++ = scale(*right_ch++);
         }
      if (src_process(xlplayer->src_state, src_data))
         {
         fprintf(stderr, "mp3decode_play: error occured during resampling\n");
         xlplayer->playmode = PM_EJECTING;
         return;
         }
      xlplayer_demux_channel_data(xlplayer, src_data->data_out, src_data->output_frames_gen, pcm->channels);
      }
   else
      {
      xlplayer->op_buffersize = (nsamples = pcm->length) * sizeof (float);
      if (!(xlplayer->leftbuffer = lp = realloc(xlplayer->leftbuffer, xlplayer->op_buffersize)) && xlplayer->op_buffersize)
         {
         fprintf(stderr, "mp3decode_play: malloc failure\n");
         exit(5);
         }
      if (!(xlplayer->rightbuffer = rp = realloc(xlplayer->rightbuffer, xlplayer->op_buffersize)) && xlplayer->op_buffersize)
         {
         fprintf(stderr, "mp3decode_play: malloc failure\n");
         exit(5);
         }
      while (nsamples--)
         {
         gain = xlplayer_get_next_gain(xlplayer);
         *lp++ = gain * scale(*left_ch++);
         if (nchannels == 2)
            *rp++ = gain * scale(*right_ch++);
         }
      if (nchannels == 1)
         memcpy(xlplayer->rightbuffer, xlplayer->leftbuffer, xlplayer->op_buffersize);
      }
   xlplayer_write_channel_data(xlplayer);
   while((frame_code = mp3decode_get_frame(xlplayer)) == -1);
   if(frame_code)
      mad_synth_frame(&(self->synth), &(self->frame));
   else
      xlplayer->playmode = PM_EJECTING;
   }

int mp3decode_reg(struct xlplayer *xlplayer)
   {
   struct mp3decode_vars *self;
   long start;

   if (!(self = xlplayer->dec_data = malloc(sizeof (struct mp3decode_vars))))
      {
      fprintf(stderr, "mp3decode_vars: malloc failure\n");
      return REJECTED;
      }
   if (!(self->fp = fopen(xlplayer->pathname, "r")))
      {
      fprintf(stderr, "mp3decode_test_mp3ness: failed to open file\n");
      free(self);
      return 0;
      }
   skip_id3v2_tag(self->fp);
   start = ftell(self->fp);
   if (!(self->read_buffer = malloc(BSIZ)))
      {
      fprintf(stderr, "mp3decode_test_mp3ness: malloc failure\n");
      fclose(self->fp);
      free(self);
      return 0;
      }
   self->bytes_in_buffer = fread(self->read_buffer, 1, BSIZ, self->fp);
   if (self->bytes_in_buffer < 8192)
      {
      fprintf(stderr, "mp3decode_test_mp3ness: file too small\n");
      fclose(self->fp);
      free(self->read_buffer);
      free(self);
      return 0;
      }
   mad_synth_init(&(self->synth));
   mad_stream_init(&(self->stream));
   self->stream.options = MAD_OPTION_IGNORECRC;
   mad_frame_init(&(self->frame));
   mad_stream_buffer(&(self->stream), self->read_buffer, self->bytes_in_buffer);
   self->playduration = 0.0F;
   while (1)
      {
      switch (mp3decode_get_frame(xlplayer))
         {
         case -1:
            if (ftell(self->fp) - start < 32768L)
               continue;
         case 0:
            mp3decode_eject(xlplayer);
            return REJECTED;
         default:
            xlplayer->dec_init = mp3decode_init;
            xlplayer->dec_play = mp3decode_play;
            xlplayer->dec_eject = mp3decode_eject;
            return ACCEPTED;
         }
      }
   }

#endif /* HAVE_MAD */
