/*
#   mp4tag.c: reads/writes mp4 metadata and gets playback time
#   Copyright (C) 2006 Stephen Fairchild
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "../config.h"
#ifdef HAVE_MP4FF

#define USE_TAGGING
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <mp4ff.h>
#include "mp4tag.h"

static FILE *fp;

static uint32_t cbmp4_read(void *user_data, void *buffer, uint32_t length)
   {
   return fread(buffer, 1, length, fp);
   }
   
static uint32_t cbmp4_write(void *user_data, void *buffer, uint32_t length)
   {
   return fwrite(buffer, 1, length, fp);
   }
   
static uint32_t cbmp4_seek(void *user_data, uint64_t position)
   {
   return fseek(fp, position, SEEK_SET);
   }

static uint32_t cbmp4_truncate(void *user_data)
   {
   off_t offset;
   /* not 100% sure what to do here but this seems to make sense */
   /* using unix style low level file objects would have made this a little less messy */
   if ((offset = ftello(fp)) >= 0)
      {
      fclose(fp);
      truncate(user_data, offset);
      fp = fopen(user_data, "a");
      return 0;		/* nothing in the header file states what return values to use :( */
      }
   else
      return -1;
   }

static mp4ff_callback_t cbstruct =
   {
   cbmp4_read,
   cbmp4_write,
   cbmp4_seek,
   cbmp4_truncate,
   NULL
   };

int mp4_file_info(char *pathname)
   {
   mp4ff_t handle;
   char *artist = NULL;
   char *title = NULL;
   int32_t timescale;
   int64_t trackduration;
   int32_t track;
   int32_t totaltracks;
   
   cbstruct.user_data = pathname;
   
   if (!(fp = fopen(pathname, "r")))
      {
      printf("idjcmixer: mp4fileinfo Not Valid\n");
      fflush(stdout);
      return 0;
      }
   if (!(handle = mp4ff_open_read(&cbstruct)))
      {
      printf("idjcmixer: mp4fileinfo Not Valid\n");
      fflush(stdout);
      fclose(fp);
      return 0;
      }
   totaltracks = mp4ff_total_tracks(handle);
   for (track = 0; track < totaltracks && mp4ff_get_audio_type(handle, track) != 64; track++);
   if (track == totaltracks)
      {
      printf("idjcmixer: mp4fileinfo Not Valid\n");
      fflush(stdout);
      mp4ff_close(handle);
      fclose(fp);
      return 0;
      }
   if (mp4ff_meta_get_artist(handle, &artist))
      {
      printf("idjcmixer: mp4fileinfo artist=%s\n", artist);
      free(artist);
      }
   if (mp4ff_meta_get_title(handle, &title))
      {
      printf("idjcmixer: mp4fileinfo title=%s\n", title);
      free(title);
      }
   trackduration = mp4ff_get_track_duration(handle, track);
   timescale = mp4ff_time_scale(handle, track);
   mp4ff_close(handle);
   fclose(fp);
   printf("idjcmixer: mp4fileinfo length=%d\n"
          "idjcmixer: mp4fileinfo end\n",
          (int)(trackduration/timescale));
   fflush(stdout);
   return 1;
   }

int mp4_tag_read(char *pathname)
   {
   mp4ff_t handle;
   int n_items;
   char *item = NULL;
   char *value = NULL;
 
   cbstruct.user_data = pathname;

   if (!(fp = fopen(pathname, "r")))
      {
      fprintf(stderr, "idjcmixer: mp4fileinfo Not Valid\n");
      return 0;
      }
   if (!(handle = mp4ff_open_read(&cbstruct)))
      {
      fprintf(stderr, "idjcmixer: mp4fileinfo Not Valid\n");
      fclose(fp);
      return 0;
      }
   n_items = mp4ff_meta_get_num_items(handle);
   while(n_items--)
      {
      mp4ff_meta_get_by_index(handle, n_items, &item, &value);
      if(strcmp(item, "coverart"))
         printf("idjcmixer: mp4tagread %s=%s\n", item, value);
      }
   printf("idjcmixer: mp4tagread end\n");
   fflush(stdout);
   if (item)
      free(item);
   if (value)
      free(value);
   return 1;
   }

int mp4_tag_write(char *pathname, char *taglist)
   {
   mp4ff_tag_t *tags = NULL, dummy;
   mp4ff_metadata_t metadata;
   int kvpcount;
   int linelength, equals;
   char *ptr, *endptr;
   
   cbstruct.user_data = pathname;
   
   if (!(fp = fopen(pathname, "r+")))
      {
      fprintf(stderr, "couldn't open file for tagging\n");
      return 0;
      }
      
   /* count the number of key/value pairs (actually just the number of newline characters) */
   for (kvpcount = 0, ptr = taglist; *ptr; ptr++)
      if (*ptr == '\n')
         kvpcount++;
   endptr = ptr;
   
   if(kvpcount)
      {
      /* allocate an array of mp4ff_tag_t to contain key/value data */
      if(!(tags = malloc(kvpcount * sizeof (mp4ff_tag_t))))
         {
         fprintf(stderr, "malloc failure\n");
         exit(5);
         }
      metadata.tags = tags;
      metadata.count = kvpcount;
      
      /* assign the pointers in the array to the appropriate parts of the string (taglist) */
      /* null termination is applied at the appropriate places */
      for(ptr = taglist; ptr != endptr; ptr += linelength + 1, tags++)
         {
         for (linelength = 0, equals = 0; ptr[linelength] != '\n'; linelength++)
            if (equals == 0 && ptr[linelength] == '=')
               equals = linelength;
         tags->item = ptr;
         tags->value = ptr + equals + 1;
         ptr[equals] = '\0';
         ptr[linelength] = '\0';
         }
      tags = metadata.tags;
      }
   else
      {
      metadata.tags = &dummy;	/* don't know how to erase all tags without causing a crash */
      metadata.count = 1;	/* using one empty pair works well though */
      dummy.item = "";
      dummy.value = "";
      }
   
   /* write out the tags */
   if(mp4ff_meta_update(&cbstruct, &metadata))
      fprintf(stderr, "file metadata updated successfully\n");
   else
      fprintf(stderr, "problem writing metadata\n");
   
   if (kvpcount)
      free(tags);
 
   fclose(fp);
   return 1;
   }
#endif
