/*
#   dbconvert.c: fast table based conversion for db to sig level and vice-versa from IDJC.
#   Copyright (C) 2005 Stephen Fairchild
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "dbconvert.h"

#define TRUE 1
#define FALSE 0

/* Comment this out to avoid using the lookup table */
#define USING_LOOKUP

#ifdef USING_LOOKUP

static float *dblookup;
static float *signallookup;

int init_dblookup_table()
   {
   int i;
   /* build a decibel lookup table to save on cpu usage */
   if (!(dblookup = malloc(sizeof (float) * 32768)))
      {
      fprintf(stderr, "Failed to allocate space for signal to db lookup table\n");
      return FALSE;
      }
   else
      {
      for (i = 0 ; i < 32768 ; i++)
	 dblookup[i] = log10f((i+1) / 32768.0F) * 20.0F;
      }
   return TRUE;
   }
      
int init_signallookup_table()
   {
   int i;
   /* the opposite of the decibel lookup table */
   if (!(signallookup = malloc(sizeof (float) * 32768)))
      {
      fprintf(stderr, "Failed to allocate space for db to signal table\n");
      return FALSE;
      }
   else
      {
      for (i=0; i < 32768; i++)
         signallookup[i] = powf(10.0F, (float)i / 5120.0F);
      }
   return TRUE;
   }
   
void free_dblookup_table()
   {
   free(dblookup);
   }
   
void free_signallookup_table()
   {
   free(signallookup);
   }

/* a table based db lookup function - considerably faster than using the maths co-processor */
inline float level2db(float signal)
   {
   int index;
    
   if (signal > 1.0F)
      return ((index = (int)(32768.005F / signal) - 1) >= 0) ? -dblookup[index] : -90.3089987F;
   else
      return ((index = (int)(signal * 32768.005F) - 1) >= 0) ? dblookup[index] : -90.3089987F;
   }
   
/* table based level lookup function taking a db level as input */
inline float db2level(float signal)
   {
   int index;
      
   if (signal < 0.0F)
      return ((index = signal * (-256.0F)) < 32768) ? 1.0F / signallookup[index] : 1.0F / signallookup[32767];
   else
      return ((index = signal * 256.0F) < 32768) ? signallookup[index] : signallookup[32767];
   }

#else

/* These cause the maths co-processor to be used */
int init_dblookup_table()
   {
   return TRUE;
   }
   
int init_signallookup_table()
   {
   return TRUE;
   }
   
void free_dblookup_table() {};
   
void free_signallookup_table() {};

/* the more accurate but more cpu intensive method */
float level2db(float signal)
   {
   return (float)(log10f(signal) * 20.0F);
   }
      
float db2level(float signal)
   {
   return powf(10.0F, signal * 0.05F);
   }
   
#endif
