/*
#   xlplayer.h: a xine-lib based player module for idjc
#   Copyright (C) 2006 Stephen Fairchild
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <jack/jack.h>
#include <jack/ringbuffer.h>
#include <pthread.h>
#include <stdlib.h>
#include <samplerate.h>
#include <FLAC/all.h>
#include <sndfile.h>

enum command_t {XL_COMPLETE, XL_PLAY, XL_EJECT, XL_CLEANUP, XL_THREADEXIT, XL_PLAYMANY};

enum playmode_t {XL_STOPPED, XL_INITIATE, XL_PLAYING, XL_EJECTING };

enum playertype_t {XL_XINE, XL_VORBIS, XL_SNDFILE, XL_FLAC, XL_BADFILE};

struct xlplayer
   {
   jack_ringbuffer_t *left_ch;		/* main playback buffer */
   jack_ringbuffer_t *right_ch;
   jack_ringbuffer_t *left_fade;	/* buffers used for fade - swapped with above when needed */
   jack_ringbuffer_t *right_fade;
   size_t rbsize;			/* the size of the jack ringbuffers in bytes */
   size_t op_buffersize;		/* the current size of the player output buffers */
   char *pathname;			/* the pathname of the music file being played */
   char **playlist;			/* the playlist as an array of pointers */
   int loop;				/* flag indicating if we loop or come to a stop */
   int seek_s;				/* the initial seek time of the song in seconds */
   int playlistmode;			/* set when we are using a local playlist */
   int playlistindex;			/* current track number we are playing */
   int playlistsize;			/* the number of tracks in the playlist */
   jack_default_audio_sample_t *leftbuffer;	/* the output buffers */
   jack_default_audio_sample_t *rightbuffer;
   int fadeout_f;			/* flag indicated if fade is applied upon stopping */
   int jack_flush;			/* tells the jack callback to flush the ringbuffers */
   int jack_is_flushed;			/* indicates true when jack callback has done the flush */
   int samplerate;			/* the audio sample rate in use by the jack transport */
   jack_nframes_t fadeindex;		/* running tally of where we are up to re. the fadeout */
   int pause;				/* flag controlling the player paused state */
   int write_deferred;			/* suppress further generation of audio data */
   u_int64_t samples_written;		/* number of samples written to the ringbuffer */
   int32_t play_progress_ms;		/* the playback progress in milliseconds */
   int32_t fadein_index;		/* used for fadein when seeking */
   char *playername;			/* identifies this particular instance -- for error messages */
   enum playmode_t playmode;		/* indicates the player mode */
   enum command_t command;		/* the command mode */
   int test2;
   int have_swapped_buffers_f;		/* controls buffer swapping for fadeout */
   size_t avail;			/* the number of samples available in the ringbuffer */
   int have_data_f;			/* indicates the presence of audio data */
   int current_audio_context;		/* bumps when started, bumps when stopped. Odd=playing */
   int initial_audio_context;		/* return code placeholder variable for above */
   int dither;				/* flag, whether to add dither to player output */
   unsigned int seed;			/* used for dither */
   pthread_t thread;
   u_int32_t sleep_samples;		/* used to count off when it is appropriate to call sleep */
   SRC_STATE *src_state;		/* give flac access to the resampler */
   SRC_DATA *src_data;
   FLAC__uint64 totalsamples;		/* used to determine the last frame */
   float *flbuf;
   int noflush;				/* suppresses ringbuffer flushes */
   int suppress_flac_output;		/* used to suppress flac output so preventing odd seek noise */
   int *jack_shutdown_f;		/* inidcator that jack has shut down */
   int watchdog_timer;			/* this will run away if the main loop has frozen */
   SNDFILE *sndfile;			/* file handle from libsndfile for handling wav type files */
   SF_INFO sf_info;			/* points to data about current libsndfile frame */
   int up;				/* set to true when the player is fully initialised */
   };

/* xlplayer_create: create an instance of the player */
struct xlplayer *xlplayer_create(int samplerate, size_t ringbuffersize, char *playername, int *shutdown_f);
/* xlplayer_destroy: the opposite of xlplayer_create */
void xlplayer_destroy(struct xlplayer *);

/* xlplayer_play: starts the player on a particular track immediately
* if a track is currently playing eject is called
* return value: a context-id for this track */
int xlplayer_play(struct xlplayer *self, char *pathname, int seek_s);

/* xlplayer_playmany: starts the player on a playlist
* if a track is currently playing eject is called, also can set looping with this function
* return value: a context-id for this playlist */
int xlplayer_playmany(struct xlplayer *self, char *playlist, int loop_f);

/* xlplayer_play_noflush: starts the player without flushing out old data from the ringbuffer */
int xlplayer_play_noflush(struct xlplayer *self, char *pathname, int seek_s);

/* xlplayer_cancelplaynext: cancels the automatic playing of the next track 
* the current track is allowed to continue playing */
void xlplayer_cancelplaynext(struct xlplayer *self);

/* xlplayer_pause: pauses the current track */
void xlplayer_pause(struct xlplayer *self);
/* xlplayer_unpause: unpause the current track */
void xlplayer_unpause(struct xlplayer *self);
/* xlplayer_dither: turns on/off dither on players */
void xlplayer_dither(struct xlplayer *self, int dither_f);
/* xlplayer_eject: stops the current track with a fadeout unless the track is paused */
/* this call will also cancel any track cued with playnext command */
/* to suppress fadeout call pause beforehand */
void xlplayer_eject(struct xlplayer *self);

/* read_from_player: reads out the audio data from the buffers */
/* this is meant to be run inside the jack callback */
size_t read_from_player(struct xlplayer *self, jack_default_audio_sample_t *left_buf, jack_default_audio_sample_t *right_buf, jack_default_audio_sample_t *left_fbuf, jack_default_audio_sample_t *right_fbuf, jack_nframes_t nframes);

/* xlplayer_create_fadein_lookup: creates the lookup table used for song fadein */
/* must be called before starting any of the players */
int xlplayer_create_fadein_lookup(float size_seconds, int samplerate);

/* xlplayer_destroy_fadein_lookup: frees the lookup table used for fading */
void xlplayer_destroy_fadein_lookup(void);
