/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "idialogimagecomposer.h"

#include "ivtk.h"
#include "ivtkwindow.h"
#include "imath.h"
#include "iqtdefs.h"
#include "ienvironment.h"

#include <vtkImageData.h>

#include <qapplication.h>
#include <qcanvas.h>
#include <qcheckbox.h>
#include <qcolordialog.h>
#include <qcombobox.h>
#include <qfiledialog.h>
#include <qframe.h>
#include <qimage.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qpushbutton.h>
#include <qspinbox.h>
#include <qtabwidget.h>


QPixmap image(char *);

#define VIEW ((ViewWidget *)View)

//
//  Helper classes
//
namespace iDialogImageComposer_Private
{

const int win_rtti = 1001;

class Window : public QCanvasRectangle
{

	friend class ViewWidget;
	friend class ::iDialogImageComposer;

protected:

	Window(int tx, int ty, iVTK *v, QCanvas *c) : QCanvasRectangle(c)
	{
		tileX = tx;
		tileY = ty;
		font.setBold(true);
		borderWidth = drawnBorderWidth = 0;
		borderColor = black;
		if(v != 0)
		{
			// foreground window
			this->setZ(zMax);
			zMax += 1.0;
			type = 1;
			borderWidth = drawnBorderWidth = 1;
		}
		else if(tx>=0 && ty>=0)
		{
			// background window
			this->setZ(0.0);
			type = 0;
		}
		else
		{
			// image window
			this->setZ(-1.0);
			type = -1;
		}
		shadow = 0;
		em = false;
		scale = 50;
		imagePosX = imagePosY = 0;
		posQuantum = 10;
		canvas = c;
		vtk = v;
		this->setPen(QPen(Qt::NoPen));
		basicBrush = QBrush(white);
	}

	virtual int rtti () const 
	{
		return win_rtti;
	}

	bool nonNull()
	{
		return (vtk!=0 || !wallpaperImage.isNull());
	}

	int getImageWidth()
	{
		if(vtk != 0) return vtk->getThisImageWidth(); else if(!wallpaperImage.isNull()) return wallpaperImage.width(); else return 0;
	}

	int getImageHeight()
	{
		if(vtk != 0) return vtk->getThisImageHeight(); else if(!wallpaperImage.isNull()) return wallpaperImage.height(); else return 0;
	}

	void setBasicBrush(QBrush b)
	{
		basicBrush = b;
	}

	virtual void update()
	{
		if(vtk != 0) this->setBrush(QBrush(vtk->getBackgroundColor())); else this->setBrush(basicBrush);
		QCanvasRectangle::update();
	}

	void emphasize(bool s)
	{
		em = s;
//		if(s) shadow = 3; else shadow = 0;
		this->update();
	}

	void setFactor(float fac)
	{
		if(type > 0)
		{
			drawnImagePosX = fac*imagePosX;
			drawnImagePosY = fac*imagePosY;
			drawnPosQuantum = round(fac*posQuantum);
			if(drawnPosQuantum < 1) drawnPosQuantum = 1;
			if(vtk != 0) this->setSize(round(0.01*scale*fac*this->getImageWidth())+2*drawnBorderWidth,round(0.01*scale*fac*this->getImageHeight())+2*drawnBorderWidth);
		}
	}

	virtual void drawShape(QPainter &p)
	{
		const int gray_s1 = 100;
		const int gray_s2 = 125;
		const int gray_s3 = 150;
		const int gray_l3 = 175;
		const int gray_l2 = 200;
		const int gray_l1 = 225;
		static QColor light1 = QColor(gray_l1,gray_l1,gray_l1);
		static QColor light2 = QColor(gray_l2,gray_l2,gray_l2);
		static QColor light3 = QColor(gray_l3,gray_l3,gray_l3);
		static QColor shadow1 = QColor(gray_s1,gray_s1,gray_s1);
		static QColor shadow2 = QColor(gray_s2,gray_s2,gray_s2);
		static QColor shadow3 = QColor(gray_s3,gray_s3,gray_s3);
		int x1, y1, x2, y2;
		x1 = (int)x() + shadow;
		y1 = (int)y() + shadow;
		int w = width() - 2*shadow;
		int h = height() - 2*shadow;

		QCanvasRectangle::drawShape(p);
		if(drawnBorderWidth > 0)
		{
			x2 = x1 + w - drawnBorderWidth;
			y2 = y1 + h - drawnBorderWidth;
			if(drawnBorderWidth > 1)
			{
				QBrush b = QBrush(borderColor);
				p.fillRect(x1,y1,w,drawnBorderWidth,b);
				p.fillRect(x1,y2,w,drawnBorderWidth,b);
				p.fillRect(x1,y1,drawnBorderWidth,h,b);
				p.fillRect(x2,y1,drawnBorderWidth,h,b);
			}
			else
			{
				p.setPen(QPen(borderColor,1));
				p.drawLine(x1,y1,x2,y1);
				p.drawLine(x2,y1,x2,y2);
				p.drawLine(x2,y2,x1,y2);
				p.drawLine(x1,y2,x1,y1);
			}
		}
		if(vtk != 0) 
		{
			int n = 1 + vtk->getWindowNumber();
			if(n == 0) n = 1;
			QString s = QString::number(n);
			float f1 = (this->height()-2*drawnBorderWidth)*0.75;
			float f2 = 1.3*(this->width()-2*drawnBorderWidth)/s.length();
			font.setPixelSize(round((f1>f2)?f2:f1));
			p.setFont(font);
			QColor c = vtk->getBackgroundColor();
			p.setPen(QPen(QColor(255-c.red(),255-c.green(),255-c.blue()),0));
			p.drawText(this->boundingRect(),Qt::AlignCenter,s);
		} 
		else if(!wallpaperImage.isNull())
		{
			int wi = w - 2*drawnBorderWidth;
			int hi = h - 2*drawnBorderWidth;
			if(wi!=drawnWallpaperImage.width() || hi!=drawnWallpaperImage.height())
			{
				drawnWallpaperImage.convertFromImage(wallpaperImage.smoothScale(wi,hi,QImage::ScaleMin));
			}
			p.drawPixmap(QPoint(x1+drawnBorderWidth,y1+drawnBorderWidth),drawnWallpaperImage);
		}
		if(shadow > 0)
		{
			const int off = 0;
			x1 = (int)x() + off;
			y1 = (int)y() + off;
			x2 = x1 + width() - 1 - 2*off;
			y2 = y1 + height() - 1 - 2*off;
			p.setPen(QPen(light1,1));
			p.drawLine(x1,y1,x2,y1);
			p.drawLine(x1,y1,x1,y2);
			p.setPen(QPen(light2,1));
			p.drawLine(x1+1,y1+1,x2,y1+1);
			p.drawLine(x1+1,y1+1,x1+1,y2);
			p.setPen(QPen(light3,1));
			p.drawLine(x1+2,y1+2,x2,y1+2);
			p.drawLine(x1+2,y1+2,x1+2,y2);
			p.setPen(QPen(shadow3,1));
			p.drawLine(x2-2,y2-2,x2-2,y1+2);
			p.drawLine(x2-2,y2-2,x1+2,y2-2);
			p.setPen(QPen(shadow2,1));
			p.drawLine(x2-1,y2-1,x2-1,y1+1);
			p.drawLine(x2-1,y2-1,x1+1,y2-1);
			p.setPen(QPen(shadow1,1));
			p.drawLine(x2,y2,x2,y1);
			p.drawLine(x2,y2,x1,y2);
		}
		if(em) 
		{
			p.drawWinFocusRect(x1,y1,w,h);//,brush().color());
			p.drawWinFocusRect(x1+1,y1+1,w-2,h-2);//,brush().color());
			p.drawWinFocusRect(x1+2,y1+2,w-4,h-4);//,brush().color());
		}

	}
	
	iVTK *vtk;
	int type;
	QCanvas *canvas;
	int tileX, tileY;
	QFont font;
	int scale, shadow;
	QColor borderColor;
	int borderWidth, drawnBorderWidth;
	int imagePosX, imagePosY, drawnImagePosX, drawnImagePosY;
	int posQuantum, drawnPosQuantum;
	static double zMax;
	QBrush basicBrush;
	bool em;
	QImage wallpaperImage;
	QPixmap drawnWallpaperImage;
	iString wallpaperImageFile;

};


double Window::zMax = 1.0;


class ViewWidget : public QCanvasView, public iObject
{

public:

	ViewWidget(QWidget *parent = 0, const char *name = 0, WFlags f = 0) : QCanvasView(parent,name,f), iObject(0)
	{
		this->setCanvas(new QCanvas(this));
		canvas()->setBackgroundColor(gray);
		canvas()->resize(100,100);
		numTilesX = numTilesY = 1;
		fac = 0.25;
		buf = 0.75;
		borderWidth = drawnBorderWidth = 0;
		borderColor = black;
		innerBorder = false;
		fullWidth = tileWidth = 640;
		fullHeight = tileHeight = 480;
		drawnFullWidth = drawnTileWidth = 160;
		drawnFullHeight = drawnTileHeight = 120;
		image = new Window(-1,-1,0,canvas());
		image->show();
		curBackground = new Window(0,0,0,canvas());
		curBackground->emphasize(true);
		curBackground->show();
		curForeground = 0;
		emBG = emFG = false;
		scaleBackgroundImages = blockImageUpdates = false;
		label = new QCanvasText("NOT ACTIVE",canvas());
		label->setColor(red);
		label->setTextFlags(Qt::AlignCenter);
		label->setZ(1.0e35);
		font.setBold(true);
		font.setFixedPitch(true);
	}

	int getFullWidth(){ return fullWidth; }
	int getFullHeight(){ return fullHeight; }
	int getNumTilesX(){ return numTilesX; }
	int getNumTilesY(){ return numTilesY; }
	bool getInnerBorder(){ return innerBorder; }
	int getBorderWidth(){ return borderWidth; }
	QColor getBorderColor(){ return borderColor; }

	void setNumTiles(int nx, int ny);
	void setInnerBorder(bool s);
	void setBorderWidth(int s); 
	void setBorderColor(QColor c);
	void setForegroundWindowBorderWidth(Window *w, int s); 
	void setForegroundWindowBorderColor(Window *w, QColor c);
	void setForegroundWindowScale(Window *w, int s);

	void emphasize(bool bg);

	iVTK* getBackgroundVTK(int tx, int ty);
	void setBackgroundVTK(int tx, int ty, iVTK* v);
	void setBackgroundWallpaperImage(int tx, int ty, QImage im, iString fn);
	void setCurrentBackgroundWindow(int tx, int ty);

	Window* getCurrentForegroundWindow(){ return curForeground; }
	Window* getForegroundWindow(iVTK *v);
	void createForegroundWindow(iVTK *v);
	void deleteForegroundWindow(Window *w);
	void deleteCurrentForegroundWindow();

	void updateWindowList(bool imageResized);

	virtual void resizeEvent(QResizeEvent *e);

	virtual void packState(iString &s);
	virtual void unpackState(iString s);

	int compose(iVTK *vtk, vtkImageData *data);

protected:

	void updateImage(bool scaleForegrounds);
    void contentsMousePressEvent(QMouseEvent*);
    void contentsMouseReleaseEvent(QMouseEvent*);
    void contentsMouseMoveEvent(QMouseEvent*);

private:

	Window *image;
	bool innerBorder;
	QColor borderColor;
	int borderWidth, drawnBorderWidth;
	int numTilesX, numTilesY;
	int fullWidth, fullHeight;
	int tileWidth, tileHeight;
	int drawnFullWidth, drawnFullHeight;
	int drawnTileWidth, drawnTileHeight;
	float fac, buf;
    Window* moving;
    QPoint moving_start;
	Window *curForeground, *curBackground;
	bool emBG, emFG, scaleBackgroundImages, blockImageUpdates;
	QCanvasText *label;
	QFont font;

};


void ViewWidget::setNumTiles(int nx, int ny)
{ 
	int i, j;
	Window *w;

	if(nx <= 0) nx = numTilesX;
	if(ny <= 0) ny = numTilesY;

	if(nx==numTilesX && ny==numTilesY) return;

	QCanvasItemList list = canvas()->allItems();
	for(i=0; i<(int)list.count(); i++) if(list[i]->rtti() == win_rtti)
	{
		w = (Window *)list[i];
		if(w->type == 0)
		{
			//  background
			if(w->tileX>=nx || w->tileY>=ny) delete w; 
		}
	}

	for(j=numTilesX; j<nx; j++)
	{
		for(i=0; i<numTilesY; i++)
		{
			w = new Window(j,i,0,canvas());
			w->show();
		}
	}
	numTilesX = nx; 

	for(j=numTilesY; j<ny; j++)
	{
		for(i=0; i<numTilesX; i++)
		{
			w = new Window(i,j,0,canvas());
			w->show();
		}
	}
	numTilesY = ny; 

	updateImage(false); 

	QResizeEvent re(this->size(),this->size());
	QApplication::sendEvent(this,&re);

}


void ViewWidget::setInnerBorder(bool s)
{
	if(s != innerBorder)
	{
		innerBorder = s;
		this->updateImage(true);
	}
}


void ViewWidget::setBorderWidth(int s) 
{ 
	if(s >= 0) 
	{ 
		borderWidth = s; 
		drawnBorderWidth = round(fac*borderWidth); 
		if(borderWidth>0 && drawnBorderWidth==0) drawnBorderWidth = 1;
		this->updateImage(true); 
	} 
}


void ViewWidget::setBorderColor(QColor c) 
{ 
	if(c.isValid()) 
	{ 
		borderColor = c; 
		this->updateImage(false); 
	} 
}


void ViewWidget::setForegroundWindowBorderWidth(Window *w, int s) 
{ 
	if(s>=0 && w!=0 && w->type>0 && s<0.4*w->width() && s<0.4*w->height()) 
	{ 
		w->borderWidth = s; 
		w->drawnBorderWidth = round(fac*w->borderWidth); 
		if(w->borderWidth>0 && w->drawnBorderWidth==0) w->drawnBorderWidth = 1;
		w->setFactor(fac);
		w->update();
		moving = curForeground;
		this->contentsMouseReleaseEvent(0);
	} 
}


void ViewWidget::setForegroundWindowBorderColor(Window *w, QColor c) 
{ 
	if(c.isValid() && w!=0 && w->type>0) 
	{ 
		w->borderColor = c; 
		w->update();
		canvas()->update();
	} 
}


void ViewWidget::setForegroundWindowScale(Window *w, int s) 
{ 
	if(s>0 && s<=100 && w!=0 && w->type>0) 
	{ 
		w->scale = s; 
		w->setFactor(fac);
		w->update();
		moving = curForeground;
		this->contentsMouseReleaseEvent(0);
	} 
}


void ViewWidget::emphasize(bool bg)
{
	emBG = bg;
	emFG = !bg;
	if(curBackground != 0) curBackground->emphasize(bg);
	if(curForeground != 0) curForeground->emphasize(!bg);
	canvas()->update();
}


iVTK* ViewWidget::getBackgroundVTK(int tx, int ty)
{
	int i;
	Window *w;

	if(tx>=0 && tx<numTilesX && ty>=0 && ty<numTilesY)
	{
		QCanvasItemList list = canvas()->allItems();
		for(i=0; i<(int)list.count(); i++) if(list[i]->rtti() == win_rtti)
		{
			w = (Window *)list[i];
			if(w->type == 0)
			{
				//  background
				if(w->tileX==tx && w->tileY==ty) return w->vtk;
			}
		}
	}
	return 0;
}


void ViewWidget::setBackgroundVTK(int tx, int ty, iVTK* vtk)
{
	unsigned int i;
	Window *w;

	if(tx>=0 && tx<numTilesX && ty>=0 && ty<numTilesY)
	{
		QCanvasItemList list = canvas()->allItems();
		for(i=0; i<list.count(); i++) if(list[i]->rtti() == win_rtti)
		{
			w = (Window *)list[i];
			if(w->type == 0)
			{
			//  background
				if(w->tileX==tx && w->tileY==ty) 
				{
					w->vtk = vtk; 
					w->update();
					break;
				}
			}
		}
		this->updateImage(true);
	}
}


void ViewWidget::setBackgroundWallpaperImage(int tx, int ty, QImage im, iString fn)
{
	unsigned int i;
	Window *w;

	if(tx>=0 && tx<numTilesX && ty>=0 && ty<numTilesY)
	{
		QCanvasItemList list = canvas()->allItems();
		for(i=0; i<list.count(); i++) if(list[i]->rtti() == win_rtti)
		{
			w = (Window *)list[i];
			if(w->type == 0)
			{
			//  background
				if(w->tileX==tx && w->tileY==ty) 
				{
					w->wallpaperImage = im; 
					w->wallpaperImageFile = fn;
					w->setFactor(fac);
					w->update();
					break;
				}
			}
		}
		this->updateImage(true);
	}
}


void ViewWidget::setCurrentBackgroundWindow(int tx, int ty)
{
	int i;
	Window *w;

	if(tx>=0 && tx<numTilesX && ty>=0 && ty<numTilesY)
	{
		QCanvasItemList list = canvas()->allItems();
		for(i=0; i<(int)list.count(); i++) if(list[i]->rtti() == win_rtti)
		{
			w = (Window *)list[i];
			if(w->type == 0)
			{
				//  background
				if(w->tileX==tx && w->tileY==ty) 
				{
					curBackground->emphasize(false);
					curBackground = w;
					if(emBG) curBackground->emphasize(true);
					canvas()->update();
					return;
				}
			}
		}
	}
}


Window* ViewWidget::getForegroundWindow(iVTK *vtk)
{
	int i;
	Window *w;

	if(vtk != 0)
	{
		QCanvasItemList list = canvas()->allItems();
		for(i=0; i<(int)list.count(); i++) if(list[i]->rtti() == win_rtti)
		{
			w = (Window *)list[i];
			if(w->type > 0)
			{
			//  foreground
				if(w->vtk == vtk) return w;
			}
		}
	}
	return 0;

}


void ViewWidget::createForegroundWindow(iVTK *vtk)
{
	if(vtk != 0)
	{
		Window *w = new Window(-1,-1,vtk,canvas());
		w->move(image->x(),image->y());
		w->setFactor(fac);
		w->update();
		w->show();
//		curForeground = w;
		this->updateImage(false);
		emit contentsMoving(0,0); // use the only available signal to indicate the change of the current VTK
	}
}


void ViewWidget::deleteCurrentForegroundWindow()
{
	this->deleteForegroundWindow(curForeground);
}


void ViewWidget::deleteForegroundWindow(Window *w0)
{
	int i;
	Window *w, *wmax = 0;

	if(w0 != 0)
	{
		if(w0 == curForeground)
		{
			QCanvasItemList list = canvas()->allItems();
			double zmax = 0.0;
			for(i=0; i<(int)list.count(); i++) if(list[i]->rtti() == win_rtti)
			{
				w = (Window *)list[i];
				if(w!=w0 && w->type>0 && w->z()>zmax)
				{
					zmax = w->z();
					wmax = w;
				}
			}
			if(wmax != 0) curForeground = wmax; else curForeground = 0;
		}
		delete w0;
		this->updateImage(false);
		emit contentsMoving(0,0);  // use the only available signal to indicate the change of the current VTK
	}
}


void ViewWidget::contentsMousePressEvent(QMouseEvent* e)
{
	Window *w;
	unsigned int i;

    QPoint p = inverseWorldMatrix().map(e->pos());
    QCanvasItemList list = canvas()->collisions(p);
	for(i=0; i<list.count(); i++) if(list[i]->rtti() == win_rtti)
	{
		w = (Window *)list[i];
		if(w->type > 0)
		{
			if(w->boundingRect().contains(p))
			{
				w->setZ(Window::zMax); Window::zMax += 1.0;
				if(curForeground != 0) curForeground->emphasize(false);
				curForeground = w;
				if(emFG) curForeground->emphasize(true);
				emit contentsMoving(0,0); // use the only available signal to indicate the change of the current VTK
				moving = w;
				moving_start = p;
				canvas()->update();
				return;
 			}
		}
	}
    moving = 0;
}


void ViewWidget::contentsMouseReleaseEvent(QMouseEvent *)
{
	if(moving!=0 && moving->vtk!=0)
	{
		int sid;

		if(moving->boundingRect().right() >= image->boundingRect().right())
		{
			moving->imagePosX = fullWidth - 0.01*moving->scale*moving->getImageWidth() - 2*moving->borderWidth;
			moving->drawnImagePosX = image->width() - moving->width();
		}
		else
		{
			moving->imagePosX = moving->posQuantum*round((moving->x()-image->x())/fac/moving->posQuantum);
			sid = round(0.01*moving->scale*moving->getImageWidth()+2*moving->borderWidth);
			if(moving->imagePosX+sid >= fullWidth) moving->imagePosX = fullWidth - sid;
			if(moving->imagePosX < 0) moving->imagePosX = 0;
			moving->drawnImagePosX = fac*moving->imagePosX;
		}
		
		if(moving->boundingRect().bottom() >= image->boundingRect().bottom())
		{
			moving->imagePosY = fullHeight - 0.01*moving->scale*moving->getImageHeight() - 2*moving->borderWidth;
			moving->drawnImagePosY = image->height() - moving->height();
		}
		else
		{
			moving->imagePosY = moving->posQuantum*round((moving->y()-image->y())/fac/moving->posQuantum);
			sid = round(0.01*moving->scale*moving->getImageHeight()+2*moving->borderWidth);
			if(moving->imagePosY+sid >= fullHeight) moving->imagePosY = fullHeight - sid;
			if(moving->imagePosY < 0) moving->imagePosY = 0;
			moving->drawnImagePosY = fac*moving->imagePosY;
		}

		moving->move(image->x()+moving->drawnImagePosX,image->y()+moving->drawnImagePosY);
		canvas()->update();
		moving = 0;
	}
}


void ViewWidget::contentsMouseMoveEvent(QMouseEvent* e)
{
    if(moving != 0) 
	{
		QPoint p = inverseWorldMatrix().map(e->pos());
		double dx = p.x() - moving_start.x();
		double dy = p.y() - moving_start.y();
		moving->moveBy(dx,dy);
		if(!image->boundingRect().contains(moving->boundingRect())) moving->moveBy(-dx,-dy);
		moving_start = p;
		canvas()->update();
    }
}


void ViewWidget::updateImage(bool scaleForegrounds)
{
	Window *w;
	unsigned int i;
	bool work = false;

	if(blockImageUpdates) return;

	int oldFullWidth = fullWidth;
	int oldFullHeight = fullHeight;
	//
	//  Compute real size
	//
	tileWidth = tileHeight = 0;
	QCanvasItemList list = canvas()->allItems();
	for(i=0; i<list.count(); i++) if(list[i]->rtti() == win_rtti)
	{
		w = (Window *)list[i];
		if(w->type==0 && w->nonNull())
		{
			work = true;
			//  background
			if(tileWidth < w->getImageWidth()) tileWidth = w->getImageWidth();
			if(tileHeight < w->getImageHeight()) tileHeight = w->getImageHeight();
		}
	}
	if(tileWidth == 0) tileWidth = 640;
	if(tileHeight == 0) tileHeight = 480;

	fullWidth = numTilesX*tileWidth + 2*borderWidth;
	fullHeight = numTilesY*tileHeight + 2*borderWidth;
	if(innerBorder)
	{
		fullWidth += (numTilesX-1)*borderWidth;
		fullHeight += (numTilesY-1)*borderWidth;
	}
	//
	//  Compute reduction factor
	//
	float f1 = buf*canvas()->width()/fullWidth; 
	float f2 = buf*canvas()->height()/fullHeight; 
	fac = (f1 > f2) ? f2 : f1;
	if(fac > 1.0) fac = 1.0;
	//
	//  Compute reduces size
	//
	drawnTileWidth = round(fac*tileWidth);
	drawnTileHeight = round(fac*tileHeight);
	drawnFullWidth = numTilesX*drawnTileWidth + 2*drawnBorderWidth;
	drawnFullHeight = numTilesY*drawnTileHeight + 2*drawnBorderWidth;
	if(innerBorder)
	{
		drawnFullWidth += drawnBorderWidth*(numTilesX-1);
		drawnFullHeight += drawnBorderWidth*(numTilesY-1);
	}
	//	
	//  update full image
	//
	image->setSize(drawnFullWidth,drawnFullHeight);
	image->setBasicBrush(QBrush(borderColor));

	double x = (canvas()->width()-image->width())/2;
	double y = (canvas()->height()-image->height())/2;
	image->move(x,y);
	image->update();
	//
	//  update background images
	//
	for(i=0; i<list.count(); i++) if(list[i]->rtti() == win_rtti)
	{
		w = (Window *)list[i];
		if(w->type == 0)
		{
			//  background
			x = image->x() + drawnBorderWidth + w->tileX*drawnTileWidth;
			y = image->y() + drawnBorderWidth + w->tileY*drawnTileHeight;
			if(innerBorder)
			{
				x += w->tileX*drawnBorderWidth;
				y += w->tileY*drawnBorderWidth;
			}
			w->setSize(drawnTileWidth,drawnTileHeight);
			w->move(x,y);
			w->update();
		}
	}
	//
	//  update foreground images
	//
	for(i=0; i<list.count(); i++) if(list[i]->rtti() == win_rtti)
	{
		w = (Window *)list[i];
		if(w->type > 0)	
		{
			work = true;
			if(scaleForegrounds)
			{
				w->imagePosX = (fullWidth*w->imagePosX)/oldFullWidth;
				w->imagePosY = (fullHeight*w->imagePosY)/oldFullHeight;
			}
			w->setFactor(fac);
			x = image->x() + w->drawnImagePosX;
			y = image->y() + w->drawnImagePosY;
			w->move(x,y);
			if(w->boundingRect().left() < image->boundingRect().left()) w->setX(image->x());
			if(w->boundingRect().right() > image->boundingRect().right()) w->setX(image->x()+image->width()-w->width());
			if(w->boundingRect().top() < image->boundingRect().top()) w->setY(image->y());
			if(w->boundingRect().bottom() > image->boundingRect().bottom()) w->setY(image->y()+image->height()-w->height());
			w->update();
		}
	}
	//
	//  Inactive label
	//
	if(work) label->hide(); else 
	{
		float f1 = canvas()->height()*0.75;
		float f2 = 1.3*canvas()->width()/10; //label->text().length();
		font.setPixelSize(round((f1>f2)?f2:f1));
		label->setFont(font);
		label->move(canvas()->width()/2,canvas()->height()/2);
		label->show();
	}
	//
	//  redraw the canvas
	//
	canvas()->update();
	
}


void ViewWidget::updateWindowList(bool imageResized)
{
	Window *w;
	iVTK *vtk;
	unsigned int i;
	int k;
	bool present;

	QCanvasItemList list = canvas()->allItems();
	for(i=0; i<list.count(); i++) if(list[i]->rtti() == win_rtti)
	{
		w = (Window *)list[i];
		vtk = w->vtk;
		present = false;
		for(k=0; !present && k<=iVTKWindow::getMaxWindowIndex(); k++)
		{
			if(vtk == iVTKWindow::getWindow(k)) present = true;
		}
		if(!present) 
		{
			if(w->type > 0)
			{
				delete w; // foreground
			}
			else
			{
				w->vtk = 0;
			}
		}
	}
	this->updateImage(imageResized);
}


void ViewWidget::resizeEvent(QResizeEvent *e)
{
	int cw = e->size().width()-4;
	int ch = e->size().height()-4;
	canvas()->resize(cw,ch);
	this->updateImage(false);
	QCanvasView::resizeEvent(e);
}


void ViewWidget::packState(iString &s)
{
	Window *w;
	unsigned int i;
	int j;

	this->packValue(s,"iDialogImageComposer::innerBorder",innerBorder);
	this->packValue(s,"iDialogImageComposer::borderColor",borderColor);
	this->packValue(s,"iDialogImageComposer::borderWidth",borderWidth);
	this->packValue(s,"iDialogImageComposer::numTilesX",numTilesX);
	this->packValue(s,"iDialogImageComposer::numTilesY",numTilesY);

	int nbg = 0;
	QCanvasItemList list = canvas()->allItems();
	for(i=0; i<list.count(); i++) if(list[i]->rtti() == win_rtti)
	{
		w = (Window *)list[i];
		if(w->type==0 && w->nonNull())  // background windows
		{
			nbg++;
		}
	}
	this->packValue(s,"iDialogImageComposer::numBG",nbg);

	if(nbg > 0)
	{
		int *tx = new int[nbg];
		int *ty = new int[nbg];
		int *vtk = new int[nbg];
		iString *fn = new iString[nbg];
		j = 0;
		for(i=0; i<list.count(); i++) if(list[i]->rtti() == win_rtti)
		{
			w = (Window *)list[i];
			if(w->type==0 && w->nonNull())  // background windows
			{
				tx[j] = w->tileX;
				ty[j] = w->tileY;
				if(w->vtk != 0) vtk[j] = w->vtk->getWindowNumber(); else 
				{
					vtk[j] = -1;
					if(!w->wallpaperImage.isNull()) fn[j] = w->wallpaperImageFile;
				}
				j++;
			}
		}

		this->packValue(s,"iDialogImageComposer::bgTileX",tx,nbg);
		this->packValue(s,"iDialogImageComposer::bgTileY",ty,nbg);
		this->packValue(s,"iDialogImageComposer::bgVTK",vtk,nbg);
		this->packValue(s,"iDialogImageComposer::bgImage",fn,nbg);

		delete [] tx;
		delete [] ty;
		delete [] vtk;
		delete [] fn;

	}

	int nfg = 0;
	for(i=0; i<list.count(); i++) if(list[i]->rtti() == win_rtti)
	{
		w = (Window *)list[i];
		if(w->type>0)  // foreground windows
		{
			nfg++;
		}
	}
	this->packValue(s,"iDialogImageComposer::numFG",nfg);

	if(nfg > 0)
	{
		int *x = new int[nfg];
		int *y = new int[nfg];
		int *sc = new int[nfg];
		int *bw = new int[nfg];
		iColor *bc = new iColor[nfg];
		int *vtk = new int[nfg];
		j = 0;
		for(i=0; i<list.count(); i++) if(list[i]->rtti() == win_rtti)
		{
			w = (Window *)list[i];
			if(w->type>0)  // foreground windows
			{
				x[j] = w->imagePosX;
				y[j] = w->imagePosY;
				sc[j] = w->scale;
				bw[j] = w->borderWidth;
				if(w->vtk != 0) vtk[j] = w->vtk->getWindowNumber(); else 
				{
					vtk[j] = -1;
				}
				bc[j] = w->borderColor;
				j++;
			}
		}

		this->packValue(s,"iDialogImageComposer::fgX",x,nfg);
		this->packValue(s,"iDialogImageComposer::fgY",y,nfg);
		this->packValue(s,"iDialogImageComposer::fgS",sc,nfg);
		this->packValue(s,"iDialogImageComposer::fgBW",bw,nfg);
		this->packValue(s,"iDialogImageComposer::fgBC",bc,nfg);
		this->packValue(s,"iDialogImageComposer::fgVTK",vtk,nfg);

		delete [] x;
		delete [] y;
		delete [] sc;
		delete [] bw;
		delete [] bc;
		delete [] vtk;

	}

}


void ViewWidget::unpackState(iString s)
{
	Window *w;
	int nbg = 0, nfg = 0;
	int i; bool b; iColor c;

	blockImageUpdates = true;

	if(this->unpackValue(s,"iDialogImageComposer::innerBorder",b)) this->setInnerBorder(b);
	if(this->unpackValue(s,"iDialogImageComposer::borderColor",c)) this->setBorderColor(c);
	if(this->unpackValue(s,"iDialogImageComposer::borderWidth",i)) this->setBorderWidth(i);
	if(this->unpackValue(s,"iDialogImageComposer::numTilesX",i)) this->setNumTiles(i,numTilesY);
	if(this->unpackValue(s,"iDialogImageComposer::numTilesY",i)) this->setNumTiles(numTilesX,i);

	if(this->unpackValue(s,"iDialogImageComposer::numBG",i)) nbg = i;
	if(nbg > 0)
	{
		int *tx = new int[nbg];
		int *ty = new int[nbg];
		int *vtk = new int[nbg];
		iString *fn = new iString[nbg];

		bool ok = true;
		ok &= this->unpackValue(s,"iDialogImageComposer::bgTileX",tx,nbg);
		ok &= this->unpackValue(s,"iDialogImageComposer::bgTileY",ty,nbg);
		ok &= this->unpackValue(s,"iDialogImageComposer::bgVTK",vtk,nbg);
		ok &= this->unpackValue(s,"iDialogImageComposer::bgImage",fn,nbg);

		if(ok)
		{
			for(i=0; i<nbg; i++)
			{
				if(vtk[i] > -1)	this->setBackgroundVTK(tx[i],ty[i],iVTKWindow::getWindow(vtk[i])); else
				{
					this->setBackgroundWallpaperImage(tx[i],ty[i],iVTK::loadImageFromFile(fn[i]),fn[i]);
				}
			}
		}

		delete [] tx;
		delete [] ty;
		delete [] vtk;
		delete [] fn;

	}

	if(this->unpackValue(s,"iDialogImageComposer::numFG",i)) nfg = i;
	if(nfg > 0)
	{
		int *x = new int[nfg];
		int *y = new int[nfg];
		int *sc = new int[nfg];
		int *bw = new int[nfg];
		iColor *bc = new iColor[nfg];
		int *vtk = new int[nfg];

		bool ok = true;
		ok &= this->unpackValue(s,"iDialogImageComposer::fgX",x,nfg);
		ok &= this->unpackValue(s,"iDialogImageComposer::fgY",y,nfg);
		ok &= this->unpackValue(s,"iDialogImageComposer::fgS",sc,nfg);
		ok &= this->unpackValue(s,"iDialogImageComposer::fgBW",bw,nfg);
		ok &= this->unpackValue(s,"iDialogImageComposer::fgBC",bc,nfg);
		ok &= this->unpackValue(s,"iDialogImageComposer::fgVTK",vtk,nfg);

		if(ok)
		{
			for(i=0; i<nfg; i++) if(iVTKWindow::getWindow(vtk[i]) != 0)
			{
				if(vtk[i] > -1) 
				{
					w = new Window(-1,-1,iVTKWindow::getWindow(vtk[i]),canvas());
				}
				else
				{
					w = new Window(-1,-1,iVTKWindow::getCurrentWindow(),canvas());
				}
				if(w != 0)
				{
					w->imagePosX = x[i];
					w->imagePosY = y[i];
					w->scale = sc[i];
					w->borderWidth = bw[i];
					w->borderColor = bc[i];
//					w->setFactor(fac);
//					w->move(image->x()+w->drawnImagePosX,image->y()+w->drawnImagePosY);
//					w->update();
					w->show();
				}
			}
		}

		delete [] x;
		delete [] y;
		delete [] sc;
		delete [] bw;
		delete [] bc;
		delete [] vtk;

	}

	blockImageUpdates = false;

	this->updateImage(false);
	canvas()->update();
	emit contentsMoving(0,0); // use the only available signal to indicate the change of the current VTK

}


int ViewWidget::compose(iVTK *vtk, vtkImageData *data)
{
	Window *w;
	static QImage qtImage;
	char rgb[3];
	unsigned int l;
	int ret, i, j, k, w1, h1, w0, h0, xoff, yoff, ioff, joff, dims[3];
	char *dPtr, *dPtr1, *wPtr, *wPtr1;
	bool work = false;

	if(data==0 || vtk==0) return -1;

	//
	//  We received a request to compose the image and put the data into data.
	//  This function uses a combination of QT and VTK methods for image manipulation
	//  for better quality and efficiency.
	//
	//  First, make sure that ViewWidget is up to date
	//
	this->updateWindowList(false);

	vtkImageData *winData, *outData = vtkImageData::New();
	outData->SetDimensions(fullWidth,fullHeight,1);
	outData->SetScalarType(VTK_UNSIGNED_CHAR);
	outData->SetNumberOfScalarComponents(3);
	outData->SetOrigin(0.0,0.0,0.0);
	outData->SetSpacing(1.0,1.0,1.0);
	outData->AllocateScalars();

	dPtr = (char *)outData->GetScalarPointer();
	QCanvasItemList list = canvas()->allItems();

	for(j=0; j<fullHeight; j++)  // fill with white
	{
		for(i=0; i<fullWidth; i++)
		{
			dPtr1 = dPtr + 3*(i+fullWidth*j);
			dPtr1[0] = dPtr1[1] = dPtr1[2] = 255;
		}
	}

	if(borderWidth > 0)
	{
		//
		//  Create the border
		//
		work = true;
		rgb[0] = borderColor.red();
		rgb[1] = borderColor.green();
		rgb[2] = borderColor.blue();
		//
		//  Vertical lines
		//
		for(j=0; j<fullHeight; j++)  // sort of a waste, but it is much simpler this way
		{
			for(i=0; i<borderWidth; i++)
			{
				dPtr1 = dPtr + 3*(i+fullWidth*j);
				for(k=0; k<3; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
			}
			for(i=fullWidth-borderWidth; i<fullWidth; i++)
			{
				dPtr1 = dPtr + 3*(i+fullWidth*j);
				for(k=0; k<3; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
			}
			
			if(innerBorder)
			{
				for(k=1; k<numTilesX; k++)
				{
					xoff = k*(borderWidth+tileWidth);
					for(i=xoff; i<xoff+borderWidth; i++)
					{
						dPtr1 = dPtr + 3*(i+fullWidth*j);
						for(k=0; k<3; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
					}

				}
			}
		}
		//
		//  Horizontal lines
		//
		for(j=0; j<borderWidth; j++)  // sort of a waste, but it is much simpler this way
		{
			for(i=0; i<fullWidth; i++)
			{
				dPtr1 = dPtr + 3*(i+fullWidth*j);
				for(k=0; k<3; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
			}
		}

		for(j=fullHeight-borderWidth; j<fullHeight; j++)
		{
			for(i=0; i<fullWidth; i++)
			{
				dPtr1 = dPtr + 3*(i+fullWidth*j);
				for(k=0; k<3; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
			}
		}
		
		if(innerBorder)
		{
			for(k=1; k<numTilesY; k++)
			{
				xoff = k*(borderWidth+tileHeight);
				for(j=xoff; j<xoff+borderWidth; j++)
				{
					for(i=0; i<fullWidth; i++)
					{
						dPtr1 = dPtr + 3*(i+fullWidth*j);
						for(k=0; k<3; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
					}
				}
			}
		}

	}

	ret = 0;

	//
	//  Background windows 
	//
	for(l=0; l<list.count(); l++) if(list[l]->rtti() == win_rtti)
	{
		w = (Window *)list[l];
		if(w->type==0 && w->nonNull())  
		{
			work = true;
			ioff = borderWidth + tileWidth*w->tileX;
			joff = borderWidth + tileHeight*(numTilesY-1-w->tileY); //  Y tile index must be inverted due to difference in canvas vs vtkImage coordinate systems
			if(innerBorder)
			{
				ioff += borderWidth*w->tileX;
				joff += borderWidth*(numTilesY-1-w->tileY);
			}
			//
			// fill the tile with the background color
			//
			if(w->vtk != 0)
			{
				rgb[0] = w->vtk->getBackgroundColor().red();
				rgb[1] = w->vtk->getBackgroundColor().green();
				rgb[2] = w->vtk->getBackgroundColor().blue();
				for(j=0; j<tileHeight; j++)  // sort of a waste, but it is much simpler this way
				{
					for(i=0; i<tileWidth; i++)
					{
						dPtr1 = dPtr + 3*(i+ioff+fullWidth*(j+joff));
						for(k=0; k<3; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
					}
				}
				//
				//  Render the image
				//
				ret = w->vtk->renderImageData(winData);
				if(ret != 0) 
				{
					data->DeepCopy(outData);
					outData->Delete();
					return ret;
				}
				//
				//  Check that dimensions are ok.
				//
				winData->GetDimensions(dims);
				if(scaleBackgroundImages && (dims[0]!=tileWidth || dims[1]!=tileHeight))
				{
					//
					//  Current implementation is to use QT's smoothScale function
					//
					iVTK::convertImage(winData,qtImage);
					iVTK::convertImage(qtImage.smoothScale(tileWidth,tileHeight,QImage::ScaleMin),winData);
					winData->GetDimensions(dims);
				}
			}
			else if(!w->wallpaperImage.isNull())
			{
				winData = vtkImageData::New();
				iVTK::convertImage(w->wallpaperImage,winData);
				winData->GetDimensions(dims);
			}

			wPtr = (char *)winData->GetScalarPointer();
			xoff = ioff + (tileWidth-dims[0])/2;
			yoff = joff + (tileHeight-dims[1])/2; 
			for(j=0; j<dims[1]; j++)  // sort of a waste, but it is much simpler this way
			{
				for(i=0; i<dims[0]; i++)
				{
					dPtr1 = dPtr + 3*(i+xoff+fullWidth*(j+yoff));
					wPtr1 = wPtr + 3*(i+dims[0]*j);
					for(k=0; k<3; k++) dPtr1[k] = wPtr1[k];
				}
			}

			if(w->vtk==0 && !w->wallpaperImage.isNull()) winData->Delete();

		}
	}

	//
	//  Foreground windows 
	//
	for(l=0; l<list.count(); l++) if(list[l]->rtti() == win_rtti)
	{
		w = (Window *)list[l];
		if(w->type>0 && w->nonNull())  
		{
			work = true;
			w0 = floor(0.01*w->scale*w->getImageWidth());
			h0 = floor(0.01*w->scale*w->getImageHeight());
			w1 = 2*w->borderWidth + w0;
			h1 = 2*w->borderWidth + h0;
			xoff = w->imagePosX;
			yoff = fullHeight - w->imagePosY - h1;  // canvas vs vtkImageData
			if(yoff < 0) yoff = 0;
			if(xoff+w1 > fullWidth) w1 = fullWidth - xoff;
			if(yoff+h1 > fullHeight) h1 = fullHeight - yoff;
			//
			// fill the tile with the border color
			//
			if(w->borderWidth > 0)
			{
				rgb[0] = w->borderColor.red();
				rgb[1] = w->borderColor.green();
				rgb[2] = w->borderColor.blue();
				for(j=0; j<h1; j++)  // sort of a waste, but it is much simpler this way
				{
					for(i=0; i<w1; i++)
					{
						dPtr1 = dPtr + 3*(i+xoff+fullWidth*(j+yoff));
						for(k=0; k<3; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
					}
				}
			}
			//
			//  Render the image
			//
			if(w->vtk != 0)
			{
				ret = w->vtk->renderImageData(winData);
				if(ret != 0) 
				{
					data->DeepCopy(outData);
					outData->Delete();
					return ret;
				}
				//
				//  Scale image down using Qt smoothScale function
				//
				winData->GetDimensions(dims);
				if(w->scale < 100)
				{
					//
					//  Current implementation is to use QT's smoothScale function
					//
					iVTK::convertImage(winData,qtImage);
					iVTK::convertImage(qtImage.smoothScale(w0,h0),winData);
					winData->GetDimensions(dims);
				}
			}

			xoff += w->borderWidth;
			yoff += w->borderWidth;

			wPtr = (char *)winData->GetScalarPointer();
			//
			//  Chop off too large a foreground
			//
			w1 = dims[0];
			h1 = dims[1];
			if(xoff+w1 > fullWidth) w1 = fullWidth - xoff;
			if(yoff+h1 > fullHeight) h1 = fullHeight - yoff;
			for(j=0; j<h1; j++)  // sort of a waste, but it is much simpler this way
			{
				for(i=0; i<w1; i++)
				{
					dPtr1 = dPtr + 3*(i+xoff+fullWidth*(j+yoff));
					wPtr1 = wPtr + 3*(i+dims[0]*j);
					for(k=0; k<3; k++) dPtr1[k] = wPtr1[k];
				}
			}
		}
	}

	if(!work && numTilesX==1 && numTilesY==1)
	{
		ret = vtk->renderImageData(winData);
		outData->ShallowCopy(winData);
	}

	data->DeepCopy(outData);
	outData->Delete();
	return ret;

}




};

using namespace iDialogImageComposer_Private;

//
//  Main class
//
void iDialogImageComposer::init()
{

	this->setIcon(image("imcomp.png"));

	View = new ViewWidget(ViewFrame);
	QHBoxLayout *l = new QHBoxLayout(ViewFrame);
	l->addWidget(View);

	updateImageSize();

	connect(View,SIGNAL(contentsMoving(int,int)),this,SLOT(updateForeground_CurrentWindowSet(int,int)));

	this->updateWidgets();

}


void iDialogImageComposer::updateWidgets()
{

	CALL_FUNCTION1(NumTilesXSpinBox,setValue,VIEW->getNumTilesX());
	CALL_FUNCTION1(NumTilesYSpinBox,setValue,VIEW->getNumTilesY());
	CurrentTileXSpinBox->setMaxValue(VIEW->getNumTilesX());
	CurrentTileYSpinBox->setMaxValue(VIEW->getNumTilesY());
	if(TabWidget->currentPageIndex() == 0)
	{
		VIEW->setCurrentBackgroundWindow(CurrentTileXSpinBox->value()-1,CurrentTileYSpinBox->value()-1);
	}
	if(VIEW->getBackgroundVTK(CurrentTileXSpinBox->value()-1,CurrentTileYSpinBox->value()-1) != 0)
	{
		CALL_FUNCTION1(CurrentTileSetWindowComboBox,setCurrentItem,VIEW->getBackgroundVTK(CurrentTileXSpinBox->value()-1,CurrentTileYSpinBox->value()-1)->getWindowNumber()+1);
		EmptyTileOptionsPushButton->setEnabled(false);
	}
	else
	{
		EmptyTileOptionsPushButton->setEnabled(true);
	}
	CALL_FUNCTION1(GlobalBorderWidthSpinBox,setValue,VIEW->getBorderWidth());
	CALL_FUNCTION1(GlobalBackgroundColorLabel,setEraseColor,VIEW->getBorderColor());
	CALL_FUNCTION1(DrawInnerBorderCheckBox,setChecked,VIEW->getInnerBorder());

	this->updateForeground_CurrentWindowSet(0,0); 

}


void iDialogImageComposer::show()
{
	QWidget::show();
}


void iDialogImageComposer::updateAll(bool imageResized)
{
	int i;
	AddWindowComboBox->clear();
	CurrentTileSetWindowComboBox->clear();
	CurrentTileSetWindowComboBox->insertItem("None");
	for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++)
	{
		CurrentTileSetWindowComboBox->insertItem("Window #"+QString::number(i+1));
		AddWindowComboBox->insertItem("Window #"+QString::number(i+1));
	}
	VIEW->updateWindowList(imageResized);
	this->updateWidgets();
	this->updateImageSize();
}


void iDialogImageComposer::updateImageSize()
{
	ImageSizeXTextLabel->setText(QString::number(VIEW->getFullWidth()));
	ImageSizeYTextLabel->setText(QString::number(VIEW->getFullHeight()));
}


void iDialogImageComposer::updateBackground_NumTilesX(int n)
{
	VIEW->setNumTiles(n,-1);
	updateImageSize();
	CurrentTileXSpinBox->setMaxValue(VIEW->getNumTilesX());
}


void iDialogImageComposer::updateBackground_NumTilesY(int n)
{
	VIEW->setNumTiles(-1,n);
	updateImageSize();
	CurrentTileYSpinBox->setMaxValue(VIEW->getNumTilesY());
}


void iDialogImageComposer::updateBackground_CurrentTile(int)
{
	VIEW->setCurrentBackgroundWindow(CurrentTileXSpinBox->value()-1,CurrentTileYSpinBox->value()-1);
	iVTK *vtk = VIEW->getBackgroundVTK(CurrentTileXSpinBox->value()-1,CurrentTileYSpinBox->value()-1);
	int n = 0;
	if(vtk != 0) n = 1 + vtk->getWindowNumber();
	CALL_FUNCTION1(CurrentTileSetWindowComboBox,setCurrentItem,n);
	EmptyTileOptionsPushButton->setEnabled(vtk == 0);
}


void iDialogImageComposer::updateBackground_CurrentTileSetWindow(int n)
{
	VIEW->setBackgroundVTK(CurrentTileXSpinBox->value()-1,CurrentTileYSpinBox->value()-1,iVTKWindow::getWindow(n-1));
	EmptyTileOptionsPushButton->setEnabled(n == 0);
	updateImageSize();
}


void iDialogImageComposer::updateBackground_WindowBorderColor()
{
	VIEW->setBorderColor(QColorDialog::getColor(VIEW->getBorderColor(),this));
	GlobalBackgroundColorLabel->setEraseColor(VIEW->getBorderColor());
}


void iDialogImageComposer::updateBackground_EmptyTile()
{
	iString filename = QFileDialog::getOpenFileName(iEnvironment::getInstance()->get_IFRIT_DIR(),"Images (*.jpg *.jpeg *.pnm *.bmp *.png *.tif *.tiff)",this,"","IFRIT - Image Composer File Dialog");
	if(filename.isEmpty()) return;
	VIEW->setBackgroundWallpaperImage(CurrentTileXSpinBox->value()-1,CurrentTileYSpinBox->value()-1,iVTK::loadImageFromFile(filename),filename);
	updateImageSize();
}


void iDialogImageComposer::updateBackground_WindowBorderWidth(int n) 
{
	VIEW->setBorderWidth(n);
	updateImageSize();
}


void iDialogImageComposer::updateBackground_DrawInnerBorder(bool s) 
{
	VIEW->setInnerBorder(s);
	updateImageSize();
}


void iDialogImageComposer::updateForeground_AddWindow() 
{
	VIEW->createForegroundWindow(iVTKWindow::getWindow(AddWindowComboBox->currentItem()));
}


void iDialogImageComposer::updateForeground_CurrentWindowSet(int, int) 
{
	Window *fw = VIEW->getCurrentForegroundWindow();
	if(fw != 0)
	{
		CALL_FUNCTION1(CurrentWindowScaleSpinBox,setValue,fw->scale);
		CALL_FUNCTION1(CurrentWindowBorderWidthSpinBox,setValue,fw->borderWidth);
		CALL_FUNCTION1(CurrentWindowBackgroundColorLabel,setEraseColor,fw->borderColor);
	}
}


void iDialogImageComposer::updateForeground_CurrentWindowRemove()
{
	VIEW->deleteCurrentForegroundWindow();
}


void iDialogImageComposer::updateForeground_CurrentWindowScale(int s)
{
	VIEW->setForegroundWindowScale(VIEW->getCurrentForegroundWindow(),s);
}


void iDialogImageComposer::updateForeground_CurrentWindowBorderColor()
{
	QColor c = QColorDialog::getColor(VIEW->getCurrentForegroundWindow()->borderColor,this);
	VIEW->setForegroundWindowBorderColor(VIEW->getCurrentForegroundWindow(),c);
	CALL_FUNCTION1(CurrentWindowBackgroundColorLabel,setEraseColor,c);
}


void iDialogImageComposer::updateForeground_CurrentWindowBorderWidth(int w) 
{
	VIEW->setForegroundWindowBorderWidth(VIEW->getCurrentForegroundWindow(),w);
}


void iDialogImageComposer::updateTabPageChanged(QWidget *w) 
{
	if(w == TabWidget->page(0))
	{
		VIEW->emphasize(true);
	}
	if(w == TabWidget->page(1))
	{
		VIEW->emphasize(false);
	}
}


void iDialogImageComposer::packState(iString &s)
{
	VIEW->packState(s);
}


void iDialogImageComposer::unpackState(iString s)
{
	VIEW->unpackState(s);
	this->updateAll(true);
}


int iDialogImageComposer::compose(iVTK *vtk, vtkImageData *data)
{
	return VIEW->compose(vtk,data);
}


int iDialogImageComposer::getImageWidth()
{
	return VIEW->getFullWidth();
}


int iDialogImageComposer::getImageHeight()
{
	return VIEW->getFullHeight();
}

