/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iglobals.h"
#include "iorthoplane.h"

#include <vtkStructuredPoints.h>
#include <vtkPolyData.h>
#include <vtkPointData.h>
#include <vtkFloatArray.h>
#include <vtkCellArray.h>

void reportNullPointer(int);


iOrthoPlane::iOrthoPlane() 
{
	
	correctFlat = false;

	QuadMode = 1;
	oldAxis = 0; 
	oldDim[0] = oldDim[1] = oldDim[2] = 0;

	newstrips=vtkCellArray::New();
	if(newstrips == 0) reportNullPointer(6610);
	newquads=vtkCellArray::New();
	if(newquads == 0) reportNullPointer(6611);

}


iOrthoPlane::~iOrthoPlane() {

	newstrips->Delete();
	newquads->Delete();

}

void iOrthoPlane::ExecuteData(vtkDataObject *)
{
    vtkPoints *newPts;
    vtkPointData *pd;
    vtkDataArray *inScalars;
    vtkStructuredPoints *input = (vtkStructuredPoints *)this->GetInput();
    vtkPolyData *output = (vtkPolyData *)this->GetOutput();
	int dim[3];
	vtkFloat ptorig[3], ptspace[3];

	if(input == 0) return;

    pd = input->GetPointData();
    if(pd == 0) return;
    inScalars = pd->GetScalars();
	if(inScalars == 0) return;

    input->GetOrigin(ptorig);
    input->GetSpacing(ptspace);
    input->GetDimensions(dim);

	int i, Axis = -1;
	for(i=0; i<3; i++)
	{
		if(dim[i] == 1) Axis = i;
	}
	if(Axis == -1) return;

    int Uidx=(Axis+1)%3,  Vidx=(Axis+2)%3;

	vtkIdType numPts = (vtkIdType)dim[Uidx]*dim[Vidx];
    newPts = vtkPoints::New(VTK_FLOAT);
	if(newPts == 0) reportNullPointer(6601);
    newPts->SetNumberOfPoints(numPts);
    
    int ijk[3];
    float xyz[3];
    xyz[Axis] = ptorig[Axis];
    for (ijk[Uidx]=0;ijk[Uidx]<dim[Uidx];ijk[Uidx]++){

		this->UpdateProgress((float)ijk[Uidx]/dim[Uidx]);
		if(this->GetAbortExecute()) break;

		for (ijk[Vidx]=0;ijk[Vidx]<dim[Vidx];ijk[Vidx]++){
			xyz[Uidx]=ptorig[Uidx]+ptspace[Uidx]*ijk[Uidx];
			xyz[Vidx]=ptorig[Vidx]+ptspace[Vidx]*ijk[Vidx];
			newPts->SetPoint(ijk[Uidx]+ijk[Vidx]*dim[Uidx],xyz);

		}
    }
    
	if(Axis!=oldAxis || dim[0]!=oldDim[0] || dim[1]!=oldDim[1] || dim[2]!=oldDim[2]) {
		
		if (QuadMode==0){
			newstrips->Delete();
			newstrips=vtkCellArray::New();
			if(newstrips == 0) reportNullPointer(6603);
			newstrips->Allocate(newstrips->EstimateSize(dim[Vidx]-1,2*dim[Uidx]));
			for (int ii=0;ii<dim[Vidx]-1;ii++){

				this->UpdateProgress((float)ii/(dim[Vidx]-0.99));
				if(this->GetAbortExecute()) break;

				newstrips->InsertNextCell(2*dim[Uidx]);
				for (int jj=0;jj<dim[Uidx];jj++){ 
					newstrips->InsertCellPoint(jj+ii*dim[Uidx]);
					newstrips->InsertCellPoint(jj+(ii+1)*dim[Uidx]);
				}
			}
		}
		else {
			newquads->Delete();
			newquads=vtkCellArray::New();
			if(newquads == 0) reportNullPointer(6604);
			newquads->Allocate(newquads->EstimateSize(dim[Vidx]-1,4*(dim[Uidx]-1)));
			for (int ii=0;ii<dim[Vidx]-1;ii++){

				this->UpdateProgress((float)ii/(dim[Vidx]-0.99));
				if(this->GetAbortExecute()) break;

				for (int jj=0;jj<dim[Uidx]-1;jj++){ 
					newquads->InsertNextCell(4);
					newquads->InsertCellPoint(jj+ii*dim[Uidx]);
					newquads->InsertCellPoint(jj+(ii+1)*dim[Uidx]);
					newquads->InsertCellPoint((jj+1)+(ii+1)*dim[Uidx]);
					newquads->InsertCellPoint((jj+1)+ii*dim[Uidx]);
				}
			}
		}
		
    }
    
	if (QuadMode==0){
		output->SetStrips(newstrips);
	}
	else {
		output->SetPolys(newquads);
	}

	oldAxis = Axis;
	oldDim[0] = dim[0];
	oldDim[1] = dim[1];
	oldDim[2] = dim[2];

    output->SetPoints(newPts);
    newPts->Delete();
	//
	//  Correct scalars for flat shading to preserve symmetry - for quads only
	//
	if(correctFlat && QuadMode!=0)
	{
		vtkFloatArray *newScalars = vtkFloatArray::New();
		if(newScalars == 0) reportNullPointer(6602);

		newScalars->SetNumberOfComponents(inScalars->GetNumberOfComponents());
		newScalars->SetNumberOfTuples(inScalars->GetNumberOfTuples());
		
		float v1, v2, v3, v4;
		for (int ii=0;ii<dim[Vidx]-1;ii++){
			for (int jj=0;jj<dim[Uidx]-1;jj++){ 
				v1 = inScalars->GetTuple1(jj+ii*dim[Uidx]);
				v2 = inScalars->GetTuple1(jj+(ii+1)*dim[Uidx]);
				v3 = inScalars->GetTuple1((jj+1)+(ii+1)*dim[Uidx]);
				v4 = inScalars->GetTuple1((jj+1)+ii*dim[Uidx]);
#ifdef I_VTK_VERSION_42
				newScalars->SetTuple1(1+jj+ii*dim[Uidx],0.25*(v1+v2+v3+v4));
#else
				//
				//  Why, for the hell's sake, did VTK guys change the order of polygon vertices in 4.4?????
				//
				newScalars->SetTuple1(jj+ii*dim[Uidx],0.25*(v1+v2+v3+v4));
#endif
			}
		}
		output->GetPointData()->SetScalars(newScalars);
		newScalars->Delete();
	}
	else output->GetPointData()->SetScalars(inScalars);
    
}

