/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*
Implementation of iparticles.h
*/

#include "iparticles.h"

#include "ivtk.h"
#include "ilimits.h"
#include "ipalette.h"
#include "iglobals.h"
#include "iobjectfactory.h"

#include "ivtkcallback.h"
#include "ipiecewisefunction.h"
#include "iparticlesdataconverter.h"
#include "ipolygonaldataextender.h"
#include "iparticlesconnector.h"
#include "ihistogrammaker.h"
#include "iparticlessplitter.h"
#include "ivisualobjectobserver.h"
#include "idatareader.h"

#include <vtkPolyData.h>
#include <vtkPolyDataMapper.h>
#include <vtkLODActor.h>
#include <vtkActor.h>
#include <vtkProperty.h>
#include <vtkLookupTable.h>


void reportNullPointer(int ec);


iParticles* iParticles::New(iVTK *mVTK)
{
	return (iParticles *)iObjectFactory::createObject("Particles",mVTK);
}


iParticles::iParticles(iVTK *mVTK) : iVisualObject(mVTK)
{
	
	myPiece = 0;
	mySplitterDataChannel = mySplitterDataStream = 0;

	attSize = attColor = 0;

	attSizeDirect = false;
	attSizeExtraFactor = 1.0;

	type = 0;
	pal = 0;
	opa = 1.0;	
	color = iColor(0,0,0);
	size = 1.0;
	logscaleToColor = logscaleToSize = true;
	shaded = shade = false;
	keepPoint = false;

	lineWidth = 1;
	
	sif = new iPiecewiseFunction();
	if(sif == 0) reportNullPointer(8801);
	sif->setFunctionLimits(0.1,1.0);
	sif->movFunctionPoint(0,0.0,0.1);
	sif->movFunctionPoint(1,1.0,0.1);
	
	clipPlaneOn = false;

	ambient = 0.2;
	diffuse = 0.6;
	specular = 0.2;
	specularPower = 50.0;

	particlesActor = vtkLODActor::New();
	if(particlesActor == 0) reportNullPointer(8802);
	particlesMapper = vtkPolyDataMapper::New();
	if(particlesMapper == 0) reportNullPointer(8803);
	particlesDataConverter = iParticlesDataConverter::New(this);
	if(particlesDataConverter == 0) reportNullPointer(8804);
	particlesConnector = iParticlesConnector::New();
	if(particlesConnector == 0) reportNullPointer(8809);
	particlesExtender = iPolygonalDataExtender::New(this);
	if(particlesExtender == 0) reportNullPointer(8805);
	colorLookupTable = vtkLookupTable::New();
	if(colorLookupTable == 0) reportNullPointer(8806);
	histogramMaker = iHistogramMaker::New();
	if(histogramMaker == 0) reportNullPointer(8807);

	iVTKAbortRenderEventObserver *obsAbortRender = myVTK->getAbortRenderEventObserver();
	particlesMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	particlesDataConverter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	particlesConnector->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	particlesExtender->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);

	particlesMapper->SetInput(particlesExtender->GetOutput());
	particlesExtender->SetInput(particlesDataConverter->GetOutput());
	particlesDataConverter->SetInput(particlesConnector->GetOutput());

	if(attColor > 0) 
	{
		particlesMapper->ColorByArrayComponent(0,attColor-1);
		particlesMapper->ScalarVisibilityOn();
	} 
	else particlesMapper->ScalarVisibilityOff();
	particlesMapper->UseLookupTableScalarRangeOn();
	particlesMapper->SetLookupTable(colorLookupTable);
	
	this->setLogScaleToColor(logscaleToColor);
	this->setLogScaleToSize(logscaleToSize);

	particlesActor->SetMapper(particlesMapper);
	particlesActor->SetPosition(0.0,0.0,0.0);

	particlesProperty = particlesActor->GetProperty();
	particlesProperty->SetOpacity(opa);
	particlesProperty->SetPointSize(size);
	particlesProperty->SetAmbient(1.0);
	particlesProperty->SetDiffuse(0.0);
	particlesProperty->SetSpecular(0.0);
	particlesProperty->SetSpecularPower(0.0);
	particlesProperty->SetLineWidth(lineWidth);
	//
	//  Add observer to keep information about this object
	//
	particlesActor->AddObserver(vtkCommand::UserEvent,objectObserver);
	//
	//  Which data type am I?
	//
	objectObserver->setTypes(IOBJECTTYPE_PART,IDATATYPE_PART);

	this->connectSplitter(mySplitterDataChannel,mySplitterDataStream);
	this->setColor(color);

}


iParticles::~iParticles()
{
	
	mySplitter->deletePiece(myPiece);

	delete sif;
	if(!init) myVTK->removeProp(particlesActor);
	particlesActor->Delete();
	particlesMapper->Delete();
	particlesDataConverter->Delete();
	particlesExtender->Delete();
	particlesConnector->Delete();
	histogramMaker->Delete();
	colorLookupTable->Delete();
	
}


void iParticles::reset()
{
	this->updateDataChannel();
	//
	//  Un-initialized, if needed
	//
	if(!init)
	{
		myVTK->removeProp(particlesActor);
		init = true;
	}
}


void iParticles::connectSplitter(int dc, int ds)
{
	iParticlesSplitter *s = myVTK->getParticlesSplitter(dc,ds);
	if(s != 0)
	{
		mySplitterDataChannel = dc;
		mySplitterDataStream = ds;
		mySplitter = s; 
		this->setParticlesSplitterPiece(myPiece);
	}
}


void iParticles::setParticlesSplitterPiece(int n)
{
	if(n>=0 && n<mySplitter->getNumberOfPieces())
	{
		myPiece = n;
		particlesConnector->SetInput(mySplitter->GetOutput(myPiece));
		histogramMaker->SetInput(mySplitter->GetOutput(myPiece),attSize);
	}
}


void iParticles::showClipPlane(bool s)
{

	if(s && !clipPlaneOn)
	{
		particlesMapper->AddClippingPlane(myVTK->getClipPlane());
		clipPlaneOn = true;
	} 
	if(!s && clipPlaneOn) 
	{
		particlesMapper->RemoveClippingPlane(myVTK->getClipPlane());
		clipPlaneOn = false;
	}

}


void iParticles::setLineWidth(int w)
{
	lineWidth = w;
	particlesProperty->SetLineWidth(lineWidth);
}


void iParticles::setShade(bool v)
{
	shade = v;
	this->setType(type); 
}


void iParticles::setAmbient(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	ambient = v;
	if(shaded) particlesProperty->SetAmbient(v);
}


void iParticles::setDiffuse(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	diffuse = v;
	if(shaded) particlesProperty->SetDiffuse(v);
}


void iParticles::setSpecular(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	specular = v;
	if(shaded) particlesProperty->SetSpecular(v);
}


void iParticles::setSpecularPower(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>100.0) ? 100.0 : v;
	specularPower = v;
	if(shaded) particlesProperty->SetSpecularPower(v);
}


void iParticles::setType(int t)
{ 

	if(t>=0 && t<=4) type = t; else type = 0;
	particlesDataConverter->setMode(type);

	switch (type) 
	{
	case 0:
	case 1: 
	case 3:
	case 4: { shaded = shade; break; }
	case 2:	{ shaded = true; break; }
	}
	if(shaded)
	{
		particlesProperty->SetAmbient(ambient);
		particlesProperty->SetDiffuse(diffuse);
		particlesProperty->SetSpecular(specular);
		particlesProperty->SetSpecularPower(specularPower);
	} 
	else 
	{
		particlesProperty->SetAmbient(1.0);
		particlesProperty->SetDiffuse(0.0);
		particlesProperty->SetSpecular(0.0);
		particlesProperty->SetSpecularPower(0.0);
	}

	keepPoint = false;
	switch (type)
	{
	case 3:
	case 4: { keepPoint = true; break; }
	}
	if(keepPoint) particlesProperty->SetPointSize(1.0); else particlesProperty->SetPointSize(size); 

}


void iParticles::setAttToColor(int a)
{ 
	int natt = this->getLimits()->getNatt();
	if(a>=0 && a<=natt) attColor = a; else attColor = 0;
	if(attColor > 0) 
	{
		particlesMapper->ScalarVisibilityOn(); 
		particlesMapper->ColorByArrayComponent(0,attColor);
	} 
	else particlesMapper->ScalarVisibilityOff(); 
	this->updateColorLookupTable();
}


void iParticles::setAttToSize(int a)
{ 
	int natt = this->getLimits()->getNatt();
	if(a>=0 && a<=natt) attSize = a; else attSize = 0;
	particlesDataConverter->Modified();
	histogramMaker->SetInputComponent(attSize);
}


void iParticles::setAttSizeDirect(bool s)
{
	attSizeDirect = s;
	particlesDataConverter->Modified();
}


void iParticles::setAttSizeExtraFactor(double f)
{
	if(f > 0.0)
	{
		attSizeExtraFactor = f;
		particlesDataConverter->Modified();
	}
}


double iParticles::getAttSizeBasicScale()
{
	return myVTK->getReader()->getParticlesBasicScale();
}


void iParticles::setPal(int p)
{ 
	p = ( p < 0 ) ? 0 : p;
	p = ( p > iPaletteList::getInstance()->getNumPalettes() ) ? iPaletteList::getInstance()->getNumPalettes() : p;
	pal = p;
	this->updateColorLookupTable();
}


void iParticles::setColor(iColor c) 
{
	color = c;
	particlesProperty->SetColor(color.red()/255.0,color.green()/255.0,color.blue()/255.0);
}


void iParticles::setOpacity(float o)
{ 
	o = (o < 0.0) ? 0.0 : o;
	o = (o > 1.0) ? 1.0 : o;
	opa = o;
	particlesProperty->SetOpacity(opa);
	particlesActor->Modified();
}


void iParticles::setFixedSize(float o)
{ 
	o = (o < 0.01) ? 0.01 : o;
	o = (o > 100.0) ? 100.0 : o;
	size = o;
	if(keepPoint) particlesProperty->SetPointSize(1.0); else particlesProperty->SetPointSize(size); 
	particlesDataConverter->setSize(size);
	particlesActor->Modified();
}


void iParticles::setLogScaleToColor(bool s)
{ 
	logscaleToColor = s;
	this->updateColorLookupTable();
}


void iParticles::updateColorLookupTable()
{
	iPaletteList::getInstance()->getPalette(pal)->fillLookupTable(colorLookupTable);
	colorLookupTable->SetTableRange(this->getLimits()->getAttLow(attColor),this->getLimits()->getAttHigh(attColor));
	if(logscaleToColor && this->getLimits()->getAttLow(attColor)>0 && this->getLimits()->getAttHigh(attColor)>0) 
	{
		colorLookupTable->SetScaleToLog10();
	} 
	else 
	{
		colorLookupTable->SetScaleToLinear();
	}
}


void iParticles::setLogScaleToSize(bool s)
{ 
	logscaleToSize = s;
	particlesDataConverter->Modified();
}


void iParticles::show(bool show)
{
	  
	if(show)
	{
		if(init)
		{
			myVTK->addProp(particlesActor);
			init = false;
		}
		if(!init)
		{
			this->setLogScaleToColor(logscaleToColor);
			particlesActor->VisibilityOn();
		}
	} 
	else 
	{
		if(!init) particlesActor->VisibilityOff();
	}
	
}


float iParticles::getMemorySize()
{
	float s = 0.0;
	s += particlesDataConverter->getMemorySize();
	s += particlesExtender->getMemorySize();
	return s;
}
//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iParticles::packState(iString &s)
{
	int nFun = sif->getFunctionN();

	iVisualObject::packState(s);

	this->packValue(s,"iParticles::myPiece",myPiece);
	this->packValue(s,"iParticles::color",color);
	this->packValue(s,"iParticles::type",type);
	this->packValue(s,"iParticles::attColor",attColor);
	this->packValue(s,"iParticles::attSize",attSize);
	this->packValue(s,"iParticles::pal",pal);
	this->packValue(s,"iParticles::lineWidth",lineWidth);
	this->packValue(s,"iParticles::particlesConnector->getAttributeToConnect()",particlesConnector->getAttributeToConnect());
	this->packValue(s,"iParticles::particlesConnector->getAttributeToSeparate()",particlesConnector->getAttributeToSeparate());
	this->packValue(s,"iParticles::mySplitterDataChannel",mySplitterDataChannel);
	this->packValue(s,"iParticles::mySplitterDataStream",mySplitterDataStream);

	this->packValue(s,"iParticles::attSizeDirect",attSizeDirect);
	this->packValue(s,"iParticles::attSizeExtraFactor",attSizeExtraFactor);

	this->packValue(s,"iParticles::logscaleToColor",logscaleToColor);
	this->packValue(s,"iParticles::logscaleToSize",logscaleToSize);
	this->packValue(s,"iParticles::particlesConnector->isConnected()",particlesConnector->isConnected());

	this->packValue(s,"iParticles::opa",opa);
	this->packValue(s,"iParticles::size",size);

	this->packValue(s,"iParticles::nFun",nFun);
	this->packValue(s,"iParticles::xFun",sif->getFunctionX(),nFun);
	this->packValue(s,"iParticles::yFun",sif->getFunctionY(),nFun);

}


void iParticles::unpackState(iString s)
{
	int i, i2; bool b; float f; iColor c;
		
	iVisualObject::unpackState(s);

	if(this->unpackValue(s,"iParticles::myPiece",i)) this->setParticlesSplitterPiece(i);
	if(this->unpackValue(s,"iParticles::color",c)) this->setColor(c);
	if(this->unpackValue(s,"iParticles::type",i)) this->setType(i);
	if(this->unpackValue(s,"iParticles::attColor",i)) this->setAttToColor(i);
	if(this->unpackValue(s,"iParticles::attSize",i)) this->setAttToSize(i);
	if(this->unpackValue(s,"iParticles::pal",i)) this->setPal(i);
	if(this->unpackValue(s,"iParticles::lineWidth",i)) this->setLineWidth(i);
	if(this->unpackValue(s,"iParticles::particlesConnector->getAttributeToConnect()",i)) this->particlesConnector->setAttributeToConnect(i);
	if(this->unpackValue(s,"iParticles::particlesConnector->getAttributeToSeparate()",i)) this->particlesConnector->setAttributeToSeparate(i);
	if(this->unpackValue(s,"iParticles::mySplitterDataChannel",i) && this->unpackValue(s,"iParticles::mySplitterDataStream",i2)) this->connectSplitter(i,i2);
	
	if(this->unpackValue(s,"iParticles::attSizeDirect",b)) this->setAttSizeDirect(b);
	if(this->unpackValue(s,"iParticles::attSizeExtraFactor",f)) this->setAttSizeExtraFactor((double)f);

	if(this->unpackValue(s,"iParticles::logscaleToColor",b)) this->setLogScaleToColor(b);
	if(this->unpackValue(s,"iParticles::logscaleToSize",b)) this->setLogScaleToSize(b);
	if(this->unpackValue(s,"iParticles::particlesConnector->isConnected()",b)) this->particlesConnector->setConnected(b);
	
	if(this->unpackValue(s,"iParticles::opa",f)) this->setOpacity(f);
	if(this->unpackValue(s,"iParticles::size",f)) this->setFixedSize(f);
	
	int nFun;
	if(this->unpackValue(s,"iParticles::nFun",nFun))
	{
		delete sif;
		sif = new iPiecewiseFunction();
		if(sif == 0) reportNullPointer(8808);
		sif->setFunctionLimits(0.1,1.0);
		sif->movFunctionPoint(0,0.0,0.1);
		sif->movFunctionPoint(1,1.0,0.1);
		
		float *xFun = new float[nFun];
		float *yFun = new float[nFun];
		
		if(this->unpackValue(s,"iParticles::xFun",xFun,nFun) && this->unpackValue(s,"iParticles::yFun",yFun,nFun))
		{
			for(i=0; i<nFun; i++)
			{
				if(i==0 || i==nFun-1) 
				{
					sif->movFunctionPoint(i,xFun[i],yFun[i]);
				}
				else
				{
					sif->addFunctionPoint(xFun[i],yFun[i]);
				}
			}
		}
	}

}

