/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/****************************************************************************

 Class containing data limits used in GUI and VTL.
 
****************************************************************************/

#include "ilimits.h"
#include "ienvironment.h"
#include "iglobals.h"

#include "ifile.h"


#define ILIMITS_CONSTRUCTOR_DECLARATIONS_STATIC(label,Label,LABEL) \
label##Name[0] = iString("None"); \
for(i=0; i<=N##LABEL##MAX; i++) \
{ \
	label##MinSaved[i] = label##Min[i]; \
	label##MaxSaved[i] = label##Max[i]; \
	label##NameSaved[i] = label##Name[i]; \
    label##Min[i] = this->stretch##Label(label##Min[i],i); label##Max[i] = this->stretch##Label(label##Max[i],i); \
} 

#define ILIMITS_CONSTRUCTOR_DECLARATIONS_LOCAL(label,LABEL) \
n##label = N##LABEL##MAX; \
for(i=0; i<=N##LABEL##MAX; i++) \
{ \
	label##Low[i] = label##Min[i]; \
	label##High[i] = label##Max[i]; \
} 



iLimits::iLimits(iVTK *m) : iObject(m)
{
	int i;

	dataChannel = 0;

	varName[1] = iString("Variable 1");
	varName[2] = iString("Variable 2");
	varName[3] = iString("Variable 3");
	attName[1] = iString("Attribute 1");
	attName[2] = iString("Attribute 2");
	attName[3] = iString("Attribute 3");
	vecName[1] = iString("Vector field");
	tenName[1] = iString("Tensor field");
	
	for(i=0; i<=NVARMAX; i++) 
	{ 
		varStretch[i] = 0;
		varMin[i] = 0.0;
		varMax[i] = 1.0;
	}
	
	for(i=0; i<=NATTMAX; i++) 
	{ 
		attStretch[i] = 0;
		attMin[i] = 0.0;
		attMax[i] = 1.0;
	}
	
	for(i=0; i<=NVECMAX; i++) 
	{ 
		vecStretch[i] = 0;
		vecMin[i] = 0.0;
		vecMax[i] = 1.0;
	}
	
	for(i=0; i<=NTENMAX; i++) 
	{ 
		tenStretch[i] = 0;
		tenMin[i] = 0.0;
		tenMax[i] = 1.0;
	}
	
	labelName = iString("File");
	labelUnit = iString("");
	labelScale = 1.0;
	
	doSurf = doVolv = doPart = doVect = doTens = false;
	for(i=0; i<3; i++) exDown[i] = exUp[i] = 0;
		
	for(i=0; i<3; i++) periodic[i] = false;
	periodicBC = false;

	voxelLocation = 0;

	this->edition();
	//    
	//  Read the ifrit.nam file here
	//    
	iFile f(iEnvironment::getInstance()->getFileName("DIR","ifrit.nam"));
	
	if(f.open(IO_ReadOnly))
	{
		iString s;
		//
		//  Read three variable names
		//
		int i;
		bool ok;
		float v;
		for(i=1; i<=NVARMAX; i++)
		{
			f.readLine(s,1024);
			if(!s.isEmpty())
			{
				s = s.simplifyWhiteSpace();
				v = (s.section(' ',0,0)).toFloat(&ok);
				if(ok) varMin[i] = v;
				v = (s.section(' ',1,1)).toFloat(&ok);
				if(ok) varMax[i] = v;
				if(varMin[i] > varMax[i])
				{
					v = varMin[i];
					varMin[i] = varMax[i];
					varMax[i] = v;
				}
				if(fabs(varMax[i]-varMin[i])/(1.0e-30+fabs(varMax[i])+fabs(varMin[i])) < 1.0e-5) varMax[i] = varMin[i] + 1.0;
				varStretch[i] = (QString("logarithmic").find(s.section(' ',2,2)) == 0) ? 0 : 1;
				varName[i] = s.section(' ',3);
			}
		}
		//
		//  Read labels format
		//
		f.readLine(s,1024);
		if(!s.isEmpty())
		{
			char sName[1024], sUnit[1024]; float sScale = 1.0;
			int ret = sscanf(s.latin1(),"%s %s %g",sName,sUnit,&sScale);
			if(ret == 3) 
			{
				labelName = sName;
				labelUnit = sUnit;
				labelScale = sScale;
			}
			else 
			{
				ret = sscanf(s.latin1(),"%s %g",sName,&sScale);
				if(ret == 2) 
				{
					labelName = sName;
					labelScale = sScale;
				}
			}
		}
		
	}
		
	ILIMITS_CONSTRUCTOR_DECLARATIONS_STATIC(var,Var,VAR)
	ILIMITS_CONSTRUCTOR_DECLARATIONS_STATIC(att,Att,ATT)
	ILIMITS_CONSTRUCTOR_DECLARATIONS_STATIC(vec,Vec,VEC)
	ILIMITS_CONSTRUCTOR_DECLARATIONS_STATIC(ten,Ten,TEN)
			
	ILIMITS_CONSTRUCTOR_DECLARATIONS_LOCAL(var,VAR)
	ILIMITS_CONSTRUCTOR_DECLARATIONS_LOCAL(att,ATT)
	ILIMITS_CONSTRUCTOR_DECLARATIONS_LOCAL(vec,VEC)
	ILIMITS_CONSTRUCTOR_DECLARATIONS_LOCAL(ten,TEN)
			
}


iLimits::~iLimits()
{
}


void iLimits::packState(iString &s)
{

	s = "";

	this->packValue(s,"iLimits::dataChannel",dataChannel);
	this->packValue(s,"iLimits::dataStream",dataStream);

	this->packValue(s,"iLimits::periodicBC",periodicBC);
	this->packValue(s,"iLimits::periodic0",periodic[0]);
	this->packValue(s,"iLimits::periodic1",periodic[1]);
	this->packValue(s,"iLimits::periodic2",periodic[2]);
	this->packValue(s,"iLimits::voxelLocation",voxelLocation);

	this->packValue(s,"iLimits::nvar",nvar);
	this->packValue(s,"iLimits::natt",natt);
	this->packValue(s,"iLimits::nvec",nvec);
	this->packValue(s,"iLimits::nten",nten);

	this->packValue(s,"iLimits::varLog",varStretch+1,nvar);
	this->packValue(s,"iLimits::varMin",varMin+1,nvar);
	this->packValue(s,"iLimits::varMax",varMax+1,nvar);
	this->packValue(s,"iLimits::varLow",varLow+1,nvar);
	this->packValue(s,"iLimits::varHigh",varHigh+1,nvar);

	this->packValue(s,"iLimits::attLog",attStretch+1,natt);
	this->packValue(s,"iLimits::attMin",attMin+1,natt);
	this->packValue(s,"iLimits::attMax",attMax+1,natt);
	this->packValue(s,"iLimits::attLow",attLow+1,natt);
	this->packValue(s,"iLimits::attHigh",attHigh+1,natt);

	this->packValue(s,"iLimits::vecLog",vecStretch+1,nvec);
	this->packValue(s,"iLimits::vecMin",vecMin+1,nvec);
	this->packValue(s,"iLimits::vecMax",vecMax+1,nvec);
	this->packValue(s,"iLimits::vecLow",vecLow+1,nvec);
	this->packValue(s,"iLimits::vecHigh",vecHigh+1,nvec);

	this->packValue(s,"iLimits::tenLog",tenStretch+1,nten);
	this->packValue(s,"iLimits::tenMin",tenMin+1,nten);
	this->packValue(s,"iLimits::tenMax",tenMax+1,nten);
	this->packValue(s,"iLimits::tenLow",tenLow+1,nten);
	this->packValue(s,"iLimits::tenHigh",tenHigh+1,nten);

}


void iLimits::unpackState(iString s)
{
	int i; bool b; 
	int *itmp; float *ftmp;
		
	if(this->unpackValue(s,"iLimits::dataChannel",i)) this->setDataChannel(i);
	if(this->unpackValue(s,"iLimits::dataStream",i)) this->setDataStream(i);

	if(this->unpackValue(s,"iLimits::periodicBC",b)) periodicBC = b;
	if(this->unpackValue(s,"iLimits::periodic0",b)) periodic[0] = b;
	if(this->unpackValue(s,"iLimits::periodic1",b)) periodic[1] = b;
	if(this->unpackValue(s,"iLimits::periodic2",b)) periodic[2] = b;
	if(this->unpackValue(s,"iLimits::voxelLocation",i)) this->setVoxelLocation(i);

	if(this->unpackValue(s,"iLimits::nvar",i)) this->setNvar(i);
	if(this->unpackValue(s,"iLimits::natt",i)) this->setNatt(i);
	if(this->unpackValue(s,"iLimits::nvec",i)) this->setNvec(i);
	if(this->unpackValue(s,"iLimits::nten",i)) this->setNten(i);
	
	itmp = new int[nvar];
	ftmp = new float[nvar];
	if(this->unpackValue(s,"iLimits::varLog",itmp,nvar)) memcpy(varStretch+1,itmp,nvar*sizeof(int));
	if(this->unpackValue(s,"iLimits::varMin",ftmp,nvar)) memcpy(varMin+1,ftmp,nvar*sizeof(float));
	if(this->unpackValue(s,"iLimits::varMax",ftmp,nvar)) memcpy(varMax+1,ftmp,nvar*sizeof(float));
	if(this->unpackValue(s,"iLimits::varLow",ftmp,nvar)) memcpy(varLow+1,ftmp,nvar*sizeof(float));
	if(this->unpackValue(s,"iLimits::varHigh",ftmp,nvar)) memcpy(varHigh+1,ftmp,nvar*sizeof(float));
	delete [] itmp;
	delete [] ftmp;

	itmp = new int[natt];
	ftmp = new float[natt];
	if(this->unpackValue(s,"iLimits::attLog",itmp,natt)) memcpy(attStretch+1,itmp,natt*sizeof(int));
	if(this->unpackValue(s,"iLimits::attMin",ftmp,natt)) memcpy(attMin+1,ftmp,natt*sizeof(float));
	if(this->unpackValue(s,"iLimits::attMax",ftmp,natt)) memcpy(attMax+1,ftmp,natt*sizeof(float));
	if(this->unpackValue(s,"iLimits::attLow",ftmp,natt)) memcpy(attLow+1,ftmp,natt*sizeof(float));
	if(this->unpackValue(s,"iLimits::attHigh",ftmp,natt)) memcpy(attHigh+1,ftmp,natt*sizeof(float));
	delete [] itmp;
	delete [] ftmp;

	itmp = new int[nvec];
	ftmp = new float[nvec];
	if(this->unpackValue(s,"iLimits::vecLog",itmp,nvec)) memcpy(vecStretch+1,itmp,nvec*sizeof(int));
	if(this->unpackValue(s,"iLimits::vecMin",ftmp,nvec)) memcpy(vecMin+1,ftmp,nvec*sizeof(float));
	if(this->unpackValue(s,"iLimits::vecMax",ftmp,nvec)) memcpy(vecMax+1,ftmp,nvec*sizeof(float));
	if(this->unpackValue(s,"iLimits::vecLow",ftmp,nvec)) memcpy(vecLow+1,ftmp,nvec*sizeof(float));
	if(this->unpackValue(s,"iLimits::vecHigh",ftmp,nvec)) memcpy(vecHigh+1,ftmp,nvec*sizeof(float));
	delete [] itmp;
	delete [] ftmp;

	itmp = new int[nten];
	ftmp = new float[nten];
	if(this->unpackValue(s,"iLimits::tenLog",itmp,nten)) memcpy(tenStretch+1,itmp,nten*sizeof(int));
	if(this->unpackValue(s,"iLimits::tenMin",ftmp,nten)) memcpy(tenMin+1,ftmp,nten*sizeof(float));
	if(this->unpackValue(s,"iLimits::tenMax",ftmp,nten)) memcpy(tenMax+1,ftmp,nten*sizeof(float));
	if(this->unpackValue(s,"iLimits::tenLow",ftmp,nten)) memcpy(tenLow+1,ftmp,nten*sizeof(float));
	if(this->unpackValue(s,"iLimits::tenHigh",ftmp,nten)) memcpy(tenHigh+1,ftmp,nten*sizeof(float));
	delete [] itmp;
	delete [] ftmp;

}


#if !EDITION
bool iLimits::edition()
{
	return false;
}
#endif



