/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

#include "iglobals.h"
#include "icaption.h"
#include "ivtk.h"

#include <vtkMath.h>
#include <vtkRenderer.h>
#include <vtkCamera.h>
#include <vtkProperty2D.h>
#include <vtkTextMapper.h>
#include <vtkPolyData.h>
#include <vtkPolyDataMapper.h>
#include <vtkPolyDataMapper2D.h>
#include <vtkActor.h>
#include <vtkActor2D.h>
#include <vtkPoints.h>
#include <vtkCellArray.h>
#include <vtkProperty.h>

#ifndef I_VTK_VERSION_40
#include <vtkTextProperty.h>
#endif

void reportNullPointer(int);


iCaption* iCaption::New()
{

	return new iCaption;

}


iCaption::iCaption()
{

	fontSize = 12;

	this->BorderOff(); // we redefine border and leader - they do not scale under magnification
	this->LeaderOff();
	this->ThreeDimensionalLeaderOn();
	this->GetPositionCoordinate()->SetReferenceCoordinate(NULL);
	this->GetPositionCoordinate()->SetCoordinateSystemToNormalizedViewport();
	this->GetPosition2Coordinate()->SetCoordinateSystemToNormalizedViewport();
	this->SetPosition(0.02,0.88);
	this->SetPosition2(0.20,0.10);
#ifndef I_VTK_VERSION_40
	this->GetCaptionTextProperty()->SetFontFamilyToArial();
	this->GetCaptionTextProperty()->ItalicOff();
	this->GetCaptionTextProperty()->ShadowOn();
	this->GetCaptionTextProperty()->BoldOn();
	this->GetCaptionTextProperty()->SetJustificationToCentered(); 
	this->GetCaptionTextProperty()->SetVerticalJustificationToCentered();
#else
	this->ItalicOff();
	this->ShadowOn();
	this->BoldOn();
	this->SetJustificationToCentered(); 
	this->SetVerticalJustificationToCentered();
#endif

	this->PickableOff();

	this->borderPolyData = vtkPolyData::New();
	vtkPoints *pts = vtkPoints::New(VTK_FLOAT);
	pts->SetNumberOfPoints(4);
	this->borderPolyData->SetPoints(pts);
	pts->Delete();
	vtkCellArray *border = vtkCellArray::New();
	border->InsertNextCell(5);
	border->InsertCellPoint(0);
	border->InsertCellPoint(1);
	border->InsertCellPoint(2);
	border->InsertCellPoint(3);
	border->InsertCellPoint(0);
	this->borderPolyData->SetLines(border);
//	this->borderPolyData->SetPolys(border);
	border->Delete();
	
	this->borderMapper = vtkPolyDataMapper2D::New();
	this->borderMapper->SetInput(this->borderPolyData);
	this->borderActor = vtkActor2D::New();
	this->borderActor->SetMapper(this->borderMapper);
	
	// This is the leader (line) from the attachment point to the caption
	this->leaderPolyData = vtkPolyData::New();
	pts = vtkPoints::New(VTK_FLOAT);
	pts->SetNumberOfPoints(2);
	this->leaderPolyData->SetPoints(pts);
	pts->Delete();
	vtkCellArray *leader = vtkCellArray::New();
	leader->InsertNextCell(2); 
	leader->InsertCellPoint(0);
	leader->InsertCellPoint(1); //at the attachment point
	this->leaderPolyData->SetLines(leader);
	leader->Delete();
	
	this->leaderMapper2D = vtkPolyDataMapper2D::New();
	this->leaderMapper2D->SetInput(this->leaderPolyData);
	this->leaderActor2D = vtkActor2D::New();
	this->leaderActor2D->SetMapper(this->leaderMapper2D);

	this->leaderMapper3D = vtkPolyDataMapper::New();
	this->leaderMapper3D->SetInput(this->leaderPolyData);
	this->leaderActor3D = vtkActor::New();
	this->leaderActor3D->SetMapper(this->leaderMapper3D);

	this->tmpMapper = vtkTextMapper::New();

	bool leader3D = false;

}


iCaption::~iCaption()
{
	
	this->borderPolyData->Delete();
	this->borderMapper->Delete();
	this->borderActor->Delete();
	
	this->leaderPolyData->Delete();
	this->leaderMapper2D->Delete();
	this->leaderActor2D->Delete();
	this->leaderMapper3D->Delete();
	this->leaderActor3D->Delete();
	
	this->tmpMapper->Delete();
}


void iCaption::ReleaseGraphicsResources(vtkWindow *win)
{
	this->vtkCaptionActor2D::ReleaseGraphicsResources(win);
	this->borderActor->ReleaseGraphicsResources(win); 
	this->leaderActor2D->ReleaseGraphicsResources(win); 
	this->leaderActor3D->ReleaseGraphicsResources(win); 
}


int iCaption::RenderOverlay(vtkViewport* viewport)
{
	int ret = 0;
	ret += this->vtkCaptionActor2D::RenderOverlay(viewport);
	ret += this->borderActor->RenderOverlay(viewport);
    if(leader3D) ret += this->leaderActor3D->RenderOverlay(viewport); else ret += this->leaderActor2D->RenderOverlay(viewport);
	return ret;
}


int iCaption::RenderOpaqueGeometry(vtkViewport* viewport)
{

	vtkCamera *cam = ((vtkRenderer *)viewport)->GetActiveCamera();
	if(cam == 0) return 0;
	
	iVTK *vtk = iVTK::convertFromViewport(viewport);

	int ret = 0;
	//
	//  Maintain the font size
	//
	vtkFloat *x1, *x2;
	x1 = this->GetPosition();
	x2 = this->GetPosition2();
	int mag = vtk->getRenderingMagnification();
	int *size = viewport->GetSize();

	float s = (fontSize+2*Padding)/fabs(size[1]*x2[1]);

	if(mag == 1)
	{
		unmagx1[0] = x1[0];
		unmagx1[1] = x1[1];
		unmagx2[0] = s*x2[0];
		unmagx2[1] = s*x2[1];
	}
	else
	{

		int winij[2];
		vtk->computePositionShiftsUnderMagnification(winij,viewport);
		
		this->SetPosition(mag*unmagx1[0]-winij[0],mag*unmagx1[1]-winij[1]);
		Padding *= mag;

	}
	
#ifndef I_VTK_VERSION_40
	tmpMapper->GetTextProperty()->SetFontSize(fontSize);
#else
	tmpMapper->SetFontSize(fontSize);
#endif
	tmpMapper->SetInput(Caption);
	int w = tmpMapper->GetWidth(viewport);
	if(w+2*Padding >= size[0]*unmagx2[0]) unmagx2[0] = 10.0*floor(1+0.1*(w+2*Padding))/size[0];
	
	this->SetPosition2(unmagx2[0]*mag,unmagx2[1]*mag);

	ret = this->vtkCaptionActor2D::RenderOpaqueGeometry(viewport);
	
	//
	//  Scale lines
	//
	borderActor->GetProperty()->SetLineWidth(mag);
	if(leader3D) leaderActor3D->GetProperty()->SetLineWidth(mag); else leaderActor2D->GetProperty()->SetLineWidth(mag);
	//
	// This piece is copied from vtkCaptionActor2D
	//
	{
		//
		// compute coordinates and set point values
		//
		int *x1, *x2, *x3;
		vtkFloat p1[4], p2[4], p3[4];
		x1 = this->AttachmentPointCoordinate->GetComputedDisplayValue(viewport);
		x2 = this->PositionCoordinate->GetComputedDisplayValue(viewport);
		x3 = this->Position2Coordinate->GetComputedDisplayValue(viewport);
		p1[0] = (float)x1[0]; p1[1] = (float)x1[1]; p1[2] = 0.0;
		p2[0] = (float)x2[0]; p2[1] = (float)x2[1]; p2[2] = p1[2];
		p3[0] = (float)x3[0]; p3[1] = (float)x3[1]; p3[2] = p1[2];
		
		// Define the border
		vtkPoints *pts = this->borderPolyData->GetPoints();
		pts->SetPoint(0, p2);
		pts->SetPoint(1, p3[0],p2[1],p1[2]);
		pts->SetPoint(2, p3[0],p3[1],p1[2]);
		pts->SetPoint(3, p2[0],p3[1],p1[2]);
		
		if(mag == 1)
		{
			// Define the leader. Have to find the closest point from the
			// border to the attachment point. We look at the four vertices
			// and four edge centers.
			vtkFloat d2, minD2, pt[3], minPt[3];
			minD2 = VTK_LARGE_FLOAT;
			
			pt[0] = p2[0]; pt[1] = p2[1]; pt[2] = minPt[2] = 0.0;
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[0] = (p2[0]+p3[0])/2.0;
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[0] = p3[0];
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[1] = (p2[1]+p3[1])/2.0;
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[1] = p3[1];
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[0] = (p2[0]+p3[0])/2.0;
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[0] = p2[0];
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[1] = (p2[1]+p3[1])/2.0;
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			// Set the leader coordinates in appropriate coordinate system
			// The pipeline is connected differently depending on the dimension
			// and availability of a leader head.
			pts = this->leaderPolyData->GetPoints();
			
			if(leader3D)
			{
				viewport->SetDisplayPoint(minPt[0],minPt[1],0.0);
				viewport->DisplayToWorld();
				viewport->GetWorldPoint(p1);
				
				double *cpos = cam->GetPosition();
				double *clip = cam->GetClippingRange();
				
				p3[0] = p1[0] - cpos[0];
				p3[1] = p1[1] - cpos[1];
				p3[2] = p1[2] - cpos[2];
				
				cam->GetDirectionOfProjection(p2);
				vtkMath::Normalize(p2);
				float s = clip[0]/vtkMath::Dot(p2,p3);
				p1[0] = cpos[0] + s*p3[0];
				p1[1] = cpos[1] + s*p3[1];
				p1[2] = cpos[2] + s*p3[2];
				
				pts->SetPoint(0,this->AttachmentPointCoordinate->GetComputedWorldValue(viewport));
				pts->SetPoint(1,p1);
				pts->Modified();
				
				this->leaderPolyData->Update();
			}
			else
			{
				pts->SetPoint(0,p1);
				pts->SetPoint(1,minPt[0],minPt[1],0.0);
			}
			
		}
		//
		// assign properties
		//
		this->borderActor->SetProperty(this->GetProperty());
		if(leader3D) this->leaderActor3D->GetProperty()->SetColor(this->GetProperty()->GetColor()); else this->leaderActor2D->GetProperty()->SetColor(this->GetProperty()->GetColor());
		
	}
	
	ret += this->borderActor->RenderOpaqueGeometry(viewport);
	if(leader3D) ret += this->leaderActor3D->RenderOpaqueGeometry(viewport); else ret += this->leaderActor2D->RenderOpaqueGeometry(viewport); 
	
	//
	//  Reset to the unmagnified state
	//
	if(mag > 1)
	{
		this->SetPosition(unmagx1[0],unmagx1[1]);
		this->SetPosition2(unmagx2[0],unmagx2[1]);
		Padding /= mag;
	}
	
	return ret;
	
}


void iCaption::setFontSize(int s)
{ 
	if(s>0 && s<99) fontSize = s; 
}




