/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

#include "ienvironment.h"
#include "iqt.h"
#include "ivtk.h"
#include "ivtkwindow.h"
#include "iqtwindow.h"
#include "iglobals.h"

#include "iqtdefs.h"
#include "iqt_data.h"
#include "iqt_surf.h"
#include "iqt_xsec.h"
#include "iqt_volv.h"
#include "iqt_view.h"

#include "idialogcalculator.h"
#include "idatagateway.h"
#include "idatareader.h"
#include "ierror.h"
#include "ifunctionparser.h"
#include "imath.h"
#include "imeshfromparticlesfilter.h"
#include "iparticles.h"
#include "ipolygonaldataextender.h"
#include "iuniformmeshdataextender.h"
#include "isurface.h"
#include "itensor.h"
#include "ivector.h"
#include "ivolume.h"
#include "ivolumedataconverter.h"
#include "ixsection.h"
#include "ilimits.h"
#include "iprogressbar.h"
#include "idialogpick.h"

#include <vtkPolyDataMapper.h>
#include <vtkLookupTable.h>
#include <vtkVolume.h>

#include <qaction.h>
#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlcdnumber.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qslider.h>
#include <qspinbox.h>
#include <qtabwidget.h>
#include <qvalidator.h>


#define CURRENT_DATACHANNEL	myQT->dataChannel
#define CURRENT_READER	iVTKWindow::getCurrentWindow()->getReader()
#define CURRENT_LIMITS	CURRENT_READER->getLimits(CURRENT_DATACHANNEL)



void reportNullPointer(int ec);


//
// *********************************************************************************
//
//  
//   Data manipulation panel widgets
//
//
// *********************************************************************************
//
#define DATA_TRAN_SETSLIDERS \
	int dims[3]; \
	dims[0] = round(pow(2.0,Data_Tran_PartToMeshXSlider->value())); \
	dims[1] = round(pow(2.0,Data_Tran_PartToMeshYSlider->value())); \
	dims[2] = round(pow(2.0,Data_Tran_PartToMeshZSlider->value())); \
	Data_Tran_PartToMeshXLCD->display(dims[0]); \
	Data_Tran_PartToMeshYLCD->display(dims[1]); \
	Data_Tran_PartToMeshZLCD->display(dims[2]); \
	CURRENT_READER->getMeshFromParticlesFilter()->SetSampleDimensions(dims); \
	Data_Tran_PartToMeshApplyNow->setEnabled(true);


void iQT_Data::init()
{

	autoReload = true;
	Data_Bound_ReloadPushButton->hide();

	myQT = (iQT *)parentWidget()->parentWidget()->parentWidget()->parentWidget()->parentWidget()->parentWidget();
	if(strcmp(myQT->name(),"iQT") != 0)
	{
		iQTWindow::getCurrentWindow()->popupError("Incorrect parenting in Tab Pages",IFRITERROR_INCORRECT_PARENTING_IN_TABS);
		exit(1);
	}

	//
	//  Helper for Data_Calc_Function
	//
	Data_Calc_FunctionList = new QString;
	if(Data_Calc_FunctionList == 0) reportNullPointer(1006);
	//
	//
	//
	if(CURRENT_LIMITS->periodicBC)
	{
		CALL_FUNCTION1(Data_Bound_BoundaryConditionsBox,setButton,0);
	}
	else
	{
		CALL_FUNCTION1(Data_Bound_BoundaryConditionsBox,setButton,1);
	}
	//
	//  Make sliders look better
	//
	//  LayoutDataLimit->setStretchFactor(LayoutDataLimitLeft,3);
	//  LayoutDataLimit->setStretchFactor(LayoutDataLimitCenter,3);
	//  LayoutDataLimit->addStretch(1);

	LayoutDataPart2MeshDimension->setStretchFactor(LayoutDataPart2MeshDimensionLeft,6);
	LayoutDataPart2MeshDimension->addStretch(1);

	LayoutDataPart2MeshKernel->setStretchFactor(LayoutDataPart2MeshKernelLeft,3);
	LayoutDataPart2MeshKernel->addStretch(1);
	//
	//  Disable what needs to be disabled
	//
	TabWidgetTransform->page(0)->setEnabled(false);
	TabWidgetTransform->page(1)->setEnabled(false);
	//
	//  Set validator for the box size line edit
	//
	QDoubleValidator *v = new QDoubleValidator(Data_Box_BoxSizeLineEdit);
	v->setBottom(0.0);
	Data_Box_BoxSizeLineEdit->setValidator(v);

}


void iQT_Data::destroy()
{

	delete Data_Calc_FunctionList;

}


void iQT_Data::doRender()
{
	//
	//  Render all windows as data widgets can affect several windows at once
	//
    if(!myQT->sliderRenderTracking)
	{
		myQT->startTask();
		myQT->render(true,true);
		myQT->endTask();
		myQT->resetCursor();
	}
}


void iQT_Data::doShift()
{
	if(CURRENT_READER->doDataShift()) myQT->doRender();
}


void iQT_Data::updateData_Widgets()
{
	//
	//  Limits panel
	//
	Data_Limit_Var1->setText(CURRENT_LIMITS->getVarName(1));
    Data_Limit_Var2->setText(CURRENT_LIMITS->getVarName(2));
    Data_Limit_Var3->setText(CURRENT_LIMITS->getVarName(3));

	QSlider *sliMin, *sliMax;
	QLCDNumber *lcdMin, *lcdMax;
	QLabel *labMin, *labMax;

	int v;
	for(v=1; v<4; v++)
	{

		switch(v)
		{
		case 1:
			{
				sliMin = Data_Limit_Var1MinSlider;
				sliMax = Data_Limit_Var1MaxSlider;
				lcdMin = Data_Limit_Var1MinLCD;
				lcdMax = Data_Limit_Var1MaxLCD;
				labMin = Data_Limit_Var1MinTextLabel;
				labMax = Data_Limit_Var1MaxTextLabel;
				break;
			}
		case 2:
			{
				sliMin = Data_Limit_Var2MinSlider;
				sliMax = Data_Limit_Var2MaxSlider;
				lcdMin = Data_Limit_Var2MinLCD;
				lcdMax = Data_Limit_Var2MaxLCD;
				labMin = Data_Limit_Var2MinTextLabel;
				labMax = Data_Limit_Var2MaxTextLabel;
				break;
			}
		case 3:
			{
				sliMin = Data_Limit_Var3MinSlider;
				sliMax = Data_Limit_Var3MaxSlider;
				lcdMin = Data_Limit_Var3MinLCD;
				lcdMax = Data_Limit_Var3MaxLCD;
				labMin = Data_Limit_Var3MinTextLabel;
				labMax = Data_Limit_Var3MaxTextLabel;
				break;
			}
		default:
			{
				sliMin = sliMax = NULL;
				lcdMin = lcdMax = NULL;
				labMin = labMax = NULL;
				break;
			}
		}
		
		if(CURRENT_LIMITS->getVarStretch(v) > 0) 
		{
			
			CALL_FUNCTION1(sliMin,setMinValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarMin(v)));
			CALL_FUNCTION1(sliMin,setMaxValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarMax(v)));
			CALL_FUNCTION1(sliMin,setValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarLow(v)));
			lcdMin->setNumDigits(4);
			lcdMin->display(round(myQT->scaleSlider1*CURRENT_LIMITS->getVarLow(v))/myQT->scaleSlider1);
			labMin->setText("Min (dex)");

			CALL_FUNCTION1(sliMax,setMinValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarMin(v)));
			CALL_FUNCTION1(sliMax,setMaxValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarMax(v)));
			CALL_FUNCTION1(sliMax,setValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarHigh(v)));
			lcdMax->setNumDigits(4);
			lcdMax->display(round(myQT->scaleSlider1*CURRENT_LIMITS->getVarHigh(v))/myQT->scaleSlider1);
			labMax->setText("Max (dex)");
			
		}
		else
		{
			
			CALL_FUNCTION1(sliMin,setMinValue,0);
			CALL_FUNCTION1(sliMin,setMaxValue,100);
			CALL_FUNCTION1(sliMin,setValue,round(100*(CURRENT_LIMITS->getVarLow(v)-CURRENT_LIMITS->getVarMin(v))/(CURRENT_LIMITS->getVarMax(v)-CURRENT_LIMITS->getVarMin(v))));
			lcdMin->setNumDigits(6);
			lcdMin->display(CURRENT_LIMITS->getVarLow(v));
			labMin->setText("Min");
			
			CALL_FUNCTION1(sliMax,setMinValue,0);
			CALL_FUNCTION1(sliMax,setMaxValue,100);
			CALL_FUNCTION1(sliMax,setValue,round(100*(CURRENT_LIMITS->getVarHigh(v)-CURRENT_LIMITS->getVarMin(v))/(CURRENT_LIMITS->getVarMax(v)-CURRENT_LIMITS->getVarMin(v))));
			lcdMax->setNumDigits(6);
			lcdMax->display(CURRENT_LIMITS->getVarHigh(v));
			labMax->setText("Max");
			
		}
		
	}
    //
	//  Smoothing panel
	//
	CALL_FUNCTION1(Data_Smoot_LocalSmoothingBox,setButton,iVTKWindow::getCurrentWindow()->getGateway()->getSmoothingFactor());
	//
	//  Box panel
	//
	CALL_FUNCTION1(Data_Box_UseOpenGLCheckBox,setChecked,iVTKWindow::getCurrentWindow()->isUsingOpenGLCoordinates());
	this->updateData_Box_UseOpenGLCheckBox(Data_Box_UseOpenGLCheckBox->isChecked());
	//
	//  Extend panel
	//
	CALL_FUNCTION1(Data_Exten_Surf,setChecked,CURRENT_LIMITS->isExtendSurf());
	CALL_FUNCTION1(Data_Exten_Volv,setChecked,CURRENT_LIMITS->isExtendVolv());
	CALL_FUNCTION1(Data_Exten_Part,setChecked,CURRENT_LIMITS->isExtendPart());
	CALL_FUNCTION1(Data_Exten_Vect,setChecked,CURRENT_LIMITS->isExtendVect());
	CALL_FUNCTION1(Data_Exten_Tens,setChecked,CURRENT_LIMITS->isExtendTens());

	CALL_FUNCTION1(Data_Exten_XdnSpinBox,setValue,CURRENT_LIMITS->getExtendDown(0));
	CALL_FUNCTION1(Data_Exten_YdnSpinBox,setValue,CURRENT_LIMITS->getExtendDown(1));
	CALL_FUNCTION1(Data_Exten_ZdnSpinBox,setValue,CURRENT_LIMITS->getExtendDown(2));
	CALL_FUNCTION1(Data_Exten_XupSpinBox,setValue,CURRENT_LIMITS->getExtendUp(0));
	CALL_FUNCTION1(Data_Exten_YupSpinBox,setValue,CURRENT_LIMITS->getExtendUp(1));
	CALL_FUNCTION1(Data_Exten_ZupSpinBox,setValue,CURRENT_LIMITS->getExtendUp(2));

	CALL_FUNCTION1(TabWidgetPlacement->page(1),setEnabled,CURRENT_LIMITS->periodicBC);
	//
	//  Shift panel
	//
	CALL_FUNCTION1(Data_Shift_Xslider,setValue,round(100.0*CURRENT_READER->getShift(0)));
	CALL_FUNCTION1(Data_Shift_Yslider,setValue,round(100.0*CURRENT_READER->getShift(1)));
	CALL_FUNCTION1(Data_Shift_Zslider,setValue,round(100.0*CURRENT_READER->getShift(2)));
	//
	//  Fit pane;
	//
	if(CURRENT_DATACHANNEL == 0)
	{
		CALL_FUNCTION1(Data_Fit_FitToBox,setButton,CURRENT_READER->getScaledDimension()+2);
	}
	//
	//  Operate on mesh panel
	//
	if(CURRENT_DATACHANNEL == 0)
	{
		CALL_FUNCTION1(Data_Calc_VarBox,setButton,CURRENT_READER->getMeshArrayOutput()-1);
		CALL_FUNCTION1(Data_Calc_Function,setText,CURRENT_READER->getMeshArrayFunction());
		CALL_FUNCTION1(Data_Calc_Name,setText,CURRENT_LIMITS->getVarName(CURRENT_READER->getMeshArrayOutput()));
	}
	//
	//   Particles to mesh panel
	//
	if(CURRENT_DATACHANNEL == 0)
	{
		CALL_FUNCTION1(Data_Tran_PartToMeshCheckBox,setChecked,CURRENT_READER->getMeshFromParticles());
		
		CALL_FUNCTION1(Data_Tran_PartToMeshMethodBox,setButton,CURRENT_READER->getMeshFromParticlesFilter()->getMethod());
		
		CALL_FUNCTION1(Data_Tran_PartToMeshKernelBox,setButton,CURRENT_READER->getMeshFromParticlesFilter()->getKernel());
		
		CALL_FUNCTION1(Data_Tran_PartToMeshXSlider,setValue,round(log((double)CURRENT_READER->getMeshFromParticlesFilter()->GetSampleDimensions()[0])/log(2.0)));
		CALL_FUNCTION1(Data_Tran_PartToMeshYSlider,setValue,round(log((double)CURRENT_READER->getMeshFromParticlesFilter()->GetSampleDimensions()[1])/log(2.0)));
		CALL_FUNCTION1(Data_Tran_PartToMeshZSlider,setValue,round(log((double)CURRENT_READER->getMeshFromParticlesFilter()->GetSampleDimensions()[2])/log(2.0)));
		DATA_TRAN_SETSLIDERS
			
		if(Data_Tran_PartToMeshXSlider->value()!=Data_Tran_PartToMeshYSlider->value() || Data_Tran_PartToMeshXSlider->value()!=Data_Tran_PartToMeshZSlider->value()) 
		{
			CALL_FUNCTION1(Data_Tran_PartToMeshKeepCubeCheckBox,setChecked,false);
		}
		else
		{
			CALL_FUNCTION1(Data_Tran_PartToMeshKeepCubeCheckBox,setChecked,true);
		}
			
		CALL_FUNCTION1(Data_Tran_PartToMeshSampleSlider,setValue,round(log10(CURRENT_READER->getMeshFromParticlesFilter()->GetMaximumDistance())*myQT->scaleSlider1));
		Data_Tran_PartToMeshSampleLCD->display(CURRENT_READER->getMeshFromParticlesFilter()->GetMaximumDistance());
	}
	//
	//  Settings panel
	//
	if(CURRENT_DATACHANNEL == 0)
	{
		if(CURRENT_LIMITS->periodicBC)
		{
			CALL_FUNCTION1(Data_Bound_BoundaryConditionsBox,setButton,0);
		}
		else
		{
			CALL_FUNCTION1(Data_Bound_BoundaryConditionsBox,setButton,1);
		}
		CALL_FUNCTION1(Data_Bound_VoxelLocationBox,setButton,CURRENT_LIMITS->getVoxelLocation());
		CALL_FUNCTION1(Data_Bound_CellToPointModeBox,setButton,CURRENT_READER->getCellToPointMode());
	}

	Data_Stretch_Var1TextLabel->setText(CURRENT_LIMITS->getVarName(1));
    Data_Stretch_Var2TextLabel->setText(CURRENT_LIMITS->getVarName(2));
    Data_Stretch_Var3TextLabel->setText(CURRENT_LIMITS->getVarName(3));

	CALL_FUNCTION1(Data_Stretch_Var1ButtonGroup,setButton,CURRENT_LIMITS->getVarStretch(1));
	CALL_FUNCTION1(Data_Stretch_Var2ButtonGroup,setButton,CURRENT_LIMITS->getVarStretch(2));
	CALL_FUNCTION1(Data_Stretch_Var3ButtonGroup,setButton,CURRENT_LIMITS->getVarStretch(3));
	//
	//  Particles subpanel
	//
	CALL_FUNCTION1(Data_Part_ResampleFactorSlider,setValue,round(log((double)CURRENT_READER->getParticlesDownsampleFactor())/log(2.0)/3.0));
	CALL_FUNCTION1(Data_Part_ResampleFactorLCD,display,pow((double)CURRENT_READER->getParticlesDownsampleFactor(),1.0/3.0));
	CALL_FUNCTION1(Data_Part_ResampleMaskButtonGroup,setButton,CURRENT_READER->getParticlesDownsampleMode());
	CALL_FUNCTION1(Data_Part_DoubleCheckBox,setChecked,!CURRENT_READER->getPointsAreFloat());
	//
	//  Erase panel
	//
	CALL_FUNCTION1(Data_Erase_MeshCheckBox,setChecked,false);
	CALL_FUNCTION1(Data_Erase_PartCheckBox,setChecked,false);
	CALL_FUNCTION1(Data_Erase_VectCheckBox,setChecked,false);
	CALL_FUNCTION1(Data_Erase_TensCheckBox,setChecked,false);

	CALL_FUNCTION1(Data_Erase_MeshCheckBox,setEnabled,CURRENT_READER->isThereMeshData(CURRENT_DATACHANNEL,0));
	CALL_FUNCTION1(Data_Erase_PartCheckBox,setEnabled,CURRENT_READER->isTherePartData(CURRENT_DATACHANNEL,0));
	CALL_FUNCTION1(Data_Erase_VectCheckBox,setEnabled,CURRENT_READER->isThereVectData(CURRENT_DATACHANNEL,0));
	CALL_FUNCTION1(Data_Erase_TensCheckBox,setEnabled,CURRENT_READER->isThereTensData(CURRENT_DATACHANNEL,0));

	CALL_FUNCTION1(Data_Erase_ErasePushButton,setEnabled,CURRENT_READER->isThereMeshData(CURRENT_DATACHANNEL,0)||CURRENT_READER->isTherePartData(CURRENT_DATACHANNEL,0)||CURRENT_READER->isThereVectData(CURRENT_DATACHANNEL,0)||CURRENT_READER->isThereTensData(CURRENT_DATACHANNEL,0));
	

	myQT->updateDataChannelComboBox(DISPLAYPAGE_DATA);
	
}


void iQT_Data::updateData_Limit_Var1MinSlider(int l)
{

	updateData_Limit_Slider(false,1,l);

}


void iQT_Data::updateData_Limit_Var2MinSlider( int l )
{

	updateData_Limit_Slider(false,2,l);

}


void iQT_Data::updateData_Limit_Var3MinSlider( int l )
{

	updateData_Limit_Slider(false,3,l);

}


void iQT_Data::updateData_Limit_Var1MaxSlider(int l)
{

	updateData_Limit_Slider(true,1,l);

}


void iQT_Data::updateData_Limit_Var2MaxSlider( int l )
{

	updateData_Limit_Slider(true,2,l);

}


void iQT_Data::updateData_Limit_Var3MaxSlider( int l )
{

	updateData_Limit_Slider(true,3,l);

}


void iQT_Data::updateData_Limit_Slider(bool m, int v, int l)
{

	QLCDNumber *wmin = 0, *wmax = 0;
	QSlider *smin = 0, *smax = 0;

	switch (v) 
	{
	case 1: 
		{ 
			smin = Data_Limit_Var1MinSlider;
			smax = Data_Limit_Var1MaxSlider;
			wmin = Data_Limit_Var1MinLCD;
			wmax = Data_Limit_Var1MaxLCD;
			break;
		}
	case 2: 
		{ 
			smin = Data_Limit_Var2MinSlider;
			smax = Data_Limit_Var2MaxSlider;
			wmin = Data_Limit_Var2MinLCD;
			wmax = Data_Limit_Var2MaxLCD;
			break;
		}
	case 3: 
		{ 
			smin = Data_Limit_Var3MinSlider;
			smax = Data_Limit_Var3MaxSlider;
			wmin = Data_Limit_Var3MinLCD;
			wmax = Data_Limit_Var3MaxLCD;
			break;
		}
	}

	float f;
    if(CURRENT_LIMITS->getVarStretch(v) > 0) 
	{
		f = l/myQT->scaleSlider1;
    } 
	else 
	{
		f = CURRENT_LIMITS->getVarMin(v) + 0.01*l*(CURRENT_LIMITS->getVarMax(v)-CURRENT_LIMITS->getVarMin(v));
    }

	if(m) 
	{
		CURRENT_LIMITS->setVarHigh(v,f);
	} 
	else 
	{
		CURRENT_LIMITS->setVarLow(v,f);
	}

	wmin->display((double)CURRENT_LIMITS->getVarLow(v));    
	wmax->display((double)CURRENT_LIMITS->getVarHigh(v));    
	if(CURRENT_LIMITS->getVarStretch(v) > 0) 
	{
		CALL_FUNCTION1(smin,setValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarLow(v)));    
		CALL_FUNCTION1(smax,setValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarHigh(v)));    
	}
	else
	{
		CALL_FUNCTION1(smin,setValue,round(100*(CURRENT_LIMITS->getVarLow(v)-CURRENT_LIMITS->getVarMin(v))/(CURRENT_LIMITS->getVarMax(v)-CURRENT_LIMITS->getVarMin(v))));
		CALL_FUNCTION1(smax,setValue,round(100*(CURRENT_LIMITS->getVarHigh(v)-CURRENT_LIMITS->getVarMin(v))/(CURRENT_LIMITS->getVarMax(v)-CURRENT_LIMITS->getVarMin(v))));
	}

	iVTKWindow::broadcastVariableLimitsChange(CURRENT_DATACHANNEL,v);
	
	myQT->TabWidgetVolv->updateVolv_Opaci_FunctionPixmap(0);

	if(myQT->sliderRenderTracking) myQT->render(true,true);

}


void iQT_Data::updateData_Limit_Var1MinToolButton()
{

	updateData_Limit_ToolButton(false,1);

}


void iQT_Data::updateData_Limit_Var2MinToolButton()
{

	updateData_Limit_ToolButton(false,2);

}


void iQT_Data::updateData_Limit_Var3MinToolButton()
{

	updateData_Limit_ToolButton(false,3);

}


void iQT_Data::updateData_Limit_Var1MaxToolButton()
{

	updateData_Limit_ToolButton(true,1);

}


void iQT_Data::updateData_Limit_Var2MaxToolButton()
{

	updateData_Limit_ToolButton(true,2);

}


void iQT_Data::updateData_Limit_Var3MaxToolButton()
{

	updateData_Limit_ToolButton(true,3);

}


void iQT_Data::updateData_Limit_ToolButton(bool m, int v)
{

	QLCDNumber *wmin = 0, *wmax = 0;
	QSlider *smin = 0, *smax = 0;

	switch (v) 
	{
	case 1: 
		{ 
			smin = Data_Limit_Var1MinSlider;
			smax = Data_Limit_Var1MaxSlider;
			wmin = Data_Limit_Var1MinLCD;
			wmax = Data_Limit_Var1MaxLCD;
			break;
		}
	case 2: 
		{ 
			smin = Data_Limit_Var2MinSlider;
			smax = Data_Limit_Var2MaxSlider;
			wmin = Data_Limit_Var2MinLCD;
			wmax = Data_Limit_Var2MaxLCD;
			break;
		}
	case 3: 
		{ 
			smin = Data_Limit_Var3MinSlider;
			smax = Data_Limit_Var3MaxSlider;
			wmin = Data_Limit_Var3MinLCD;
			wmax = Data_Limit_Var3MaxLCD;
			break;
		}
	}

	float vMin, vMax;
    if(CURRENT_LIMITS->getVarStretch(v) > 0) 
	{
		if(m)
		{
			CURRENT_LIMITS->setVarMax(v,CURRENT_LIMITS->getVarMax(v)+1);
			CURRENT_LIMITS->setVarMin(v,CURRENT_LIMITS->getVarMin(v)-1);
		}
		else
		{
			vMax = CURRENT_LIMITS->getVarMax(v) - 1;
			vMin = CURRENT_LIMITS->getVarMin(v) + 1;
			if(vMax > vMin)
			{
				CURRENT_LIMITS->setVarMax(v,vMax);
				CURRENT_LIMITS->setVarMin(v,vMin);
			}
		}
    } 
	else 
	{
		vMin = CURRENT_LIMITS->getVarMin(v);
		vMax = CURRENT_LIMITS->getVarMax(v);
		if(m)
		{
			CURRENT_LIMITS->setVarMax(v,vMax+0.5*(vMax-vMin)); 
			CURRENT_LIMITS->setVarMin(v,vMin-0.5*(vMax-vMin)); 
		}
		else
		{
			CURRENT_LIMITS->setVarMax(v,vMax-0.25*(vMax-vMin)); 
			CURRENT_LIMITS->setVarMin(v,vMin+0.25*(vMax-vMin)); 
		}
    }

	wmin->display((double)CURRENT_LIMITS->getVarLow(v));    
	wmax->display((double)CURRENT_LIMITS->getVarHigh(v));    
	if(CURRENT_LIMITS->getVarStretch(v) > 0) 
	{
		CALL_FUNCTION1(smin,setMinValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarMin(v)));
		CALL_FUNCTION1(smin,setMaxValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarMax(v)));
		CALL_FUNCTION1(smin,setValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarLow(v)));    
		CALL_FUNCTION1(smax,setMinValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarMin(v)));
		CALL_FUNCTION1(smax,setMaxValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarMax(v)));
		CALL_FUNCTION1(smax,setValue,round(myQT->scaleSlider1*CURRENT_LIMITS->getVarHigh(v)));    
	}
	else
	{
		CALL_FUNCTION1(smin,setValue,round(100*(CURRENT_LIMITS->getVarLow(v)-CURRENT_LIMITS->getVarMin(v))/(CURRENT_LIMITS->getVarMax(v)-CURRENT_LIMITS->getVarMin(v))));
		CALL_FUNCTION1(smax,setValue,round(100*(CURRENT_LIMITS->getVarHigh(v)-CURRENT_LIMITS->getVarMin(v))/(CURRENT_LIMITS->getVarMax(v)-CURRENT_LIMITS->getVarMin(v))));
	}

	iVTKWindow::broadcastVariableLimitsChange(CURRENT_DATACHANNEL,v);
	
	myQT->TabWidgetVolv->updateVolv_Opaci_FunctionPixmap(0);

	if(!m) myQT->render(true,true);

}


void iQT_Data::updateData_Shift_X( int v )
{

	updateData_Shift(0,v);

}

void iQT_Data::updateData_Shift_Y( int v )
{

	updateData_Shift(1,v);

}


void iQT_Data::updateData_Shift_Z( int v )
{

	updateData_Shift(2,v);

}


void iQT_Data::updateData_Shift( int d, int v )
{

	CURRENT_READER->setDataShift(d,0.01*v);
	if(myQT->sliderRenderTracking) this->doShift();

}



void iQT_Data::updateData_Exten_Surf( bool s )
{

	if(s != CURRENT_LIMITS->isExtendSurf()) 
	{
		myQT->startTask();
		iVTKWindow::broadcastExtentSettingsChange(CURRENT_DATACHANNEL,s,IVTKWINDOW_BROADCASTADDRESS_SURF);
		myQT->render(true,true);
		myQT->endTask();
	}   

}


void iQT_Data::updateData_Exten_Volv( bool s )
{

	if(s != CURRENT_LIMITS->isExtendVolv()) 
	{
		myQT->startTask();
		iVTKWindow::broadcastExtentSettingsChange(CURRENT_DATACHANNEL,s,IVTKWINDOW_BROADCASTADDRESS_VOLV);
		myQT->render(true,true);
		myQT->endTask();
	}     

}


void iQT_Data::updateData_Exten_Part( bool s )
{

	if(s != CURRENT_LIMITS->isExtendPart()) 
	{
		myQT->startTask();
		iVTKWindow::broadcastExtentSettingsChange(CURRENT_DATACHANNEL,s,IVTKWINDOW_BROADCASTADDRESS_PART);
		myQT->render(true,true);
		myQT->endTask();
	}     

}


void iQT_Data::updateData_Exten_Vect( bool s )
{

	if(s != CURRENT_LIMITS->isExtendVect()) 
	{
		myQT->startTask();
		iVTKWindow::broadcastExtentSettingsChange(CURRENT_DATACHANNEL,s,IVTKWINDOW_BROADCASTADDRESS_VECT);
		myQT->render(true,true);
		myQT->endTask();
	}     

}


void iQT_Data::updateData_Exten_Tens( bool s )
{

	if(s != CURRENT_LIMITS->isExtendTens()) 
	{
		myQT->startTask();
		iVTKWindow::broadcastExtentSettingsChange(CURRENT_DATACHANNEL,s,IVTKWINDOW_BROADCASTADDRESS_TENS);
		myQT->render(true,true);
		myQT->endTask();
	}     

}


void iQT_Data::updateData_Exten_Zdn( int v )
{

	this->updateData_Exten(false,2,v);

}


void iQT_Data::updateData_Exten_Ydn( int v )
{

	this->updateData_Exten(false,1,v);

}


void iQT_Data::updateData_Exten_Xdn( int v )
{

	this->updateData_Exten(false,0,v);

}


void iQT_Data::updateData_Exten_Zup( int v )
{

	this->updateData_Exten(true,2,v);

}


void iQT_Data::updateData_Exten_Yup( int v )
{

	this->updateData_Exten(true,1,v);

}


void iQT_Data::updateData_Exten_Xup( int v )
{

	this->updateData_Exten(true,0,v);

}


void iQT_Data::updateData_Exten(bool s, int d, int v)
{
	bool changed = false;
	if(s) 
	{
		if(v != CURRENT_LIMITS->getExtendUp(d)) 
		{
			CURRENT_LIMITS->setExtendUp(d,v);
			changed = true;
		}
	} 
	else 
	{
		if(v != CURRENT_LIMITS->getExtendDown(d)) 
		{
			CURRENT_LIMITS->setExtendDown(d,v);
			changed = true;
		}
	}

	if(!changed) return;

	myQT->startTask();
	iVTKWindow::broadcastExtentSettingsChange(CURRENT_DATACHANNEL,false,IVTKWINDOW_BROADCASTADDRESS_EXT);
	myQT->render(true,true);
	myQT->endTask();

}


void iQT_Data::updateData_Smoot_LocalSmoothingBox( int v )
{

	myQT->startTask();
	iVTKWindow::getCurrentWindow()->getGateway()->setSmoothingFactor(v);
	iVTKWindow::getCurrentWindow()->getGateway()->Modified();
	myQT->render(true,true);
	myQT->endTask();

}


void iQT_Data::updateData_Calc_StartEditor()
{

	iDialogCalculator d;
	d.setParentLineEdit(Data_Calc_Function,Data_Calc_FunctionList);
	iEnvironment::getInstance()->directErrorMessagesToWindow();
	iEnvironment::getInstance()->setErrorMessageFirstString("vtkFunctionParser ");
	iEnvironment::getInstance()->setErrorMessageFirstSymbol(31);
	d.exec();
	iEnvironment::getInstance()->setErrorMessageFirstString((char *)0);
	iEnvironment::getInstance()->setErrorMessageFirstSymbol(0);
	iEnvironment::getInstance()->directErrorMessagesToLog();

	if(Data_Calc_Function->text().isEmpty())
	{
		CURRENT_LIMITS->restoreVarNames();
		this->updateVariableNames();
		Data_Calc_Name->clear();
		Data_Calc_Name->setEnabled(false);
		CURRENT_READER->setMeshArrayFunction(0);
	} 
	else 
	{
		Data_Calc_Name->setEnabled(true);
		CURRENT_READER->setMeshArrayFunction(Data_Calc_Function->text().latin1());
	}

}


void iQT_Data::updateData_Calc_Var( int v )
{

	CURRENT_LIMITS->restoreVarNames();
	this->updateVariableNames();
	Data_Calc_Name->clear();
	CURRENT_READER->setMeshArrayOutput(v+1);

}


void iQT_Data::updateData_Calc_NewName( const QString &s )
{

	if(!s.isEmpty())
	{
		CURRENT_LIMITS->restoreVarNames();
		CURRENT_LIMITS->setVarName(CURRENT_READER->getMeshArrayOutput(),s);
		this->updateVariableNames();
	}

}


void iQT_Data::updateData_Calc_ApplyNow()
{

	myQT->startTask();
	myQT->reloadLastSet(IDATAREADER_MESH);
	myQT->render(true,true);
	myQT->endTask();

}


void iQT_Data::updateData_Tran_PartToMeshOn(bool s)
{

	if(CURRENT_READER->getLastPartFileName().isEmpty()) return; 

	CURRENT_READER->setMeshFromParticles(s);
	if(CURRENT_READER->getLastMeshFileName().isEmpty())
	{
		CALL_FUNCTION1(myQT->TabWidgetView->View_Surf,setEnabled,s);
		CALL_FUNCTION1(myQT->TabWidgetView->View_Xsec,setEnabled,s);
		CALL_FUNCTION1(myQT->TabWidgetView->View_Volv,setEnabled,s);
		CALL_FUNCTION1(myQT->TabWidgetSurf->View_SurfLocal,setEnabled,s);
		CALL_FUNCTION1(myQT->TabWidgetXsec->View_XsecLocal,setEnabled,s);
		CALL_FUNCTION1(myQT->TabWidgetVolv->View_VolvLocal,setEnabled,s);
		CALL_FUNCTION1(myQT->actionShowSurf,setEnabled,s);
		CALL_FUNCTION1(myQT->actionShowXsec,setEnabled,s);
		CALL_FUNCTION1(myQT->actionShowVolv,setEnabled,s);
	}
	if(!s)
	{
		myQT->TabWidgetView->View_Surf->setChecked(false); //  this must send signals
		myQT->TabWidgetView->View_Xsec->setChecked(false);
		myQT->TabWidgetView->View_Volv->setChecked(false);
	}

}


void iQT_Data::updateData_Tran_PartToMeshKeepCube(bool s)
{

	if(s)
	{
		CALL_FUNCTION1(Data_Tran_PartToMeshYSlider,setValue,Data_Tran_PartToMeshXSlider->value());
		CALL_FUNCTION1(Data_Tran_PartToMeshZSlider,setValue,Data_Tran_PartToMeshXSlider->value());
	}
	DATA_TRAN_SETSLIDERS

}


void iQT_Data::updateData_Tran_PartToMeshXSlider(int v)
{

	if(Data_Tran_PartToMeshKeepCubeCheckBox->isChecked())
	{
		CALL_FUNCTION1(Data_Tran_PartToMeshYSlider,setValue,v);
		CALL_FUNCTION1(Data_Tran_PartToMeshZSlider,setValue,v);
	}
	DATA_TRAN_SETSLIDERS

}


void iQT_Data::updateData_Tran_PartToMeshYSlider(int v)
{

	if(Data_Tran_PartToMeshKeepCubeCheckBox->isChecked())
	{
		CALL_FUNCTION1(Data_Tran_PartToMeshXSlider,setValue,v);
		CALL_FUNCTION1(Data_Tran_PartToMeshZSlider,setValue,v);
	}
	DATA_TRAN_SETSLIDERS

}


void iQT_Data::updateData_Tran_PartToMeshZSlider(int v)
{

	if(Data_Tran_PartToMeshKeepCubeCheckBox->isChecked())
	{
		CALL_FUNCTION1(Data_Tran_PartToMeshYSlider,setValue,v);
		CALL_FUNCTION1(Data_Tran_PartToMeshXSlider,setValue,v);
	}
	DATA_TRAN_SETSLIDERS

}


void iQT_Data::updateData_Tran_PartToMeshSampleSlider(int v)
{
	float f = pow10(v/myQT->scaleSlider1);
	CURRENT_READER->getMeshFromParticlesFilter()->SetMaximumDistance(f);
	Data_Tran_PartToMeshSampleLCD->display((double)f);
	Data_Tran_PartToMeshApplyNow->setEnabled(true);
}


void iQT_Data::updateData_Tran_PartToMeshApplyNow()
{

	myQT->startTask();
	CURRENT_READER->createMeshFromParticles();
	CURRENT_READER->Update();

	Data_Tran_PartToMeshApplyNow->setEnabled(false);

	int i, j, nnew = CURRENT_LIMITS->getNvar();
	for(i=myQT->nAttButtons; i<myQT->nButtons; i++)
	{
		for(j=0; j<nnew; j++) if(myQT->button[i][j]) myQT->button[i][j]->setEnabled(true);
	}

	updateData_Widgets();

	myQT->render(true);
	myQT->endTask();

}


void iQT_Data::updateData_Tran_PartToMeshMethod(int v)
{

	CURRENT_READER->getMeshFromParticlesFilter()->setMethod(v);
	Data_Tran_PartToMeshApplyNow->setEnabled(true);

}


void iQT_Data::updateData_Tran_PartToMeshKernel(int v)
{

	CURRENT_READER->getMeshFromParticlesFilter()->setKernel(v);
	Data_Tran_PartToMeshApplyNow->setEnabled(true);

}


void iQT_Data::updateData_Fit_FitToBox(int v)
{

	myQT->startTask();
	CURRENT_READER->setScaledDimension(v-2);
	myQT->render(true,true);
	myQT->endTask();

}


void iQT_Data::updateData_Bound_BoundaryConditions(int n)
{
    bool reloadfile = false;
	
    if(n == 0) 
	{
		if(!CURRENT_LIMITS->periodicBC) reloadfile = true;
		CURRENT_LIMITS->periodicBC = true;
	}
    if(n == 1) 
	{
		if(CURRENT_LIMITS->periodicBC) reloadfile = true;
		CURRENT_LIMITS->periodicBC = false;
    }
    
    if(reloadfile)
	{
		CALL_FUNCTION1(TabWidgetPlacement->page(1),setEnabled,CURRENT_LIMITS->periodicBC);
		myQT->setReloadLastSetMode(IDATAREADER_ALL);
		if(autoReload)
		{
			myQT->startTask();
			myQT->reloadLastSet();
			myQT->endTask();
		}
	}
	
}


void iQT_Data::updateData_Bound_VoxelLocation(int n)
{
    if(n != CURRENT_LIMITS->getVoxelLocation()) 
	{
		CURRENT_LIMITS->setVoxelLocation(n);
		myQT->setReloadLastSetMode(IDATAREADER_MESH|IDATAREADER_VECT|IDATAREADER_TENS);
		if(autoReload)
		{
			myQT->startTask();
			myQT->reloadLastSet();
			myQT->endTask();
		}
	}
}


void iQT_Data::updateData_Bound_CellToPointMode(int m)
{
	static bool firstTime = true;

    if(m != CURRENT_READER->getCellToPointMode()) 
	{
		CURRENT_READER->setCellToPointMode(m);
		if(m==5 && firstTime) 
		{
			firstTime = false;
			myQT->popupInfo("This option will result in the mesh data being displaced by\n half the cell size in each direction.\n");
		}
		myQT->setReloadLastSetMode(IDATAREADER_MESH|IDATAREADER_VECT|IDATAREADER_TENS);
		if(autoReload)
		{
			myQT->startTask();
			myQT->reloadLastSet();
			myQT->endTask();
		}
	}
}


void iQT_Data::updateVariableNames()
{

}


void iQT_Data::updateData_Part_ResampleMaskButtonGroup(int m)
{
    if(m != CURRENT_READER->getParticlesDownsampleMode()) 
	{
		myQT->setReloadLastSetMode(IDATAREADER_PART);
		CURRENT_READER->setParticlesDownsampleMode(m);
		if(autoReload)
		{
			myQT->startTask();
			myQT->reloadLastSet();
			myQT->resetProgressBar();
			myQT->render(true,true); // render all windows so the clones are also updated
			myQT->endTask();
		}
	}
}


void iQT_Data::updateData_Part_ResampleFactorSlider(int v)
{
	int vv = round(pow(2.0,(double)v));
	CALL_FUNCTION1(Data_Part_ResampleFactorLCD,display,vv);
}


void iQT_Data::updateData_Part_ResampleFactorSliderFinal()
{
	float m = pow(2.0,3*(double)Data_Part_ResampleFactorSlider->value());
    if(fabs(m-CURRENT_READER->getParticlesDownsampleFactor())>1.0e-5) 
	{
		CURRENT_READER->setParticlesDownsampleFactor(m);
		myQT->setReloadLastSetMode(IDATAREADER_PART);
		if(autoReload)
		{
			myQT->reloadLastSet();
			myQT->startTask();
			myQT->resetProgressBar();
			myQT->render(true,true); // render all windows so the clones are also updated
			myQT->endTask();
		}
	}
}


void iQT_Data::updateData_Stretch_Var1ButtonGroup(int l)
{
	this->updateData_Stretch_VarButtonGroup(1,l);
}


void iQT_Data::updateData_Stretch_Var2ButtonGroup(int l)
{
	this->updateData_Stretch_VarButtonGroup(2,l);
}


void iQT_Data::updateData_Stretch_Var3ButtonGroup(int l)
{
	this->updateData_Stretch_VarButtonGroup(3,l);
}


void iQT_Data::updateData_Stretch_VarButtonGroup(int v, int l)
{
	if(CURRENT_LIMITS->getVarStretch(v) != l)
	{
		
		CURRENT_READER->setVarStretch(v,l);
		myQT->setReloadLastSetMode(IDATAREADER_MESH);
		iVTKWindow::broadcastVariableLimitsChange(CURRENT_DATACHANNEL,v);
		if(autoReload)
		{
			myQT->startTask();
			myQT->blockRenderer = true; // block renderer so that limits can be set correctly
			myQT->reloadLastSet();
			myQT->blockRenderer = false;
//			iVTKWindow::broadcastVariableLimitsChange(CURRENT_DATACHANNEL,v);
			myQT->render(true,true); // render all windows so the clones are also updated
			myQT->endTask();
		}
		this->updateData_Widgets();
	}
}
	

void iQT_Data::updateData_Erase_ErasePushButton()
{
	int k;

	if(myQT->idiosyncratic && myQT->isVisible())
	{
		switch( QMessageBox::warning(this,"IFRIT","Are you sure you want to erase the data?\n\n","Yes", "Cancel",0,0,1) ) 
		{
		case 0: // The user clicked the Retry again button or pressed Enter
			{ break; }
		case 1: // The user clicked the Quit or pressed Escape
			{ return; }
		}
	}

	int flag = 0;
	if(Data_Erase_MeshCheckBox->isChecked()) flag |= IDATAREADER_MESH;
	if(Data_Erase_PartCheckBox->isChecked()) flag |= IDATAREADER_PART;
	if(Data_Erase_VectCheckBox->isChecked()) flag |= IDATAREADER_VECT;
	if(Data_Erase_TensCheckBox->isChecked()) flag |= IDATAREADER_TENS;
	CURRENT_READER->eraseData(CURRENT_DATACHANNEL,flag);
	//
	//  Synchronize all visual objects for all clones
	//
	iDataReader *r = CURRENT_READER;
	for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) if(iVTKWindow::getWindow(k)->getReader() == r)
	{
		iVTKWindow::getWindow(k)->synchronizeObjectsAndData();
	}

	myQT->updateWidgets(-1);
	myQT->render(true,true);

}


void iQT_Data::updateData_Box_UseOpenGLCheckBox(bool s)
{
	Data_Box_BoxSizeLineEdit->setEnabled(!s);
	iVTKWindow::getCurrentWindow()->useOpenGLCoordinates(s);
	myQT->updateCoordinateDisplays();
	if(s) Data_Box_BoxSizeLineEdit->setText(""); else Data_Box_BoxSizeLineEdit->setText(QString::number(iVTKWindow::getCurrentWindow()->getBoxSize()));
}


void iQT_Data::updateData_Box_BoxSizeLineEdit(const QString &s)
{
	bool ok;
	float v = s.toFloat(&ok);
	if(ok) 
	{
		iVTKWindow::getCurrentWindow()->setBoxSize(v);
		myQT->updateCoordinateDisplays();
		if(iVTKWindow::getCurrentWindow()->isRulerVisible()) iVTKWindow::getCurrentWindow()->render(true);
	}
	else CALL_FUNCTION1(Data_Box_BoxSizeLineEdit,setText,"");
}


void iQT_Data::updateData_Part_DoubleCheckBox(bool s)
{
    if(s != !CURRENT_READER->getPointsAreFloat()) 
	{
		CURRENT_READER->setPointsAreFloat(!s);
		myQT->setReloadLastSetMode(IDATAREADER_PART);
		if(autoReload)
		{
			myQT->startTask();
			myQT->reloadLastSet();
			myQT->endTask();
		}
	}
}



