/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#include "idatareader.h"


#include "ibasicparticlesdatasubject.h"
#include "ibuffer.h"
#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idirectory.h"
#include "iedition.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ieventobserver.h"
#include "iexpressionparser.h"
#include "iextensionfactory.h"
#include "ifile.h"
#include "iparallelmanager.h"
#include "iparallelworker.h"
#include "iuniformgriddatasubject.h"
#include "iviewmodule.h"

#include <vtkArrayCalculator.h>

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


//
//  Define static members
//
IOBJECT_DEFINE_TYPE(iDataReader,DataReader,dr,iObjectType::_Helper);

IOBJECT_DEFINE_KEY(iDataReader,BoundaryConditions,bc,Int,1);
IOBJECT_DEFINE_KEY(iDataReader,CellToPointMode,cp,Int,1);
IOBJECT_DEFINE_KEY(iDataReader,EraseData,e,Int,1);
IOBJECT_DEFINE_KEY(iDataReader,ErrorMessage,em,String,1); 
IOBJECT_DEFINE_KEY(iDataReader,IsSet,is,Bool,1); 
IOBJECT_DEFINE_KEY(iDataReader,LoadData,l,String,2); 
IOBJECT_DEFINE_KEY(iDataReader,ParticlesDownsampleFactor,pdf,Int,1);
IOBJECT_DEFINE_KEY(iDataReader,ParticlesDownsampleMode,pdm,Int,1);
IOBJECT_DEFINE_KEY(iDataReader,ParticlesOrderInFileIsAttribute,poa,Bool,1);
IOBJECT_DEFINE_KEY(iDataReader,ShiftData,s,Double,3);
IOBJECT_DEFINE_KEY(iDataReader,ScalarFieldCalculatorFunction,scf,String,1);
IOBJECT_DEFINE_KEY(iDataReader,ScalarFieldCalculatorOutputVariable,scv,OffsetInt,1);
IOBJECT_DEFINE_KEY(iDataReader,ScaledDimension,sd,Int,1);
IOBJECT_DEFINE_KEY(iDataReader,VoxelLocation,vl,Int,1);


//
//  Main class
//
iDataReader* iDataReader::New(iViewModule* vm)
{
	int i;

	iDataReader *tmp = new iDataReader(vm); IERROR_ASSERT(tmp);
	iExtensionFactory::InstallExtensions(tmp);

	iDataSubject *s;
	for(i=0; i<tmp->mExtensions.Size(); i++)
	{
		while((s = iDataReaderExtension::RequiredCast(tmp->mExtensions[i])->AddNewSubject()) != 0)
		{
			tmp->mSubjects.Add(s);
		}
	}

	iEdition::ApplySettings(tmp,iDataReader::Type());

	return tmp;
}


iDataReader::iDataReader(iViewModule *vm) : iViewModuleComponent(vm), mSetSubjects(0), mSubjects(0), mVisitedFilesList(VisitedFile())
{
	mTwoCopies = mInExtensionUpdate = false;

	mCellToPointMode = 2;
	mVoxelLocation = 0;
	mBoundaryConditions = _BoundaryConditionsWall;

    mUniformVectorsSubject = new iUniformVectorsDataSubject(this); IERROR_ASSERT(mUniformVectorsSubject);
    mUniformScalarsSubject = new iUniformScalarsDataSubject(this,mUniformVectorsSubject); IERROR_ASSERT(mUniformScalarsSubject);
    mUniformTensorsSubject = new iUniformTensorsDataSubject(this); IERROR_ASSERT(mUniformTensorsSubject);
    mBasicParticlesSubject = new iBasicParticlesDataSubject(this); IERROR_ASSERT(mBasicParticlesSubject);

	iEdition::ApplySettings(mUniformScalarsSubject,mUniformScalarsSubject->RealType());
	iEdition::ApplySettings(mUniformVectorsSubject,mUniformVectorsSubject->RealType());
	iEdition::ApplySettings(mUniformTensorsSubject,mUniformTensorsSubject->RealType());
	iEdition::ApplySettings(mBasicParticlesSubject,mBasicParticlesSubject->RealType());

	mSubjects.Add(mUniformScalarsSubject);
	mSubjects.Add(mUniformVectorsSubject);
	mSubjects.Add(mUniformTensorsSubject);
	mSubjects.Add(mBasicParticlesSubject);

	mCurrentRecord = -1;
	mPeriodicityLeader = 0;

	mDataShift[0] = mDataShift[1] = mDataShift[2] = 0.0f;

	mScaledDim = -1; //max dim

//	mLastFileName = mLastAttemptedFileName = "";

	mDirectory = new iDirectory; IERROR_ASSERT(mDirectory);
}


iDataReader::~iDataReader()
{
	int i;

	delete mDirectory;

	for(i=0; i<mSubjects.Size(); i++) mSubjects[i]->Delete();
}


bool iDataReader::IsAborted(const iDataType &type) const
{
	int i;

	for(i=0; i<mSubjects.Size(); i++)
	{
		if(mSubjects[i]->IsUsingData(type) && mSubjects[i]->IsAborted()) return true;
	}
	return false;
}


void iDataReader::EraseData(const iDataType &type)
{
	int i;

	for(i=0; i<mSubjects.Size(); i++)
	{
		if(mSubjects[i]->IsUsingData(type) && mSubjects[i]->IsThereData()) mSubjects[i]->EraseData();
	}
	this->ClearCache();
}


void iDataReader::DetachData(const iDataType &type)
{
	int i;

	if(mTwoCopies) return; // if we keep two copies, the data are already detached

	for(i=0; i<mSubjects.Size(); i++)
	{
		if(mSubjects[i]->IsUsingData(type)) mSubjects[i]->DetachData();
	}
	this->ClearCache();
}


void iDataReader::ReloadData(const iDataType &type)
{
	int i;

	this->GetErrorStatus()->Clear();
	for(i=0; this->GetErrorStatus()->NoError() && i<mSubjects.Size(); i++)
	{
		if(mSubjects[i]->IsUsingData(type) && mSubjects[i]->IsThereData() && mSubjects[i]->OwnsData())
		{
			this->GetErrorStatus()->Monitor(mSubjects[i]);
			mSubjects[i]->ReadFile(mSubjects[i]->GetLastFileName());
			if(mSubjects[i]->GetObserver()->IsAborted()) this->GetErrorStatus()->SetAbort();

			if(this->GetErrorStatus()->NoError()) mSubjects[i]->Finalize();
			if(mSubjects[i]->GetObserver()->IsAborted()) this->GetErrorStatus()->SetAbort();

			if(this->GetErrorStatus()->NoError()) mSubjects[i]->ShiftData(mDataShift);
			if(mSubjects[i]->GetObserver()->IsAborted()) this->GetErrorStatus()->SetAbort();

			if(this->GetErrorStatus()->IsError())
			{
				mSubjects[i]->EraseData();
				this->ClearCache();
			}
		}
	}
}


void iDataReader::ResetPipeline()
{
	int i;

	for(i=0; i<mSubjects.Size(); i++)
	{
		mSubjects[i]->Reset();
	}
	this->ClearCache();
}


bool iDataReader::IsThereData(const iDataInfo &info) const
{
	int i;

	for(i=0; i<info.Count(); i++)
	{
		if(this->IsThereData(info.Type(i))) return true;
	}
	return false;
}

	
bool iDataReader::IsThereData(const iDataType &type) const
{
	int i;

	for(i=0; i<mSubjects.Size(); i++)
	{
		if(mSubjects[i]->IsUsingData(type) && mSubjects[i]->IsThereData()) return true;
	}
	return false;
}


const iString& iDataReader::GetLastFileName(const iDataType &type) const
{
	static const iString null;
	int i;

	for(i=0; i<mSubjects.Size(); i++)
	{
		if(mSubjects[i]->IsUsingData(type)) return mSubjects[i]->GetLastFileName();
	}
	return null;
}


iDataLimits* iDataReader::GetLimits(const iDataType &type) const
{
	int i;

	for(i=0; i<mSubjects.Size(); i++)
	{
		if(mSubjects[i]->IsUsingData(type)) return mSubjects[i]->GetLimits();
	}
	return 0;
}


iDataSubject* iDataReader::GetSubject(const iDataType &type) const
{
	int i;

	for(i=0; i<mSubjects.Size(); i++)
	{
		if(mSubjects[i]->IsUsingData(type)) return mSubjects[i];
	}
	return 0;
}


bool iDataReader::IsSetLeader(const iDataSubject *subject) const
{
	return (mSetSubjects.Size()>0 && mSetSubjects[0]==subject);
}


void iDataReader::LoadFile(const iDataType &type, const iString &fname, bool dryrun, bool savename)
{
	int i, newrec;
//	iString root, suffix, name;

	this->GetErrorStatus()->Clear();
	this->ClearCache();
	mLastAttemptedFileName = fname;
	
	//
	//  Ask subjects to read the data
	//
	iDataSubject *subject = 0;
	for(i=0; subject==0 && i<mSubjects.Size(); i++)
	{
		if(mSubjects[i]->IsUsingData(type))
		{
			if(dryrun)
			{
				if(!iFile::IsLoadable(fname)) this->GetErrorStatus()->Set("File does not exist.");
			}
			else
			{
				iString root, suffix;
				newrec = mSubjects[i]->DissectFileName(fname,root,suffix);

				this->GetErrorStatus()->Monitor(mSubjects[i]);
				mSubjects[i]->ReadFile(fname);
				if(this->GetErrorStatus()->NoError())
				{
					subject = mSubjects[i];
				}
				else
				{
					if(!mTwoCopies) mSubjects[i]->EraseData(); // mark data as absent if we keep only one copy
				}
			}
			break;
		}
	}

	//
	//  Was there an error?
	//
	if(subject == 0) return;

	//
	//  Update the set
	//
	if(newrec >= 0)
	{
		if(newrec == mCurrentRecord)
		{
			//
			//  Adding to the existent set if the subject is not there yet
			//
			if(mSetSubjects.Find(subject) < 0) mSetSubjects.Add(subject);
		}
		else
		{
			mCurrentRecord = newrec;
			if(savename) this->InsertIntoVisitedFilesList(subject,fname);
			//
			//  Re-establish the set if we are not loading a new one
			//
			if(!this->IsSetLeader(subject))
			{
				this->BreakSet();
				mSetSubjects.Add(subject);
			}
		}
	}
	else
	{
		this->BreakSet();
		if(savename) this->InsertIntoVisitedFilesList(subject,fname);
	}

	if(mPeriodicityLeader==0 || subject->GetDataType().GetPriority()<mPeriodicityLeader->GetDataType().GetPriority()) mPeriodicityLeader = subject;
	mLastFileName = fname;
	
	if(dryrun) return; // we are done in that case

	//
	//  Do we have a set? If yes, load the rest of it too
	//
	if(this->IsSetLeader(subject))
	{
		//
		//Is the filename consistent?
		//
		if(!mSetSubjects[0]->IsSeriesFileName(fname))
		{
			this->BreakSet();
		}
		else
		{
			//
			//  Load the data
			//
			for(i=1; this->GetErrorStatus()->NoError() && i<mSetSubjects.Size(); i++) // start at 1 since the leader has been loaded already
			{
				this->GetErrorStatus()->Monitor(mSetSubjects[i]);
				mSetSubjects[i]->ReadFile(mSetSubjects[i]->GetFileName(newrec));
				if(this->GetErrorStatus()->IsError() && !mTwoCopies) mSetSubjects[i]->EraseData(); // mark data as absent if we keep only one copy
			}
		}
	}

	this->GetViewModule()->GetParallelManager()->StartCounters();

	//
	//  Finish updating the data
	//
	subject->Finalize();
	if(subject->GetObserver()->IsAborted()) { subject->EraseData(); this->GetErrorStatus()->SetAbort(); }
	if(this->IsSetLeader(subject))
	{
		for(i=1; this->GetErrorStatus()->NoError() && i<mSetSubjects.Size(); i++) // the leader has been loaded already
		{
			mSetSubjects[i]->Finalize();
			if(mSetSubjects[i]->GetObserver()->IsAborted()) { mSetSubjects[i]->EraseData(); this->GetErrorStatus()->SetAbort(); }
		}
	}
		
	//
	//  Shift the data
	//
	subject->ShiftData(mDataShift);
	if(subject->GetObserver()->IsAborted()) { subject->EraseData(); this->GetErrorStatus()->SetAbort(); }
	if(this->IsSetLeader(subject))
	{
		for(i=1; this->GetErrorStatus()->NoError() && i<mSetSubjects.Size(); i++) // the leader has been loaded already
		{
			mSetSubjects[i]->ShiftData(mDataShift);
			if(mSetSubjects[i]->GetObserver()->IsAborted()) { mSetSubjects[i]->EraseData(); this->GetErrorStatus()->SetAbort(); }
		}
	}

	this->GetViewModule()->GetParallelManager()->StopCounters();
	this->GetViewModule()->UpdateAfterFileLoad();

	//
	//  Ask extensions do to their share of work, but guard against loops if they call this function
	//
	if(mInExtensionUpdate) return;
	mInExtensionUpdate = true;

	for(i=0; i<mExtensions.Size(); i++) 
	{
		iDataReaderExtension *ext = iDataReaderExtension::RequiredCast(mExtensions[i]);
		if(ext->IsUsingData(type)) ext->AfterLoadFile(type,fname,dryrun);
	}

	mInExtensionUpdate = false;
}


void iDataReader::BreakSet()
{
	mSetSubjects.Clear();
	mDirectory->Close();
	this->ClearCache();
}


void iDataReader::LoadRecord(int rec, int skip, bool dryrun, bool savename)
{
	if(rec < 0) rec = mCurrentRecord;

	this->GetErrorStatus()->Clear();
	this->ClearCache();
	if(mSetSubjects.Size() == 0)
	{
		this->GetErrorStatus()->Set("File set has not been established.");
		return;
	}

	iString fname = mSetSubjects[0]->GetFileName(rec);
	if(!iFile::IsLoadable(fname))
	{
		this->GetErrorStatus()->Set("File is not accessible.");
		return;
	}

	if(skip != 0)
	{
		if(!mDirectory->IsOpened() && !mDirectory->Open(fname))
		{
			this->GetErrorStatus()->Set("Unable to read the directory.");
			return;
		}
	
		while(skip>0 && !fname.IsEmpty())
		{
			fname = mDirectory->GetFile(fname,1);
			if(this->IsAnotherSetLeader(fname)) skip--;
		}

		//
		//  Is this filename from the same series?
		//
		if(!this->IsAnotherSetLeader(fname))
		{
			this->GetErrorStatus()->Set("File does not exist.",10);
			return;
		}
	}

	if(!fname.IsEmpty())
	{
		this->LoadFile(mSetSubjects[0]->GetDataType(),fname,dryrun,savename);
	}
	else
	{
		this->GetErrorStatus()->Set("File does not exist.",10);
	}
}


const iDataType& iDataReader::GetFileSetDataType(int member) const
{
	if(member<0 || member>=mSetSubjects.Size())
	{
		return iData::None();
	}
	else
	{
		return mSetSubjects[member]->GetDataType();
	}
}


const iString& iDataReader::GetLastFileSetName() const
{
	static const iString none;

	if(mSetSubjects.Size() == 0) return none; else return mSetSubjects[0]->GetLastFileName();
}


const iString iDataReader::GetFileSetName(int rec) const
{
	static const iString none;
	
	if(mSetSubjects.Size()>0 && rec>=0) return mSetSubjects[0]->GetFileName(rec); else return none;
}


bool iDataReader::IsCurrentSetMember(const iString &fname) const
{
	int i;
	for(i=0; i<mSetSubjects.Size(); i++)
	{
		if(mSetSubjects[i]->GetLastFileName() == fname) return true;
	}
	return false;
}


bool iDataReader::IsAnotherSetLeader(const iString &fname) const
{
	if(mSetSubjects.Size() > 0)
	{
		return mSetSubjects[0]->IsSeriesFileName(fname);
	}
	else return false;
}


void iDataReader::SetDataShift(int d, double dx)
{
	if(d>=0 && d<=2) mDataShift[d] = dx;
	this->ClearCache();
}


void iDataReader::ShiftData(double *dr)
{
	int i;

	if(dr == 0) dr = mDataShift;

	this->GetViewModule()->GetParallelManager()->StartCounters();

	for(i=0; i<mSubjects.Size(); i++) if(mSubjects[i]->IsThereData())
	{	
		mSubjects[i]->ShiftData(dr);
	}

	this->GetViewModule()->GetParallelManager()->StopCounters();
}


float iDataReader::GetMemorySize() const
{
	int i;
    float s = 0.0;
	
	for(i=0; i<mSubjects.Size(); i++)
	{	
		s += mSubjects[i]->GetMemorySize();
	}

	return s;
}


void iDataReader::ExtendableObjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyParticlesOrderInFileIsAttribute(),this->GetParticlesOrderInFileIsAttribute());
	this->PackValue(s,KeyCellToPointMode(),mCellToPointMode);
	this->PackValue(s,KeyScaledDimension(),mScaledDim);
	this->PackValue(s,KeyVoxelLocation(),mVoxelLocation);
	this->PackValue(s,KeyBoundaryConditions(),mBoundaryConditions);

	this->PackValue(s,KeyScalarFieldCalculatorOutputVariable(),this->GetScalarFieldCalculatorOutputVariable());
	this->PackValue(s,KeyScalarFieldCalculatorFunction(),this->GetScalarFieldCalculatorFunction());

	this->PackValue(s,KeyParticlesDownsampleMode(),this->GetParticlesDownsampleMode());
	this->PackValue(s,KeyParticlesDownsampleFactor(),this->GetParticlesDownsampleFactor());

	this->PackValue(s,KeyShiftData(),mDataShift,3);

	//
	//  Special "query" keys
	//
	this->PackValue(s,KeyErrorMessage(),this->GetErrorStatus()->GetErrorMessage());
	this->PackValue(s,KeyIsSet(),this->IsSet());
}


void iDataReader::ExtendableObjectUnPackStateBody(const iString &s)
{
	int i; bool b; double ds[3]; iString s1;

	if(this->UnPackValue(s,KeyParticlesOrderInFileIsAttribute(),b)) this->SetParticlesOrderInFileIsAttribute(b);

	if(this->UnPackValue(s,KeyCellToPointMode(),i)) this->SetCellToPointMode(i);
	if(this->UnPackValue(s,KeyScaledDimension(),i)) this->SetScaledDimension(i);
	if(this->UnPackValue(s,KeyScalarFieldCalculatorOutputVariable(),i)) this->SetScalarFieldCalculatorOutputVariable(i);
	if(this->UnPackValue(s,KeyVoxelLocation(),i)) this->SetVoxelLocation(i);
	if(this->UnPackValue(s,KeyBoundaryConditions(),i)) this->SetBoundaryConditions(i);

	if(this->UnPackValue(s,KeyParticlesDownsampleMode(),i)) this->SetParticlesDownsampleMode(i);
	if(this->UnPackValue(s,KeyParticlesDownsampleFactor(),i)) this->SetParticlesDownsampleFactor(i);

	if(this->UnPackValue(s,KeyScalarFieldCalculatorFunction(),s1)) this->SetScalarFieldCalculatorFunction(s1);

	for(i=0; i<3; i++) ds[i] = mDataShift[i];
	if(this->UnPackValue(s,KeyShiftData(),ds,3)) 
	{
		for(i=0; i<3; i++) this->SetDataShift(i,ds[i]);
		this->ShiftData();
	}

	//
	//  Special "action" keys
	//
	iObject::ReportMissingKeys(false); //action keys are not part of the states

	if(this->UnPackValue(s,KeyEraseData(),i)) if(i>=0 && i<mSubjects.Size()) this->EraseData(mSubjects[i]->GetDataType()); 

	iString ws[2];
	if(this->UnPackValue(s,KeyLoadData(),ws,2))
	{
		this->LoadFileByStringKey(iData::UniformScalars(),_EnvironmentScalarFieldData,ws);
		this->LoadFileByStringKey(iData::UniformVectors(),_EnvironmentVectorFieldData,ws);
		this->LoadFileByStringKey(iData::UniformTensors(),_EnvironmentTensorFieldData,ws);
		this->LoadFileByStringKey(iData::BasicParticles(),_EnvironmentParticleSetData,ws);
	}

	iObject::ReportMissingKeys(true);
}


void iDataReader::PackCompleteState(iString &s) const
{
	int i; iString s1;

	this->PackState(s);

	//
	//  Pack all subjects for state saving
	//
	for(i=0; i<mSubjects.Size(); i++)
	{
		mSubjects[i]->PackCompleteState(s1);
		s += iObjectKey::SpecialSeparator() + mSubjects[i]->GetDataType().GetName() + iObjectKey::Delimiter() + s1;
	}
}


void iDataReader::UnPackCompleteState(const iString &s)
{
	int i, j, n;
	iString s1, name;

	this->UnPackState(s);

	//
	//  UnPack all subjects for state restoring
	//
	n = s.Contains(iObjectKey::SpecialSeparator());
	for(j=0; j<n; j++)
	{
		s1 = s.Section(iObjectKey::SpecialSeparator(),j+1,j+1);
		name = s1.Section(iObjectKey::Delimiter(),0,0);
		for(i=0; i<mSubjects.Size(); i++) if(mSubjects[i]->GetDataType().IsOfType(name))
		{
			mSubjects[i]->UnPackCompleteState(s1);
		}
	}
}
	

void iDataReader::LoadFileByStringKey(const iDataType &type, int envkey, iString ws[2])
{
	if(type.IsOfType(ws[0])) 
	{
		if(ws[1][0] == '+') ws[1] = this->GetViewModule()->GetControlModule()->GetFileName(envkey,ws[1].Part(1));
		this->LoadFile(type,ws[1]);
	}
}


void iDataReader::InsertIntoVisitedFilesList(const iDataSubject *subject, const iString &name)
{
	this->InsertStringIntoVisitedFilesList(subject,name);
}


void iDataReader::InsertStringIntoVisitedFilesList(const iDataSubject *subject, const iString &string)
{
	VisitedFile v;
	v.Name = string;
	v.Subject = subject;
	mVisitedFilesList.Add(v);
}


iProgressEventObserver* iDataReader::GetProgressEventObserver(const iDataType &type)
{
	int i;

	for(i=0; i<mSubjects.Size(); i++)
	{
		if(mSubjects[i]->IsUsingData(type)) return mSubjects[i]->GetObserver();
	}

	return 0;
}


const iDataType& iDataReader::GetDataType(int i) const
{
	if(i>=0 && i<mSubjects.Size()) return mSubjects[i]->GetDataType(); else return iData::None();
}


//
//  Outputs
//
vtkDataSet* iDataReader::GetOutput(const iDataType &type) const
{
	int i;

	for(i=0; i<mSubjects.Size(); i++)
	{
		if(mSubjects[i]->IsUsingData(type))
		{
			return mSubjects[i]->GetData();
		}
	}

	return 0;
}


//
//  Decorator functions
//
void iDataReader::SetParticlesOrderInFileIsAttribute(bool s)
{ 
	mBasicParticlesSubject->SetOrderInFileIsAttribute(s); 
	this->ClearCache();
}


bool iDataReader::GetParticlesOrderInFileIsAttribute() const
{ 
	return mBasicParticlesSubject->GetOrderInFileIsAttribute();
}


//
void iDataReader::SetParticlesDownsampleMode(int v)
{
	int i;

	iGenericParticlesDataSubject *ds;
	for(i=0; i<mSubjects.Size(); i++)
	{
		ds = dynamic_cast<iGenericParticlesDataSubject*>(mSubjects[i]);
		if(ds != 0)
		{
			ds->SetDownsampleMode(v);
		}
	}
	this->ClearCache();
}


int iDataReader::GetParticlesDownsampleMode() const
{
	return mBasicParticlesSubject->GetDownsampleMode();
}


//
void iDataReader::SetParticlesDownsampleFactor(int v)
{
	int i;

	iGenericParticlesDataSubject *ds;
	for(i=0; i<mSubjects.Size(); i++)
	{
		ds = dynamic_cast<iGenericParticlesDataSubject*>(mSubjects[i]);
		if(ds != 0)
		{
			ds->SetDownsampleFactor(v);
		}
	}
	this->ClearCache();
}


int iDataReader::GetParticlesDownsampleFactor() const
{
	return mBasicParticlesSubject->GetDownsampleFactor();
}


//
void iDataReader::SetScalarFieldCalculatorOutputVariable(int n)
{
	mUniformScalarsSubject->SetCalculatorOutputVariable(n);
	this->ClearCache();
}


int iDataReader::GetScalarFieldCalculatorOutputVariable() const
{
	return mUniformScalarsSubject->GetCalculatorOutputVariable();
}


//
void iDataReader::SetScalarFieldCalculatorFunction(const iString &s)
{ 
	mUniformScalarsSubject->SetCalculatorFunction(s); 
	this->ClearCache();
}


const iString& iDataReader::GetScalarFieldCalculatorFunction() const
{ 
	return mUniformScalarsSubject->GetCalculatorFunction();
}


void iDataReader::SetBoundaryConditions(int v)
{
	int i;

	if(v>=0 && v<__NumBoundaryConditions)
	{
		mBoundaryConditions = v;
		for(i=0; i<mSubjects.Size(); i++)
		{
			mSubjects[i]->SetBoundaryConditions(v);
		}
		this->ClearCache();
	}
}


//
void iDataReader::SetScaledDimension(int v)
{
	int i;

	iUniformGridDataSubject *ds;
	for(i=0; i<mSubjects.Size(); i++)
	{
		ds = dynamic_cast<iUniformGridDataSubject*>(mSubjects[i]);
		if(ds != 0)
		{
			ds->SetScaledDimension(v);
			mScaledDim = ds->GetScaledDimension();
		}
	}
	this->ClearCache();
}


//
void iDataReader::SetVoxelLocation(int v)
{
	int i;

	iUniformGridDataSubject *ds;
	for(i=0; i<mSubjects.Size(); i++)
	{
		ds = dynamic_cast<iUniformGridDataSubject*>(mSubjects[i]);
		if(ds != 0)
		{
			ds->SetVoxelLocation(v);
			mVoxelLocation = ds->GetVoxelLocation();
		}
	}
	this->ClearCache();
}


//
void iDataReader::SetCellToPointMode(int m)
{ 
	int i;

	iUniformGridDataSubject *ds;
	for(i=0; i<mSubjects.Size(); i++)
	{
		ds = dynamic_cast<iUniformGridDataSubject*>(mSubjects[i]);
		if(ds != 0)
		{
			ds->SetCellToPointMode(m);
			mCellToPointMode = ds->GetCellToPointMode();
		}
	}
	this->ClearCache();
}


//
//  Extension
//
IOBJECTEXTENSION_DEFINE_ABSTRACT(iDataReader);


void iDataReaderExtension::Define()
{
	mNextSubjectToCreate = 0;
}


iDataSubject* iDataReaderExtension::AddNewSubject()
{
	iDataSubject *s = this->CreateSubject(mNextSubjectToCreate);
	if(s != 0) mNextSubjectToCreate++;
	return s;
}

