/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
// Class containing data limits for a single DataSubject
//
#include "idatalimits.h"


#include "idatastretch.h"
#include "idatasubject.h"
#include "ifileloader.h"
#include "iviewmodule.h"

//
//  templates
//
#include "iarraytemplate.h"


using namespace iParameter;


iDataLimits* iDataLimits::New(const iDataSubject *subject, int num, const iString &name)
{
	IERROR_ASSERT(subject);
	return new iDataLimits(subject,num,name,true,false);
}


iDataLimits::iDataLimits(const iDataSubject *subject, int num, const iString &name, bool resizable, bool namesfixed) : mSubject(subject), mIsResizable(resizable), mAreNamesFixed(namesfixed), mMinNumListed(num)
{
	IERROR_ASSERT(mSubject);
	if(num < 1) 
	{
		IERROR_FATAL("Number of vars must be > 0.");
	}

	mBlockChecking = false;
	mBlockNotifications = true; // must block here to avoid a loop

	this->Configure(num,name);  
	this->AssignVars();  // assign all when there is no data present

	mBlockNotifications = false;
}


iDataLimits::~iDataLimits()
{
}


const iDataType& iDataLimits::GetDataType() const
{
	return mSubject->GetDataType();
}


void iDataLimits::Resize(int n)
{
	if(mIsResizable)
	{
		if(n < mMinNumListed) n = mMinNumListed;
		this->Configure(n,"");
	}
}


void iDataLimits::BackupVars()
{
	int i;
	mVarsBackup.Clear();
	for(i=0; i<mVars.Size(); i++) mVarsBackup.Add(mVars[i]);
}


void iDataLimits::RestoreVars()
{
	int i;
	mVars.Clear();
	for(i=0; i<mVarsBackup.Size(); i++) mVars.Add(mVarsBackup[i]);
}


void iDataLimits::Configure(int num, const iString &name)
{
	mNamePrefix = name + " ";

	if(num > 0)
	{
		while(mListedVars.Size() > num)
		{
			mVars.Remove(&mListedVars.Last());
			mListedVars.RemoveLast();
		}
		while(mListedVars.Size() < num)
		{
			Var tmp;
			tmp.Included = true;
			if(num > 1)
			{
				tmp.Name = mNamePrefix + iString::FromNumber(mListedVars.Size()+1);
			}
			else
			{
				tmp.Name = name;
			}
			tmp.Stretch = 0;
			tmp.Min = 0.1f;
			tmp.Max = 10.0f;
			tmp.FixedStretch = false;
			tmp.Class = 0;
			this->Initialize(tmp);
			mListedVars.Add(tmp);
		}
	}
}


void iDataLimits::ResetVars()
{
	mVars.Clear();
}


void iDataLimits::AssignVars(int n)
{
	int i;

	bool bn = mBlockNotifications;
	if(!bn) mBlockNotifications = true;

	//
	//  Try to adjust size
	//
	this->Resize(n);

	if(n<0 || n>mListedVars.Size()) n = mListedVars.Size();

	this->ResetVars();
	for(i=0; i<n; i++) this->AddVar(i);

	if(!bn)
	{
		mBlockNotifications = false;
		this->NotifyDependencies(-1);
	}
}


void iDataLimits::AddVar(int n)
{
	if(n>=0 && n<mListedVars.Size())
	{
		mVars.AddUnique(&mListedVars[n]);
		this->NotifyDependencies(mVars.MaxIndex());
	}
	else this->Invalid(0);
}


int iDataLimits::Invalid(int v) const
{
	if(!mBlockChecking)
	{
		IERROR_LOW("Invalid access to a DataLimits member.");
	}
	return v;
}


float iDataLimits::Invalid(float v) const
{
	if(!mBlockChecking)
	{
		IERROR_LOW("Invalid access to a DataLimits member.");
	}
	return v;
}


const iString& iDataLimits::Invalid(const iString &v) const
{
	if(!mBlockChecking)
	{
		IERROR_LOW("Invalid access to a DataLimits member");
	}
	return v;
}


void iDataLimits::AdjustMinMax(Var &v)
{
	if(!(v.Max > v.Min))
	{
		float q = 0.5f*(v.Min+v.Max);
		if(q > 0.0f)
		{
			v.Min = q*0.99f;
			v.Max = q*1.01f;
		}
		else if(q < 0.0f)
		{
			v.Min = q*1.01f;
			v.Max = q*0.99f;
		}
		else
		{
			v.Min = -0.01f;
			v.Max =  0.01f;
		}
	}
		
	if(v.UpperLimit > v.Max) v.UpperLimit = v.Max;
	if(v.UpperLimit < v.Min) v.UpperLimit = v.Min;
	if(v.LowerLimit < v.Min) v.LowerLimit = v.Min;
	if(v.LowerLimit > v.Max) v.LowerLimit = v.Max;
}


void iDataLimits::SetMax(int n, float v)
{
	if(n>=0 && n<mVars.Size())
	{
		mVars[n]->Max = v;
		this->AdjustMinMax(*mVars[n]);
		this->NotifyDependencies(n);
	}
	else this->Invalid(0);
}


void iDataLimits::SetMin(int n, float v)
{
	if(n>=0 && n<mVars.Size())
	{
		mVars[n]->Min = v;
		this->AdjustMinMax(*mVars[n]);
		this->NotifyDependencies(n);
	}
	else this->Invalid(0);
}


void iDataLimits::SetName(int n, const iString &v)
{
	if(mAreNamesFixed) return;

	if(n>=0 && n<mVars.Size())
	{
		mVars[n]->Name = v;
		this->NotifyDependencies(n);
	}
	else this->Invalid(0);
}


void iDataLimits::SetStretch(int n, int v)
{
	if(n>=0 && n<mVars.Size() && v>=0 && v<2)
	{
		if(!mVars[n]->FixedStretch)
		{
			mVars[n]->Stretch = v;
			this->NotifyDependencies(n);
		}
	}
	else this->Invalid(0);
}


void iDataLimits::SetUpperLimit(int n, float v)
{
	this->UpdateUpperLimit(v,n,mVars[n]->LowerLimit,mVars[n]->UpperLimit);
}


void iDataLimits::SetLowerLimit(int n, float v)
{
	this->UpdateLowerLimit(v,n,mVars[n]->LowerLimit,mVars[n]->UpperLimit);
}


void iDataLimits::UpdateUpperLimit(float v, int n, float &ll, float &ul)
{
	if(n>=0 && n<mVars.Size())
    {
		if(v > mVars[n]->Max) v = mVars[n]->Max;
		ul = v;
		if(v < ll) ll = v;
		this->NotifyDependencies(n);
	}
	else this->Invalid(0);
}


void iDataLimits::UpdateLowerLimit(float v, int n, float &ll, float &ul)
{
	if(n>=0 && n<mVars.Size())
    {
		if(v < mVars[n]->Min) v = mVars[n]->Min;
		ll = v;
		if(v > ul) ul = v;
		this->NotifyDependencies(n);
	}
	else this->Invalid(0);
}


void iDataLimits::Initialize(Var &v)
{
	v.LowerLimit = v.Min;
	v.UpperLimit = v.Max;
}


void iDataLimits::GetPeriodicities(bool per[3]) const
{
	int i;
	for(i=0; i<3; i++) per[i] = mSubject->GetLoader()->IsDirectionPeriodic(i);
}


bool iDataLimits::IsBoxPeriodic() const
{
	return mSubject->GetLoader()->IsBoxPeriodic();
}


void iDataLimits::NotifyDependencies(int var)
{
	if(!mBlockNotifications)
	{
		mSubject->GetViewModule()->NotifyDataConsumers(mSubject->Request(var));
	}
}

