/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icontrolmodule.h"


#include "ianimator.h"
#include "icamera.h"
#include "icolorbars.h"
#include "icommoneventobservers.h"
#include "icontrolscript.h"
#include "icrosssectionviewsubject.h"
#include "idata.h"
#include "idatareader.h"
#include "idatasubject.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "ierrorstatus.h"
#include "iextensionfactory.h"
#include "ifile.h"
#include "iimagecomposer.h"
#include "imarker.h"
#include "imarkerfamily.h"
#include "ipalette.h"
#include "iparallel.h"
#include "iparallelmanager.h"
#include "iparticlegroup.h"
#include "iparticlesviewsubject.h"
#include "ipicker.h"
#include "ipiecewisefunction.h"
#include "irendertool.h"
#include "ishell.h"
#include "isurfaceviewsubject.h"
#include "itensorfieldviewsubject.h"
#include "ivectorfieldviewsubject.h"
#include "iviewmodule.h"
#include "iviewobject.h"
#include "iviewobjectfamily.h"
#include "ivolumeviewsubject.h"

#include <vtkCamera.h>
#include <vtkDataObject.h>
#include <vtkInteractorStyle.h>
#include <vtkMapper.h>
#include <vtkRenderer.h>
#include <vtkRenderWindow.h>
#include <vtkRenderWindowInteractor.h>

//
//  templates
//
#include "iarraytemplate.h"
#include "iviewfamilytemplate.h"


using namespace iParameter;


int iControlModule::mIdCounter = 0;


IOBJECT_DEFINE_TYPE(iControlModule,ControlModule,cm,iObjectType::_Module);

IOBJECT_DEFINE_KEY(iControlModule,AutoRender,ar,Bool,1);
IOBJECT_DEFINE_KEY(iControlModule,OptimizationMode,om,Int,1);
IOBJECT_DEFINE_KEY(iControlModule,SynchronizeCameras,sc,Bool,1);
IOBJECT_DEFINE_KEY(iControlModule,SynchronizeInteractors,si,Bool,1);


#define BACKWARD_COMPATIBLE
#define COMPATIBILITY_MODE	12


//
//  Helper functions hidden in a private namespace
//
namespace iControlModule_Private
{
	iPalette* Palette_Rainbow();
	iPalette* Palette_Temperature();
	iPalette* Palette_Greyscale();
	iPalette* Palette_Bluewhite();
	iPalette* Palette_Prizm();
	iPalette* Palette_Greenwhite();
	iPalette* Palette_Bluered();
	iPalette* Palette_Stern();
	iPalette* Palette_Haze();
	iPalette* Palette_Starlight();
	iPalette* Palette_3color();
	iPalette* Palette_4color();

	template<class Family>
	bool SetCurrentObject(Family *fam, int obj)
	{
		if(obj>=0 && obj<=fam->GetMaxMemberIndex())
		{
			fam->SetCurrentMemberIndex(obj);
			return true;
		}
		else return false;
	}

	template<class Family>
	void ExecuteForObject(Family *fam, const iString &command, int objOption, bool &executeOk)
	{
		switch(objOption)
		{
		case _ObjectOptionOne:
			{
				fam->GetCurrentMember()->UnPackState(command);
				executeOk = fam->GetCurrentMember()->UnPackedSomething();
				break;
			}
		case _ObjectOptionAll:
			{
				int j;
				for(j=0; j<=fam->GetMaxMemberIndex(); j++)
				{
					fam->GetMember(j)->UnPackState(command);
					executeOk = executeOk || fam->GetMember(j)->UnPackedSomething();
				}
				break;
			}
		}
	}

	template<class Family>
	bool CreateObject(Family *fam, iViewModule *module)
	{
		int	i = fam->CreateMember();
		if(i == -1) return false;
		fam->GetMember(i)->CopyState(fam->GetCurrentMember());
		fam->Show(fam->IsVisible());
		fam->SetCurrentMemberIndex(i);
		module->ClearCache();
		return true;
	}

	template<class Family>
	bool DeleteObject(Family *fam, iViewModule *module, int obj)
	{
		if(obj == -1) obj = fam->GetCurrentMemberIndex();
		if(!fam->DeleteMember(obj)) return false;
		module->ClearCache();
		return true;
	}

	template<class Family>
	bool SaveObject(Family *fam, const iObjectType &type, iFile &F, const iString &prefix, int k)
	{
		int i;
		iString ws, state;

		for(i=0; i<=fam->GetMaxMemberIndex(); i++)
		{
			ws = prefix + type.FullName() + " " + iString::FromNumber(k) + " " + iString::FromNumber(i) + " ";
			fam->GetMember(i)->PackCompleteState(state);
			ws += state;
			if(!F.WriteLine(ws)) return false;
		}
		return true;
	}

	bool Error(iFile &F)
	{
		F.Close();
		return false;
	}

	bool OnQueryFailed()
	{
#ifdef I_DEBUG
		int ooo = 0;
#endif
		return false;
	}
};


using namespace iControlModule_Private;


//
//
#define LOAD_OBJECT(_getfam_,_type_,_show_) \
	if(ws == _type_.FullName()) \
	{ \
		k = line.Section(" ",1,1).ToInt(ok); \
		if(!ok) return Error(F); \
		if(k>=0 && k<nvm) \
		{ \
			iViewModule *vm = mViewModules->GetMember(k); \
			i = line.Section(" ",2,2).ToInt(ok); \
			if(!ok) return Error(F); \
			while(i > vm->_getfam_->GetMaxMemberIndex()) if(vm->_getfam_->CreateMember()==-1) return Error(F); \
			vm->_getfam_->GetMember(i)->UnPackCompleteState(line.Section(" ",3)); \
			this->GetShell()->OnInitAtom(); \
			if(_show_) \
			{ \
				vm->_getfam_->Show(_show_); \
				this->GetShell()->OnLoadStateFileAtom(60+(20*(kk+1)/nvm)); \
				k++; \
			} \
		} \
	}


//
//  Helper class. Probably should not be here.
//
class iInteractorEventObserver : public iEventObserver
{

	IPOINTER_AS_PART(ControlModule);

public:
	
	vtkTypeMacro(iInteractorEventObserver,iEventObserver);
	static iInteractorEventObserver* New(iControlModule *cm = 0)
	{
		IERROR_ASSERT(cm);
		return new iInteractorEventObserver(cm);
	}

	void Activate(bool s)
	{
		mActive = s;
	}

protected:

	iInteractorEventObserver(iControlModule *cm) : iEventObserver(cm->GetShell()), mControlModule(cm)
	{
		mInUpdate = mActive = false;
	}

	virtual void ExecuteBody(vtkObject *caller, unsigned long event1, void *callData)
	{
		if(!mActive || mInUpdate) return;

		vtkRenderWindowInteractor *iren, *driver = vtkRenderWindowInteractor::SafeDownCast(caller);
		if(this->GetControlModule()!=0 && driver!=0)
		{
			int i;
			mInUpdate = true;
			for(i=0; i<this->GetControlModule()->GetNumberOfViewModules(); i++) if((iren=this->GetControlModule()->GetViewModule(i)->GetInteractor()) != driver)
			{
				iren->SetControlKey(driver->GetControlKey());
				iren->SetShiftKey(driver->GetShiftKey());
				iren->SetEventPosition(driver->GetEventPosition());
				iren->SetKeySym(driver->GetKeySym());
				iren->InvokeEvent(event1,callData);
			}
			mInUpdate = false;
		}
	}

	bool mInUpdate, mActive;
};


//
//  Main class
//
iControlModule* iControlModule::New(iShell *s)
{
	IERROR_ASSERT(s);
	iControlModule *cm = new iControlModule(s);
	if(cm != 0)
	{
		cm->FinishInitialization();
	}
	return cm;
}


iControlModule::iControlModule(iShell *s) : iObject("ControlModule"), mShell(s)
{
	mId = mIdCounter++;

	mInExecute = false;

	mBlockBroadcasts = true;
	mAutoRender = mSyncInteractors = mExecuteOk = false;

	//
	//  Palettes must be created before the ViewModule family is created, since they are used then
	//
	this->CreateDefaultPalettes();

	//
	//  Parallel performance - this must be first, since many components are parallel workers as well
	//
	mParallelManager = iParallelManager::New(this); IERROR_ASSERT(mParallelManager);
	mParallelObserver = iParallelUpdateEventObserver::New(this); IERROR_ASSERT(mParallelObserver);
	mParallelManager->AddObserver(iParallel::InformationEvent,mParallelObserver);

	mCurrentSubject = 0;

	//
	//  Start with NULL view module family
	//
	mViewModules = 0;
}


void iControlModule::FinishInitialization()
{
	//
	//  Create observer for synchronizing interactors
	//
	mInteractorObserver = iInteractorEventObserver::New(this); IERROR_ASSERT(mInteractorObserver);
	//
	//  Create script - must be created before VM family, since VM Animators will be children of CM script
	//
	mControlScript = iControlScript::New(this);
	//
	//  Composer must be created before view modules
	//
	mComposer = iImageComposer::New(this); IERROR_ASSERT(mComposer);
	//
	//  ViewModule family must be created after ControlModule is fully created
	//
	mViewModules = iFamily<iViewModule,iControlModule>::New(this); IERROR_ASSERT(mViewModules);
	//
	//  Allow broadcasts
	//
	mBlockBroadcasts = false;
}


iControlModule::~iControlModule()
{
	int i;

	mControlScript->Delete();
	mComposer->Delete();
	mInteractorObserver->Delete();

	//
	//  Deleting view modules is tricky: we need to delete clones first so that
	//  data reader and limits are not deleted before the view object are.
	//
	for(i=0; i<=mViewModules->GetMaxMemberIndex(); i++) if(mViewModules->GetMember(i)->IsClone()) mViewModules->DeleteMember(i);
	mViewModules->Delete();

	mParallelObserver->Delete();
	mParallelManager->Delete();

	for(i=0; i<mPalList.Size(); i++) delete mPalList[i];
}


bool iControlModule::Execute(const iString &command, bool render, int option, int mod)
{
	if(mInExecute) return true; // Protect from concurrent calls from a multitheaded shell
	mInExecute = true;

	int j;
	mExecuteOk = false;

	render = render || mAutoRender;

	//
	//  Is it a ControlModule command?
	//
	iString pre = command.Section(iObjectKey::PrefixSeparator(),0,0);
	if(iControlModule::Type().IsMatch(pre)) 
	{
		iObject::ReportMissingKeys(false); //commands are not complete states
		this->UnPackState(command);
		iObject::ReportMissingKeys(true);
		mExecuteOk = true;

		if(render)
		{
			for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) 
			{
				mViewModules->GetMember(j)->Render();
			}
		}
		mInExecute = false;
		return mExecuteOk;
	}

	//
	//  Set modes
	//
	int objOption = option & _ObjectOptionMask;
	int modOption = option & _ModuleOptionMask;
	int renOption = option & _RenderOptionMask;

	//
	//  Check that modes are valid
	//
	if(
		(objOption!=_ObjectOptionOne && objOption!=_ObjectOptionAll) ||
		(modOption!=_ModuleOptionOne && modOption!=_ModuleOptionAll && modOption!=_ModuleOptionClones) || 
		(renOption!=_RenderOptionOne && renOption!=_RenderOptionAll && renOption!=_RenderOptionClones && renOption!=_RenderOptionAuto))
	{
		IERROR_LOW("Invalid option.");
		//
		//  Continue after an error with default ones
		//
		objOption = _ObjectOptionOne;
		modOption = _ModuleOptionOne;
		renOption = _RenderOptionAuto;
	}

	//
	//  Loop over view modules
	//
	iDataReader *r = mViewModules->GetCurrentMember()->GetReader();

	iObject::ReportMissingKeys(false); //commands are not complete states
	//
	//  Execute and Set renOption to prefered value if it is Set to Auto initially
	//
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) 
	{
		switch(modOption)
		{
		case _ModuleOptionOne:
			{
				if(j == mod) this->ExecuteForViewModule(mViewModules->GetMember(j),command,render,objOption,renOption);
				break;
			}
		case _ModuleOptionAll:
			{
				if(renOption == _RenderOptionAuto) renOption = _RenderOptionAll;
				this->ExecuteForViewModule(mViewModules->GetMember(j),command,render,objOption,renOption);
				break;
			}
		case _ModuleOptionClones:
			{
				if(renOption == _RenderOptionAuto) renOption = _RenderOptionClones;
				if(r == mViewModules->GetMember(j)->GetReader()) this->ExecuteForViewModule(mViewModules->GetMember(j),command,render,objOption,renOption);
				break;
			}
		}
	}

	iObject::ReportMissingKeys(true);
#ifdef I_CHECK1
	if(renOption == _RenderOptionAuto) IERROR_LOW("Improper RenderOption.");
#endif

	//
	//  Execute changes the state of the object. So, if the command is non-empty, we need to
	//  clear the cached state
	//
	if(!command.IsEmpty())
	{
		this->ClearCache();
	}
	
	//
	//  Don't render if not asked explicitly
	//
	if(!render)
	{
		mInExecute = false;
		return mExecuteOk;
	}

	//
	//  Render
	//
	for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) 
	{
		switch(renOption)
		{
		case _RenderOptionOne:
			{
				if(j == mod) mViewModules->GetMember(j)->Render();
				break;
			}
		case _RenderOptionAll:
			{
				mViewModules->GetMember(j)->Render();
				break;
			}
		case _RenderOptionClones:
			{
				if(r == mViewModules->GetMember(j)->GetReader()) mViewModules->GetMember(j)->Render();
				break;
			}
		}
	}

	mInExecute = false;
	return mExecuteOk;
}


iObject* iControlModule::FindObject(const iObjectType &type, int mod, int obj)
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return 0;

	iObject *object = this->FindSingleObject(type,module);
	if(object != 0) return object;

	if(type.IsMatch(iMarker::Type()))
	{
		return module->GetMarker(obj);
	}

	if(type.IsMatch(iParticleGroup::Type()))
	{
		if(obj == -1) return module->GetParticlesViewSubject()->GetCurrentGroup(); else return module->GetParticlesViewSubject()->GetGroup(obj);
	}

	object = this->FindViewSubject(type,module,obj);
	return object;
}


iObject* iControlModule::FindSingleObject(const iObjectType &type, iViewModule *module)
{
	if(type.IsMatch(iControlModule::Type()))
	{
		return this;
	}
	if(type.IsMatch(iViewModule::Type()))
	{
		return module;
	}
	if(type.IsMatch(iAnimator::Type()))
	{
		return module->GetAnimator();
	}
	if(type.IsMatch(iCamera::Type()))
	{
		return module->GetRenderTool()->GetCamera();
	}
	if(type.IsMatch(iColorBars::Type()))
	{
		return module->GetColorBars();
	}
	if(type.IsMatch(iDataReader::Type()))
	{
		return module->GetReader();
	}
	if(iDataSubject::IsTypeMatch(type.FullName()))
	{
		mDataSubjectName = iDataSubject::GetDataSubjectName(type.FullName());
		return module->GetReader()->GetSubject(iDataType::FindTypeByName(mDataSubjectName));
	}
	if(type.IsMatch(iImageComposer::Type()))
	{
		return mComposer;
	}
	if(type.IsMatch(iPicker::Type()))
	{
		return module->GetPicker();
	}

	return 0;
}


iViewObjectFamily* iControlModule::FindViewObjectFamily(const iObjectType &type, iViewModule *module) const
{
	int i = 0;
	iViewObjectFamily *fam;

	while((fam = module->GetViewObjectFamily(i++)) != 0)
	{
		if(type.IsMatch(fam->GetParent()->GetObjectType()))
		{
			return fam;
		}
	}

	return 0;
}


iViewSubject* iControlModule::FindViewSubject(const iObjectType &type, iViewModule *module, int obj) const
{
	iViewObjectFamily *fam = this->FindViewObjectFamily(type,module);
	if(fam != 0)
	{
		if(obj == -1) return fam->GetCurrentMember()->GetViewSubject(); else return fam->GetMember(obj)->GetViewSubject();
	}
	return 0;
}


void iControlModule::ExecuteForViewModule(iViewModule *module, const iString &command, bool /*render*/, int objOption, int &renOption)
{
	const iObjectType *ptype = iObjectTypeRegistry::FindType(command.Section(iObjectKey::PrefixSeparator(),0,0));
	if(ptype == 0)
	{
		if(renOption == _RenderOptionAuto) renOption = _RenderOptionOne;
		return;
	}

	iObject *object = this->FindSingleObject(*ptype,module);
	if(object != 0)
	{
		this->GetErrorStatus()->Monitor(object->GetErrorStatus());
		object->UnPackState(command);
		mExecuteOk = object->UnPackedSomething();
		if(renOption == _RenderOptionAuto) renOption = _RenderOptionOne;

		if(ptype->IsMatch(iDataReader::Type()) || ptype->IsMatch(iDataSubject::Type()))
		{
			if(renOption == _RenderOptionAuto) renOption = _RenderOptionClones;
		}

		return;
	}

	if(ptype->IsMatch(iMarker::Type()))
	{
		iMarkerFamily *fam = module->GetMarkerFamily();
		ExecuteForObject(fam,command,objOption,mExecuteOk);
		if(mExecuteOk) module->GetMarkerFamily()->UpdateLegend();
		if(renOption == _RenderOptionAuto) renOption = _RenderOptionOne;
		return;
	}

	if(ptype->IsMatch(iParticleGroup::Type()))
	{
		iParticlesViewSubject *fam = module->GetParticlesViewSubject();
		ExecuteForObject(fam,command,objOption,mExecuteOk);
		if(renOption == _RenderOptionAuto) renOption = _RenderOptionOne;
		return;
	}

	iViewObjectFamily *fam = this->FindViewObjectFamily(*ptype,module);
	if(fam != 0)
	{
		ExecuteForObject(fam,command,objOption,mExecuteOk);
	}

	if(renOption == _RenderOptionAuto) renOption = _RenderOptionOne;
}


bool iControlModule::CreateObject(const iObjectType &type, bool render, int mod, int vmtype)
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return false;

	render = render || mAutoRender;

	if(type.IsMatch(iViewModule::Type()))
	{
		int i = mViewModules->CreateMember(); 
		if(i != -1) 
		{
			//
			//  Set the type of the instance created
			//
			if(vmtype==_ModuleTypeClone || vmtype==_ModuleTypeCopy)
			{
				mViewModules->GetMember(i)->CopyState(module);
				mViewModules->GetMember(i)->BecomeClone(module);
			}
			//
			//  If a copy, unclone
			//
			if(vmtype == _ModuleTypeCopy) mViewModules->GetMember(i)->BecomeClone(0);
			//
			//  Update composer
			//
			mComposer->UpdateWindowList();
			if(render) mViewModules->GetMember(i)->Render();
			int j;
			for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) mViewModules->GetMember(j)->UpdateWindowNumber();
			mViewModules->GetCurrentMember()->UpdateIcon();
		}
		return i != -1;
	}

	if(type.IsMatch(iMarker::Type()))
	{
		iMarkerFamily *fam = module->GetMarkerFamily();
		if(fam->GetMaxMemberIndex()==0 && !fam->IsVisible())
		{
			fam->Show(true);
			module->ClearCache();
		}
		else
		{
			if(!iControlModule_Private::CreateObject(fam,module)) return false;
		}
		if(render) module->Render();
		return true;
	}

	if(type.IsMatch(iParticleGroup::Type()))
	{
		int i = module->GetParticlesViewSubject()->CreateGroup();
		if(i == -1) return false;
		module->ClearCache();
		if(render) module->Render();
		return true;
	}

	iViewObjectFamily *fam = this->FindViewObjectFamily(type,module);
	if(fam!=0 && !fam->IsSingleInstance())
	{
		return iControlModule_Private::CreateObject(fam,module);
	}

#ifdef I_CHECK1
	IERROR_LOW("This object cannot be created.");
#endif
	return false;
}


bool iControlModule::DeleteObject(const iObjectType &type, bool render, int mod, int obj)
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return false;

	if(type.IsMatch(iViewModule::Type()))
	{
		int j;
		bool ok;
		//
		//  First check whether it is the only non-cloned one, then it cannot be deleted
		//
		if(!module->IsClone())
		{
			ok = true;
			for(j=0; ok && j<=mViewModules->GetMaxMemberIndex(); j++) if(!mViewModules->GetMember(j)->IsClone()) ok = false;
			if(ok) return false;
		}

		//
		//  If this is not a clone, delete all its clones
		//
		if(!module->IsClone())
		{
			while(module->mClones.Size() > 0)
			{
				if(!mViewModules->DeleteMember(module->mClones[0]->GetWindowNumber()))
				{
					for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) mViewModules->GetMember(j)->UpdateWindowNumber();
					mComposer->UpdateWindowList();
					return false;
				}
			}
		}
		//
		//  Now delete the window itself
		//
		ok = mViewModules->DeleteMember(mod);
		for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) mViewModules->GetMember(j)->UpdateWindowNumber();
		mComposer->UpdateWindowList();

		this->SetCurrentViewModuleIndex(-1);

		return ok;
	}

	if(type.IsMatch(iMarker::Type()))
	{
		iMarkerFamily *fam = module->GetMarkerFamily();
		if(fam->GetMaxMemberIndex()==0 && fam->IsVisible())
		{
			fam->Show(false);
			module->ClearCache();
		}
		else
		{
			iControlModule_Private::DeleteObject(fam,module,obj);
		}
		if(render) module->Render();
		return true;
	}

	if(type.IsMatch(iParticleGroup::Type()))
	{
		if(obj == -1) obj = module->GetParticlesViewSubject()->GetCurrentGroupIndex();
		if(!module->GetParticlesViewSubject()->DeleteGroup(obj)) return false;
		module->ClearCache();
		if(render) module->Render();
		return true;
	}

	iViewObjectFamily *fam = this->FindViewObjectFamily(type,module);
	if(fam!=0 && !fam->IsSingleInstance())
	{
		iControlModule_Private::DeleteObject(fam,module,obj);
		return true;
	}

#ifdef I_CHECK1
	IERROR_LOW("This object cannot be deleted.");
#endif
	return false;
}


bool iControlModule::SetCurrentViewModuleIndex(int mod)
{
	int oldcw = mViewModules->GetCurrentMemberIndex();

	if(mod>=-1 && mod<=mViewModules->GetMaxMemberIndex() && mod!=oldcw)
	{
		mViewModules->SetCurrentMemberIndex(mod);
		mViewModules->GetMember(oldcw)->UpdateIcon();
		mViewModules->GetCurrentMember()->UpdateIcon();
		return true;
	}
	else return (mod == oldcw);
}


void iControlModule::Show(const iObjectType &type, bool show, bool render, int mod)
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return;

	if(type.IsMatch(iMarker::Type()))
	{
		module->GetMarkerFamily()->Show(show);
		if(render || mAutoRender) module->Render();
		return;
	}

	iViewObjectFamily *fam = this->FindViewObjectFamily(type,module);
	if(fam != 0)
	{
		fam->Show(show);
		if(render || mAutoRender) module->Render();
		return;
	}
}


float iControlModule::GetMemorySize(int mod) const
{
	int i;
	float s = 0.0;
	if(mod == -2)
	{
		for(i=0; i<=mViewModules->GetMaxMemberIndex(); i++) s += mViewModules->GetMember(i)->GetMemorySize();
	}
	else if(mod>-2 && mod<=mViewModules->GetMaxMemberIndex())
	{
		if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
		mViewModules->GetMember(mod)->GetMemorySize();
	}
	return s;
}


float iControlModule::GetUpdateTime(int mod) const
{
	float s = 0.0;
	if(mod>-1 && mod<=mViewModules->GetMaxMemberIndex())
	{
		if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
		mViewModules->GetMember(mod)->GetUpdateTime();
	}
	return s;
}


void iControlModule::FlyTo(const iObjectType &type, float time, int mod, int obj) const
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return;

	const double *pos;

	iViewSubject *object = this->FindViewSubject(type,module,obj);
	if(object != 0)
	{
		pos = object->GetPosition();
	}
	else
	{
		if(type == iPicker::Type()) pos = module->GetPicker()->GetPosition(); else return;
	}

	this->FlyTo(pos,time,mod);
}


void iControlModule::FlyTo(const double *pos, float time, int mod) const
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module==0 || pos==0) return;

	if(time > 1.0e-30)
	{
		//
		//  Gradual fly-to
		//
		int n = round(time/module->GetRenderer()->GetLastRenderTimeInSeconds());
		if(n > 60) n = 60;
		if(n < 10) n = 0;
		module->GetInteractor()->SetNumberOfFlyFrames(n);
		module->GetInteractor()->FlyTo(module->GetRenderer(),pos[0],pos[1],pos[2]);
	}
	else
	{
		//
		//  A jump
		//
		int j;
		vtkCamera *cam = module->GetRenderer()->GetActiveCamera();
		double cf[3], cp[3];

		cam->GetPosition(cp);
		cam->GetFocalPoint(cf);
		for(j=0; j<3; j++) cp[j] += (pos[j]-cf[j]);
		cam->SetFocalPoint(pos);
		cam->SetPosition(cp);
		module->Render();
	}
}


void iControlModule::PackStateBody(iString &s) const
{
	this->PackValue(s,KeyAutoRender(),mAutoRender);
	this->PackValue(s,KeySynchronizeInteractors(),mSyncInteractors);
	this->PackValue(s,KeyOptimizationMode(),this->GetOptimizationMode());
}


void iControlModule::UnPackStateBody(const iString &s)
{
	int i; bool b;

	if(this->UnPackValue(s,KeyAutoRender(),b)) this->SetAutoRender(b);
	if(this->UnPackValue(s,KeySynchronizeInteractors(),b)) this->SetSynchronizeInteractors(b);
	if(this->UnPackValue(s,KeyOptimizationMode(),i)) this->SetOptimizationMode(i);

	//
	//  Action keys
	//
	iObject::ReportMissingKeys(false); //action keys are not part of the states

	if(this->UnPackValue(s,KeySynchronizeCameras(),b) && b) this->SynchronizeCameras();

	iObject::ReportMissingKeys(true);
}


void iControlModule::QueryState(const iObjectType &type, iString &state, int vis, int obj)
{ 
	iObject *o = this->FindObject(type,vis,obj);
	o->PackState(state);
}


//
//  Give access to piecewise functions directly for higher efficiency
//
bool iControlModule::QueryValue(const iObjectKey &key, iFunctionMapping* &f, int mod, int obj) const
{ 
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return OnQueryFailed();

	if(key.Type().IsMatch(iParticleGroup::Type()))
	{
		if(obj == -1) obj = module->GetParticlesViewSubject()->GetCurrentGroupIndex();
		if(key == iParticleGroup::KeySizeFunction())
		{
			f = module->GetParticlesViewSubject()->GetGroup(obj)->GetSizeFunction();
			return true;
		}
	}

	if(key.Type().IsMatch(iParticlesViewSubject::Type()))
	{
		if(obj == -1) obj = module->GetParticlesViewSubject()->GetCurrentGroupIndex();
		if(key == iParticlesViewSubject::KeySizeFunction())
		{
			f = module->GetParticlesViewSubject()->GetGroup(obj)->GetSizeFunction();
			return true;
		}
	}
	
	if(key.Type().IsMatch(iVolumeViewSubject::Type()))
	{
		if(obj == -1) obj = module->GetViewObjectFamily(_ObjectTypeVolume)->GetCurrentMemberIndex();
		if(key == iVolumeViewSubject::KeyOpacityFunction())
		{
			f = module->GetVolumeViewSubject(obj)->GetOpacityFunction();
			return true;
		}
	}

	f = 0;
	return OnQueryFailed();
}


//
//  Give access to range collections directly for higher efficiency
//
bool iControlModule::QueryValue(const iObjectKey &key, iRangeMapping* &f, int mod, int /*obj*/) const
{ 
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return OnQueryFailed();

	if(key.Type().IsMatch(iParticlesViewSubject::Type()))
	{
		if(key == iParticlesViewSubject::KeySplitRanges())
		{
			f = module->GetParticlesViewSubject()->GetSplitRanges();
			return true;
		}
	}

	f = 0;
	return OnQueryFailed();
}


//
//  Query images
//
bool iControlModule::QueryValue(const iObjectKey &key, iImage &im, int mod, int /*obj*/) const
{ 
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return OnQueryFailed();

	if(key.Type().IsMatch(iAnimator::Type()))
	{
		if(key == iAnimator::KeyTitlePageImage())
		{
			im = module->GetAnimator()->GetTitlePageImage();
			return true;
		}
		if(key == iAnimator::KeyLogoImage())
		{
			im = module->GetAnimator()->GetLogoImage();
			return true;
		}
	}

	return OnQueryFailed();
}


//
//  Get any query as a portion of the state string
//
bool iControlModule::QueryValueAsString(const iObjectKey &key, iString &val, int mod, int obj)
{
	iObject *o = this->FindObject(key,mod,obj);
	if(o != 0)
	{
		iString ws;
		o->PackState(ws);
		int index, ind = key.FindInString(ws,index);
		if(ind >=0)
		{
			if(index < 0)
			{
				val = key.FieldSeparator() + ws.Part(ind).Section(key.FieldSeparator(),1).Section(key.FieldSeparator()+key.Delimiter(),0,0) + key.FieldSeparator();
			}
			else
			{
				val = ws.Part(ind).Section(key.FieldSeparator(),index+1,index+1);
			}
			return true;
		}
		else return OnQueryFailed();
	}
	else return OnQueryFailed();
}


//
//  generic QueryValue functions
//
#define ICONTROLMODULE_QUERYVALUE_DEFINITION(_type_) \
	bool iControlModule::QueryValue(const iObjectKey &key, _type_ *val, int n, int mod, int obj) \
	{ \
		iObject *o = this->FindObject(key,mod,obj); \
		if(o != 0) \
		{ \
			iString ws; \
			o->PackState(ws); \
			iObject::ReportMissingKeys(false); \
			bool ret = o->UnPackValue(ws,key,val,n); \
			iObject::ReportMissingKeys(true); \
			return (ret?true:OnQueryFailed()); \
		} \
		else return OnQueryFailed(); \
	} \
	bool iControlModule::QueryValue(const iObjectKey &key, int index, _type_ &val, int n, int mod, int obj) \
	{ \
		if(index<0 || index>=n) return OnQueryFailed(); \
		iObject *o = this->FindObject(key,mod,obj); \
		if(o != 0) \
		{ \
			iString ws; \
			o->PackState(ws); \
			_type_ *tmp = new _type_[n]; IERROR_ASSERT(tmp); \
			iObject::ReportMissingKeys(false); \
			bool ret = o->UnPackValue(ws,key,tmp,n); \
			iObject::ReportMissingKeys(true); \
			val = tmp[index]; \
			delete [] tmp; \
			return (ret?true:OnQueryFailed()); \
		} \
		else return OnQueryFailed(); \
	} 


//
//  Packing/unpacking helpers - mostly moved iObject helpers into public interface
//
#define ICONTROLMODULE_PACKCOMMAND_DEFINITION(_type1_,_type2_) \
	void iControlModule::PackCommand(iString &s, const iObjectKey &key, _type1_ *val, int n) const \
	{ \
		this->PackValue(s,key,val,n,-1,true); \
	} \
	void iControlModule::PackCommand(iString &s, const iObjectKey &key, _type2_ val) const \
	{ \
		this->PackValue(s,key,val,-1,true); \
	} \
	void iControlModule::PackCommand(iString &s, const iObjectKey &key, int index, _type2_ val) const \
	{ \
		this->PackValue(s,key,val,index,true); \
	}


ICONTROLMODULE_QUERYVALUE_DEFINITION(int);
ICONTROLMODULE_QUERYVALUE_DEFINITION(bool);
ICONTROLMODULE_QUERYVALUE_DEFINITION(float);
ICONTROLMODULE_QUERYVALUE_DEFINITION(double);
ICONTROLMODULE_QUERYVALUE_DEFINITION(iColor);
ICONTROLMODULE_QUERYVALUE_DEFINITION(iString);


ICONTROLMODULE_PACKCOMMAND_DEFINITION(int,int);
ICONTROLMODULE_PACKCOMMAND_DEFINITION(bool,bool);
ICONTROLMODULE_PACKCOMMAND_DEFINITION(float,float);
ICONTROLMODULE_PACKCOMMAND_DEFINITION(double,double);
ICONTROLMODULE_PACKCOMMAND_DEFINITION(iColor,iColor);
ICONTROLMODULE_PACKCOMMAND_DEFINITION(iString,const iString&);


//
// Palette manipulation
//
void iControlModule::CreateDefaultPalettes()
{
	if(mPalList.Size() > 0) return;

	mPalList.Add(Palette_Rainbow());
	mPalList.Add(Palette_Temperature());
	mPalList.Add(Palette_Bluewhite());
	mPalList.Add(Palette_Prizm());
	mPalList.Add(Palette_Starlight());
	mPalList.Add(Palette_Greenwhite());
	mPalList.Add(Palette_Bluered());
	mPalList.Add(Palette_Haze());
	mPalList.Add(Palette_Stern());
	mPalList.Add(Palette_Greyscale());
	mPalList.Add(Palette_3color());
	mPalList.Add(Palette_4color());
}


iPalette* iControlModule::GetPalette(int n) const
{
	return mPalList[n];
}


int iControlModule::GetNumberOfPalettes() const
{ 
	return mPalList.Size(); 
}


bool iControlModule::AddEmptyPalette()
{
	iPalette *tmp = new iPalette; if(tmp == 0) return false;
	mPalList.Add(tmp);
	return true;
}


//
//  Saving and restoring the state
//
bool iControlModule::SaveStateToFile(const iString &filename) const
{
    int i, j, k;
	iViewObjectFamily *fam;
	iString prefix = "";
	iString state, ws;
	iFile F(filename);
	
	if(!F.Open(iFile::_Write,iFile::_Text)) return false;
	
	this->GetShell()->OnLoadStateFileAtom(0);

	//
	//  Compatibility mode
	//
	if(!F.WriteLine("CompatibilityMode "+iString::FromNumber(COMPATIBILITY_MODE))) return Error(F);
	//
	//  Save the state of this ControlModule
	//
	this->PackCompleteState(state);
	ws = prefix + iControlModule::Type().FullName() + " " + state;
	if(!F.WriteLine(ws)) return Error(F);
	//
	//  Save ViewModules with their window geometries, animators, data readers, limits,
	//  particle splitters, and all view objects
	//
	this->GetShell()->OnLoadStateFileAtom(5);
	for(k=0; k<=mViewModules->GetMaxMemberIndex(); k++) 
	{
		//
		//  Visual module location
		//
		ws = prefix + iViewModule::Type().FullName() + " " + iString::FromNumber(k) + " " + iString::FromNumber(mViewModules->GetMember(k)->GetRenderWindow()->GetPosition()[0]) + " " + iString::FromNumber(mViewModules->GetMember(k)->GetRenderWindow()->GetPosition()[1]) + " ";
		//
		//  Visual module parameters
		//
		mViewModules->GetMember(k)->PackCompleteState(state);
		ws += state;
		if(!F.WriteLine(ws)) return Error(F);
		this->GetShell()->OnLoadStateFileAtom(5+(75*k+5)/(mViewModules->GetMaxMemberIndex()+1));
		//
		//  Animator
		//
		ws = prefix + iAnimator::Type().FullName() + " " + iString::FromNumber(k) + " ";
		mViewModules->GetMember(k)->GetAnimator()->PackCompleteState(state);
		ws += state;
		if(!F.WriteLine(ws)) return Error(F);
		this->GetShell()->OnLoadStateFileAtom(5+(75*k+10)/(mViewModules->GetMaxMemberIndex()+1));
		//
		//  Camera
		//
		ws = prefix + iCamera::Type().FullName() + " " + iString::FromNumber(k) + " ";
		mViewModules->GetMember(k)->GetRenderTool()->GetCamera()->PackCompleteState(state);
		ws += state;
		if(!F.WriteLine(ws)) return Error(F);
		this->GetShell()->OnLoadStateFileAtom(5+(75*k+15)/(mViewModules->GetMaxMemberIndex()+1));
		//
		//  Color Bars
		//
		ws = prefix + iColorBars::Type().FullName() + " " + iString::FromNumber(k) + " ";
		mViewModules->GetMember(k)->GetColorBars()->PackCompleteState(state);
		ws += state;
		if(!F.WriteLine(ws)) return Error(F);
		this->GetShell()->OnLoadStateFileAtom(5+(75*k+20)/(mViewModules->GetMaxMemberIndex()+1));
		//
		//  Data reader
		//
		ws = prefix + iDataReader::Type().FullName() + " " + iString::FromNumber(k) + " ";
		mViewModules->GetMember(k)->GetReader()->PackCompleteState(state);
		ws += state;
		if(!F.WriteLine(ws)) return Error(F);
		this->GetShell()->OnLoadStateFileAtom(5+(75*k+25)/(mViewModules->GetMaxMemberIndex()+1));
		//
		//  Picker
		//
		ws = prefix + iPicker::Type().FullName() + " " + iString::FromNumber(k) + " ";
		mViewModules->GetMember(k)->GetPicker()->PackCompleteState(state);
		ws += state;
		if(!F.WriteLine(ws)) return Error(F);
		this->GetShell()->OnLoadStateFileAtom(5+(75*k+30)/(mViewModules->GetMaxMemberIndex()+1));
		//
		//  Save view objects
		//
		for(j=0; j<mViewModules->GetMember(k)->GetNumberOfViewObjectFamilies(); j++)
		{
			fam = mViewModules->GetMember(k)->GetViewObjectFamily(j); 
			if(!SaveObject(fam,fam->GetParent()->GetObjectType(),F,prefix,k)) return Error(F);
			this->GetShell()->OnLoadStateFileAtom(5+(75*k+30+(40*(j+1))/mViewModules->GetMember(k)->GetNumberOfViewObjectFamilies())/(mViewModules->GetMaxMemberIndex()+1));
		}
		//
		//  Don't save single invisible marker
		//
		if(mViewModules->GetMember(k)->GetMarkerFamily()->GetMaxMemberIndex()>0 || mViewModules->GetMember(k)->GetMarkerFamily()->GetMember(0)->IsVisible())
		{
			if(!SaveObject(mViewModules->GetMember(k)->GetMarkerFamily(),iMarker::Type(),F,prefix,k)) return Error(F);
		}
		this->GetShell()->OnLoadStateFileAtom(5+(75*k+75)/(mViewModules->GetMaxMemberIndex()+1));
	}
	//
	//  Save the state of ImageComposer
	//
	mComposer->PackCompleteState(state);
	ws = prefix + iImageComposer::Type().FullName() + " " + state;
	if(!F.WriteLine(ws)) return Error(F);
	this->GetShell()->OnLoadStateFileAtom(85);
	//
	//  Save the palettes
	//
	for(i=0; i<mPalList.Size(); i++)
	{
		this->PackValuePalette(ws,mPalList[i]);
		ws = prefix + "Palette " + iString::FromNumber(i) + " " + ws;
		if(!F.WriteLine(ws)) 
		{
			F.Close();
			return false;
		}
	}
	
	this->GetShell()->OnLoadStateFileAtom(90);

	bool ret = this->GetShell()->SaveShellStateToFile(F);

	this->GetShell()->OnLoadStateFileAtom(100);

	F.Close();	
	return ret;
}


bool iControlModule::LoadStateFromFile(const iString &filename)
{
	bool ok;
	int k, i, j, i1, i2;
	int CompatibilityMode = 0;
	iViewObjectFamily *fam;
	iString line, ws;
	iFile F(filename);
	
	if(!F.Open(iFile::_Read,iFile::_Text)) return false;
	
	//
	// ***************** Pre-Pass ****************************
	//
	//  Find the Compatibility parameter and restore this control module
	//
	F.Rewind();
	while(F.ReadLine(line)) 
	{
		line.ReduceWhiteSpace();
		ws = line.Section(" ",0,0);
		
		if(ws == "CompatibilityMode")
		{
			i = line.Section(" ",1,1).ToInt(ok);
			if(!ok || i<10 || i>COMPATIBILITY_MODE) return Error(F);
			CompatibilityMode = i;
			break;
		}

		if(ws == iControlModule::Type().FullName()) 
		{
			this->UnPackCompleteState(line.Section(" ",1));
			this->GetShell()->OnInitAtom();
		}
	}
	this->GetShell()->OnLoadStateFileAtom(5);
	//
	// ***************** Pass 1 ****************************
	//
	//  Count the number of view modules.
	//
	F.Rewind();
	int nvm = 0, kvmMax = 0;
	while(F.ReadLine(line))
	{
		line.ReduceWhiteSpace();
		ws = line.Section(" ",0,0);
		
		if(ws == iViewModule::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) return Error(F);
			if(k > kvmMax) kvmMax = k;
			nvm++;
		}
	}
	
	if(kvmMax+1 != nvm) return Error(F);
	//
	//  Create missing view modules, destroy extra ones.
	//
	while(mViewModules->GetMaxMemberIndex() > kvmMax)
	{
		if(!mViewModules->DeleteMember(mViewModules->GetMaxMemberIndex())) return Error(F);
	}
	this->GetShell()->OnLoadStateFileAtom(10);

	while(mViewModules->GetMaxMemberIndex() < kvmMax)
	{
		if(mViewModules->CreateMember() == -1) return Error(F);
		this->GetShell()->OnLoadStateFileAtom(10+(30*(mViewModules->GetMaxMemberIndex()+1)/nvm));
	}
	//
	//  set all window numbers
	//
	for(k=0; k<nvm; k++)
	{
		mViewModules->GetMember(k)->UpdateWindowNumber();
	}
	//
	// ***************** Pass 2 ****************************
	//
	//  Restore view modules
	//
	int kk = 0;
	F.Rewind();
	while(F.ReadLine(line)) 
	{
		line.ReduceWhiteSpace();
		ws = line.Section(" ",0,0);
		
		if(ws == iViewModule::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) return Error(F);
			if(k>=0 && k<nvm)
			{		
				//
				//  Visual module position
				//
				i1 = line.Section(" ",2,2).ToInt(ok);
				if(!ok) return Error(F);
				i2 = line.Section(" ",3,3).ToInt(ok);
				if(!ok) return Error(F);
				mViewModules->GetMember(k)->GetRenderWindow()->SetPosition(i1,i2);
				//
				//  View module parameters
				//
				mViewModules->GetMember(k)->UnPackCompleteState(line.Section(" ",4));
				this->GetShell()->OnInitAtom();
				this->GetShell()->OnLoadStateFileAtom(40+(10*(kk+1)/nvm));
				kk++;
			}
		}
	}
	this->GetShell()->OnLoadStateFileAtom(50);
	//	
	// ***************** Pass 3 ****************************
	//
	//  Restore animators, cameras, color bars, data readers, and pickers
	//
	kk = 0;
	F.Rewind();
	while(F.ReadLine(line)) 
	{
		line.ReduceWhiteSpace();
		ws = line.Section(" ",0,0);
		
		if(ws == iAnimator::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) return Error(F);
			if(k>=0 && k<nvm)
			{		
				mViewModules->GetMember(k)->GetAnimator()->UnPackCompleteState(line.Section(" ",2));
				this->GetShell()->OnInitAtom();
			}
		}		
	
		if(ws == iCamera::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) return Error(F);
			if(k>=0 && k<nvm)
			{		
				mViewModules->GetMember(k)->GetRenderTool()->GetCamera()->UnPackCompleteState(line.Section(" ",2));
				this->GetShell()->OnInitAtom();
			}
		}		
		
		if(ws == iColorBars::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) return Error(F);
			if(k>=0 && k<nvm)
			{		
				mViewModules->GetMember(k)->GetColorBars()->UnPackCompleteState(line.Section(" ",2));
				this->GetShell()->OnInitAtom();
			}
		}		
		
		if(ws == iDataReader::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) return Error(F);
			if(k>=0 && k<nvm)
			{		
				mViewModules->GetMember(k)->GetReader()->UnPackCompleteState(line.Section(" ",2));
				this->GetShell()->OnInitAtom();
			}
		}		

		if(ws == iPicker::Type().FullName()) 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) return Error(F);
			if(k>=0 && k<nvm)
			{		
				mViewModules->GetMember(k)->GetPicker()->UnPackCompleteState(line.Section(" ",2));
				this->GetShell()->OnInitAtom();
				this->GetShell()->OnLoadStateFileAtom(50+(10*(kk+1)/nvm));
			}
		}		
	}
	this->GetShell()->OnLoadStateFileAtom(60);
	//	
	// ***************** Pass 4 ****************************
	//
	//  Restore view objects, palettes, and the image composer
	//
	kk = 0;
	F.Rewind();
	while(F.ReadLine(line)) 
	{
		line.ReduceWhiteSpace();
		ws = line.Section(" ",0,0);
		
		j = 0;
		while((fam = mViewModules->GetMember(0)->GetViewObjectFamily(j)) != 0)
		{
			LOAD_OBJECT(GetViewObjectFamily(j),fam->GetParent()->GetObjectType(),false);
			j++;
		}

		LOAD_OBJECT(GetMarkerFamily(),iMarker::Type(),true);
		
		if(ws == "Palette") 
		{
			k = line.Section(" ",1,1).ToInt(ok);
			if(!ok) return Error(F);
			ok = true;
			if(k<0 || k>=100000) ok = false;
	
			//
			//  Do not read reversed palettes saved under CompatibilityMode=11
			//
			if(CompatibilityMode==11 && line.Section(" ",2,2).Section("/",0,0).Find("*reversed")>-1) ok = false;

			while(k>=mPalList.Size() && ok)
			{
				ok = this->AddEmptyPalette();
			}
			if(ok)
			{
				ws = line.Section(" ",2);
				if(!this->UnPackValuePalette(ws,mPalList[k])) return Error(F);
			}
		}		

		if(ws == iImageComposer::Type().FullName()) 
		{
			mComposer->UnPackCompleteState(line.Section(" ",1));
			this->GetShell()->OnInitAtom();
		}
	}

	this->GetShell()->OnLoadStateFileAtom(80);

	for(k=0; k<=mViewModules->GetMaxMemberIndex(); k++) 
	{
		j = 0;
		while((fam = mViewModules->GetMember(k)->GetViewObjectFamily(j++)) != 0)
		{
			fam->Show(false);
		}
	}

	this->GetShell()->OnLoadStateFileAtom(90);

	bool ret = this->GetShell()->LoadShellStateFromFile(F);

	this->GetShell()->OnLoadStateFileAtom(100);

	F.Close();
	return ret;
}


void iControlModule::PackValuePalette(iString &s, const iPalette *p) const
{
	iViewSubject *tmpObject = mViewModules->GetMember(0)->GetMarkerFamily()->GetMember(0);
	const iObjectKey& keyRed = iObjectKeyRegistry::GetTempKey(Type(),"Red");
	const iObjectKey& keyGreen = iObjectKeyRegistry::GetTempKey(Type(),"Green");
	const iObjectKey& keyBlue = iObjectKeyRegistry::GetTempKey(Type(),"Blue");

	s = p->GetName().Substitute(" ","*") + iObjectKey::FieldSeparator();
	tmpObject->PackValuePiecewiseFunction(s,keyRed,p->GetRedComponent());
	tmpObject->PackValuePiecewiseFunction(s,keyGreen,p->GetGreenComponent());
	tmpObject->PackValuePiecewiseFunction(s,keyBlue,p->GetBlueComponent());

	iObjectKeyRegistry::ReleaseTempKey(keyRed);
	iObjectKeyRegistry::ReleaseTempKey(keyGreen);
	iObjectKeyRegistry::ReleaseTempKey(keyBlue);
}


bool iControlModule::UnPackValuePalette(const iString &s, iPalette *p)
{
	bool ok = true;

	iViewSubject *tmpObject = mViewModules->GetMember(0)->GetMarkerFamily()->GetMember(0);
	const iObjectKey& keyRed = iObjectKeyRegistry::GetTempKey(Type(),"Red");
	const iObjectKey& keyGreen = iObjectKeyRegistry::GetTempKey(Type(),"Green");
	const iObjectKey& keyBlue = iObjectKeyRegistry::GetTempKey(Type(),"Blue");

	iString ws;
	ws = s.Section(iObjectKey::FieldSeparator(),0,0);
	ws.Replace("*"," ");
	if(!ws.IsEmpty()) p->SetName(ws); else ok = false;

	if(ok)
	{
		//
		//  Loop over 3 color functions
		//
		ws = s.Section(iObjectKey::FieldSeparator(),1);
		iPiecewiseFunction *r = p->GetRedComponent();
		iPiecewiseFunction *g = p->GetGreenComponent();
		iPiecewiseFunction *b = p->GetBlueComponent();

		ok = tmpObject->UnPackValuePiecewiseFunction(ws,keyRed,r) && 
		 	tmpObject->UnPackValuePiecewiseFunction(ws,keyGreen,g) &&
			tmpObject->UnPackValuePiecewiseFunction(ws,keyBlue,b);

		if(ok) p->Update();
	}

	iObjectKeyRegistry::ReleaseTempKey(keyRed);
	iObjectKeyRegistry::ReleaseTempKey(keyGreen);
	iObjectKeyRegistry::ReleaseTempKey(keyBlue);

	return ok;
}


iEventObserver* iControlModule::GetInteractorObserver() const
{
	return mInteractorObserver;
}


int iControlModule::GetNumberOfViewModules() const
{
	return 1 + mViewModules->GetMaxMemberIndex();
}


int iControlModule::GetCurrentViewModuleIndex() const
{
	return mViewModules->GetCurrentMemberIndex();
}


iViewModule* iControlModule::GetViewModule() const
{
	return mViewModules->GetCurrentMember();
}


iViewModule* iControlModule::GetViewModule(int mod) const
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	return mViewModules->GetMember(mod);
}


iAnimatorScript* iControlModule::GetAnimatorScript(int mod) const
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return 0; else return module->GetAnimator()->GetScript();
}


void iControlModule::SynchronizeCameras(int mod)
{
	if(mod == -1) mod = mViewModules->GetCurrentMemberIndex();
	iViewModule *module = mViewModules->GetMember(mod);
	if(module == 0) return;

	int i;
	vtkCamera *c = module->GetRenderer()->GetActiveCamera();
	double *fp = c->GetFocalPoint();
	double *cp = c->GetPosition();
	double *vu = c->GetViewUp();
	double ps = c->GetParallelScale();
	for(i=0; i<=mViewModules->GetMaxMemberIndex(); i++)
	{
		c = mViewModules->GetMember(i)->GetRenderer()->GetActiveCamera();
		c->SetFocalPoint(fp);
		c->SetPosition(cp);
		c->SetViewUp(vu);
		c->SetParallelScale(ps);
		mViewModules->GetMember(i)->Render();
	}
}

//
//  We add/remove observers so that in the non-sync mode no extra callbacks take place.
//
void iControlModule::SetSynchronizeInteractors(bool s)
{
	mSyncInteractors = s; 
	mInteractorObserver->Activate(s);
	this->ClearCache();
}


void iControlModule::SetOptimizationMode(int m)
{
	iDataConsumer::SetGlobalOptimizationMode(m);
	vtkMapper::SetGlobalImmediateModeRendering((m==_DataConsumerOptimizeForMemory)?1:0);
	vtkDataObject::SetGlobalReleaseDataFlag((m==_DataConsumerOptimizeForMemory)?1:0);

	this->ClearCache();
	//
	//  We need to reset everything
	//
	int j;
	for(j=0; j<=mViewModules->GetMaxMemberIndex(); j++) mViewModules->GetMember(j)->GetReader()->ResetPipeline();
}


int iControlModule::GetOptimizationMode() const
{
	return iDataConsumer::GetGlobalOptimizationMode();
}


//
//  Helper functions hidden in a private namespace
//
namespace iControlModule_Private
{
	iPalette* Palette_Rainbow()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT(tmp);
		tmp->SetName("Rainbow");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.1);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,0.9);
		b->MovePoint(1,1.0,0.9);

		r->AddPoint(1.0/6.0,0.0);
		g->AddPoint(1.0/6.0,0.0);
		b->AddPoint(1.0/6.0,1.0);

		r->AddPoint(2.0/6.0,0.0);
		g->AddPoint(2.0/6.0,1.0);
		b->AddPoint(2.0/6.0,1.0);

		r->AddPoint(3.0/6.0,0.0);
		g->AddPoint(3.0/6.0,1.0);
		b->AddPoint(3.0/6.0,0.0);

		r->AddPoint(4.0/6.0,1.0);
		g->AddPoint(4.0/6.0,1.0);
		b->AddPoint(4.0/6.0,0.0);

		r->AddPoint(5.0/6.0,1.0);
		g->AddPoint(5.0/6.0,0.0);
		b->AddPoint(5.0/6.0,0.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Temperature()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT(tmp);
		tmp->SetName("Temperature");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.3);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,0.8);
		b->MovePoint(1,1.0,0.8);

		r->AddPoint(0.7,1.0);
		g->AddPoint(0.6,0.0);
		b->AddPoint(0.8,0.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Greyscale()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT(tmp);
		tmp->SetName("Greyscale");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,1.0);
		b->MovePoint(1,1.0,1.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Bluewhite()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT(tmp);
		tmp->SetName("Blue-white");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.1);

		r->MovePoint(1,1.0,0.9);
		g->MovePoint(1,1.0,0.9);
		b->MovePoint(1,1.0,1.0);

		r->AddPoint(0.75,0.0);
		g->AddPoint(0.38,0.0);
		b->AddPoint(0.78,1.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Prizm()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT(tmp);
		tmp->SetName("Prizm");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,0.0);
		g->MovePoint(1,1.0,0.0);
		b->MovePoint(1,1.0,0.0);

		r->AddPoint(0.25,1.0);
		g->AddPoint(0.25,0.0);
		b->AddPoint(0.25,0.0);

		r->AddPoint(0.50,0.0);
		g->AddPoint(0.50,1.0);
		b->AddPoint(0.50,0.0);

		r->AddPoint(0.75,0.0);
		g->AddPoint(0.75,0.0);
		b->AddPoint(0.75,1.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Greenwhite()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT(tmp);
		tmp->SetName("Green-white");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.1);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,1.0);
		b->MovePoint(1,1.0,1.0);

		r->AddPoint(0.375,0.0);
		b->AddPoint(0.750,0.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Bluered()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT(tmp);
		tmp->SetName("Blue-red");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,0.0);
		b->MovePoint(1,1.0,0.0);

		r->AddPoint(0.25,0.0);
		g->AddPoint(0.25,1.0);
		b->AddPoint(0.25,1.0);

		r->AddPoint(0.5,0.0);
		g->AddPoint(0.5,0.0);
		b->AddPoint(0.5,1.0);

		r->AddPoint(0.75,1.0);
		g->AddPoint(0.75,0.0);
		b->AddPoint(0.75,1.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Stern()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT(tmp);
		tmp->SetName("Stern");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,1.0);
		b->MovePoint(1,1.0,1.0);

		r->AddPoint(0.05,1.0);
		r->AddPoint(0.25,0.0);
		b->AddPoint(0.50,1.0);
		b->AddPoint(0.75,0.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Haze()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT(tmp);
		tmp->SetName("Haze");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,1.0);
		g->MovePoint(0,0.0,0.8);
		b->MovePoint(0,0.0,1.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,0.8);
		b->MovePoint(1,1.0,0.0);

		r->AddPoint(0.5,0.0);
		g->AddPoint(0.5,0.1);
		b->AddPoint(0.5,0.5);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_Starlight()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT(tmp);
		tmp->SetName("Starlight");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,0.5);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,1.0);
		g->MovePoint(1,1.0,1.0);
		b->MovePoint(1,1.0,0.7);

		r->AddPoint(0.5,0.9);
		g->AddPoint(0.5,0.7);
		b->AddPoint(0.5,0.2);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_3color()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT(tmp);
		tmp->SetName("3 color");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,1.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,0.0);
		g->MovePoint(1,1.0,0.0);
		b->MovePoint(1,1.0,1.0);

		r->AddPoint(0.3333,1.0);
		g->AddPoint(0.3333,0.0);
		b->AddPoint(0.3333,0.0);
		r->AddPoint(0.3334,0.0);
		g->AddPoint(0.3334,1.0);
		b->AddPoint(0.3334,0.0);

		r->AddPoint(0.6666,0.0);
		g->AddPoint(0.6666,1.0);
		b->AddPoint(0.6666,0.0);
		r->AddPoint(0.6667,0.0);
		g->AddPoint(0.6667,0.0);
		b->AddPoint(0.6667,1.0);

		tmp->Update();
		return tmp;
	}


	iPalette* Palette_4color()
	{
		iPalette* tmp = new iPalette; IERROR_ASSERT(tmp);
		tmp->SetName("4 color");

		iPiecewiseFunction *r = tmp->GetRedComponent();
		iPiecewiseFunction *g = tmp->GetGreenComponent();
		iPiecewiseFunction *b = tmp->GetBlueComponent();

		r->MovePoint(0,0.0,1.0);
		g->MovePoint(0,0.0,0.0);
		b->MovePoint(0,0.0,0.0);

		r->MovePoint(1,1.0,0.0);
		g->MovePoint(1,1.0,0.0);
		b->MovePoint(1,1.0,1.0);

		r->AddPoint(0.24,1.0);
		g->AddPoint(0.24,0.0);
		b->AddPoint(0.24,0.0);
		r->AddPoint(0.25,1.0);
		g->AddPoint(0.25,1.0);
		b->AddPoint(0.25,0.0);

		r->AddPoint(0.49,1.0);
		g->AddPoint(0.49,1.0);
		b->AddPoint(0.49,0.0);
		r->AddPoint(0.50,0.0);
		g->AddPoint(0.50,1.0);
		b->AddPoint(0.50,0.0);

		r->AddPoint(0.74,0.0);
		g->AddPoint(0.74,1.0);
		b->AddPoint(0.74,0.0);
		r->AddPoint(0.75,0.0);
		g->AddPoint(0.75,0.0);
		b->AddPoint(0.75,1.0);

		tmp->Update();
		return tmp;
	}
};

