/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ihelpfactory.h"


#include "iconfigure.h"
#include "ierror.h"
#include "ifile.h"
#include "iobject.h"
#include "iobjecthelp.h"
#include "iparticledatasubject.h"
#include "iversion.h"

#include <vtkCriticalSection.h>
#include <vtkZLibDataCompressor.h>


#define I_CHECK_MISSING_REFERENCES


namespace iHelpFactory_Private
{
	const int _IconTag = 0;
	const int _BeginExtraTag = 1;
	const int _EndExtraTag = 2;

	void Uncompress(const char *data, unsigned int csize, iString &str, int usize)
	{
		static vtkCriticalSection *dataCompressorMutex = vtkCriticalSection::New();
		static vtkZLibDataCompressor *dataCompressor = vtkZLibDataCompressor::New();
		
		unsigned char *destData = (unsigned char *)str.GetWritePointer(usize);
		if(destData == 0)
		{
			str.Clear();
			return;
		}
		//
		//  Thread-safe
		//
		dataCompressorMutex->Lock();
		dataCompressor->Uncompress((unsigned char *)data,csize,destData,usize);
		dataCompressorMutex->Unlock();
		destData[usize] = 0;
	}

	void WriteFile(const iString &fname, const iString &text)
	{
		iFile f(fname);
		if(f.Open(iFile::_Write,iFile::_Text))
		{
			f.WriteLine(text);
			f.Close();
		}
	}

	//
	//  A generic function for querying data
	//
	iHelpData* GetData(const char *tag, iHelpData *data, int &num, bool &sorted)
	{
		//
		//  Measure the number of records
		//
		if(num == -1)
		{
			num = 0;
			while(data[num].Length > -1) num++;
			//
			//  Check that the data are ordered
			//
			int i;
#ifdef I_DEBUG
			const char *is = 0;
			bool ss = sorted;
#endif
			sorted = true;
			for(i=1; i<num; i++)
			{
				if(strcmp(data[i-1].Tag,data[i].Tag) >= 0) 
				{
					sorted = false;
#ifdef I_DEBUG
					is = data[i].Tag;
#endif
				}
			}
#ifdef I_DEBUG
			if(ss && !sorted)
			{
				IERROR_LOW("Help data are not sorted.");
			}
#endif
		}
		
		if(sorted)
		{
			//
			//  If the data are ordered, use a fast binary search 
			//
			int c, ia, ib, ic;
			
			ia = 0; 
			ib = num - 1;
			while(ib > ia+1)
			{
				ic = (ia+ib)/2;
				c = strcmp(data[ic].Tag,tag);
				if(c < 0) ia = ic; else if(c > 0) ib = ic; else return &data[ic];
			}
			if(strcmp(data[ia].Tag,tag) == 0) 
			{
				return &data[ia];
			}
			else if(strcmp(data[ib].Tag,tag) == 0) 
			{
				return &data[ib];
			}
		}
		else
		{
			//
			//  If the data are not ordered, use a slow direct search 
			//
			int i;
			for(i=0; i<num; i++) if(strcmp(data[i].Tag,tag) == 0) 
			{
				return &data[i];
			}
		}
		
		return 0;
	}

	void CreateChapter(iString &text, const iHelpFactory::ChapterInfo info, iHelpFactory::InfoSource source, bool nl = false)
	{
		int i;
		iHelpDataBuffer *pb;
		
		text += "<p>&nbsp;";
		text += iString("<p><a name=\"") + info.Tag + "\"><h2>" + info.Name + "</h2></a>\n";
		
		for(i=0; info.SectionTags[i]!=0; i++)
		{
			pb = iHelpFactory::FindData(info.SectionTags[i],source);
			if(pb != 0)
			{
				text += pb->GetExtraTag(_BeginExtraTag) + "<a name=\"" + info.SectionTags[i] + "\"></a><p><h3>" + pb->GetTitle() + "</h3><p>\n" + pb->GetHTML(true) + pb->GetExtraTag(_EndExtraTag);
				delete pb;
			}
		}
		//
		//  Images
		//
		text.Replace("\"images::","\"images/");
		text.Replace("\"ug::","\"images/");

		if(nl) text += "<!-- PAGE BREAK -->\n";
	}

	void CreateTopicEntry(iString &text, const iHelpFactory::ChapterInfo info, iHelpFactory::InfoSource source)
	{
		int i;
		iHelpDataBuffer *pb;
		
		text += "<p><b>" + info.Name + "</b><ul>";
		
		for(i=0; info.SectionTags[i]!=0; i++)
		{
			pb = iHelpFactory::FindData(info.SectionTags[i],source);
			if(pb != 0)
			{
				text += pb->GetExtraTag(_BeginExtraTag) + "<li><a href=\"" + info.SectionTags[i] + "\">" + pb->GetTitle() + "</a></li>" + pb->GetExtraTag(_EndExtraTag);
				delete pb;
			}
		}
		text += "</ul>";
	}

	void CreateCompressedHeader(const iString &outroot, iHelpData *data)
	{
		//
		//  Actually create the data
		//
		iString ws, wlist;

		iFile f("Code/"+outroot+".h");
		if(!f.Open(iFile::_Write,iFile::_Text))  return;

		f.WriteLine("//LICENSE A");
		f.WriteLine("#if 0");
		f.WriteLine("#include \"../Work/"+outroot+".txt\"");
		f.WriteLine("#else");
		f.WriteLine("static iHelpData data[] = {");

		int i, j, usize, csize;
		unsigned long cmax;
		//
		//  Thread-safe
		//
		vtkZLibDataCompressor *dataCompressor = vtkZLibDataCompressor::New();
		for(i=0; data[i].Length>-1; i++)
		{
			if(data[i].Buffer[0] == '&')
			{
				//
				//  Reference, write as is
				//
				wlist = "{ 0U, 0, \"" + iString(data[i].Tag) + "\",\"" + iString(data[i].Title) + "\",";
				if(data[i].Extra == 0) wlist += "0"; else wlist += iString("\"") + data[i].Extra + "\"";
				wlist += ",\n\"" + iString(data[i].Buffer) + "\"},";
			}
			else
			{
				//
				//  Not a reference, compress
				//
				usize = int(strlen(data[i].Buffer));
				cmax = dataCompressor->GetMaximumCompressionSpace(usize);
				csize = dataCompressor->Compress((const unsigned char *)data[i].Buffer,usize,(unsigned char*)ws.GetWritePointer(cmax),cmax);

				//
				//  Add list entry
				//
				wlist = "{" + iString::FromNumber(csize,"%u") + "," + iString::FromNumber(usize) + ",\"" + data[i].Tag + "\",\"" + data[i].Title + "\",";
				if(data[i].Extra == 0) wlist += "0"; else wlist += iString("\"") + data[i].Extra + "\"";
				wlist += ",\n\"";
				for(j=0; j<(int)csize; j++)
				{
					wlist += iString::FromNumber(int((unsigned char)ws[j]),"\\%o");
					if(j%300 == 299) wlist += "\"\n\"";
				}
				wlist += "\"},";
			}
			f.WriteLine(wlist);
		}
		dataCompressor->Delete();
		f.WriteLine("{ 0U, -1, 0, 0, 0, 0 }");
		f.WriteLine("};");
		f.WriteLine("#endif");
		f.Close();
		exit(0);
	}

	//
	//  Help data live inside these functions, hidden in a private namespace to avoid
	//  cluttering the class namespace. Is there a better design?
	//
	iHelpData* GetDataUG(const char *tag)
	{
		static bool sorted = false;
		static int num = -1;
		//
		//  data live here
		//
#include "ihelpfactorydataUG.h"
		if(data!=0 && data[0].CompressedSize==0U)
		{
			CreateCompressedHeader("help/ihelpfactorydataUG",data);
		}

		return GetData(tag,data,num,sorted);
	}

	iHelpData* GetDataOR(const char *tag)
	{
		static bool sorted = true;
		static int num = -1;
		//
		//  data live here
		//
#include "ihelpfactorydataOR.h"
		if(data!=0 && data[0].CompressedSize==0U)
		{
			CreateCompressedHeader("help/ihelpfactorydataOR",data);
		}

		return GetData(tag,data,num,sorted);
	}

	iHelpData* GetDataSR(const char *tag)
	{
		static bool sorted = false;
		static int num = -1;
		//
		//  data live here
		//
#include "ihelpfactorydataSR.h"
		if(data!=0 && data[0].CompressedSize==0U)
		{
			CreateCompressedHeader("help/ihelpfactorydataSR",data);
		}

		return GetData(tag,data,num,sorted);
	}

	void CreateTypeReference(const iObjectType *type, iString &text, bool hidden, bool extraType, bool inherited = true)
	{
		iString stmp;
		const iObjectKey *key;
		iHelpDataBuffer *dtmp1, *dtmp2;

		iObjectKeyRegistry::InitTraversal(type,inherited);
		while((key=iObjectKeyRegistry::GetNextKey(hidden)) != 0) if(key->GetHelp()->GetText() != "#") // if not for internal use
		{
			dtmp1 = iHelpFactory::FindData(key->GetHelp()->GetTag(),iHelpFactory::_All,false);
			stmp = dtmp1->GetHTML(true);
			if(stmp[0] == '&')
			{
				//
				//  Reference
				//
				dtmp2 = iHelpFactory::FindData(stmp.ToCharPointer(2),iHelpFactory::_All,true);
#ifdef I_DEBUG
				if(dtmp2->GetTitle().IsEmpty())
				{
					IERROR_LOW("Referenced item is missing a title.");
				}
#endif
				stmp = iString("<br>See <a href=\"") + dtmp2->GetTag() + "\">" + dtmp2->GetTitle() + "</a>.";
				delete dtmp2;
			}
			else
			{
				stmp = "<br>" + stmp;
			}

			text += iHelpFactory::FormKeyDescription(*key,stmp,true,!extraType);
			delete dtmp1;
		}
	}

	//
	//  Chapter components (they are small, so are kept here
	//
	const char *gdTags[] = { "ug.gd.ov", "ug.gd.gs", "ug.gd.cc", 0 };
	const iHelpFactory::ChapterInfo ugOverview("ug.gd","Overview",gdTags);

	const char *cnTags[] = { "ug.cn.i", "ug.cn.ms", "ug.cn.ev", "ug.cn.cl", "ug.cn.ef", 0 };
	const iHelpFactory::ChapterInfo ugControls("ug.cn","Controlling IFrIT",cnTags);

	const char *ffTags[] = { "ug.ff.i", "ug.ff.us", "ug.ff.uv", "ug.ff.ut", "ug.ff.bp", 0 };
	const iHelpFactory::ChapterInfo ugFormats("ug.ff","File Formats",ffTags);

	const char *asTags[] = { "ug.as.i", "ug.as.f", 0 };
	const iHelpFactory::ChapterInfo ugAnimation("ug.as","Animation Support",asTags);

	const char *scTags[] = { "ug.sc.i", "ug.sc.ex", "ug.sc.cc", "ug.sc.cs", "ug.sc.as", "ug.sc.av", 0 };
	const iHelpFactory::ChapterInfo ugScripts("ug.sc","Animation and Control Scripts",scTags);

	const char *paTags[] = { "ug.pa.i", 0 };
	const iHelpFactory::ChapterInfo ugPalettes("ug.pa","IFrIT Palettes",paTags);

	const char *ceTags[] = { "ug.ce.i", "ug.ce.f", "ug.ce.c", "ug.ce.l", 0 };
	const iHelpFactory::ChapterInfo ugCodes("ug.ce","Codes For Writing IFrIT Data Files",ceTags);

	const char *liTags[] = { "ug.li.i", "ug.li.g", "ug.li.h", 0 };
	const iHelpFactory::ChapterInfo ugLicense("ug.li","License Agreement",liTags);

	const char *clTags[] = { "sr.cl.i", 0 };
	const iHelpFactory::ChapterInfo ugCLShell("sr.cl","Command-line Shell Reference",clTags);

	const char *ggTags[] = { "sr.gg.i", "sr.gg.ds", "sr.gg.da", "sr.gg.dc", "sr.gg.dd", "sr.gg.de", "sr.gg.df", "sr.gg.di", "sr.gg.dk", "sr.gg.dp", "sr.gg.dr", "sr.gg.hcw", "sr.gg.hfo", "sr.gg.hmc", "sr.gg.hme", "sr.gg.cl", 0 };
	const iHelpFactory::ChapterInfo ugGGShell("sr.gg","GUI Shell Reference",ggTags);

	const char *orTags[] = { "or.i.i", "or.i.r", 0 };
	const iHelpFactory::ChapterInfo ugObjects("or.i","Overview",orTags);
};


using namespace iHelpFactory_Private;

//
//  Helper class
//
iHelpDataBuffer::iHelpDataBuffer(iHelpData &data, bool null) : mData(data), mNull(null)
{
	if(mData.CompressedSize == 0)
	{
		//
		//  Uncompressed data, set the length
		//
		mData.Length = int(strlen(mData.Buffer));
	}
}


iString iHelpDataBuffer::GetTitle() const
{
	return mData.Title;
}


iString iHelpDataBuffer::GetText(int length) const
{
	iString s;

	this->TranformDataToString(s);
	//
	//  Default implementation uses HTML formatting, so strip formatting
	//
	s.ReformatHTMLToText(length);

	return s;
}


iString iHelpDataBuffer::GetHTML(bool untagged) const
{
	iString s;

	this->TranformDataToString(s);
	//
	//  Default implementation uses HTML formatting, so thus return the data, but
	//  add an anchor with the tag in front
	//
	if(untagged || s.IsEmpty() || s[0]=='&') return s; else return iString("<a name=\"") + mData.Tag + "\"></a>" + s;
}


void iHelpDataBuffer::TranformDataToString(iString &s) const
{
	if(mData.CompressedSize > 0)
	{
		//
		//  Compressed data
		//
		Uncompress(mData.Buffer,mData.CompressedSize,s,mData.Length);
	}
	else
	{
		s = iString(mData.Buffer);
	}
}


const iString iHelpDataBuffer::GetExtraTag(int type) const
{
	static const iString none;

	switch(type)
	{
	case _IconTag:
		{
			return iHelpFactory::GetIconTag(mData.Extra);
		}
	case _BeginExtraTag:
		{
			if(mData.Extra!=0 && mData.Extra[0]=='-' && (mData.Extra[1]=='-' || mData.Extra[1]=='[')) return iString(iHelpFactory::IsTagPrivate(mData.Extra+2)?"<private>":"") + "<extra type="+iString(mData.Extra+2)+">"; else return none;
		}
	case _EndExtraTag:
		{
			if(mData.Extra!=0 && mData.Extra[0]=='-' && (mData.Extra[1]=='-' || mData.Extra[1]==']')) return "</extra>" + iString(iHelpFactory::IsTagPrivate(mData.Extra+2)?"</private>":""); else return none;
		}
	default:
		{
			return none;
		}
	}
}


//
//  iHelpFactory implementation
//
iHelpDataBuffer* iHelpFactory::FindData(const char *tag, InfoSource source, bool replaceRefs, int level)
{
	static iHelpData null = { 0U, 0, " ", 0, 0, "No help is available for this item." };
	iHelpData *tmp = 0;

	if(level > 9)
	{
		IERROR_LOW("References are looped.");
		return new iHelpDataBuffer(null,true);
	}

	if(tmp==0 && (source==_All || source==_UserGuide))
	{
		tmp = GetDataUG(tag);
	}

	if(tmp==0 && (source==_All || source==_ShellReference))
	{
		tmp = GetDataSR(tag);
	}

	if(tmp==0 && (source==_All || source==_ObjectReference))
	{
		tmp = GetDataOR(tag);
	}

	if(tmp == 0)
	{
#if defined(I_DEBUG) && defined (I_CHECK_MISSING_REFERENCES)
		if(iString(tag).Contains(".-") == 0)
		{
			IERROR_LOW(("Missing reference:"+iString(tag)).ToCharPointer());
		}
#endif
		return new iHelpDataBuffer(null,true);
	}
	else
	{
		if(tmp->Buffer[0]=='&' && tmp->Buffer[1]=='=')
		{
			//
			//  '=' causes immediate replacement of data
			//
			return FindData(tmp->Buffer+2,_All,replaceRefs,level+1);
		}
		else
		{
			if(replaceRefs && tmp->Buffer[0]=='&' && tmp->Buffer[1]==':')
			{
				return FindData(tmp->Buffer+2,_All,replaceRefs,level+1);
			}
			return new iHelpDataBuffer(*tmp);
		}
	}
}


#ifdef I_DEBUG
void iHelpFactory::CreateUserGuide(UGForm form)
{
	static const iString afterPartTitle("<p>&nbsp;");
	iString text, stmp;
	iHelpDataBuffer *dtmp1;

	if(form == _Publish)
	{
		iHelpFactory::CreateUserGuide(iHelpFactory::_Public);
		iHelpFactory::CreateUserGuide(iHelpFactory::_Private);
		system("Work\\scripts\\createug.csh");
		return;
	}

	iString kind;
	switch(form)
	{
	case _Base:
		{
			kind = "Base";
			break;
		}
	case _Public:
		{
			kind = "Public";
			break;
		}
	case _Private:
		{
			kind = "Private";
			break;
		}
	}

	//
	//  Create the title page
	//
	stmp = "<html><head><title>Title page</title></head><body><center><img src=\"../images/genie2.png\"><p><img src=\"images/_logoug.png\" alt=\"<h1>IFrIT<br>User Guide</h1>\"><p><h2>by</h2><h1>Nick Gnedin and IFrIT</h1><p>&nbsp;<p>&nbsp;<p>&nbsp;<p><h3>IFrIT Version " + iVersion::GetVersion() + "</center>";
	switch(form)
	{
	case _Public:
		{
			stmp += "<p><center><h3>(Including Public Extensions)</h3></center>";
			break;
		}
	case _Private:
		{
			stmp += "<p><center><h3>(Including All Extensions)</h3></center>";
			break;
		}
	}

	//
	//  Create preface
	//
	dtmp1 = iHelpFactory::FindData("ug.pf");
	if(dtmp1 != 0)
	{
		stmp += "<!-- NEW SHEET -->";
		stmp += "<!-- HALF PAGE -->";
		stmp += dtmp1->GetHTML();
		delete dtmp1;
	}
	stmp += "</body></html>";
	WriteFile("Work/docs/ugTitle"+kind+".html",stmp);

	//
	//  Create part on user guide
	//
	text += "<center><a name=\"ug\"><h1 TYPE=\"1\">User Guide</h1></a></center>\n";
	text += afterPartTitle;

	//
	//  Create overview chapter
	//
	CreateChapter(text,ugOverview,_UserGuide);

	//
	//  Create chapter on controls
	//
	CreateChapter(text,ugControls,_UserGuide);

	//
	//  Create chapter on file formats
	//
	CreateChapter(text,ugFormats,_UserGuide);

	//
	//  Create chapter on palettes
	//
	CreateChapter(text,ugPalettes,_UserGuide);

	//
	//  Create chapter on animation support
	//
	CreateChapter(text,ugAnimation,_UserGuide);

	//
	//  Create chapter on script syntax
	//
	CreateChapter(text,ugScripts,_UserGuide);

	//
	//  Create part on shell reference
	//
	text += "<center><a name=\"sr\"><h1>Shell Reference</h1></a></center>\n";
	text += afterPartTitle;

	//
	//  Command-line shell
	//
	CreateChapter(text,ugCLShell,_ShellReference);

	//
	//  GUI shell
	//
	CreateChapter(text,ugGGShell,_ShellReference);

	//
	//  Create object reference
	//
	const iObjectType *type;
	
	text += "<center><a name=\"ug\"><h1>Object Reference</h1></a></center>\n";
	text += afterPartTitle;

	//
	//  Intro
	//
	CreateChapter(text,ugObjects,_ObjectReference,false);
	
	text += "<h2>Available objects</h2>\n";

	text += "<p><a name=\"or.mod\"><b>Modules</b></a>:<ul>\n";
	iObjectTypeRegistry::InitTraversal(iObjectType::_Module);
	while((type=iObjectTypeRegistry::GetNextType(false)) != 0)
	{
		text += FormTypeReference(*type,true);
	}
	text += "</ul>\n";

	text += "<p><a name=\"or.vis\"><b>View objects</b></a>:<ul>\n";
	iObjectTypeRegistry::InitTraversal(iObjectType::_View);
	while((type=iObjectTypeRegistry::GetNextType(false)) != 0)
	{
		text += FormTypeReference(*type,true);
	}
	text += "</ul>\n";

	text += "<p><a name=\"or.hel\"><b>Helper objects</b></a>:<ul>\n";
	iObjectTypeRegistry::InitTraversal(iObjectType::_Helper);
	while((type=iObjectTypeRegistry::GetNextType(false)) != 0)
	{
		text += FormTypeReference(*type,true);
	}
	text += "</ul>\n";
	
	//
	//  Data objects are special
	//
	text += "<p><a name=\"ordat\"><b>Data objects and data types</b></a>";
	text += "<p>Names of data objects are derived from the corresponding names of data types by adding \"Data-\" in front. The following data objects are available:<ul>\n";
	iObjectTypeRegistry::InitTraversal(iObjectType::_Data);
	while((type=iObjectTypeRegistry::GetNextType(false))!=0)
	{
		if(type->GetHelp() != 0) text += type->GetHelp()->GetExtraTag(_BeginExtraTag);
		text += "<li><b>" + type->FullName() + "</b> object (short form: <b>" + type->ShortName() + "</b>)";
		if(type->GetHelp() != 0)
		{
			text += type->GetHelp()->GetExtraTag(_IconTag) + "<br>" + type->GetHelp()->GetHTML();
		}
		text += "</li>";
		if(type->GetHelp() != 0) text += type->GetHelp()->GetExtraTag(_EndExtraTag);
	}
	text += "</ul>\n";
	text += "All Data objects have the same set of <a href=\"or.d-a\">propeties</a>\n";
	
	text += "<p><hr><p>\n";

	//
	//  Include all objects except data objects.
	//
	bool extraType;
	iObjectTypeRegistry::InitTraversal();
	while((type=iObjectTypeRegistry::GetNextType(false)) != 0) if(type->Class() != iObjectType::_Data)
	{
		extraType = !type->GetHelp()->GetExtraTag(_BeginExtraTag).IsEmpty();
		text += type->GetHelp()->GetExtraTag(_BeginExtraTag) + "<p>" + "<a name=\"or."+type->ShortName()+"\"><h2>" + type->FullName() + " object "  + type->GetHelp()->GetExtraTag(_IconTag) + "</h2><br>" + type->GetHelp()->GetHTML() + "<p>Short form: <b>" + type->ShortName() + "</b><p>Available properties:<p><ul>\n";

		CreateTypeReference(type,text,false,extraType);

		text += "</ul><p><hr>" + type->GetHelp()->GetExtraTag(_EndExtraTag) + "<p>\n";
	}

	//
	//  Include the Data object.
	//
	text += "<p><a name=\"or.d-a\"><h2>Properties of Data Objects</h2></a><br>All data objects support the following common properties.<ul>\n";
	CreateTypeReference(&iDataSubject::Type(),text,true,false);
	text += "</ul>\n";

	//
	//  Include the ParticleData object.
	//
	text += "<p>All particle data objects also support additional properties:<ul>";
	CreateTypeReference(&iParticleDataSubject::Type(),text,true,false);
	text += "</ul>\n";

	//
	//  Include special properties of specific Data objects.
	//
	text += "<p>Some data objects can also have properties that are specific just for them:<ul>";

	const iObjectKey *key;
	iObjectTypeRegistry::InitTraversal();
	while((type=iObjectTypeRegistry::GetNextType(false)) != 0) if(type->Class() == iObjectType::_Data)
	{
		iObjectKeyRegistry::InitTraversal(type,false);
		bool hasKeys = false;
		while((key=iObjectKeyRegistry::GetNextKey(false)) != 0) if(key->GetHelp()->GetText() != "#") // if not for internal use
		{
			hasKeys = true;
		}

		if(hasKeys)
		{
			extraType = !type->GetHelp()->GetExtraTag(_BeginExtraTag).IsEmpty();
			text += type->GetHelp()->GetExtraTag(_BeginExtraTag) + "<li><a name=\"or."+type->ShortName()+"\"><b>" + type->FullName() + " object "  + type->GetHelp()->GetExtraTag(_IconTag) + "</b>(short form: <b>" + type->ShortName() + "</b>)<br>" + type->GetHelp()->GetHTML() + "<br>Available properties:<ul>\n";

			CreateTypeReference(type,text,false,extraType,false);

			text += "</ul>" + type->GetHelp()->GetExtraTag(_EndExtraTag) + "</li><p>\n";
		}
	}
	text += "</ul>\n";

	//
	//  Create appendices
	//
	text += "<center><a name=\"ap\"><h1 VALUE=\"1\" TYPE=\"A\">Appendices</h1></a></center>\n";

	CreateChapter(text,ugCodes,_UserGuide);
	CreateChapter(text,ugLicense,_UserGuide);

	//
	//  Verify integrity
	//
#if defined(I_DEBUG) && defined (I_CHECK_MISSING_REFERENCES)
	int i = -1;
	bool ok = true;
	iString mr;
	while((i = text.Find("<a href=\"",i+1)) > -1)
	{
		stmp = text.Part(i+9).Section("\"",0,0);
		if(stmp[2] == '.')  // omit volume references
		{
			dtmp1 = FindData(stmp.ToCharPointer(),_All,true);
			if(dtmp1->IsNull())
			{
				mr += stmp + ",";
				ok = false;
			}
			delete dtmp1;
		}
	}
	if(!ok)
	{
		IERROR_LOW(("Missing references: "+mr).ToCharPointer());
	}
#endif

	//
	//  Define what to create
	//
	switch(form)
	{
	case _Base:
		{
			//
			//  Remove everything between <extra type=...> and </extra> tags 
			//
			text.RemoveFormatting("<extra type=","</extra>");
			//
			//  No break, keep going
			//
		}
	case _Public:
		{
			//
			//  Remove everything between <private> and </private> tags 
			//
			text.RemoveFormatting("<private>","</private>");
			break;
		}
	case _Private:
		{
			//
			//  If extra info is included, remove special <extra type=...> </extra> and <private></private> tags 
			//
			text.RemoveFormatting("<extra type=",">");
			text.Replace("</extra>","");
			text.Replace("<private>","");
			text.Replace("</private>","");
			break;
		}
	}

	//
	//  Create the html file
	//
	WriteFile("Work/docs/ug"+kind+".html","<html><title>&copy; 2005-2008 by Nick Gnedin</title><body>"+text+"</body></html>");
}
#endif


void iHelpFactory::CreateTopicList(iString &text)
{
	text.Clear();

	text = "<center><h3>Available Help Topics</h3></center><p>";

	//
	//  Create overview chapter
	//
	CreateTopicEntry(text,ugOverview,_UserGuide);

	//
	//  Create chapter on controls
	//
	CreateTopicEntry(text,ugControls,_UserGuide);

	//
	//  Create chapter on file formats
	//
	CreateTopicEntry(text,ugFormats,_UserGuide);

	//
	//  Create chapter on palettes
	//
	CreateTopicEntry(text,ugPalettes,_UserGuide);

	//
	//  Create chapter on animation support
	//
	CreateTopicEntry(text,ugAnimation,_UserGuide);

	//
	//  Create chapter on script syntax
	//
	CreateTopicEntry(text,ugScripts,_UserGuide);

	//
	//  GUI shell
	//
	CreateTopicEntry(text,ugGGShell,_ShellReference);

	//
	//  Create object reference
	//
	const iObjectType *type;
	
	text += "<p><b>Objects</b><ul>";

	iObjectTypeRegistry::InitTraversal();
	while((type=iObjectTypeRegistry::GetNextType(false)) != 0)
	{
		text += FormTypeReference(*type,false);
	}
	text += "</ul>";

	//
	//  Create appendices
	//
	CreateTopicEntry(text,ugCodes,_UserGuide);
	CreateTopicEntry(text,ugLicense,_UserGuide);

	//
	//  Finalize
	//
	InterpretExtraElements(text);
}


const iString iHelpFactory::FormTypeReference(const iObjectType &type, bool bold)
{
	iString tmp;
	if(type.GetHelp() != 0) tmp += type.GetHelp()->GetExtraTag(_BeginExtraTag);
	tmp += "<li><a href=\"or." + type.ShortName() + "\">" + (bold?"<b>":"") + type.FullName() + (bold?"</b>":"") + "</a></li>";
	if(type.GetHelp() != 0) tmp += type.GetHelp()->GetExtraTag(_EndExtraTag);
	return tmp;
}


const iString iHelpFactory::FormKeyReference(const iObjectKey &key, bool bold)
{
	iString tmp;
	if(key.GetHelp() != 0) tmp += key.GetHelp()->GetExtraTag(_BeginExtraTag);
	tmp += "<li><a href=\"or." + key.HelpEntryKey() + "\">" + (bold?"<b>":"") + key.UnprefixedFullName() + (bold?"</b>":"") + "</a></li>";
	if(key.GetHelp() != 0) tmp += key.GetHelp()->GetExtraTag(_EndExtraTag);
	return tmp;
}


const iString iHelpFactory::FormKeyDescription(const iObjectKey &key, const iString &body, bool line, bool full)
{
	iString text;

	if(key.GetHelp()!=0 && full) text += key.GetHelp()->GetExtraTag(_BeginExtraTag);

	if(line)
	{
		text += "<li><a name=\"or." + key.HelpEntryKey() + "\">";
		if(key.GetHelp() != 0)
		{
			if(full) text += key.GetHelp()->GetExtraTag(_BeginExtraTag);
			text += key.GetHelp()->GetExtraTag(_IconTag);
		}
		text += "<b>" + key.UnprefixedFullName() + "</b>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;(short form: <b>" + key.UnprefixedShortName() + "</b>;&nbsp;&nbsp;type: ";
	}
	else
	{
		text += "<h4>Property:</h4><br><table>";
		text += "<tr><td><b>Complete long name:</b></td><td><tt>" + key.UnprefixedFullName() + "</tt></td></tr>";
		text += "<tr><td><b>Complete short name:</b></td><td><tt>" + key.UnprefixedShortName() + "</tt></td></tr>";
		text += "<tr><td><b>Type:</b></td><td><tt>";
	}

	switch(key.Argument())
	{
	case iObjectKey::_OffsetInt:
	case iObjectKey::_Int:
		{
			text += "<b><tt>int</tt></b>";
			break;
		}
	case iObjectKey::_Bool:
		{
			text += "<b><tt>bool</tt></b>";
			break;
		}
	case iObjectKey::_Float:
		{
			text += "<b><tt>float</tt></b>";
			break;
		}
	case iObjectKey::_Double:
		{
			text += "<b><tt>double</tt></b>";
			break;
		}
	case iObjectKey::_Color:
		{
			text += "<b><tt>color</tt></b>";
			break;
		}
	case iObjectKey::_String:
		{
			text += "<b><tt>string</tt></b>";
			break;
		}
	case iObjectKey::_Any:
		{
			text += "any";
			break;
		}
	}

	if(line)
	{
		text += ";&nbsp;&nbsp;# of arguments: <b><tt>";
		if(key.Dimension() > 0) text += iString::FromNumber(key.Dimension()); else text += "any";
		text += "</tt></b>)\n" + body + "</li>\n";
	}
	else
	{
		text += "</tt></td></tr>";
		text += "<tr><td><b>Dimension:</b></td><td><tt>" + (key.Dimension()>0?iString::FromNumber(key.Dimension()):"any") + "</tt></td></tr></table>";
		text += "<p>" + body;
	}

	if(key.GetHelp()!=0 && full) text += key.GetHelp()->GetExtraTag(_EndExtraTag);

	return text;
}


//
//  Explicitly configued members
//
bool iHelpFactory::IsTagPrivate(const char *tag)
{
#if IEXTENSION_INCLUDED(IEXTENSION_ART)
	static const char *hTag = "ART";
	if(strcmp(tag,hTag) == 0) return true;
#endif
	return false;
}


void iHelpFactory::InterpretExtraElements(iString &text, const iString &tag)
{
	if(tag.IsEmpty() || tag=="ART")
	{
#if IEXTENSION_INCLUDED(IEXTENSION_ART)
		text.Replace("<extra type=ART>","");
		text.Replace("</extra>","");
#else
		text.RemoveFormatting("<extra type=ART>","</extra>");
#endif
	}
	if(tag.IsEmpty() || tag=="GADGET")
	{
#if IEXTENSION_INCLUDED(IEXTENSION_GADGET)
		text.Replace("<extra type=GADGET>","");
		text.Replace("</extra>","");
#else
		text.RemoveFormatting("<extra type=GADGET>","</extra>");
#endif
	}
	if(tag.IsEmpty() || tag=="VTK")
	{
#if IEXTENSION_INCLUDED(IEXTENSION_VTK)
		text.Replace("<extra type=VTK>","");
		text.Replace("</extra>","");
#else
		text.RemoveFormatting("<extra type=VTK>","</extra>");
#endif
	}
}


const iString& iHelpFactory::GetIconTag(const char *extra)
{
	static const iString none;

#if IEXTENSION_INCLUDED(IEXTENSION_ART)
	static const iString hTag("<img src=\"images::art.png\">");
	if(extra!=0 && ((extra[0]=='-' && strcmp(extra+2,"ART")==0) || strcmp(extra,"ART")==0)) return hTag;
#endif
#if IEXTENSION_INCLUDED(IEXTENSION_GADGET)
	static const iString gTag("<img src=\"images::gadget.png\">");
	if(extra!=0 && ((extra[0]=='-' && strcmp(extra+2,"GADGET")==0) || strcmp(extra,"GADGET")==0)) return gTag;
#endif
#if IEXTENSION_INCLUDED(IEXTENSION_VTK)
	static const iString vtkTag("<img src=\"images::vtk.png\">");
	if(extra!=0 && ((extra[0]=='-' && strcmp(extra+2,"VTK")==0) || strcmp(extra,"VTK")==0)) return vtkTag;
#endif

	return none;
}

