/***************************************
  $Header: /cvsroot/petscgraphics/tsview-ng.c,v 1.10 2004/05/21 22:08:44 hazelsct Exp $

  This program views the output of a time series saved using
  +latex+{\tt IlluMultiSave()}.
  +html+ <tt>IlluMultiSave()</tt>.
  It basically just switches between timesteps; future versions may be more
  interesting.  The neat part of it is that it loads multiprocessor data and
  displays it on a single CPU.
  ***************************************/

static char help[] = "Displays the output of of a timestep series saved using IlluMultiSave().\n\
Usage:\n\
\n\
  tsview <basename> [-no_transparency]\n\
\n\
Then interactively flip through the timesteps (h or ? lists commands).\n";

#include "illuminator.h"
#include <glade/glade.h>
#include <gnome.h>
#include <libgnomeui/libgnomeui.h>
#include <sys/dir.h> /* For scandir(), alphasort, struct dirent */
#include <libgen.h>  /* For dirname(), basename() */
#include <string.h>  /* For strdup() */

/* Build with -DDEBUG for debugging output */
#undef DPRINTF
#ifdef DEBUG
#define DPRINTF(fmt, args...) PetscPrintf (PETSC_COMM_WORLD, "%s: " fmt, __FUNCT__, args)
#else
#define DPRINTF(fmt, args...)
#endif

GladeXML *xml;
/* Filename list */
int entrynum=0, total_entries=0, current_timestep;
char *the_basename, *basedirname, **stepnames=NULL;
double current_time;

/* Window parameters and drawables */
int width=0, height=0, nx, ny, dataview_count=1;
GtkWidget *dataviews [1];
guchar *rgbbuf [1] = { NULL };
double sizemag;
PetscTruth transp;

/* Maximum intensity for hueintense plots */
PetscScalar vecmax=-1.;

/* PETSc structures etc. */
DA theda;
Vec global;
PetscScalar minmax[6] = { 0.,1., 0.,1., 0.,1. };
field_plot_type *fieldtypes;
int dimensions, num_fields, num_variables[1], **variable_indices;

/* First some main functions which do stuff, then callbacks, then main(). */
static inline void pseudocolor (PetscScalar val, PetscScalar min,
				   PetscScalar max, guchar *pixel)
{
  PetscScalar shade = (val - min) / (max - min);
  /* Old stuff *
  if (shade < 0.2) {      /* red -> yellow *
    *pixel++ = 255;
    *pixel++ = 1275*shade;
    *pixel++ = 0; }
  else if (shade < 0.4) { /* yellow -> green *
    *pixel++ = 510-1275*shade;
    *pixel++ = 255;
    *pixel++ = 0; }
  else if (shade < 0.6) { /* green -> cyan *
    *pixel++ = 0;
    *pixel++ = 255;
    *pixel++ = 1275*shade-510; }
  else if (shade < 0.8) { /* cyan -> blue *
    *pixel++ = 0;
    *pixel++ = 1020-1275*shade;
    *pixel++ = 255; }
  else {                  /* blue -> magenta *
    *pixel++ = 1275*shade-1020;
    *pixel++ = 0;
    *pixel++ = 255; }
  /* New stuff */
  if (shade < 0.25) {      /* red -> yellow */
    *pixel++ = 255;
    *pixel++ = 1020*shade;
    *pixel++ = 0; }
  else if (shade < 0.5) { /* yellow -> green */
    *pixel++ = 510-1020*shade;
    *pixel++ = 255;
    *pixel++ = 0; }
  else if (shade < 0.75) { /* green -> cyan */
    *pixel++ = 0;
    *pixel++ = 255;
    *pixel++ = 1020*shade-510; }
  else { /* cyan -> blue */
    *pixel++ = 0;
    *pixel++ = 1020-1020*shade;
    *pixel++ = 255; }
}

static inline void pseudohueintcolor
(PetscScalar vx, PetscScalar vy, PetscScalar refmag, guchar *pixel)
{
  PetscScalar mag=sqrt(vx*vx+vy*vy), theta=atan2(vy,vx), red, green, blue;
  if (refmag <= 0.)
    {
      *pixel = *(pixel+1) = *(pixel+2) = 0;
      return;
    }
  mag = (mag > refmag) ? 1.0 : mag/refmag;
  /* Old pseudocolor */
  /* red = (theta<-M_PI*5./6.)? 1.5/M_PI*(-M_PI*5./6.-theta):
    ((theta<-M_PI/6.)? 0.:
     ((theta<M_PI/2.)? 1.5/M_PI*(theta+M_PI/6.): 1.5/M_PI*(M_PI*7./6.-theta)));
  green = (theta<-M_PI*5./6.)? 1.5/M_PI*(theta+M_PI*3./2.):
    ((theta<-M_PI/6.)? 1.5/M_PI*(-M_PI/6.-theta):
     ((theta<M_PI/2.)? 0.: 1.5/M_PI*(theta-M_PI/2.)));
  blue = (theta<-M_PI*5./6.)? 0.:
    ((theta<-M_PI/6.)? 1.5/M_PI*(theta+M_PI*5./6.):
    ((theta<M_PI/2.)? 1.5/M_PI*(M_PI/2.-theta): 0.)); */

  /* New pseudocolor: green, yellow, red, blue at 0, 90, 180, 270 degrees */
  red = 2./M_PI * ((theta<-M_PI/2.) ? -M_PI/2.-theta :
		   ((theta<0.) ? 0. : ((theta<M_PI/2.) ? theta : M_PI/2.)));
  green = 2./M_PI * ((theta<-M_PI/2.) ? 0. :
		     ((theta<0.) ? theta+M_PI/2. :
		      ((theta<M_PI/2.) ? M_PI/2. : M_PI-theta)));
  blue = 2./M_PI * ((theta<-M_PI/2.) ? theta+M_PI :
		    ((theta<0.) ? -theta : 0.));

  *pixel++ = 255*mag*red;
  *pixel++ = 255*mag*green;
  *pixel++ = 255*mag*blue;
}

static inline void pseudoternarycolor
(PetscScalar A, PetscScalar B, PetscScalar Corner1A, PetscScalar Corner1B,
 PetscScalar Corner2A, PetscScalar Corner2B, PetscScalar Corner3A,
 PetscScalar Corner3B, guchar *pixel)
{
  PetscScalar x1, x2, inverse_det;

  /* Transform A,B into x1,x2 based on corners */
  A        -= Corner3A;
  Corner1A -= Corner3A;
  Corner2A -= Corner3A;
  B        -= Corner3B;
  Corner1B -= Corner3B;
  Corner2B -= Corner3B;
  inverse_det = 1./(Corner1A*Corner2B - Corner2A*Corner1B);
  x1 = (A*Corner2B - B*Corner2A) * inverse_det;
  x2 = (B*Corner1A - A*Corner1B) * inverse_det;

  /* Now colorize */
  *pixel++ = 255*(1.-x1);
  *pixel++ = 255*(x1+x2);
  *pixel++ = 255*(1.-x2);
}

/* Find the maximum magnitude of a 2-D vector field in an array */
PetscScalar ArrayMaxMag (PetscScalar *global, int points, int fields,int field)
{
  int i;
  PetscScalar retval;

  retval = global[field]*global[field] + global[field+1]*global[field+1];

  for (i=1; i<points; i++)
    retval = PetscMax (retval, global[i*fields+field]*global[i*fields+field] +
		       global[i*fields+field+1]*global[i*fields+field+1]);
  return sqrt (retval);
}

void paint_rgb (guchar *rgb, int rwidth, int rheight, DA rgbda, Vec rgbglobal,
		int display_field)
{
  int nx,ny,nz, ix,iy, ierr;
  PetscScalar *global_array, minval, maxval, refmag;

  ierr = DAGetInfo (rgbda, PETSC_NULL, &nx,&ny,&nz,
		    PETSC_NULL,PETSC_NULL,PETSC_NULL, PETSC_NULL,
		    PETSC_NULL,PETSC_NULL,PETSC_NULL); CHKERRQ (ierr);

  ierr = VecStrideMin (rgbglobal, display_field, PETSC_NULL, &minval);
  CHKERRQ (ierr);
  ierr = VecStrideMax (rgbglobal, display_field, PETSC_NULL, &maxval);
  CHKERRQ (ierr);

  ierr = VecGetArray (rgbglobal, &global_array);

  /* If a vector field, then find max magnitude in fields 0-1 */
  if (fieldtypes[display_field] == FIELD_VECTOR_HUEINTENSE ||
      fieldtypes[display_field] == FIELD_VECTOR_ARROWS)
    {
      if (vecmax > 0)
	refmag = vecmax;
      else
	refmag = ArrayMaxMag (global_array, nx*ny, num_fields, display_field);
      printf ("Reference vector magnitude = %g\n", refmag);
    }

  for (iy=0; iy<rheight; iy++)
    for (ix=0; ix<rwidth; ix++)
      {
	int vecindex;
	guchar *pixel;
	vecindex = (((rheight-iy-1)*ny/rheight)*nx+ ix*nx/rwidth)*num_fields+
	  display_field;
	pixel = rgb + 3*(iy*rwidth + ix);

	if (fieldtypes[display_field] == FIELD_TERNARY_RGB)
	  pseudoternarycolor
	    (global_array [vecindex], global_array [vecindex+1],
	     0.,0., 1.,0., 0.,1., pixel);
	else if (fieldtypes[display_field] == FIELD_VECTOR_HUEINTENSE ||
		 fieldtypes[display_field] == FIELD_VECTOR_ARROWS)
	  pseudohueintcolor
	    (global_array [vecindex], global_array [vecindex+1], refmag,
	     pixel);
	else
	  pseudocolor (global_array [vecindex], minval, maxval, pixel);
      }
  ierr = VecRestoreArray (rgbglobal, &global_array);
}

void render_dataviews ()
{
  int viewnum, nx,ny,nz, ierr;
  GtkWidget *thebar = glade_xml_get_widget(xml, "appbar1");

  if (dataview_count != 1)
    {
      printf ("dataview_count != 1 is not yet supported\n");
      exit(0);
    }
  for (viewnum=0; viewnum<dataview_count; viewnum++)
    {
      int ix,iy, display_field=0;
      PetscScalar minval, maxval, refmag, *global_array;
      GtkType type;
      char thestatus [100];

      /* (Re)allocate buffer */
      if (!(rgbbuf [viewnum] =
	    (guchar *) realloc (rgbbuf [viewnum],
				3*width*height*sizeof(guchar)))) {
	  printf ("ERROR: can't reallocate RGB buffer\n");
	  exit (1); }

      sprintf (thestatus, "Rendering view %d...", viewnum);
      gnome_appbar_set_default (GNOME_APPBAR (thebar), thestatus);

      /* Render into rgbbuf */
      paint_rgb (rgbbuf [viewnum], width,height, theda, global, display_field);

      /* Draw buffer onto window */
      if (!dataviews [viewnum])
	dataviews [viewnum] = glade_xml_get_widget (xml, "plot_area");
      gtk_drawing_area_size (GTK_DRAWING_AREA (dataviews [viewnum]),
			     width, height);
      gdk_draw_rgb_image (dataviews [viewnum]->window,
			  dataviews [viewnum]->style->fg_gc[GTK_STATE_NORMAL],
			  0, 0, width, height, GDK_RGB_DITHER_MAX,
			  rgbbuf [viewnum], width * 3);
    }
  gnome_appbar_set_default (GNOME_APPBAR (thebar), "Done.");
}


#undef __FUNCT__
#define __FUNCT__ "myfilter"

/*+ Little variable for myfilter() and refresh_stepnames(). +*/
static char *basefilename;

/*++++++++++++++++++++++++++++++++++++++
  This function returns non-zero for "qualifying" file names which start with
  the stored files' basename.time and end with
  +latex+{\tt .cpu0000.meta}.
  +html+ <tt>.cpu0000.meta</tt>.
  It is used as the
  +latex+{\tt select()} function for {\tt scandir()} in {\tt main()}.
  +html+ <tt>select()</tt> function for <tt>scandir()</tt> in <tt>main()</tt>.

  int myfilter Returns non-zero for qualifying filenames.

  const struct dirent *direntry Directory entry with filename to test.
  ++++++++++++++++++++++++++++++++++++++*/

int myfilter (const struct dirent *direntry)
{
  if ((!strncmp (direntry->d_name, basefilename, strlen(basefilename))) &&
      (!strncmp (direntry->d_name + strlen(basefilename), ".time", 5)))
    return (!strncmp (direntry->d_name + strlen(direntry->d_name) - 13,
		      ".cpu0000.meta", 13));
  return 0;
}


void on_plot_area_expose_event (GtkWidget *widget, GdkEventExpose *event,
				   gpointer user_data)
{
  gdk_draw_rgb_image (widget->window, widget->style->fg_gc[GTK_STATE_NORMAL],
		      0, 0, width, height, GDK_RGB_DITHER_MAX, rgbbuf [0],
		      width * 3);
}

/*++++++++++++++++++++++++++++++++++++++
  This loads the names of the files into a long list.
  ++++++++++++++++++++++++++++++++++++++*/

int refresh_stepnames ()
{
  struct dirent **namelist;
  char *filec, *dirc;
  int i, ierr;

  filec = strdup (the_basename);
  dirc = strdup (the_basename);
  basefilename = basename (filec);
  basedirname = dirname (dirc);

  total_entries = scandir (basedirname, &namelist, myfilter, alphasort);
  if (!total_entries)
    {
      ierr = PetscPrintf (PETSC_COMM_WORLD, "No such files\n");
      CHKERRQ (ierr);
      return 1;
    }
  if (total_entries < 0)
    {
      ierr = PetscPrintf (PETSC_COMM_WORLD, "Error scanning directory %s\n",
			  basedirname); CHKERRQ (ierr);
      ierr = PetscFinalize (); CHKERRQ(ierr);
      return 1;
    }
  ierr = PetscPrintf (PETSC_COMM_WORLD, "%d eligible files:\n", total_entries);
  CHKERRQ (ierr);

  if (!(stepnames = (char **) realloc
	(stepnames, total_entries*sizeof (char *))))
    {
      ierr = PetscPrintf (PETSC_COMM_WORLD, "Error allocating memory\n");
      CHKERRQ (ierr);
      return 1;
    }
  for (i=0; i<total_entries; i++)
    {
      int filength = strlen(namelist[i]->d_name);

      stepnames [i] = (char *) malloc ((filength-12)*sizeof(char));
      strncpy (stepnames [i], namelist[i]->d_name, filength-13);
      stepnames [i] [filength-13] = '\0';
      free (namelist[i]);
      ierr = PetscPrintf (PETSC_COMM_WORLD, "[%d] %s\n", i, stepnames [i]);
      CHKERRQ (ierr);
    }

  free (namelist);
  return 0;
}


void on_mag_spin_value_changed (GtkWidget *mag_spin, gpointer user_data) {
  G_CONST_RETURN gchar *entrytext;
  entrytext = gtk_entry_get_text (GTK_ENTRY (mag_spin));
  sscanf (entrytext, "%lf", &sizemag);
  width = (int) (minmax [1] * sizemag);
  height = (int) (minmax [3] * sizemag);

  render_dataviews(dataviews, dataview_count);
}


void display_timestep (int usermetacount, char **usermetanames,
		       char **usermetadata)
{
  int i;
  static char step_buffer [20], time_buffer [20];

  for (i=0; i<usermetacount; i++)
    {
      if (!strncmp (usermetanames [i], "timestep", 8))
	sscanf (usermetadata [i], "%d", &current_timestep);
      else if (!strncmp (usermetanames [i], "time", 4))
	sscanf (usermetadata [i], "%lf", &current_time);
    }
  snprintf (step_buffer, 19, "Timestep: %d", current_timestep);
  gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (xml, "timestep_label")),
		      step_buffer);
  snprintf (time_buffer, 19, "Time: %g", current_time);
  gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (xml, "time_label")),
		      time_buffer);

  on_mag_spin_value_changed (glade_xml_get_widget (xml, "mag_spin"), NULL);
}


void on_save_activate (GtkWidget *widget, gpointer user_data)
{
  int i, ierr;
  char **usermetanames, **usermetadata, filename [200], thestatus [200];
  GtkWidget *thebar = glade_xml_get_widget(xml, "appbar1");
  FILE *outppm;

  strncpy (filename, basedirname, 198);
  strcat (filename, "/");
  strncat (filename, stepnames [entrynum], 198 - strlen (filename));
  strncat (filename, ".ppm", 198 - strlen (filename));

  sprintf (thestatus, "Saving entry %d in filename %s", entrynum, filename);
  gnome_appbar_set_default (GNOME_APPBAR (thebar), thestatus);

  if (!(outppm = fopen (filename, "w")))
    printf ("Error opening file %s\n", filename);
  fprintf (outppm, "P6\n%d %d\n255\n", width, height);
  fwrite (rgbbuf [0], sizeof (guchar), 3*width*height, outppm);
  fclose (outppm);

  sprintf (thestatus, "Saved entry %d in filename %s", entrynum, filename);
  gnome_appbar_set_default (GNOME_APPBAR (thebar), thestatus);
}


void on_timestep_spin_value_changed (GtkWidget *timestep_spin, gpointer user_data) {
  int usermetacount, ierr;
  G_CONST_RETURN gchar *entrytext;
  char **usermetanames, **usermetadata, filename [200], thestatus [200];
  GtkWidget *thebar = glade_xml_get_widget(xml, "appbar1");

  entrytext = gtk_entry_get_text (GTK_ENTRY (timestep_spin));
  sscanf (entrytext, "%d", &entrynum);

  /* Bound the entrynum between 0 and total_entries-1; -11 is the minimum of
     the widget (from jump), 1000001 is the maximum. */
  if ((entrynum < 0 && entrynum != -11) || entrynum == 1000001)
    entrynum = total_entries-1;
  if ((entrynum >= total_entries && entrynum != 1000001) || entrynum == -11)
    entrynum = 0;
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (timestep_spin),
			     (gfloat) entrynum);

  strncpy (filename, basedirname, 198);
  strcat (filename, "/");
  strncat (filename, stepnames [entrynum], 198 - strlen (filename));

  sprintf (thestatus, "Loading entry %d, basename %s\n", entrynum, filename);
  gnome_appbar_set_default (GNOME_APPBAR (thebar), thestatus);

  ierr = IlluMultiRead (theda, global, filename, &usermetacount,&usermetanames,
			&usermetadata); CHKERRQ (ierr);

  display_timestep (usermetacount, usermetanames, usermetadata);
}


void on_refresh_activate (GtkWidget *none, gpointer user_data) {
  if (refresh_stepnames ()) exit (1); }


void on_about_activate (GtkWidget *none, gpointer user_data) {
  gtk_widget_show (glade_xml_get_widget (xml, "about")); }


/*
#define HELP_STRING "tsview commands:\n\
  <enter> Display next timestep\n\
  b       Display previous timestep\n\
  ###     Jump to timestep ###\n\
  t       Toggle Geomview transparency (3-D only)\n\
  v       Change field displayed (3-D only)\n\
  h/?     Print this information\n\
  q/x     Quit tsview\n"

int start (int argc, char *argv[])
{
  int total_entries, current_entry, dims, i, ierr;
  PetscViewer theviewer;
  PetscTruth loaded = PETSC_FALSE, transp=PETSC_TRUE; */

  /*+In the main loop, the various timesteps are displayed, with options:
    +latex+\begin{itemize} \item
    +html+ <ul><li>
    A number jumps to that entry in the files table.
    +latex+\item {\stt <return>}
    +html+ <li><tt>&lt;return&gt;</tt>
    loads the next file.
    +latex+\item {\tt b}
    +html+ <li><tt>b</tt>
    goes back one file.
    +latex+\item {\tt q}
    +html+ <li><tt>q</tt>
    quits the program.
    +latex+\end{itemize}
    +html+ </ul>
    +*/
  /*  current_entry=0;
  while (1)
    {

      else
	{

	  if (dims<3)
	    {
	      int width=300, height=300;

	      if (minmax[1]<minmax[3])
		width  *= minmax[1]/minmax[3];
	      else
		height *= minmax[3]/minmax[1];

	      ierr = PetscViewerDrawOpen
		(PETSC_COMM_WORLD, 0, "", PETSC_DECIDE, PETSC_DECIDE,
		 width, height, &theviewer); CHKERRQ (ierr);
	    }
	  else
	    {
	      ierr = GeomviewBegin (PETSC_COMM_WORLD);
	    }
	}

      ierr = PetscPrintf (PETSC_COMM_WORLD, "User data:\n"); CHKERRQ (ierr);
      for (i=0; i<usermetacount; i++)
	{
	  ierr = PetscPrintf (PETSC_COMM_WORLD, "%s = %s\n", usermetanames [i],
			      usermetadata [i]); CHKERRQ (ierr);
	}

      if (dims<3)
	{
	  ierr = VecView (global, theviewer); CHKERRQ (ierr);
	}
      else
      { */
	  /*+ The Illuminator-based 3-D viewer can only display one field at a
	    time.  At the beginning, that is field 0, and is cycled using the
	    +latex+{\tt v}
	    +html+ <tt>v</tt>
	    command. +*/
	  /* PetscScalar minval, maxval;
	  char *fieldname;

	  ierr = VecStrideMin (global, display_field, PETSC_NULL, &minval);
	  CHKERRQ (ierr);
	  ierr = VecStrideMax (global, display_field, PETSC_NULL, &maxval);
	  CHKERRQ (ierr);
	  ierr = PetscPrintf (PETSC_COMM_WORLD,
			      "Displaying field %d [%g-%g]: %s\n",
			      display_field, minval, maxval, fieldname);
	  CHKERRQ (ierr);

	  DPRINTF ("Calculating triangle locations\n",0);
	  ierr = DATriangulate (theda, global, display_field, minmax,
				PETSC_DECIDE, PETSC_NULL, PETSC_NULL);
	  CHKERRQ (ierr);
	  ierr = DAGetFieldName (theda, display_field, &fieldname);
	  CHKERRQ (ierr);
	  DPRINTF ("Consolidating triangles on head node and visualizing\n",0);
	  ierr = GeomviewDisplayTriangulation
	    (PETSC_COMM_WORLD, minmax, fieldname, transp);
	  CHKERRQ (ierr);
	  }

      for (i=0; i<usermetacount; i++)
	{
	  free (usermetanames [i]);
	  free (usermetadata [i]);
	}
      free (usermetanames);
      free (usermetadata);

      ierr = PetscPrintf (PETSC_COMM_WORLD, "What to do? (h for options) ");
      CHKERRQ (ierr);
      ierr = PetscSynchronizedFGets (PETSC_COMM_WORLD, stdin, 99, instring);
      CHKERRQ (ierr);

      switch (instring [0])
	{
	case 'q':
	case 'Q':
	case 'x':
	case 'X':
	  {
	    if (dims < 3)
	      {
		ierr = PetscViewerDestroy (theviewer); CHKERRQ (ierr);
	      }
	    else
	      {
		ierr = GeomviewEnd (PETSC_COMM_WORLD); CHKERRQ (ierr);
	      }
	    ierr = PetscFinalize(); CHKERRQ (ierr);
	    return 0;
	  }
	case 't':
	case 'T':
	  {
	    transp=!transp;
	    break;
	  }
	case 'h':
	case 'H':
	case '?':
	  {
	    ierr = PetscPrintf (PETSC_COMM_WORLD, HELP_STRING);
	    break;
	  }
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	  {
	    current_entry = atoi (instring);
	    break;
	  }
	case 'b':
	case 'B':
	  {
	    current_entry--;
	    break;
	  }
	case 'v':
	case 'V':
	  {
	    if (dims == 3)
	      display_field = (display_field+1) % fields;
	  }
	default:
	  current_entry++;
	}
      if (current_entry < 0)
	current_entry = total_entries-1;
      if (current_entry >= total_entries)
	current_entry = 0;
    }

    } */

#undef __FUNCT__
#define __FUNCT__ "main"

/*++++++++++++++++++++++++++++++++++++++
  This is
  +latex+{\tt main()}.
  +html+ <tt>main()</tt>.

  int main It returns an int to the OS.

  int argc Argument count.

  char *argv[] Arguments.
  ++++++++++++++++++++++++++++++++++++++*/

int main (int argc, char *argv[])
{
  /* GnomeProgram *app; */
  /* struct poptOption options [] = {
    { "vector_max", 'vm', POPT_ARG_FLOAT, &vecmax, 0, "Reference vector length", "VECMAX" },
    { NULL, '\0', 0, NULL, 0, NULL, NULL }}; */
  int usermetacount=0, i, ierr;
  char **usermetanames, **usermetadata, filename [200];

  /*+ After
    +latex+{\tt PETSc}
    +html+ <tt>PETSc</tt>
    and glade/GNOME initialization, it gets the list of files matching the
    basename. +*/
  ierr = PetscInitialize (&argc, &argv, (char *)0, help); CHKERRQ (ierr);

  if (argc<2)
    {
      ierr = PetscPrintf (PETSC_COMM_WORLD, "Usage: tsview basename\n");
      CHKERRQ (ierr);
      return 1;
    }

#ifdef DEBUG
  ierr = PetscPrintf (PETSC_COMM_WORLD, "Command line:"); CHKERRQ (ierr);
  for (i=0; i<argc; i++)
    {
      ierr = PetscPrintf (PETSC_COMM_WORLD, " %s", argv[i]); CHKERRQ (ierr);
    }
  ierr = PetscPrintf (PETSC_COMM_WORLD, "\n"); CHKERRQ (ierr);
#endif

  vecmax = -1.0;
  ierr = PetscOptionsGetScalar (PETSC_NULL, "-vector_max", &vecmax,PETSC_NULL);
  CHKERRQ (ierr);
  ierr = PetscOptionsHasName (PETSC_NULL, "-no_transparency", &transp);
  CHKERRQ (ierr);
  transp = !transp;

  /* Kludge alert!  Setting argc to avoid gnome_program_init errors;
     fix: use GNOME arguments instead of PETSc. */
  argc=2;

  gnome_program_init ("TSView", VERSION, LIBGNOMEUI_MODULE, argc, argv, NULL);

  strncpy (filename, GLADE_DIRECTORY, 187);
  strcat (filename, "/tsview.glade");
  xml = glade_xml_new (filename, NULL, NULL);
  glade_xml_signal_autoconnect (xml);

  if (argc>1)
    the_basename = argv[1];
  else
    {
      /* Put in filter for .time0000000.cpu0000.meta */
      gtk_widget_show (glade_xml_get_widget (xml, "open_fileselect"));
    }

  if (refresh_stepnames ())
    exit (1);

  DPRINTF ("Loading first timestep, creating distributed array\n",0);
  strncpy (filename, basedirname, 198);
  strcat (filename, "/");
  strncat (filename, stepnames [0], 198 - strlen (stepnames [0]));
  ierr = IlluMultiLoad (filename, &theda, minmax+1,minmax+3,minmax+5,
			&fieldtypes, &usermetacount, &usermetanames,
			&usermetadata);
  CHKERRQ (ierr);

  /* Usermetadata xwidth, ywidth, zwidth override minmax in case IlluMulti
     version of saved data is 0.1. */
  for (i=0; i<usermetacount; i++)
    {
      if (!strncmp (usermetanames [i], "xwidth", 6))
	sscanf (usermetadata [i], "%lf", minmax+1);
      else if (!strncmp (usermetanames [i], "ywidth", 6))
	sscanf (usermetadata [i], "%lf", minmax+3);
      else if (!strncmp (usermetanames [i], "zwidth", 6))
	sscanf (usermetadata [i], "%lf", minmax+5);
    }

  ierr = DAGetGlobalVector (theda, &global); CHKERRQ (ierr);
  ierr = DAGetInfo (theda, &dimensions, PETSC_NULL,PETSC_NULL,PETSC_NULL,
		    PETSC_NULL,PETSC_NULL,PETSC_NULL, &num_fields,
		    PETSC_NULL,PETSC_NULL,PETSC_NULL); CHKERRQ (ierr);

  /* Main window title */
  {
    char main_window_title[100] = "TSView: ";
    strncat (main_window_title, basename (the_basename), 90);
    gtk_window_set_title
      (GTK_WINDOW (glade_xml_get_widget (xml, "main_window")),
       main_window_title);
  }
  gtk_widget_set_sensitive (glade_xml_get_widget (xml, "toolbar"), TRUE);
  gtk_widget_show (glade_xml_get_widget (xml, "vbox1"));

  display_timestep (usermetacount, usermetanames, usermetadata);

  gtk_main();

  DPRINTF ("Finalizing and exitting.\n",0);
  ierr = PetscFinalize (); CHKERRQ(ierr);
  return 0;
}
