/***************************************
  $Header: /cvsroot/petscgraphics/tsview-ng.c,v 1.35 2004/11/30 14:35:54 hazelsct Exp $

  This program views the output of a time series saved using
  +latex+{\tt IlluMultiSave()}.
  +html+ <tt>IlluMultiSave()</tt>.
  It basically just switches between timesteps; future versions may be more
  interesting.  The neat part of it is that it loads multiprocessor data and
  displays it on a single CPU.
  ***************************************/

static char help[] = "Displays the output of of a timestep series saved using IlluMultiSave().\n\
Usage:\n\
\n\
  tsview <basename> [-no_transparency]\n\
\n\
Then interactively flip through the timesteps (h or ? lists commands).\n";

#include "illuminator.h"
#include <glade/glade.h>
#include <gnome.h>
#include <libgnomeui/libgnomeui.h>
#include <sys/dir.h> /* For scandir(), alphasort, struct dirent */
#include <libgen.h>  /* For dirname(), basename() */
#include <string.h>  /* For strdup() */

/* Build with -DDEBUG for debugging output */
#undef DPRINTF
#ifdef DEBUG
#define DPRINTF(fmt, args...) PetscPrintf (PETSC_COMM_WORLD, "%s: " fmt, __FUNCT__, args)
#else
#define DPRINTF(fmt, args...)
#endif

/*+ Declared in illuminator.c, these give the current number of triangles on
  this node and corner coordinates and color information for each triangle. +*/
extern int num_triangles;
extern PetscScalar vertices[];

GladeXML *xml;
/* Filename list */
int entrynum=0, total_entries=0, current_timestep;
char *the_basename, *basedirname, **stepnames=NULL;
double current_time;

/* Window parameters and drawables */
int width=0, height=0, bpp=3, nx, ny, dataview_count=1;
GtkWidget *dataviews [1];
guchar *rgbbuf [1] = { NULL }, scalar_rgb[120000], ternary_rgb[120000],
  vector_rgb[120000], shear_rgb[120000];
gboolean scalar_auto_set=TRUE, ternary_auto_set=TRUE, vector_auto_set=TRUE,
  shear_auto_set=TRUE;
double sizemag;
PetscTruth transp;

/* PETSc structures etc. */
DA theda;
Vec global;
PetscScalar minmax[6] = { 0.,1., 0.,1., 0.,1. };
PetscScalar scales[11]; /* 2 scalar, 2 vector, 6 ternary, 1 tensor */
field_plot_type *fieldtypes;
int dimensions, num_fields, current_field, *field_index, num_variables[1],
  **variable_indices;

/* First some primary functions which do stuff, then callbacks, then main(). */

#undef __FUNCT__
#define __FUNCT__ "render_dataviews"

void render_dataviews ()
{
  int viewnum, nx,ny,nz, ierr;

  DPRINTF ("Rendering dataviews\n",0);
  if (dataview_count != 1)
    {
      printf ("dataview_count != 1 is not yet supported\n");
      exit(0);
    }
  for (viewnum=0; viewnum<dataview_count; viewnum++)
    {
      int nx,ny, xs,ys, xm,ym, i;
      PetscScalar minval, maxval, refmag, *global_array;
      GtkType type;
      char thestatus [100];

      /* (Re)allocate buffer */
      DPRINTF ("(Re)allocating RGB buffer\n",0);
      if (!(rgbbuf [viewnum] =
	    (guchar *) realloc (rgbbuf [viewnum],
				bpp*width*height*sizeof(guchar)))) {
	  printf ("ERROR: can't reallocate RGB buffer\n");
	  exit (1); }

      /* Render into rgbbuf [viewnum] */
      ierr = DAGetInfo (theda, PETSC_NULL, &nx,&ny,PETSC_NULL,
			PETSC_NULL,PETSC_NULL,PETSC_NULL, PETSC_NULL,
			PETSC_NULL,PETSC_NULL,PETSC_NULL); CHKERRQ (ierr);
      ierr = DAGetCorners (theda, &xs,&ys,PETSC_NULL, &xm,&ym,PETSC_NULL);
      CHKERRQ (ierr);
      if (dimensions == 2)
	{
	  char maxes [6][20];

	  ierr = VecGetArray (global, &global_array); CHKERRQ (ierr);

	  /* Plot with automatic or manual scaling, as appropriate */
	  switch (fieldtypes [current_field])
	    {
	    case FIELD_SCALAR:
	      if (scalar_auto_set)
		{
		  auto_scale (global_array, xm*ym, num_fields, current_field,
			      fieldtypes [current_field], 2, scales);
		  snprintf (maxes[0], 19, "%g", scales[0]);
		  snprintf (maxes[1], 19, "%g", scales[1]);
		  gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget
						 (xml, "scalar_min_entry")),
				      maxes[0]);
		  gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget
						 (xml, "scalar_max_entry")),
				      maxes[1]);
		}
	      ierr = render_rgb_local_2d
		(rgbbuf [viewnum], width,height,width, bpp, global_array,
		 num_fields, current_field, fieldtypes [current_field],
		 scales, nx,ny, xs,ys, xm,ym);
	      CHKERRQ (ierr);
	      break;

	    case FIELD_TERNARY:
	      if (ternary_auto_set)
		{
		  auto_scale (global_array, xm*ym, num_fields, current_field,
			      fieldtypes [current_field], 2, scales+2);
		  snprintf (maxes[0], 19, "%g", scales[2]);
		  snprintf (maxes[1], 19, "%g", scales[3]);
		  snprintf (maxes[2], 19, "%g", scales[4]);
		  snprintf (maxes[3], 19, "%g", scales[5]);
		  snprintf (maxes[4], 19, "%g", scales[6]);
		  snprintf (maxes[5], 19, "%g", scales[7]);
		  gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget
						 (xml, "ternary_1A_entry")),
				      maxes[0]);
		  gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget
						 (xml, "ternary_1B_entry")),
				      maxes[1]);
		  gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget
						 (xml, "ternary_2A_entry")),
				      maxes[2]);
		  gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget 
						 (xml, "ternary_2B_entry")),
				      maxes[3]);
		  gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget
						 (xml, "ternary_3A_entry")),
				      maxes[4]);
		  gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget
						 (xml, "ternary_3B_entry")),
				      maxes[5]);
		}
	      ierr = render_rgb_local_2d
		(rgbbuf [viewnum], width,height,width, bpp, global_array,
		 num_fields, current_field, fieldtypes [current_field],
		 scales+2, nx,ny, xs,ys, xm,ym);
	      CHKERRQ (ierr);
	      break;

	    case FIELD_VECTOR:
	      if (vector_auto_set)
		{
		  auto_scale (global_array, xm*ym, num_fields, current_field,
			      fieldtypes [current_field], 2, scales+8);
		  snprintf (maxes[0], 19, "%g", scales[9]);
		  gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget
						 (xml, "vector_max_entry")),
				      maxes[0]);
		}
	      ierr = render_rgb_local_2d
		(rgbbuf [viewnum], width,height,width, bpp, global_array,
		 num_fields, current_field, fieldtypes [current_field],
		 scales+8, nx,ny, xs,ys, xm,ym);
	      CHKERRQ (ierr);
	      break;

	    case FIELD_TENSOR_SHEAR:
	      if (shear_auto_set)
		{
		  auto_scale (global_array, xm*ym, num_fields, current_field,
			      fieldtypes [current_field], 2, scales+10);
		  snprintf (maxes[0], 19, "%g", scales[10]);
		  gtk_entry_set_text (GTK_ENTRY (glade_xml_get_widget
						 (xml, "shear_max_entry")),
				      maxes[0]);
		}
	      ierr = render_rgb_local_2d
		(rgbbuf [viewnum], width,height,width, bpp, global_array,
		 num_fields, current_field, fieldtypes [current_field],
		 scales+10, nx,ny, xs,ys, xm,ym);
	      CHKERRQ (ierr);
	      break;
	    }

	  ierr = VecRestoreArray (global, &global_array); CHKERRQ (ierr);
	}
      else
	{
	  /* Fixed negative z-direction for now */
	  PetscScalar eye[3], dir[3] = {0, 0, -1.}, right[3] = {.5, 0., 0.};
	  eye[0] = 0.5*(minmax[0]+minmax[1]);
	  eye[1] = 0.5*(minmax[2]+minmax[3]);
	  eye[2] = minmax[5] + minmax[1]-minmax[0];

	  ierr = DATriangulate
	    (theda, global, current_field, minmax, PETSC_DECIDE, PETSC_NULL,
	     PETSC_NULL, PETSC_FALSE, PETSC_FALSE, PETSC_FALSE); CHKERRQ(ierr);

	  for (i=0; i<width*height; i++)
	    {
	      rgbbuf[viewnum][4*i] = 0x50;
	      rgbbuf[viewnum][4*i+1] = 0x50;
	      rgbbuf[viewnum][4*i+2] = 0x50;
	      rgbbuf[viewnum][4*i+3] = 0xFF;
	    }
	  ierr = render_rgb_local_3d
	    (rgbbuf [viewnum], width,height, bpp, num_triangles, vertices,
	     eye, dir, right);
	  for (i=0; i<width*height; i++)
	    {
	      guchar tmp = rgbbuf[viewnum][4*i];
	      rgbbuf[viewnum][4*i] = rgbbuf[viewnum][4*i+2];
	      rgbbuf[viewnum][4*i+2] = tmp;
	    }
	  num_triangles = 0;
	}

      /* Draw buffer onto window */
      DPRINTF ("Painting rgb%s buffer onto window\n", (bpp==4) ? "_32" : "");
      if (!dataviews [viewnum])
	dataviews [viewnum] = glade_xml_get_widget (xml, "plot_area");
      gtk_drawing_area_size (GTK_DRAWING_AREA (dataviews [viewnum]),
			     width, height);
      if (bpp == 3)
	gdk_draw_rgb_image (dataviews [viewnum]->window,
			    dataviews[viewnum]->style->fg_gc[GTK_STATE_NORMAL],
			    0, 0, width, height, GDK_RGB_DITHER_MAX,
			    rgbbuf [viewnum], width * bpp);
      else if (bpp == 4)
	gdk_draw_rgb_32_image (dataviews [viewnum]->window,
			    dataviews[viewnum]->style->fg_gc[GTK_STATE_NORMAL],
			    0, 0, width, height, GDK_RGB_DITHER_MAX,
			    rgbbuf [viewnum], width * bpp);
    }
}


#undef __FUNCT__
#define __FUNCT__ "myfilter"

/*+ Little variable for myfilter() and refresh_stepnames(). +*/
static char *basefilename;

/*++++++++++++++++++++++++++++++++++++++
  This function returns non-zero for "qualifying" file names which start with
  the stored files' basename.time and end with
  +latex+{\tt .cpu0000.meta}.
  +html+ <tt>.cpu0000.meta</tt>.
  It is used as the
  +latex+{\tt select()} function for {\tt scandir()} in {\tt main()}.
  +html+ <tt>select()</tt> function for <tt>scandir()</tt> in <tt>main()</tt>.

  int myfilter Returns non-zero for qualifying filenames.

  const struct dirent *direntry Directory entry with filename to test.
  ++++++++++++++++++++++++++++++++++++++*/

int myfilter (const struct dirent *direntry)
{
  if ((!strncmp (direntry->d_name, basefilename, strlen(basefilename))) &&
      (!strncmp (direntry->d_name + strlen(basefilename), ".time", 5)))
    return (!strncmp (direntry->d_name + strlen(direntry->d_name) - 13,
		      ".cpu0000.meta", 13));
  return 0;
}


void on_plot_area_expose_event (GtkWidget *widget, GdkEventExpose *event,
				gpointer user_data)
{
  if (bpp == 3)
    gdk_draw_rgb_image (widget->window, widget->style->fg_gc[GTK_STATE_NORMAL],
			0, 0, width, height, GDK_RGB_DITHER_MAX, rgbbuf [0],
			width * bpp);
  else if (bpp == 4)
    gdk_draw_rgb_32_image (widget->window,
			   widget->style->fg_gc[GTK_STATE_NORMAL],
			   0, 0, width, height, GDK_RGB_DITHER_MAX, rgbbuf [0],
			   width * bpp);
}

void on_scalar_scale_area_expose_event
(GtkWidget *widget, GdkEventExpose *event, gpointer user_data)
{ gdk_draw_rgb_image (widget->window, widget->style->fg_gc[GTK_STATE_NORMAL],
		      0, 0, 200, 200, GDK_RGB_DITHER_MAX, scalar_rgb, 600); }

void on_ternary_scale_area_expose_event
(GtkWidget *widget, GdkEventExpose *event, gpointer user_data)
{ gdk_draw_rgb_image (widget->window, widget->style->fg_gc[GTK_STATE_NORMAL],
		      0, 0, 200, 200, GDK_RGB_DITHER_MAX, ternary_rgb, 600); }

void on_vector_scale_area_expose_event
(GtkWidget *widget, GdkEventExpose *event, gpointer user_data)
{ gdk_draw_rgb_image (widget->window, widget->style->fg_gc[GTK_STATE_NORMAL],
		      0, 0, 200, 200, GDK_RGB_DITHER_MAX, vector_rgb, 600); }

void on_shear_scale_area_expose_event
(GtkWidget *widget, GdkEventExpose *event, gpointer user_data)
{ gdk_draw_rgb_image (widget->window, widget->style->fg_gc[GTK_STATE_NORMAL],
		      0, 0, 200, 200, GDK_RGB_DITHER_MAX, shear_rgb, 600); }

void on_scalar_auto_checkbutton_toggled
(GtkWidget *thebutton, gpointer user_data)
{
  scalar_auto_set = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON (thebutton));
  if (scalar_auto_set)
    {
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "scalar_min_label"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "scalar_max_label"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "scalar_min_entry"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "scalar_max_entry"), FALSE);
    }
  else
    {
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "scalar_min_label"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "scalar_max_label"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "scalar_min_entry"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "scalar_max_entry"), TRUE);
    }
}

void on_ternary_auto_checkbutton_toggled
(GtkWidget *thebutton, gpointer user_data)
{
  ternary_auto_set = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON (thebutton));
  if (ternary_auto_set)
    {
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_1A_label"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_1B_label"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_2A_label"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_2B_label"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_3A_label"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_3B_label"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_1A_entry"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_1B_entry"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_2A_entry"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_2B_entry"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_3A_entry"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_3B_entry"), FALSE);
    }
  else
    {
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_1A_label"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_1B_label"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_2A_label"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_2B_label"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_3A_label"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_3B_label"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_1A_entry"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_1B_entry"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_2A_entry"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_2B_entry"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_3A_entry"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "ternary_3B_entry"), TRUE);
    }
}

void on_vector_auto_checkbutton_toggled
(GtkWidget *thebutton, gpointer user_data)
{
  vector_auto_set = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON (thebutton));
  if (vector_auto_set)
    {
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "vector_max_label"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "vector_symm_label"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "vector_max_entry"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "vector_symm_spinbutton"), FALSE);
    }
  else
    {
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "vector_max_label"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "vector_symm_label"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "vector_max_entry"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "vector_symm_spinbutton"), TRUE);
    }
}

void on_shear_auto_checkbutton_toggled
(GtkWidget *thebutton, gpointer user_data)
{
  shear_auto_set = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON (thebutton));
  if (shear_auto_set)
    {
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "shear_max_label"), FALSE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "shear_max_entry"), FALSE);
    }
  else
    {
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "shear_max_label"), TRUE);
      gtk_widget_set_sensitive
	(glade_xml_get_widget (xml, "shear_max_entry"), TRUE);
    }
}

void on_scalar_min_entry_changed (GtkWidget *theentry, gpointer user_data)
{ G_CONST_RETURN gchar *entrytext = gtk_entry_get_text (GTK_ENTRY (theentry));
  sscanf (entrytext, "%lf", scales); }

void on_scalar_max_entry_changed (GtkWidget *theentry, gpointer user_data)
{ G_CONST_RETURN gchar *entrytext = gtk_entry_get_text (GTK_ENTRY (theentry));
  sscanf (entrytext, "%lf", scales+1); }

void on_ternary_1A_entry_changed (GtkWidget *theentry, gpointer user_data)
{ G_CONST_RETURN gchar *entrytext = gtk_entry_get_text (GTK_ENTRY (theentry));
  sscanf (entrytext, "%lf", scales+2); }

void on_ternary_1B_entry_changed (GtkWidget *theentry, gpointer user_data)
{ G_CONST_RETURN gchar *entrytext = gtk_entry_get_text (GTK_ENTRY (theentry));
  sscanf (entrytext, "%lf", scales+3); }

void on_ternary_2A_entry_changed (GtkWidget *theentry, gpointer user_data)
{ G_CONST_RETURN gchar *entrytext = gtk_entry_get_text (GTK_ENTRY (theentry));
  sscanf (entrytext, "%lf", scales+4); }

void on_ternary_2B_entry_changed (GtkWidget *theentry, gpointer user_data)
{ G_CONST_RETURN gchar *entrytext = gtk_entry_get_text (GTK_ENTRY (theentry));
  sscanf (entrytext, "%lf", scales+5); }

void on_ternary_3A_entry_changed (GtkWidget *theentry, gpointer user_data)
{ G_CONST_RETURN gchar *entrytext = gtk_entry_get_text (GTK_ENTRY (theentry));
  sscanf (entrytext, "%lf", scales+6); }

void on_ternary_3B_entry_changed (GtkWidget *theentry, gpointer user_data)
{ G_CONST_RETURN gchar *entrytext = gtk_entry_get_text (GTK_ENTRY (theentry));
  sscanf (entrytext, "%lf", scales+7); }

void on_vector_max_entry_changed (GtkWidget *theentry, gpointer user_data)
{ G_CONST_RETURN gchar *entrytext = gtk_entry_get_text (GTK_ENTRY (theentry));
  sscanf (entrytext, "%lf", scales+9); }

void on_vector_symm_spinbutton_changed (GtkWidget *theentry,gpointer user_data)
{ int symmetry;
  G_CONST_RETURN gchar *entrytext = gtk_entry_get_text (GTK_ENTRY (theentry));
  sscanf (entrytext, "%d", &symmetry);
  /* Because this sometimes sends events with outrageous symmetry numbers */
  if (symmetry <= 10) {
    render_scale_2d (vector_rgb, 200, 200, 3, FIELD_VECTOR, symmetry);
    on_vector_scale_area_expose_event
      (glade_xml_get_widget (xml, "vector_scale_area"), NULL, user_data); }}

void on_shear_max_entry_changed (GtkWidget *theentry, gpointer user_data)
{ G_CONST_RETURN gchar *entrytext = gtk_entry_get_text (GTK_ENTRY (theentry));
  sscanf (entrytext, "%lf", scales+10); }

/*++++++++++++++++++++++++++++++++++++++
  This loads the names of the files into a long list.
  ++++++++++++++++++++++++++++++++++++++*/

int refresh_stepnames ()
{
  struct dirent **namelist;
  char *filec, *dirc;
  int i, ierr;

  filec = strdup (the_basename);
  dirc = strdup (the_basename);
  basefilename = basename (filec);
  basedirname = dirname (dirc);

  total_entries = scandir (basedirname, &namelist, myfilter, alphasort);
  if (!total_entries)
    {
      ierr = PetscPrintf (PETSC_COMM_WORLD, "No such files\n");
      CHKERRQ (ierr);
      return 1;
    }
  if (total_entries < 0)
    {
      ierr = PetscPrintf (PETSC_COMM_WORLD, "Error scanning directory %s\n",
			  basedirname); CHKERRQ (ierr);
      ierr = PetscFinalize (); CHKERRQ(ierr);
      return 1;
    }
  DPRINTF ("%d eligible files:\n", total_entries);

  if (!(stepnames = (char **) realloc
	(stepnames, total_entries*sizeof (char *))))
    {
      ierr = PetscPrintf (PETSC_COMM_WORLD, "Error allocating memory\n");
      CHKERRQ (ierr);
      return 1;
    }
  for (i=0; i<total_entries; i++)
    {
      int filength = strlen(namelist[i]->d_name);

      stepnames [i] = (char *) malloc ((filength-12)*sizeof(char));
      strncpy (stepnames [i], namelist[i]->d_name, filength-13);
      stepnames [i] [filength-13] = '\0';
      free (namelist[i]);
      DPRINTF ("[%d] %s\n", i, stepnames [i]);
      CHKERRQ (ierr);
    }

  free (namelist);
  return 0;
}

void change_variable (GtkWidget *widget, gpointer user_data)
{
  current_field = GPOINTER_TO_INT (widget);
  DPRINTF ("Switching to variable %d\n", current_field);
  render_dataviews();
  gtk_notebook_set_current_page
    (GTK_NOTEBOOK (glade_xml_get_widget (xml, "scale_notebook")),
     fieldtypes [current_field] >> 4);
}

void on_mag_spin_value_changed (GtkWidget *mag_spin, gpointer user_data) {
  G_CONST_RETURN gchar *entrytext;
  entrytext = gtk_entry_get_text (GTK_ENTRY (mag_spin));
  sscanf (entrytext, "%lf", &sizemag);
  width = (int) (minmax [1] * sizemag);
  height = (int) (minmax [3] * sizemag);

  render_dataviews();
}


void display_timestep (int usermetacount, char **usermetanames,
		       char **usermetadata)
{
  int i;
  static char step_buffer [20], time_buffer [20];

  for (i=0; i<usermetacount; i++)
    {
      if (!strncmp (usermetanames [i], "timestep", 8))
	sscanf (usermetadata [i], "%d", &current_timestep);
      else if (!strncmp (usermetanames [i], "time", 4))
	sscanf (usermetadata [i], "%lf", &current_time);
    }
  snprintf (step_buffer, 19, "Timestep: %d", current_timestep);
  gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (xml, "timestep_label")),
		      step_buffer);
  snprintf (time_buffer, 19, "Time: %g", current_time);
  gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (xml, "time_label")),
		      time_buffer);

  on_mag_spin_value_changed (glade_xml_get_widget (xml, "mag_spin"), NULL);
}


void on_save_activate (GtkWidget *widget, gpointer user_data)
{
  int i, ierr;
  char **usermetanames, **usermetadata, filename [200], number[10];
  FILE *outppm;

  strncpy (filename, basedirname, 198);
  strcat (filename, "/");
  strncat (filename, stepnames [entrynum], 198 - strlen (filename));
  snprintf (number, 9, "-f%d", current_field);
  strncat (filename, number, 198 - strlen (filename));
  strncat (filename, ".ppm", 198 - strlen (filename));

  DPRINTF ("Saving image with filename %s\n", filename);
  if (!(outppm = fopen (filename, "w")))
    {
      PetscPrintf (PETSC_COMM_WORLD, "Error opening file %s\n", filename);
      return;
    }
  fprintf (outppm, "P6\n%d %d\n255\n", width, height);
  /* If more than 3 bytes per pixel, write just the first three (RGB) */
  if (bpp == 3)
    fwrite (rgbbuf [0], sizeof (guchar), 3*width*height, outppm);
  else
    for (i=0; i<width*height; i++)
      fwrite (rgbbuf [0] + i*bpp, sizeof (guchar), 3, outppm);
  fclose (outppm);
}


void on_timestep_spin_value_changed (GtkWidget *timestep_spin, gpointer user_data) {
  int usermetacount, ierr;
  G_CONST_RETURN gchar *entrytext;
  char **usermetanames, **usermetadata, filename [200], **field_name;
  GtkWidget *variable_options, *variable_menu, **variable_item;

  entrytext = gtk_entry_get_text (GTK_ENTRY (timestep_spin));
  sscanf (entrytext, "%d", &entrynum);

  /* Bound the entrynum between 0 and total_entries-1; -11 is the minimum of
     the widget (from jump), 1000001 is the maximum. */
  if ((entrynum < 0 && entrynum != -11) || entrynum == 1000001)
    entrynum = total_entries-1;
  if ((entrynum >= total_entries && entrynum != 1000001) || entrynum == -11)
    entrynum = 0;
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (timestep_spin),
			     (gfloat) entrynum);

  strncpy (filename, basedirname, 198);
  strcat (filename, "/");
  strncat (filename, stepnames [entrynum], 198 - strlen (filename));

  ierr = IlluMultiRead (theda, global, filename, &usermetacount,&usermetanames,
			&usermetadata); CHKERRQ (ierr);

  display_timestep (usermetacount, usermetanames, usermetadata);
}


void on_refresh_activate (GtkWidget *none, gpointer user_data) {
  if (refresh_stepnames ()) exit (1); }


void on_about_activate (GtkWidget *none, gpointer user_data) {
  gtk_widget_show (glade_xml_get_widget (xml, "about")); }


#undef __FUNCT__
#define __FUNCT__ "main"

/*++++++++++++++++++++++++++++++++++++++
  This is
  +latex+{\tt main()}.
  +html+ <tt>main()</tt>.

  int main It returns an int to the OS.

  int argc Argument count.

  char *argv[] Arguments.
  ++++++++++++++++++++++++++++++++++++++*/

int main (int argc, char *argv[])
{
  int usermetacount=0, i, ierr;
  char **usermetanames, **usermetadata, filename [200], **field_name;
  GtkWidget *variable_options, *variable_menu, **variable_item;

  /*+ After
    +latex+{\tt PETSc}
    +html+ <tt>PETSc</tt>
    and glade/GNOME initialization, it gets the list of files matching the
    basename. +*/
  ierr = PetscInitialize (&argc, &argv, (char *)0, help); CHKERRQ (ierr);

  if (argc<2)
    {
      ierr = PetscPrintf (PETSC_COMM_WORLD, "Usage: tsview basename\n");
      CHKERRQ (ierr);
      return 1;
    }

#ifdef DEBUG
  ierr = PetscPrintf (PETSC_COMM_WORLD, "Command line:"); CHKERRQ (ierr);
  for (i=0; i<argc; i++)
    {
      ierr = PetscPrintf (PETSC_COMM_WORLD, " %s", argv[i]); CHKERRQ (ierr);
    }
  ierr = PetscPrintf (PETSC_COMM_WORLD, "\n"); CHKERRQ (ierr);
#endif

  ierr = PetscOptionsHasName (PETSC_NULL, "-no_transparency", &transp);
  CHKERRQ (ierr);
  transp = !transp;

  /* Kludge alert!  Setting argc to avoid gnome_program_init errors;
     fix: use GNOME arguments instead of PETSc. */
  argc=2;

  DPRINTF ("Running gnome_program_init with argc=%d\n", argc);
  gnome_program_init ("TSView", VERSION, LIBGNOMEUI_MODULE, argc, argv, NULL);

  strncpy (filename, GLADE_DIRECTORY, 187);
  strcat (filename, "/tsview.glade");
  DPRINTF ("Loading Glade file %s\n", filename);
  xml = glade_xml_new (filename, NULL, NULL);
  glade_xml_signal_autoconnect (xml);

  if (argc>1)
    the_basename = argv[1];
  else
    {
      /* Put in filter for .time0000000.cpu0000.meta */
      gtk_widget_show (glade_xml_get_widget (xml, "open_fileselect"));
    }

  DPRINTF ("Loading list of timestep names\n",0);
  if (refresh_stepnames ())
    exit (1);

  DPRINTF ("Loading first timestep, creating distributed array\n",0);
  strncpy (filename, basedirname, 198);
  strcat (filename, "/");
  strncat (filename, stepnames [0], 198 - strlen (stepnames [0]));
  ierr = IlluMultiLoad (filename, &theda, minmax+1,minmax+3,minmax+5,
			&fieldtypes, &usermetacount, &usermetanames,
			&usermetadata);
  CHKERRQ (ierr);

  /* Usermetadata xwidth, ywidth, zwidth override minmax in case IlluMulti
     version of saved data is 0.1. */
  DPRINTF ("Checking usermetadata for width information\n",0);
  for (i=0; i<usermetacount; i++)
    {
      if (!strncmp (usermetanames [i], "xwidth", 6))
	sscanf (usermetadata [i], "%lf", minmax+1);
      else if (!strncmp (usermetanames [i], "ywidth", 6))
	sscanf (usermetadata [i], "%lf", minmax+3);
      else if (!strncmp (usermetanames [i], "zwidth", 6))
	sscanf (usermetadata [i], "%lf", minmax+5);
    }

  DPRINTF ("Getting distributed array global vector and info\n",0);
  ierr = DAGetGlobalVector (theda, &global); CHKERRQ (ierr);
  ierr = DAGetInfo (theda, &dimensions, PETSC_NULL,PETSC_NULL,PETSC_NULL,
		    PETSC_NULL,PETSC_NULL,PETSC_NULL, &num_fields,
		    PETSC_NULL,PETSC_NULL,PETSC_NULL); CHKERRQ (ierr);
  if (dimensions == 1)
    SETERRQ (PETSC_ERR_ARG_OUTOFRANGE, "tsview-ng only supports 2-D or 3-D distributed arrays.")
  else if (dimensions == 2)
    bpp = 3;
  else
    bpp = 4;

  /* Build menu of field variables */
  variable_options = glade_xml_get_widget (xml, "variable_menu");
  gtk_option_menu_remove_menu (GTK_OPTION_MENU (variable_options));
  variable_menu = gtk_menu_new ();
  variable_item = (GtkWidget **) malloc (num_fields * sizeof (GtkWidget *));
  field_name = (char **) malloc (num_fields * sizeof (char *));
  field_index = (int *) malloc (num_fields * sizeof (int));
  /* For now, use scalar plots for all types in 3-D */
  if (dimensions == 2)
    field_indices (num_fields, dimensions, fieldtypes, field_index);
  else
    for (i=0; i<num_fields; i++)
      field_index[i] = i;
  DPRINTF ("Field indices:\n",0);
  for (i=0; i<num_fields && field_index [i] != -1; i++)
    {
      ierr = DAGetFieldName (theda, field_index [i], field_name+i);
      CHKERRQ (ierr);
      DPRINTF ("%d index %d name %s\n", i, field_index [i], field_name [i]);
      variable_item [i] = gtk_menu_item_new_with_label (field_name [i]);
      gtk_menu_append (GTK_MENU (variable_menu), variable_item [i]);
      gtk_signal_connect_object (GTK_OBJECT (variable_item [i]), "activate",
				 GTK_SIGNAL_FUNC (change_variable),
				 GINT_TO_POINTER (field_index [i]));
      gtk_widget_show (variable_item [i]);
    }
  gtk_option_menu_set_menu (GTK_OPTION_MENU (variable_options), variable_menu);
  gtk_widget_show (variable_menu);
  gtk_widget_show (variable_options);

  /* Scale images */
  render_scale_2d (scalar_rgb, 200, 200, 3, FIELD_SCALAR, 1);
  render_scale_2d (ternary_rgb, 200, 200, 3, FIELD_TERNARY, 1);
  render_scale_2d (vector_rgb, 200, 200, 3, FIELD_VECTOR, 1);
  render_scale_2d (shear_rgb, 200, 200, 3, FIELD_TENSOR_SHEAR, 1);

  gtk_notebook_set_current_page
    (GTK_NOTEBOOK (glade_xml_get_widget (xml, "scale_notebook")),
     fieldtypes [0] >> 4);

  /* Main window title */
  {
    char main_window_title[100] = "TSView: ";
    GtkWidget *main_window = glade_xml_get_widget (xml, "main_window");

    strncat (main_window_title, basename (the_basename), 90);
    gtk_window_set_title (GTK_WINDOW (main_window), main_window_title);
    gtk_widget_show (main_window);
  }

  DPRINTF ("Displaying first timestep\n",0);
  display_timestep (usermetacount, usermetanames, usermetadata);

  DPRINTF ("Running main loop\n",0);
  gtk_main();

  DPRINTF ("Finalizing and exitting.\n",0);
  ierr = PetscFinalize (); CHKERRQ(ierr);
  return 0;
}
