-- options
options.info = true
options.close = true


-- check()
function check(account, mbox)
	checktype_aux(account, 'table')
	checktype_aux(mbox, 'string')

	checktype_aux(account.server, 'string')
	checktype_aux(account.username, 'string')
	checktype_aux(account.password, 'string')

	if (ifcore.login(account) ~= true) then
		return
	end

	local _, exist, recent, unseen = ifcore.status(account, mbox)

	if (type(options) == 'table' and options.info == true) then
		print(string.format("%d messages, %d recent, %d unseen, in %s@%s/%s.",
		    exist, recent, unseen, account.username, account.server, mbox))
	end

	return exist, recent, unseen
end


-- match()
function match(account, mbox, criteria)
	checktype_aux(account, 'table')
	checktype_aux(mbox, 'string')
	checktype_aux(criteria, 'table')

	checktype_aux(account.server, 'string')
	checktype_aux(account.username, 'string')
	checktype_aux(account.password, 'string')

	if (ifcore.login(account) ~= true) then
		return
	end

	ifcore.select(account, mbox)

	local s = 'ALL '
	if (criteria.invert ~= true) then
		for ka, va in ipairs(criteria) do
			if (type(va) == 'string') then
				s = s .. '' .. '(' .. va .. ')' .. ' '
			elseif (type(va) == 'table') then
				for i = 1, table.getn(va) - 1 do
					s = s .. 'OR '
				end
				for ko, vo in ipairs(va) do
					if (type(vo) ~= 'string') then
						error('filter rule not a string', 2)
					end
					s = s .. '(' .. vo .. ') '
				end
			else
				error('filter element not a string or table', 2)
			end
		end
	else
		for i = 1, table.getn(criteria) - 1 do
			s = s .. 'OR '
		end
		for ko, vo in ipairs(criteria) do
			if (type(vo) == 'string') then
				s = s .. '' .. '(' .. vo .. ')' .. ' '
			elseif (type(vo) == 'table') then
				s = s .. '('
				for ka, va in ipairs(vo) do
					if (type(va) ~= 'string') then
						error('filter rule not a string', 2)
					end
					s = s .. va .. ' '
				end
				s = string.gsub(s, '(.+) ', '%1')
				s = s .. ') '
			else
				error('filter rule not a string or table', 2)
			end
		end
	end

	s = string.gsub(s, '(.+) ', '%1')

	local _, results = ifcore.search(account, s)

	if (type(options) == 'table' and options.close == true) then
		ifcore.close(account)
	end

	if (results == nil) then
		return {}
	end

	local t = {}
	for n in string.gfind(results, '%d+') do
		table.insert(t, n)
	end

	return t
end


-- flag()
function flag(account, mbox, mode, flags, messages)
	checktype_aux(account, 'table')
	checktype_aux(mbox, 'string')
	checktype_aux(mode, 'string')
	checktype_aux(flags, 'table')
	checktype_aux(messages, 'table')

	checktype_aux(account.server, 'string')
	checktype_aux(account.username, 'string')
	checktype_aux(account.password, 'string')

	local r = flag_aux(account, mbox, mode, flags, messages)

	if (type(options) == 'table' and options.info == true and messages ~= nil and
	    r == true) then
		print(string.format("%d messages flagged in %s@%s/%s.", table.getn(messages),
		    account.username, account.server, mbox))
	end

	return r
end

function flag_aux(account, mbox, mode, flags, messages)
	if (table.getn(messages) == 0) then
		return
	end

	if (mode ~= 'add' and mode ~= 'remove' and mode ~= 'replace') then
		error('"add", "remove" or "replace" expected for mode', 3)
	end


	if (ifcore.login(account) ~= true) then
		return
	end

	ifcore.select(account, mbox)

	local f = ''
	if (table.getn(flags) ~= 0) then
		f = '\\' .. table.concat(flags, ' \\')
	end

	local r = ifcore.store(account, table.concat(messages, ','), mode, f)

	if (type(options) == 'table' and options.close == true) then
		ifcore.close(account)
	end

	return r
end


-- copy()
function copy(srcaccount, srcmbox, dstaccount, dstmbox, messages)
	checktype_aux(srcaccount, 'table')
	checktype_aux(srcmbox, 'string')
	checktype_aux(dstaccount, 'table')
	checktype_aux(dstmbox, 'string')
	checktype_aux(messages, 'table')

	checktype_aux(srcaccount.server, 'string')
	checktype_aux(srcaccount.username, 'string')
	checktype_aux(srcaccount.password, 'string')

	checktype_aux(dstaccount.server, 'string')
	checktype_aux(dstaccount.username, 'string')
	checktype_aux(dstaccount.password, 'string')

	local r = copy_aux(srcaccount, srcmbox, dstaccount, dstmbox, messages)

	if (type(options) == 'table' and options.info == true and messages ~= nil and
	    r == true) then
		print(string.format("%d messages copied from %s@%s/%s to %s@%s/%s.",
		    table.getn(messages), srcaccount.username, srcaccount.server, srcmbox,
		    dstaccount.username, dstaccount.server, dstmbox))
	end

	return r
end

function copy_aux(srcaccount, srcmbox, dstaccount, dstmbox, messages)
	if (table.getn(messages) == 0) then
		return
	end

	if (ifcore.login(srcaccount) ~= true) then
		return
	end

	local r = false
	if (srcaccount == dstaccount) then
		ifcore.select(srcaccount, srcmbox)
		r = ifcore.copy(srcaccount, table.concat(messages, ','), dstmbox)
		if (type(options) == 'table' and options.close == true) then
			ifcore.close(srcaccount)
		end
	else
		local fast = fetchfast(srcaccount, srcmbox, messages)
		local msgs = fetchtext(srcaccount, srcmbox, messages)

		if (ifcore.login(dstaccount) ~= true) then
			return
		end

		for i in pairs(fast) do
			for k, v in ipairs(fast[i]['flags']) do
				if (string.lower(v) == 'recent') then
					table.remove(fast[i]['flags'], k)
				end
			end

			local f = ''
			if (table.getn(fast[i]['flags']) ~= 0) then
				f = '\\' .. table.concat(fast[i]['flags'], ' \\')
			end
			r = ifcore.append(dstaccount, dstmbox, msgs[i], f, fast[i]['date'])
		end
	end

	return r
end


-- delete()
function delete(account, mbox, messages)
	checktype_aux(account, 'table')
	checktype_aux(mbox, 'string')
	checktype_aux(messages, 'table')

	checktype_aux(account.server, 'string')
	checktype_aux(account.username, 'string')
	checktype_aux(account.password, 'string')

	local r = flag_aux(account, mbox, 'add', { 'Deleted' }, messages)

	if (type(options) == 'table' and options.info == true and messages ~= nil and
		r == true) then
		print(string.format("%d messages deleted in %s@%s/%s.", table.getn(messages),
		    account.username, account.server, mbox))
	end

	return r
end


-- move()
function move(srcaccount, srcmbox, dstaccount, dstmbox, messages)
	checktype_aux(srcaccount, 'table')
	checktype_aux(srcmbox, 'string')
	checktype_aux(dstaccount, 'table')
	checktype_aux(dstmbox, 'string')
	checktype_aux(messages, 'table')

	checktype_aux(srcaccount.server, 'string')
	checktype_aux(srcaccount.username, 'string')
	checktype_aux(srcaccount.password, 'string')

	checktype_aux(dstaccount.server, 'string')
	checktype_aux(dstaccount.username, 'string')
	checktype_aux(dstaccount.password, 'string')

	local rc = copy_aux(srcaccount, srcmbox, dstaccount, dstmbox, messages)
	local rf = false
	if (rc == true) then
		rf = flag_aux(srcaccount, srcmbox, 'add', { 'Deleted' }, messages)
	end

	if (type(options) == 'table' and options.info == true and messages ~= nil and
	    rc == true and rf == true) then
		print(string.format("%d messages moved from %s@%s/%s to %s@%s/%s.",
		    table.getn(messages), srcaccount.username, srcaccount.server, srcmbox,
		    dstaccount.username, dstaccount.server, dstmbox))
	end

	return rc == true and rf == true
end


-- fetchheader()
function fetchheader(account, mbox, messages)
	checktype_aux(account, 'table')
	checktype_aux(mbox, 'string')
	checktype_aux(messages, 'table')

	checktype_aux(account.server, 'string')
	checktype_aux(account.username, 'string')
	checktype_aux(account.password, 'string')

	if (table.getn(messages) == 0) then
		return
	end

	if (ifcore.login(account) ~= true) then
		return
	end
	ifcore.select(account, mbox)

	local results = {}
	for i, v in ipairs(messages) do
		local _, fetch = ifcore.fetch(account, tostring(v), 'BODY.PEEK[HEADER]')
		local _, _, header = string.find(fetch, '{%d+}\r\n(.*)%)\r\n')
		if (header ~= nil) then
			results[v] = header
		end
	end

	if (type(options) == 'table' and options.close == true) then
		ifcore.close(account)
	end

	return results
end


-- fetchbody()
function fetchbody(account, mbox, messages)
	checktype_aux(account, 'table')
	checktype_aux(mbox, 'string')
	checktype_aux(messages, 'table')

	checktype_aux(account.server, 'string')
	checktype_aux(account.username, 'string')
	checktype_aux(account.password, 'string')

	if (table.getn(messages) == 0) then
		return
	end

	if (ifcore.login(account) ~= true) then
		return
	end

	ifcore.select(account, mbox)

	local results = {}
	for i, v in ipairs(messages) do
		local _, fetch = ifcore.fetch(account, tostring(v), 'BODY.PEEK[TEXT]')
		local _, _, body = string.find(fetch, '{%d+}\r\n(.*)%)\r\n')
		if (body ~= nil) then
			results[v] = body
		end
	end

	if (type(options) == 'table' and options.close == true) then
		ifcore.close(account)
	end

	return results
end


-- fetchtext()
function fetchtext(account, mbox, messages)
	checktype_aux(account, 'table')
	checktype_aux(mbox, 'string')
	checktype_aux(messages, 'table')

	checktype_aux(account.server, 'string')
	checktype_aux(account.username, 'string')
	checktype_aux(account.password, 'string')

	if (table.getn(messages) == 0) then
		return
	end

	if (ifcore.login(account) ~= true) then
		return
	end

	ifcore.select(account, mbox)

	local results = {}
	for i, v in ipairs(messages) do
		local _, fetch = ifcore.fetch(account, tostring(v), 'BODY.PEEK[HEADER]')
		local _, _, header = string.find(fetch, '{%d+}\r\n(.*)%)\r\n')
		local _, fetch = ifcore.fetch(account, tostring(v), 'BODY.PEEK[TEXT]')
		local _, _, body = string.find(fetch, '{%d+}\r\n(.*)%)\r\n')
		if (header ~= nil and body ~= nil) then
			results[v] = header .. body
		end
	end

	if (type(options) == 'table' and options.close == true) then
		ifcore.close(account)
	end

	return results
end


-- fetchfast()
function fetchfast(account, mbox, messages)
	checktype_aux(account, 'table')
	checktype_aux(mbox, 'string')
	checktype_aux(messages, 'table')

	checktype_aux(account.server, 'string')
	checktype_aux(account.username, 'string')
	checktype_aux(account.password, 'string')

	if (table.getn(messages) == 0) then
		return
	end

	if (ifcore.login(account) ~= true) then
		return
	end

	ifcore.select(account, mbox)

	local results = {}
	for i, v in ipairs(messages) do
		local _, fetch = ifcore.fetch(account, tostring(v), 'FAST')
		if (fetch ~= nil) then
			local _, _, flags = string.find(fetch, 'FLAGS %((.*)%) ')
			local _, _, date = string.find(fetch, 'INTERNALDATE "(.*)" ')
			local _, _, size = string.find(fetch, 'RFC822%.SIZE (%d+)')
			local f = {}
			for s in string.gfind(flags, '%w+') do
				table.insert(f, s)
			end
			results[v] = {}
			results[v]['flags'] = f
			results[v]['date'] = date
			results[v]['size'] = size
		end
	end

	if (type(options) == 'table' and options.close == true) then
		ifcore.close(account)
	end

	return results
end


-- fetchheader()
function fetchheaders(account, mbox, headers, messages)
	checktype_aux(account, 'table')
	checktype_aux(mbox, 'string')
	checktype_aux(headers, 'table')
	checktype_aux(messages, 'table')

	checktype_aux(account.server, 'string')
	checktype_aux(account.username, 'string')
	checktype_aux(account.password, 'string')

	if (table.getn(messages) == 0) then
		return
	end

	if (ifcore.login(account) ~= true) then
		return
	end

	ifcore.select(account, mbox)

	local results = {}
	for i, v in ipairs(messages) do
		local _, fetch = ifcore.fetch(account, tostring(v), 'BODY.PEEK[HEADER.FIELDS (' ..
		    table.concat(headers, ' ') .. ')]')
		local _, _, hdrs = string.find(fetch, '{%d+}\r\n(.*)%)\r\n')
		if (hdrs ~= nil) then
			results[v] = hdrs
		end
	end

	if (type(options) == 'table' and options.close == true) then
		ifcore.close(account)
	end

	return results
end


-- other functions
function checktype_aux(arg, argtype)
	if (type(arg) ~= argtype) then
		error(string.format('%s expected, got %s', argtype, type(arg)), 3)
	end
end
