-- options
options.info = true
options.close = true


-- check()
function check(account, mbox)
    check_type(account, 'table')
    check_type(mbox, 'string')

    check_type(account.server, 'string')
    check_type(account.username, 'string')
    check_type(account.password, 'string')

    if (ifcore.login(account) ~= true) then
	return
    end

    local _, exist, recent, unseen = ifcore.status(account, mbox)

    if (type(options) == 'table' and options.info == true) then
	print(string.format("%d messages, %d recent, %d unseen, in %s@%s/%s.",
			    exist, recent, unseen, account.username,
			    account.server, mbox))
    end

    return exist, recent, unseen
end


-- match()
function match(account, mbox, criteria)
    check_type(account, 'table')
    check_type(mbox, 'string')
    check_type(criteria, 'table')

    check_type(account.server, 'string')
    check_type(account.username, 'string')
    check_type(account.password, 'string')

    if (ifcore.login(account) ~= true) then
	return
    end

    if (ifcore.select(account, mbox) ~= true) then
	return {}
    end

    local s = 'ALL '
    if (criteria.invert ~= true) then
	for ka, va in ipairs(criteria) do
	    if (type(va) == 'string') then
		s = s .. '' .. '(' .. va .. ')' .. ' '
	    elseif (type(va) == 'table') then
		for i = 1, table.getn(va) - 1 do
		    s = s .. 'OR '
		end
		for ko, vo in ipairs(va) do
		    if (type(vo) ~= 'string') then
			error('filter rule not a string', 2)
		    end
		    s = s .. '(' .. vo .. ') '
		end
	    else
		error('filter element not a string or table', 2)
	    end
	end
    else
	for i = 1, table.getn(criteria) - 1 do
	    s = s .. 'OR '
	end
	for ko, vo in ipairs(criteria) do
	    if (type(vo) == 'string') then
		s = s .. '' .. '(' .. vo .. ')' .. ' '
	    elseif (type(vo) == 'table') then
		s = s .. '('
		for ka, va in ipairs(vo) do
		    if (type(va) ~= 'string') then
			error('filter rule not a string', 2)
		    end
		    s = s .. va .. ' '
		end
		s = string.gsub(s, '(.+) ', '%1')
		s = s .. ') '
	    else
		error('filter rule not a string or table', 2)
	    end
	end
    end

    s = string.gsub(s, '(.+) ', '%1')

    local _, results = ifcore.search(account, s)

    if (type(options) == 'table' and options.close == true) then
	ifcore.close(account)
    end

    if (results == nil) then
	return {}
    end

    local t = {}
    for n in string.gfind(results, '%d+') do
	table.insert(t, tonumber(n))
    end

    return t
end


-- flag()
function flag(account, mbox, mode, flags, messages)
    check_type(account, 'table')
    check_type(mbox, 'string')
    check_type(mode, 'string')
    check_type(flags, 'table')
    check_type(messages, 'table')

    check_type(account.server, 'string')
    check_type(account.username, 'string')
    check_type(account.password, 'string')

    local r = flag_aux(account, mbox, mode, flags, messages)

    if (type(options) == 'table' and options.info == true and
    	messages ~= nil and r == true) then
	print(string.format("%d messages flagged in %s@%s/%s.",
	                    table.getn(messages), account.username,
	                    account.server, mbox))
    end

    return r
end

function flag_aux(account, mbox, mode, flags, messages)
    if (table.getn(messages) == 0) then
	return
    end

    if (mode ~= 'add' and mode ~= 'remove' and mode ~= 'replace') then
	error('"add", "remove" or "replace" expected for mode', 3)
    end


    if (ifcore.login(account) ~= true) then
	return
    end

    if (ifcore.select(account, mbox) ~= true) then
	return
    end

    local f = ''
    if (table.getn(flags) ~= 0) then
	f = '\\' .. table.concat(flags, ' \\')
    end

    local m = range_sequence(messages)
    n = table.getn(m)
    local r = false
    for i = 1, n, 50 do
	j = i + 49
	if (n < j) then
	    j = n
	end
	r = ifcore.store(account, table.concat(m, ',', i, j), mode, f)
	if r == false then
	    break
	end
    end

    if (type(options) == 'table' and options.close == true) then
	ifcore.close(account)
    end

    return r
end


-- copy()
function copy(srcaccount, srcmbox, dstaccount, dstmbox, messages)
    check_type(srcaccount, 'table')
    check_type(srcmbox, 'string')
    check_type(dstaccount, 'table')
    check_type(dstmbox, 'string')
    check_type(messages, 'table')

    check_type(srcaccount.server, 'string')
    check_type(srcaccount.username, 'string')
    check_type(srcaccount.password, 'string')

    check_type(dstaccount.server, 'string')
    check_type(dstaccount.username, 'string')
    check_type(dstaccount.password, 'string')

    local r = copy_aux(srcaccount, srcmbox, dstaccount, dstmbox, messages)

    if (type(options) == 'table' and options.info == true and
    	messages ~= nil and r == true) then
	print(string.format("%d messages copied from %s@%s/%s to %s@%s/%s.",
			    table.getn(messages), srcaccount.username,
			    srcaccount.server, srcmbox, dstaccount.username,
			    dstaccount.server, dstmbox))
    end

    return r
end

function copy_aux(srcaccount, srcmbox, dstaccount, dstmbox, messages)
    if (table.getn(messages) == 0) then
	return
    end

    if (ifcore.login(srcaccount) ~= true) then
	return
    end

    local r = false
    if (srcaccount == dstaccount) then
	if (ifcore.select(srcaccount, srcmbox) ~= true) then
	    return
	end
	
	local m = range_sequence(messages)
	n = table.getn(m)
	for i = 1, n, 50 do
	    j = i + 49
	    if (n < j) then
		j = n
	    end
	    r = ifcore.copy(srcaccount, table.concat(m, ',', i, j), dstmbox)
	    if r == false then
		break
	    end
	end

	if (type(options) == 'table' and options.close == true) then
	    ifcore.close(srcaccount)
	end
    else
	local fast = fetchfast(srcaccount, srcmbox, messages)
	local msgs = fetchtext(srcaccount, srcmbox, messages)

	if (ifcore.login(dstaccount) ~= true) then
	    return
	end

	for i in pairs(fast) do
	    for k, v in ipairs(fast[i]['flags']) do
		if (string.lower(v) == 'recent') then
		    table.remove(fast[i]['flags'], k)
		end
	    end

	    local f = ''
	    if (table.getn(fast[i]['flags']) ~= 0) then
		f = '\\' .. table.concat(fast[i]['flags'], ' \\')
	    end
	    r = ifcore.append(dstaccount, dstmbox, msgs[i], f, fast[i]['date'])
	end
    end

    return r
end


-- delete()
function delete(account, mbox, messages)
    check_type(account, 'table')
    check_type(mbox, 'string')
    check_type(messages, 'table')

    check_type(account.server, 'string')
    check_type(account.username, 'string')
    check_type(account.password, 'string')

    local r = flag_aux(account, mbox, 'add', { 'Deleted' }, messages)

    if (type(options) == 'table' and options.info == true and
    	messages ~= nil and r == true) then
	print(string.format("%d messages deleted in %s@%s/%s.",
			    table.getn(messages), account.username,
			    account.server, mbox))
    end

    return r
end


-- move()
function move(srcaccount, srcmbox, dstaccount, dstmbox, messages)
    check_type(srcaccount, 'table')
    check_type(srcmbox, 'string')
    check_type(dstaccount, 'table')
    check_type(dstmbox, 'string')
    check_type(messages, 'table')

    check_type(srcaccount.server, 'string')
    check_type(srcaccount.username, 'string')
    check_type(srcaccount.password, 'string')

    check_type(dstaccount.server, 'string')
    check_type(dstaccount.username, 'string')
    check_type(dstaccount.password, 'string')

    local rc = copy_aux(srcaccount, srcmbox, dstaccount, dstmbox, messages)
    local rf = false
    if (rc == true) then
	rf = flag_aux(srcaccount, srcmbox, 'add', { 'Deleted' }, messages)
    end

    if (type(options) == 'table' and options.info == true and
    	messages ~= nil and rc == true and rf == true) then
	print(string.format("%d messages moved from %s@%s/%s to %s@%s/%s.",
			    table.getn(messages), srcaccount.username,
			    srcaccount.server, srcmbox, dstaccount.username,
			    dstaccount.server, dstmbox))
    end

    return rc == true and rf == true
end


-- fetchheader()
function fetchheader(account, mbox, messages)
    check_type(account, 'table')
    check_type(mbox, 'string')
    check_type(messages, 'table')

    check_type(account.server, 'string')
    check_type(account.username, 'string')
    check_type(account.password, 'string')

    if (table.getn(messages) == 0) then
	return
    end

    if (ifcore.login(account) ~= true) then
	return
    end

    if (ifcore.select(account, mbox) ~= true) then
	return
    end

    local results = {}
    for i, v in ipairs(messages) do
	local _, header = ifcore.fetchheader(account, tostring(v))

	if (header ~= nil) then
	    results[tonumber(v)] = header
	end
    end

    if (type(options) == 'table' and options.close == true) then
	ifcore.close(account)
    end

    return results
end


-- fetchbody()
function fetchbody(account, mbox, messages)
    check_type(account, 'table')
    check_type(mbox, 'string')
    check_type(messages, 'table')

    check_type(account.server, 'string')
    check_type(account.username, 'string')
    check_type(account.password, 'string')

    if (table.getn(messages) == 0) then
	return
    end

    if (ifcore.login(account) ~= true) then
	return
    end

    if (ifcore.select(account, mbox) ~= true) then
	return
    end

    local results = {}
    for i, v in ipairs(messages) do
	local _, body = ifcore.fetchbody(account, tostring(v))

	if (body ~= nil) then
	    results[tonumber(v)] = body
	end
    end

    if (type(options) == 'table' and options.close == true) then
	ifcore.close(account)
    end

    return results
end


-- fetchtext()
function fetchtext(account, mbox, messages)
    check_type(account, 'table')
    check_type(mbox, 'string')
    check_type(messages, 'table')

    check_type(account.server, 'string')
    check_type(account.username, 'string')
    check_type(account.password, 'string')

    if (table.getn(messages) == 0) then
	return
    end

    if (ifcore.login(account) ~= true) then
	return
    end

    if (ifcore.select(account, mbox) ~= true) then
	return
    end

    local results = {}
    for i, v in ipairs(messages) do
	local _, header = ifcore.fetchheader(account, tostring(v))
	local _, body = ifcore.fetchbody(account, tostring(v))

	if (header ~= nil and body ~= nil) then
	    results[tonumber(v)] = header .. body
	end
    end

    if (type(options) == 'table' and options.close == true) then
	ifcore.close(account)
    end

    return results
end


-- fetchfields()
function fetchfields(account, mbox, fields, messages)
    check_type(account, 'table')
    check_type(mbox, 'string')
    check_type(fields, 'table')
    check_type(messages, 'table')

    check_type(account.server, 'string')
    check_type(account.username, 'string')
    check_type(account.password, 'string')

    if (table.getn(messages) == 0) then
	return
    end

    if (ifcore.login(account) ~= true) then
	return
    end

    if (ifcore.select(account, mbox) ~= true) then
	return
    end

    local results = {}
    for i, v in ipairs(messages) do
	local _, headerfields = ifcore.fetchfields(account, tostring(v),
					       table.concat(fields, ' '))

	if (headerfields ~= nil) then
	    results[tonumber(v)] = headerfields
	end
    end

    if (type(options) == 'table' and options.close == true) then
	ifcore.close(account)
    end

    return results
end

fetchheaders = fetchfields


-- fetchfast()
function fetchfast(account, mbox, messages)
    check_type(account, 'table')
    check_type(mbox, 'string')
    check_type(messages, 'table')

    check_type(account.server, 'string')
    check_type(account.username, 'string')
    check_type(account.password, 'string')

    if (table.getn(messages) == 0) then
	return
    end

    if (ifcore.login(account) ~= true) then
	return
    end

    if (ifcore.select(account, mbox) ~= true) then
	return
    end

    local results = {}
    for i, v in ipairs(messages) do
	local _, flags, date, size = ifcore.fetchfast(account, tostring(v))
	if (flags ~= nil and date ~= nil and size ~= nil ) then
	    local f = {}
	    for s in string.gfind(flags, '%w+') do
		table.insert(f, s)
	    end
	    results[tonumber(v)] = {}
	    results[tonumber(v)]['flags'] = f
	    results[tonumber(v)]['date'] = date
	    results[tonumber(v)]['size'] = size
	end
    end

    if (type(options) == 'table' and options.close == true) then
	ifcore.close(account)
    end

    return results
end


-- list()
function list(account, name)
    check_type(account, 'table')

    check_type(account.server, 'string')
    check_type(account.username, 'string')
    check_type(account.password, 'string')

    if (name == nil) then
	name = ''
    else
	check_type(name, 'string')
	if (type(options) == 'table' and options.namespace == true) then
	    if (name == '/') then
		name = ''
	    end
	    if (name ~= '') then
		name = name .. '/'
	    end
	end
    end

    if (ifcore.login(account) ~= true) then
	return
    end

    local _, mboxs, folders = ifcore.list(account, '', name .. '%')
    
    local m = {}
    for s in string.gfind(mboxs, '[%w%p]+') do
	table.insert(m, s)
    end

    local f = {}
    for s in string.gfind(folders, '[%w%p]+') do
	table.insert(f, s)
    end

    return m, f
end


-- lsub()
function lsub(account, name)
    check_type(account, 'table')

    check_type(account.server, 'string')
    check_type(account.username, 'string')
    check_type(account.password, 'string')

    if (name == nil) then
	name = ''
    else
	check_type(name, 'string')
	if (type(options) == 'table' and options.namespace == true) then
	    if (name == '/') then
		name = ''
	    end
	    if (name ~= '') then
		name = name .. '/'
	    end
	end
    end

    if (ifcore.login(account) ~= true) then
	return
    end

    local _, mboxs, folders = ifcore.lsub(account, '', name .. '%')
    
    local m = {}
    for s in string.gfind(mboxs, '[%w%p]+') do
	table.insert(m, s)
    end

    local f = {}
    for s in string.gfind(folders, '[%w%p]+') do
	table.insert(f, s)
    end

    return m, f
end


-- ping()
function ping(account)
    check_type(account, 'table')
    
    check_type(account.server, 'string')
    check_type(account.username, 'string')
    check_type(account.password, 'string')

    if (ifcore.login(account) ~= true) then
	return
    end

    local r = ifcore.noop(account)

    return r
end


-- other functions
function check_type(arg, argtype)
    if (type(arg) ~= argtype) then
	error(string.format('%s expected, got %s', argtype, type(arg)), 3)
    end
end


function range_sequence(messages)
    table.sort(messages)

    local t = {}
    local a, z
    for _, m in ipairs(messages) do
	if a == nil or z == nil then
	    a = m
	    z = m
	else
	    if m == z + 1 then
		z = m
	    else
		if a == z then
		    table.insert(t, tostring(a))
		else
		    table.insert(t, a .. ':' .. z)
		end
		a = m
		z = m
	    end
	end
    end

    if a == z then
	table.insert(t, tostring(a))
    else
	table.insert(t, a .. ':' .. z)
    end

    return t
end
