--
-- This file contains examples on how IMAPFilter can be extended using the Lua
-- programming language.
--


-- When filtering based on a message's date, the argument must be of the
-- strftime(3) form "%d-%b-%Y".  Sometimes it is useful to generate the date
-- argument, based on the system's current date (a kind of "older than" or
-- "newer than" functionality).
--
-- The auxiliary function date_before() is supplied for conveniency.  For
-- example the following filter matches messages newer than 30 days:

myfilter = {
    'since ' .. date_before(30),
}


-- Passwords could be supplied during execution time.
--
-- The auxiliary function get_pass() is supplied for conveniency.  The
-- following example displays a prompt and reads the password from the user:

myaccount = {
    server = 'imap.mail.server',
    username = 'me',
    password = get_pass('Enter password for me@imap.mail.server: ')
}


-- One can list all the available mailboxes residing on a server recursively.
--
-- This can be accomplished by use of the list() and lsub() commands.  An
-- example function that traverses the folder tree, finds the available
-- mailboxes, inserts them in a table and prints the result, is demonstrated:

mailboxes = {}
function list_all(account, path)
    mboxes, folders = list(account, path)
    for _, mbox in ipairs(mboxes) do
        table.insert(mailboxes, mbox)
    end
    for _, folder in ipairs(folders) do
	if (folder ~= path .. '/') then
	    list_all(account, folder)
	end
    end
end

list_all(myaccount, '')
print(unpack(mailboxes))


-- IMAPFilter can be detached from the controlling terminal and run in the
-- background as a system daemon.
--
-- The auxiliary function daemon_mode() is supplied for conveniency.  The
-- following example puts imapfilter in the background and runs endlessly,
-- executing the commands in the forever() function and sleeping for 600
-- seconds between intervals:

function forever()
    results = match(myaccount, 'mymailbox', myfilter)
    delete(myaccount, 'mymailbox', results)
end

daemon_mode(600, forever)


-- On certain occasions, the IMAP search mechanism is too restrictive; it
-- matches strings ignoring case distinction and does not understand regular
-- expressions.  In some other cases, the mail server administrator may have
-- restricted or disabled the IMAP search functionality (although this is an
-- IMAP protocol violation).
--
-- Therefore, it may be a better (or the only) solution, to download the
-- messages (or parts of them) and perform the searching locally.  Thus, using
-- the fetch*() family of functions, the header, body, text (header and body)
-- or specific header lines of a message can be downloaded and then processed
-- using Lua's builtin string functions.
--
-- The following example matches messages based on the "From" and "Subject"
-- headers, inserts their sequence number in a table and requests from the
-- server to delete them:

results = match(myaccount, 'mymailbox', {})
fields = fetchfields(myaccount, 'mymailbox', { 'from', 'subject' }, results)

results = {}
for msgid, msghdrs in pairs(fields) do
    if (string.find(msghdrs, 'From: .*user1@domain1') or
	string.find(msghdrs, 'From: .*user2@domain2') and not
	string.find(msghdrs, 'Subject: [imapfilter-devel] .*')) then
	table.insert(results, msgid)
    end
end

delete(myaccount, 'mymailbox', results)


-- IMAPFilter can take advantage of all those filtering utilities that are
-- available and use a wide range of heuristic tests, text analysis,
-- internet-based realtime blacklists, advanced learning algorithms, etc. to
-- classify mail.  IMAPFilter can pipe a message to a program and act on the
-- message based on the program's exit status.
--
-- The auxiliary function pipe_to() is supplied for conveniency.  For example
-- if there was a utility named "bayesian-spam-filter", which returned 1 when
-- it considered the message "spam" and 0 otherwise:
 
results = match(myaccount, 'mymailbox', {})
text = fetchtext(myaccount, 'mymailbox', results)

results = {}
for msgid, msgtxt in pairs(text) do
    if (pipe_to('bayesian-spam-filter', msgtxt) == 1) then
	table.insert(results, msgid)
    end
end

delete(myaccount, 'mymailbox', results)


-- Passwords could be extracted during execution time from an encrypted file.
--
-- The file is encrypted using the openssl(1) command line tool.  For example
-- the "passwords.txt" file:
--
--   secret1
--   secret2
--
-- ... is encrypted and saved to a file named "passwords.enc" with the command:
--
--   $ openssl bf -salt -in passwords.txt -out passwords.enc
--
-- The auxiliary function pipe_from() is supplied for conveniency.  The user is
-- prompted to enter the decryption password, the file is decrypted and the
-- account passwords are set accordingly:
 
account1 = {
    server = 'imap1.mail.server',
    username = 'user1',
}

account2 = {
    server = 'imap2.mail.server',
    username = 'user2',
}

status, output = pipe_from('openssl bf -d -salt -in ~/passwords.enc')

_, _, account1.password, account2.password = string.find(output,
							 '(%w+)\n(%w+)\n')
