/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 * 
 * Released under the GPL v2. */

#include "imspector.h"

#include <sqlite3.h>

#define PLUGIN_NAME "DB responder plugin"
#define PLUGIN_SHORT_NAME "DB"

#define SQLITE_SOCKET "/tmp/.imspectorrespondersqlite"

#define CREATE_TABLE "CREATE TABLE IF NOT EXISTS responder ( " \
	"id integer PRIMARY KEY AUTOINCREMENT, " \
	"protocolname text, " \
	"userid text, " \
	"type integer NOT NULL, " \
	"timestamp integer NOT NULL );" \

#define TEST_STATEMENT "SELECT COUNT(*) FROM responder WHERE " \
	"protocolname=? AND userid=? AND type=? AND timestamp>?"

#define CLEAR_STATEMENT "DELETE FROM responder WHERE " \
	"protocolname=? AND userid=? AND type=?"
	
#define ADD_STATEMENT "INSERT INTO responder " \
	"(id, protocolname, userid, type, timestamp) " \
	"VALUES (NULL, ?, ?, ?, ?)"
	
#define TYPE_NOTICE 1
#define TYPE_FILTERED 2

/* This is a handy struct to pass about. */
struct dbinfo
{
	sqlite3 *db;
	sqlite3_stmt *teststatement;
	sqlite3_stmt *clearstatement;
	sqlite3_stmt *addstatement;
};

extern "C"
{
	bool initresponderplugin(struct responderplugininfo &presponderplugininfo,
		class Options &options, bool debugmode);
	void closeresponderplugin(void);
	int generateresponses(std::vector<struct imevent> &imevents,
		std::vector<struct response> &responses);
};

bool initdb(struct dbinfo &dbinfo, std::string filename);
int checkandadd(std::string protocol, std::string userid, int type, int timestamp);
int dbclient(std::string commandline);
bool dbserver(struct dbinfo &dbinfo, std::string filename);
int processcommand(struct dbinfo &dbinfo, std::string command, std::vector<std::string> args, int argc);
int bindstatement(sqlite3_stmt *statement, std::string protocolname, std::string userid,
	int type, int timestamp);

std::string noticeresponse;
int noticedays = 0;
std::string filteredresponse;
int filteredmins = 0;
bool localdebugmode = false;

bool initresponderplugin(struct responderplugininfo &responderplugininfo,
	class Options &options, bool debugmode)
{
	/* This is the SQL file. */
	std::string filename = options["responder_filename"];
	if (filename.empty()) return false;
	
	/* The time (days) that must pass between sending notices. 0 to disable. */
	std::string noticedaysstring = options["notice_days"];
	if (!noticedaysstring.empty())
		noticedays = atol(noticedaysstring.c_str());
	
	noticeresponse = options["notice_response"];
	if (noticeresponse.empty()) noticeresponse = "Your activities are being logged";

	/* The time (mins) that must pass between sending filtered hints. 0 to disable. */
	std::string filteredminsstring = options["filtered_mins"];
	if (!filteredminsstring.empty())
		filteredmins = atol(filteredminsstring.c_str());
	
	filteredresponse = options["filtered_response"];
	if (filteredresponse.empty()) filteredresponse = "The message or action was blocked";

	/* If neither are wanted, then disable the plugin completely. */
	if (!noticedays && !filteredmins) return false;

	syslog(LOG_INFO, PLUGIN_SHORT_NAME ": Notice every %d days; Filtered every %d mins",
		noticedays, filteredmins);

	localdebugmode = debugmode;

	responderplugininfo.pluginname = PLUGIN_NAME;
	
	struct dbinfo dbinfo;
	
	if (!initdb(dbinfo, filename)) return false;

	/* Fork off the  server process. */
	switch (fork())
	{
		/* An error occured. */
		case -1:
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Error: Fork failed: %s", strerror(errno));
			return false;
		
		/* In the child. */
		case 0:
			dbserver(dbinfo, filename);
			debugprint(localdebugmode,  PLUGIN_SHORT_NAME ": Error: We should not come here");
			exit(0);
	
		/* In the parent. */
		default:
			break;
	}
	
	return true;
}

void closeresponderplugin(void)
{
	return;
}

/* The main plugin function. See responderplugin.cpp. */
int generateresponses(std::vector<struct imevent> &imevents,
	std::vector<struct response> &responses)
{
	for (std::vector<struct imevent>::iterator i = imevents.begin();
		i != imevents.end(); i++)
	{
		/* Determine the user ID. This will be the originator of the event. */
		std::string userid;
	
		if ((*i).outgoing)
			userid = (*i).localid;
		else
			userid = (*i).remoteid;
		
		/* See if we are interested in generating "warning notices". */
		if (noticedays)
		{
			if (checkandadd((*i).protocolname, userid, TYPE_NOTICE,
				(int) time(NULL) - noticedays * 24 * 3600) > 0)
			{
				struct response response;
				
				response.outgoing = !(*i).outgoing;	
				response.text = noticeresponse;
			
				responses.push_back(response);
			}
		}
		
		/* Same idea for filtered events, but minutes instead of days. This is
		 * to stop a flood of "your message was blocked" messages, and also
		 * nicely cures MSNs file transfer requests, which will stupidly 
		 * retry. */
		if (filteredmins && (*i).filtered)
		{
			if (checkandadd((*i).protocolname, userid, TYPE_FILTERED,
				(int) time(NULL) - filteredmins * 60) > 0)
			{
				struct response response;
				
				response.outgoing = !(*i).outgoing;	
				response.text = filteredresponse;
				if (!(*i).eventdata.empty())
					response.text += " (" + (*i).eventdata + ")";
			
				responses.push_back(response);
			}
		}
	}

	return 0;
}

/* Simple stuff: inits SQLite and makes the statements etc. */
bool initdb(struct dbinfo &dbinfo, std::string filename)
{
	int rc = sqlite3_open(filename.c_str(), &dbinfo.db);
	if (rc != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't open DB, Error: %s", sqlite3_errmsg(dbinfo.db));
		return false;
	}

	rc = sqlite3_exec(dbinfo.db, CREATE_TABLE, NULL, NULL, NULL);
	if (rc != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't create table, Error: %s", sqlite3_errmsg(dbinfo.db));
		return false;
	}
		
	rc = sqlite3_prepare(dbinfo.db, TEST_STATEMENT, -1, &dbinfo.teststatement, 0);	
	if (rc != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": sqlite3_preapre() TEST_STATEMENT, Error: %s", sqlite3_errmsg(dbinfo.db));
		return false;
	}

	rc = sqlite3_prepare(dbinfo.db, CLEAR_STATEMENT, -1, &dbinfo.clearstatement, 0);	
	if (rc != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": sqlite3_preapre() CLEAR_STATEMENT, Error: %s", sqlite3_errmsg(dbinfo.db));
		return false;
	}

	rc = sqlite3_prepare(dbinfo.db, ADD_STATEMENT, -1, &dbinfo.addstatement, 0);	
	if (rc != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": sqlite3_preapre() ADD_STATEMENT, Error: %s", sqlite3_errmsg(dbinfo.db));
		return false;
	}
	
	return true;
}

/* Wrapper for the only client function. */
int checkandadd(std::string protocol, std::string userid, int type, int timestamp)
{
	return (dbclient(stringprintf("CHECK_AND_ADD %s %s %d %d\n",
		protocol.c_str(), userid.c_str(), type, timestamp)));
}

/* Client for the DB. Returns whatever the DB server gives it, which will always
 * be a number or -1 for an error. */
int dbclient(std::string commandline)
{
	class Socket sqlsock(AF_UNIX, SOCK_STREAM);
	
	/* Complete the connection. */
	if (!(sqlsock.connectsocket(SQLITE_SOCKET, ""))) return -1;
	
	/* Add on a CR as the server needs these for end of line. */
	std::string commandlinecr = commandline + "\n";
	
	if (!sqlsock.sendalldata(commandlinecr.c_str(), commandlinecr.length())) return -1;
	
	char buffer[BUFFER_SIZE];
	
	memset(buffer, 0, BUFFER_SIZE);
	
	if (sqlsock.recvline(buffer, BUFFER_SIZE) < 0)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't get command line from SQL client");
		return -1;
	}
		
	stripnewline(buffer);
	
	sqlsock.closesocket();
	
	return (atol(buffer));
}

bool dbserver(struct dbinfo &dbinfo, std::string filename)
{
	class Socket sqlsock(AF_UNIX, SOCK_STREAM);
	
	if (!sqlsock.listensocket(SQLITE_SOCKET))
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Error: Couldn't bind to SQL socket");
		return false;
	}

	/* This loop has no exit, except when the parent kills it off. */
	while (true)
	{
		std::string clientaddress;
		class Socket clientsock(AF_UNIX, SOCK_STREAM);
		char buffer[BUFFER_SIZE];
		
		if (!sqlsock.awaitconnection(clientsock, clientaddress)) continue;

		memset(buffer, 0, BUFFER_SIZE);
		if (clientsock.recvline(buffer, BUFFER_SIZE) < 0)
		{
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't get command line from SQL client");
			continue;
		}
		
		stripnewline(buffer);
		
		std::string command;
		std::vector<std::string> args;
		int argc;
		
		chopline(buffer, command, args, argc);
				
		int result = processcommand(dbinfo, command, args, argc);
		std::string resultstring = stringprintf("%d\n", result);
				
		if (clientsock.sendline(resultstring.c_str(), resultstring.length()) < 0)
		{
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Couldn't send result to SQL client");
			continue;
		}

		clientsock.closesocket();
	}
	
	return true;
}

int processcommand(struct dbinfo &dbinfo, std::string command, std::vector<std::string> args, int argc)
{
	/* We could add more commands but for now there is just one. */
	if (command != "CHECK_AND_ADD")
		return -1;
	if (argc < 4)
		return -1;
	
	/* Grab the arguments. */
	std::string protocolname = args[0];
	std::string userid = args[1];
	int type = atol(args[2].c_str());
	int timestamp = atol(args[3].c_str());
	
	/* See if we have a matching row. */
	sqlite3_stmt *statement = dbinfo.teststatement;
	
	if (bindstatement(statement, protocolname, userid, type, timestamp) < 0)
		return -1;
	
	int result = 0;
	
	if (sqlite3_step(statement) == SQLITE_ROW) 
		result = sqlite3_column_int(statement, 0);
	
	sqlite3_reset(statement);
	
	/* If we have a row already, then ok. This means no automated message is needed.*/
	if (result)
		return 0;
	
	/* Otherwise, we first remove... */
	statement = dbinfo.clearstatement;
	
	if (bindstatement(statement, protocolname, userid, type, 0) < 0)
		return -1;
	
	while (sqlite3_step(statement) == SQLITE_ROW); /* Empty loop!*/

	sqlite3_reset(statement);
	
	/* Then add a new row, timestamped to "now". */
	statement = dbinfo.addstatement;
	
	if (bindstatement(statement, protocolname, userid, type, (int) time(NULL)) < 0)
		return -1;

	while (sqlite3_step(statement) == SQLITE_ROW); /* Empty loop!*/

	sqlite3_reset(statement);
	
	/* Caller (client) will make a message for us. */
	return 1;	
}

/* Binds each column in the query.  Final arg (timestamp) is optional, because the
 * delete query dosn't use it. */
int bindstatement(sqlite3_stmt *statement, std::string protocolname, std::string userid,
	int type, int timestamp)
{
	if (sqlite3_bind_text(statement, 1, protocolname.c_str(), -1, SQLITE_STATIC) != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Unable to bind protocolname");
		return -1;
	}
	if (sqlite3_bind_text(statement, 2, userid.c_str(), -1, SQLITE_STATIC) != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Unable to bind userid");
		return -1;
	}
	if (sqlite3_bind_int(statement, 3, type) != SQLITE_OK)
	{
		syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Unable to bind type");
		return -1;
	}

	if (timestamp)
	{
		if (sqlite3_bind_int(statement, 4, timestamp) != SQLITE_OK)
		{
			syslog(LOG_ERR, PLUGIN_SHORT_NAME ": Unable to bind timestamp");
			return -1;
		}
	}
	
	return 0;
}
